/* Samba Server for eCS (OS/2) init script Version 3.0.2
   Copyright (C) netlabs.org 2007-2009

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  

   Description: Starts and stops the Samba smbd and nmbd 
                daemons used to provide CIFS/SMB network services.
                Run without parameter in order to display usage!
*/

call _LoadOtherFuncs

call _InitTempdir

call _SambaInit

call _SambaGetProperties

call _SambaExtendSearchPath

OldDir = directory()

call directory samba.!bin

/* Check if winbindd daemon exists */
HaveWinbindd = (stream(samba.!bin'\winbindd.exe', 'c', 'query exists') <> "")

/* get the commandline arguments */
arg UArg Service

/* Samba 3.3.x already supported! */
if Service = "ALL" then Service = "" 

if \(Service = "") & \(Service = "SMBD") & \(Service = "NMBD") & \(Service = "WINBINDD") then do
    say "Invalid Service - assuming ALL"
    Service = ""
end

select 
    when UArg = 'START' then do
        call _SambaStartDaemons Service
    end
    when UArg = 'STOP' then do
        call _SambaStopDaemons Service
    end
    when UArg = 'STATUS' then do
        if \_SambaRunning() then do
            say 'Samba services are not running. Aborting...'
            ok = directory(OldDir)
            exit -3
        end
        samba.!bin'\smbstatus.exe smbd 2>NUL'
    end
    when UArg = 'RESTART' then do
        say 'Restarting SMB services: '
        call _SambaStopDaemons Service
        call _SambaStartDaemons Service
    end
    when UArg = 'RELOAD' then do
        say 'Reloading smb.conf.'
        samba.!bin'\smbcontrol.exe smbd reload-config 1>NUL'
        say 'Done.'
    end
    when UArg = 'TEST' then do
        say 'Testing 'samba.!smb.conf
        samba.!bin'\testparm.exe'
        say 'Done.'
    end
    
    otherwise do /* display usage info */
        say 'Samba Server for eCS (OS/2) init script'
        say
        say '  Starts, stops, and controls the Samba services used to'
        say '  provide SMB/CIFS (LAN Server style) network services.'
        say 
        say '  Usage: smb {START | STOP | RESTART | RELOAD | STATUS | TEST }'
        say '   - START    start Samba services'
        say '   - STOP     stop Samba services'
        say '   - RESTART  stop Samba then start it again'
        say '   - RELOAD   put changes to smb.conf into effect'
        say '   - STATUS   show connections & shares in use'
        say '   - TEST     validate smb.conf'
        say
        call charout , '  Samba services are currently '
        if \_SambaRunning() then say 'not running!'; else say 'running!'
        say
        say '  Samba binaries:   'samba.!bin
        say '  Samba tools:      'samba.!tools
        say '  Samba config:     'samba.!smbconf
        say '  Samba logfiles:   'samba.!log_path
        say '  Samba lock files: 'samba.!lock_dir
        say '  Samba debuglevel: 'samba.!debuglevel
        
        ok = directory(OldDir)
        exit 1
    end
end
ok = directory(OldDir)

exit 0

_LoadOtherFuncs:
    '@echo off'

    call RxFuncAdd 'SysLoadFuncs', 'RexxUtil','SysLoadFuncs'
    call SysLoadFuncs

    call RxFuncAdd 'VRLoadFuncs',  'VROBJ',   'VRLoadFuncs'
    call VRLoadFuncs

    call RxFuncAdd 'PRLoadFuncs',  'PR1UTIL', 'PRLoadFuncs'
    call PRLoadFuncs

return

_InitTempDir: /* make sure we have a valid temporary directory */
    /* Get temporary directory */
    HaveNoTMPDIR  = 0
    TempDir       = translate(value('TMPDIR',,'OS2ENVIRONMENT'),'\','/')
    if TempDir = '' then do
        HaveNoTMPDIR  = 1
        TempDir = translate(value('TEMP',,'OS2ENVIRONMENT'),'\','/')
    end
    if TempDir = '' then TempDir = translate(value('TMP',,'OS2ENVIRONMENT'),'\','/')
    if TempDir = '' then TempDir = directory()
    if HaveNoTMPDIR then do
        say 'Setting missing TMPDIR   variable to "'TempDir'".'
        ok = value('TMPDIR',TempDir,'OS2ENVIRONMENT')
    end    
    TempDir       = strip(TempDir,'T','\')||'\' /* make sure there is trailing "\" */
return

_SambaInit: /* Detect Samba components, initialize samba. stem */
    ETC = value('ETC',,'OS2ENVIRONMENT')
    if ETC = "" then do
        say 'ETC environment variable is missing. Aborting...'
        exit -1
    end
    UnixRoot = value("UNIXROOT",,'OS2ENVIRONMENT')
    if UnixRoot = "" then do
        UnixRoot = left(ETC,length(ETC)-4)
        say 'Setting missing UNIXROOT variable to "'UnixRoot'".'
        ok = value("UNIXROOT",UnixRoot,'OS2ENVIRONMENT')
    end
    UnixETC  = UnixRoot'\etc'

    /* Find out various directories */
    parse source . . smbcmd
    samba.=''
    samba.!tools = strip(filespec('D',smbcmd)||filespec('P',smbcmd),,'\')

    /* smbd.exe */
    samba.!smbd = ""
    if samba.!smbd = "" then do /* in current directory? */
        ok = SysFileTree('.\smbd.exe', exist.,'FO')
        if exist.0 = 1 then samba.!smbd = exist.1
    end
    if samba.!smbd = "" then do /* in parent directory? */
        ok = SysFileTree('..\smbd.exe', exist.,'FO')
        if exist.0 = 1 then samba.!smbd = exist.1
    end
    if samba.!smbd = "" then do /* in ..\BIN directory? */
        ok = SysFileTree('..\bin\smbd.exe', exist.,'FO')
        if exist.0 = 1 then samba.!smbd = exist.1
    end
    if samba.!smbd = "" then do /* in SMB_EXE ? */
        samba.!smbd = SysSearchPath("SMB_EXE","SMBD.EXE")
    end
    if samba.!smbd = "" then do /* in PATH ? */
        samba.!smbd = SysSearchPath("PATH","SMBD.EXE")
    end

    if samba.!smbd = "" then do
        say "Cannot find smbd.exe - aborting..."
        exit -1
    end

    samba.!bin = VRParseFileName(samba.!smbd,'DP')

    if samba.!bin = "" then do
        say "Cannot find smbd.exe - aborting..."
        exit -1
    end
    
    /* Check if smb.conf exists. */
    samba.!smbconf = stream(ETC'\samba\smb.conf','c','query exists')
    if samba.!smbconf = "" then do
        say ETC'\samba\smb.conf is missing. Aborting...'
        exit -2
    end
return

_SambaRunning: procedure expose running. pids. TempDir HaveWinbindd
    /* Checks whether any Samba daemon is currently running
       
       Parameters: none or "ALL" 
       
       Returns: 1 = at least one Samba daemon is running
                1 = Parameter "ALL" all Samba daemons are running
                0 = no Samba daemon is running
                0 = Parameter "ALL" not all Samba daemons are running
                
       Stems: running. stores which daemon is running
              pids.    stores pids of running daemons
     */
    restype = arg(1)
    running. = 0
    pids. = ""

    ok = PRProcessList(proc)

    do I = 1 to proc.0
        CurProc = VRParseFileName(proc.i.name,'NE')
        if CurProc = "SMBD.EXE" then do
            running.!smbd = 1
            pids.!smbd = pids.!smbd||proc.i.pid||' '
        end
        if CurProc = "NMBD.EXE" then do
            running.!nmbd = 1
            pids.!nmbd = pids.!nmbd||proc.i.pid||' '
        end
        if CurProc = "WINBINDD.EXE" then do
            running.!winb = 1
            pids.!winb = pids.!winb||proc.i.pid||' '
        end
    end
    
    if restype = "ALL" then do
        runres = (running.!smbd & running.!nmbd & (running.!winb | \HaveWinbindd ))
    end 
    else do
        runres = (running.!smbd | running.!nmbd |running.!winb)
    end
return runres

_SambaGetProperties: procedure expose samba.
    /* Initialize stems from smb.conf */
    call LoadSmbConf
	/* Read some values from smb.conf into samba. stem */
    samba.!log_file   = translate(IniGet("log file", "global", samba.!smbconf),'\','/')
    samba.!log_path   = strip(filespec("D",samba.!log_file)||filespec("P",samba.!log_file),,'\')
    samba.!lock_dir   = translate(IniGet("lock directory", "global", samba.!smbconf),'\','/')
    samba.!debuglevel = IniGet("log level", "global", samba.!smbconf)
    
    ok = value("SMB_LOGS",samba.!log_path,"OS2ENVIRONMENT")
    ok = value("SMB_LOCK",samba.!lock_dir,"OS2ENVIRONMENT")
    
    address cmd samba.!bin'\smbd.exe -V >smb.ver'
    samba.!version = linein('smb.ver')
    ok = stream('smb.ver','c','close')
    ok = SysFileDelete('smb.ver')
return

_SambaStopDaemons:
    say 'Shutting down SMB services'
    svc = arg(1)

    Red='1B'x||'[31;m'
    Green='1B'x||'[32;m'
    Normal='1B'x||'[0m'
    
    ok = _SambaRunning()

    if running.!smbd = 1 & (svc = "SMBD" | svc = "") then do
        'smbcontrol smbd shutdown >' samba.!log_path'\log.smbd.shtdn 2>&1'
        ok = SysFileDelete(ETC'\samba\pid\smbd.pid')
        ok = SysSleep(0.1)
        ok = _SambaRunning()
        if words(pids.!smbd) > 0 then do I = 1 to words(pids.!smbd)
            call charout ,"Trying to kill smbd.exe PID "word(pids.!smbd,I)
            ok = PRKillProcess("1", word(pids.!smbd,I))
            if ok = 0 then say ", success!"; else say ", error "ok
        end
    end

    if running.!nmbd = 1 & (svc = "NMBD" | svc = "") then do
        'smbcontrol nmbd shutdown >' samba.!log_path'\log.nmbd.shtdn 2>&1'
        ok = SysFileDelete(ETC'\samba\pid\nmbd.pid')
        ok = SysSleep(0.1)
        ok = _SambaRunning()
        if words(pids.!nmbd) > 0 then do I = 1 to words(pids.!nmbd)
            call charout ,"Trying to kill nmbd.exe "word(pids.!nmbd,I)
            ok = PRKillProcess("1", word(pids.!nmbd,I))
            if ok = 0 then say ", success!"; else say ", error "ok
        end
    end

    if running.!winb = 1 & (svc = "WINBINDD" | svc = "") then do
        'smbcontrol winbindd shutdown >' samba.!log_path'\log.winbindd.shtdn 2>&1'
        ok = SysFileDelete(ETC'\samba\pid\winbindd.pid')
        ok = SysSleep(0.1)
        ok = _SambaRunning()
        if words(pids.!winb) > 0 then do I = 1 to words(pids.!winb)
            call charout ,"Trying to kill winbindd "word(pids.!winb,I)
            ok = PRKillProcess("1", word(pids.!winb,I))
            if ok = 0 then say ", success!"; else say ", error "ok
        end
    end
    ok = SysSleep(0.2)
    if \_SambaRunning() then say Green||"Samba "samba.!version" services stopped successfully."||Normal
    else do
        if running.!smbd = 1 then say Red||"Warning! An instance of smbd.exe is still running."||Normal
        if running.!nmbd = 1 then say Red||"Warning! An instance of nmbd.exe is still running."||Normal
        if running.!winb = 1 then say Red||"Warning! An instance of winbindd.exe is still running."||Normal
    end

    /* Plugin script to be executed after Samba is stopped */
    smbexit = stream(samba.!tools'\smbexit.cmd', 'c', 'query exists')
    if smbexit <> '' & svc = "" then do
        say 'Finishing shutdown using smbexit.cmd'
        interpret '"call 'smbexit'";'
        drop smbexit
    end
    say 'Done.'
return

_SambaStartDaemons:
    svc = arg(1)

    Red='1B'x||'[31;m'
    Green='1B'x||'[32;m'
    Normal='1B'x||'[0m'

    if svc = "" then do
        if _SambaRunning() then do
            say "Samba services are already running."
            return
        end
    end
    else do
        ok = _SambaRunning()
        if running.!smbd = 1 & svc = "SMBD" then do
            say Red||"Warning! smbd.exe already running!"||Normal
            return
        end
        if running.!nmbd = 1 & svc = "NMBD" then do
            say Red||"Warning! nmbd.exe already running!"||Normal
            return
        end
        if running.!winb = 1 & svc = "WINBINDD" then do
            say Red||"Warning! winbindd.exe already running!"||Normal
            return
        end
    end
    /* Plugin script to be executed before Samba is started */
    b4smb = stream(samba.!tools'\b4smb.cmd', 'c', 'query exists')
    if b4smb <> '' & svc = "" then do
        say 'Configuring SMB services using b4smb.cmd'
        interpret '"call 'b4smb'";'
        drop b4smb
    end

    say 'Starting SMB services'

    if running.!smbd = 0 & (svc = "SMBD" | svc = "") then do
        /* Just to be sure - this pid file should not be there anyway! */
        ok = SysFileDelete(ETC'\samba\pid\smbd.pid')
        ok = charout(, '  SMBD:     ')
        'detach smbd'
    end
    if running.!nmbd = 0 & (svc = "NMBD" | svc = "") then do
        /* Just to be sure - this pid file should not be there anyway! */    
        ok = SysFileDelete(ETC'\samba\pid\nmbd.pid')
        ok = charout(, '  NMBD:     ')
        'detach nmbd -l 'samba.!log_path' -d 'samba.!debugLevel
    end
    if haveWinbindd & running.!winb = 0 & (svc = "WINBINDD" | svc = "") then do
        /* Just to be sure - this pid file should not be there anyway! */    
        ok = SysFileDelete(ETC'\samba\pid\winbindd.pid')
        ok = charout(, '  WINBINDD: ')
        'detach winbindd -l 'samba.!log_path' -d 'samba.!debugLevel
    end
    ok = SysSleep(0.3)
    if svc = "" then do
        if _SambaRunning("ALL") then say Green||"Samba "samba.!version" services started successfully."||Normal
        else do
            if running.!smbd = 0 then say Red||"Warning! smbd.exe apparently stopped."||Normal
            if running.!nmbd = 0 then say Red||"Warning! nmbd.exe apparently stopped."||Normal
         /* if running.!winb = 0 then say Red||"Warning! winbindd.exe apparently stopped."||Normal */
        end
    end
    say 'Done.'
return

_SambaExtendSearchPath:
    /* Add binary and tools path to the PATH variable */
    old_path = value('PATH',, 'OS2ENVIRONMENT')
    if pos(translate(samba.!bin';'),translate(old_path)) = 0 then do
        if samba.!bin = samba.!tools then new_path = samba.!bin';'
        else new_path = samba.!bin';'samba.!tools';'
        ok = value('PATH', new_path || old_path, 'OS2ENVIRONMENT')
        drop new_path
    end
    drop old_path
    
    /* Add binary and tools path to the BEGINLIBPATH variable */
    old_beginlibpath = SysQueryEXtLibPath("B")
    if pos(translate(samba.!bin';'),translate(old_beginlibpath)) = 0 then do
        if samba.!bin = samba.!tools then new_beginlibpath = samba.!bin';'old_beginlibpath
        else new_beginlibpath = samba.!bin';'samba.!tools';'old_beginlibpath
        ok = SysSetExtLibPath( new_beginlibpath, "B")
        drop new_beginlibpath
    end
    drop old_beginlibpath
return

LoadSmbConf: /* Load smb.conf into stems, we do this to replace RexxINI DLL
                with own routines and minimal impact on existing code */

    sections.0 = 0 /* This stem holds all sections of smb.conf */
    s = 0
    p = 0

    do until lines(samba.!smbconf) = 0
        smbline = strip(translate(linein(samba.!smbconf),' ','09'x))
        if length(smbline) = 0 then iterate
        if left(smbline,1) = ';'|left(smbline,1) = '#' then iterate
        if left(smbline,1) = '[' then do /* new section */
            parse var smbline '['sname']'
            s = s + 1
            sections.0 = s
            sections.s = sname
            p = 0
        end
        else do /* new parm */
            parse var smbline parm'='pvalue
            parm = translate(strip(parm),'_',' ')
            pvalue = strip(pvalue)
            p = p + 1
            if pos('"',pvalue) > 0 
                then istr = sname"."parm"='"pvalue"'"
                else istr = sname'.'parm'="'pvalue'"'
            interpret istr
            istr = sections.s'.'p'="'parm'"'
            interpret istr
            istr = sections.s'.0='p
            interpret istr
        end
    end
    ok = stream(samba.!smbconf,'c','close')
return 0

IniGet: /* Replacement IniGet routine to access the value a known parameter */
    iparm = translate(arg(1),'_',' ')
    isection = arg(2)
    /* Third parameter is ignored, as the file is not required here */
    istr = 'ivalue = 'isection'.'iparm
    interpret istr
    /* Deal with empty/non existing parameters */
    if ivalue = translate(isection'.'iparm) then ivalue = ''
return ivalue    
