/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include <private/qcursor_p.h>
#include <qbitmap.h>
#include <qcursor.h>

#ifndef QT_NO_CURSOR

#include <qicon.h>
#include <qapplication.h>
#include <qdesktopwidget.h>
#include <qt_os2.h>

QT_BEGIN_NAMESPACE

extern QCursorData *qt_cursorTable[Qt::LastCursor + 1]; // qcursor.cpp

/*****************************************************************************
  Internal QCursorData class
 *****************************************************************************/

QCursorData::QCursorData(Qt::CursorShape s)
  : cshape(s), bm(0), bmm(0), hx(0), hy(0), hptr(NULLHANDLE), isSysPtr(false)
{
    ref = 1;
}

QCursorData::~QCursorData()
{
    delete bm;
    delete bmm;
    if (hptr && !isSysPtr)
        WinDestroyPointer(hptr);
}


QCursorData *QCursorData::setBitmap(const QBitmap &bitmap, const QBitmap &mask, int hotX, int hotY)
{
    if (!QCursorData::initialized)
        QCursorData::initialize();
    if (bitmap.depth() != 1 || mask.depth() != 1 || bitmap.size() != mask.size()) {
        qWarning("QCursor: Cannot create bitmap cursor; invalid bitmap(s)");
        QCursorData *c = qt_cursorTable[0];
        c->ref.ref();
        return c;
    }
    QCursorData *d = new QCursorData;
    d->bm  = new QBitmap(bitmap);
    d->bmm = new QBitmap(mask);
    d->hptr = NULLHANDLE;
    d->cshape = Qt::BitmapCursor;
    d->hx = hotX >= 0 ? hotX : bitmap.width()/2;
    d->hy = hotY >= 0 ? hotY : bitmap.height()/2;
    return d;
}

HPOINTER QCursor::handle() const
{
    if (!QCursorData::initialized)
        QCursorData::initialize();
    if (d->hptr == NULLHANDLE)
        d->update();
    return d->hptr;
}

QCursor::QCursor(HPOINTER handle)
{
    d = new QCursorData(Qt::CustomCursor);
    d->hptr = handle;
}

#endif //QT_NO_CURSOR

QPoint QCursor::pos()
{
    POINTL p;
    WinQueryPointerPos(HWND_DESKTOP, &p);
    // flip y coordinate
    p.y = QApplication::desktop()->height() - (p.y + 1);
    return QPoint(p.x, p.y);
}

void QCursor::setPos(int x, int y)
{
    // flip y coordinate
    y = QApplication::desktop()->height() - (y + 1);
    WinSetPointerPos(HWND_DESKTOP, x, y);
}

#ifndef QT_NO_CURSOR

enum { cursor_uparrow_x = 11, cursor_uparrow_y = 1 };
static char const * const cursor_uparrow_xpm[] = {
"24 24 3 1",
"       c None",
".      c #000000",
"+      c #FFFFFF",
"                        ",
"           ++           ",
"          +..+          ",
"         +....+         ",
"        +......+        ",
"       +........+       ",
"        +++..+++        ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"          +..+          ",
"           ++           ",
"                        ",
"                        "};

enum { cursor_cross_x = 11, cursor_cross_y = 11 };
static char const * const cursor_cross_xpm[] = {
"24 24 3 1",
"       c None",
".      c #FFFFFF",
"+      c #000000",
"                        ",
"           ..           ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"  ........   ........   ",
" .++++++++   ++++++++.  ",
" .++++++++   ++++++++.  ",
"  ........   ........   ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"          .++.          ",
"           ..           ",
"                        "};

enum { cursor_vsplit_x = 11, cursor_vsplit_y = 11 };
static char const * const cursor_vsplit_xpm[] = {
"24 24 3 1",
"       c None",
".      c #000000",
"+      c #FFFFFF",
"                        ",
"           ++           ",
"          +..+          ",
"         +....+         ",
"        +......+        ",
"       +........+       ",
"        +++..+++        ",
"          +..+          ",
"  +++++++++..+++++++++  ",
" +....................+ ",
" +....................+ ",
"  ++++++++++++++++++++  ",
"  ++++++++++++++++++++  ",
" +....................+ ",
" +....................+ ",
"  +++++++++..+++++++++  ",
"          +..+          ",
"        +++..+++        ",
"       +........+       ",
"        +......+        ",
"         +....+         ",
"          +..+          ",
"           ++           ",
"                        "};

enum { cursor_hsplit_x = 11, cursor_hsplit_y = 11 };
static char const * const cursor_hsplit_xpm[] = {
"24 24 3 1",
"       c None",
".      c #000000",
"+      c #FFFFFF",
"                        ",
"         ++++++         ",
"        +..++..+        ",
"        +..++..+        ",
"        +..++..+        ",
"        +..++..+        ",
"        +..++..+        ",
"     +  +..++..+  +     ",
"    +.+ +..++..+ +.+    ",
"   +..+ +..++..+ +..+   ",
"  +...+++..++..+++...+  ",
" +.........++.........+ ",
" +.........++.........+ ",
"  +...+++..++..+++...+  ",
"   +..+ +..++..+ +..+   ",
"    +.+ +..++..+ +.+    ",
"     +  +..++..+  +     ",
"        +..++..+        ",
"        +..++..+        ",
"        +..++..+        ",
"        +..++..+        ",
"        +..++..+        ",
"         ++++++         ",
"                        "};

enum { cursor_blank_x = 0, cursor_blank_y = 0 };
static char const * const cursor_blank_xpm[] = {
"1 1 1 1",
"       c None",
" "};

enum { cursor_hand_x = 9, cursor_hand_y = 0 };
static char const * const cursor_hand_xpm[] = {
"24 24 3 1",
"       c None",
".      c #FFFFFF",
"+      c #000000",
"         ++             ",
"        +..+            ",
"        +..+            ",
"        +..+            ",
"        +..+            ",
"        +..+            ",
"        +..+++          ",
"        +..+..+++       ",
"        +..+..+..++     ",
"     ++ +..+..+..+.+    ",
"    +..++..+..+..+.+    ",
"    +...+..........+    ",
"     +.............+    ",
"      +............+    ",
"      +............+    ",
"       +..........+     ",
"       +..........+     ",
"        +........+      ",
"        +........+      ",
"        ++++++++++      ",
"        ++++++++++      ",
"        ++++++++++      ",
"                        ",
"                        "};

enum { cursor_whatsthis_x = 9, cursor_whatsthis_y = 0 };
static char const * const cursor_whatsthis_xpm[] = {
"24 24 3 1",
"       c None",
".      c #000000",
"+      c #FFFFFF",
".                       ",
"..          ++++++      ",
".+.        +......+     ",
".++.      +........+    ",
".+++.    +....++....+   ",
".++++.   +...+  +...+   ",
".+++++.  +...+  +...+   ",
".++++++.  +.+  +....+   ",
".+++++++.  +  +....+    ",
".++++++++.   +....+     ",
".+++++.....  +...+      ",
".++.++.      +...+      ",
".+. .++.     +...+      ",
"..  .++.      +++       ",
".    .++.    +...+      ",
"     .++.    +...+      ",
"      .++.   +...+      ",
"      .++.    +++       ",
"       ..               ",
"                        ",
"                        ",
"                        ",
"                        ",
"                        "};

enum { cursor_openhand_x = 7, cursor_openhand_y = 7 };
static char const * const cursor_openhand_xpm[] = {
"16 16 3 1",
"       g None",
".      g #000000",
"+      g #FFFFFF",
"       ..       ",
"   .. .++...    ",
"  .++..++.++.   ",
"  .++..++.++. . ",
"   .++.++.++..+.",
"   .++.++.++.++.",
" ..+.+++++++.++.",
".++..++++++++++.",
".+++.+++++++++. ",
" .++++++++++++. ",
"  .+++++++++++. ",
"  .++++++++++.  ",
"   .+++++++++.  ",
"    .+++++++.   ",
"     .++++++.   ",
"                "};

enum { cursor_closedhand_x = 7, cursor_closedhand_y = 7 };
static char const * const cursor_closedhand_xpm[] = {
"16 16 3 1",
"       g None",
".      g #000000",
"+      g #FFFFFF",
"                ",
"                ",
"                ",
"    .. .. ..    ",
"   .++.++.++..  ",
"   .++++++++.+. ",
"    .+++++++++. ",
"   ..+++++++++. ",
"  .+++++++++++. ",
"  .++++++++++.  ",
"   .+++++++++.  ",
"    .+++++++.   ",
"     .++++++.   ",
"     .++++++.   ",
"                ",
"                "};

void QCursorData::update()
{
    if (!QCursorData::initialized)
        QCursorData::initialize();

    if (cshape == Qt::BitmapCursor) {
        QPixmap pm = pixmap;
        if (pm.isNull()) {
            Q_ASSERT(bm && bmm);
            pm = *bm;
            pm.setMask(*bmm);
        }
        hptr = QPixmap::toPmHPOINTER(QIcon(pm), true, hx, hy);
        Q_ASSERT(hptr);
        isSysPtr = false;
        return;
    }

    LONG id = 0;
    char const * const * xpm = 0;
    int xpm_hx = 0;
    int xpm_hy = 0;

    switch (cshape) { // map to OS/2 cursor
    case Qt::ArrowCursor:
        id = SPTR_ARROW;
        break;
    case Qt::UpArrowCursor:
        xpm = cursor_uparrow_xpm;
        xpm_hx = cursor_uparrow_x;
        xpm_hy = cursor_uparrow_y;
        break;
    case Qt::CrossCursor:
        xpm = cursor_cross_xpm;
        xpm_hx = cursor_cross_x;
        xpm_hy = cursor_cross_y;
	    break;
    case Qt::WaitCursor:
        id = SPTR_WAIT;
	    break;
    case Qt::IBeamCursor:
    	id = SPTR_TEXT;
    	break;
    case Qt::SizeVerCursor:
    	id = SPTR_SIZENS;
    	break;
    case Qt::SizeHorCursor:
    	id = SPTR_SIZEWE;
    	break;
    case Qt::SizeBDiagCursor:
    	id = SPTR_SIZENESW;
    	break;
    case Qt::SizeFDiagCursor:
    	id = SPTR_SIZENWSE;
    	break;
    case Qt::SizeAllCursor:
    	id = SPTR_MOVE;
    	break;
    case Qt::BlankCursor:
        xpm = cursor_blank_xpm;
        xpm_hx = cursor_blank_x;
        xpm_hy = cursor_blank_y;
        break;
    case Qt::SplitVCursor:
        xpm = cursor_vsplit_xpm;
        xpm_hx = cursor_vsplit_x;
        xpm_hy = cursor_vsplit_y;
        break;
    case Qt::SplitHCursor:
        xpm = cursor_hsplit_xpm;
        xpm_hx = cursor_hsplit_x;
        xpm_hy = cursor_hsplit_y;
        break;
    case Qt::PointingHandCursor:
        xpm = cursor_hand_xpm;
        xpm_hx = cursor_hand_x;
        xpm_hy = cursor_hand_y;
        break;
    case Qt::ForbiddenCursor:
    	id = SPTR_ILLEGAL;
    	break;
    case Qt::WhatsThisCursor:
        xpm = cursor_whatsthis_xpm;
        xpm_hx = cursor_whatsthis_x;
        xpm_hy = cursor_whatsthis_y;
        break;
    case Qt::BusyCursor:
        id = SPTR_WAIT;
        break;
    case Qt::OpenHandCursor:
        xpm = cursor_openhand_xpm;
        xpm_hx = cursor_openhand_x;
        xpm_hy = cursor_openhand_y;
        break;
    case Qt::ClosedHandCursor:
        xpm = cursor_closedhand_xpm;
        xpm_hx = cursor_closedhand_x;
        xpm_hy = cursor_closedhand_y;
        break;
    default:
        qWarning("QCursor::update: Invalid cursor shape %d", cshape);
        return;
    }

    if (!id) {
#ifndef QT_NO_IMAGEFORMAT_XPM
        QPixmap pm(xpm);
        hptr = QPixmap::toPmHPOINTER(QIcon(pm), true, xpm_hx, xpm_hy);
        Q_ASSERT(hptr);
        isSysPtr = false;
        return;
#else
        id = SPTR_ARROW;
#endif
    }

    Q_ASSERT(id);

    hptr = WinQuerySysPointer(HWND_DESKTOP, id, FALSE);
    Q_ASSERT(hptr);
    isSysPtr = true;
}

#endif // QT_NO_CURSOR

QT_END_NAMESPACE
