/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qapplication.h"

#include "qapplication_p.h"
#include "qevent.h"
#include "qpainter.h"
#include "qwidget.h"
#include "qbuffer.h"
#include "qdatastream.h"
#include "qcursor.h"
#include "qdesktopwidget.h"
#include "qfile.h"
#include "qdnd_p.h"
#include "qdebug.h"

#include "qt_os2.h"

//#define QDND_DEBUG // in pair with qmime_pm.cpp

#ifdef QDND_DEBUG
#   define DEBUG(a) qDebug a
#else
#   define DEBUG(a) do {} while(0)
#endif

QT_BEGIN_NAMESPACE

#if !defined(QT_NO_DRAGANDDROP) && !defined(QT_NO_CLIPBOARD)

extern void qt_pmMouseButtonUp(); // defined in qapplication_pm.cpp
extern void qt_DrgFreeDragtransfer(DRAGTRANSFER *xfer); // defined in qmime_pm.cpp

/** \internal
 *  Data for QDragEnterEvent/QDragMoveEvent/QPMDropEvent.
 */
class QPMDragData
{
public:
    QPMDragData();
    ~QPMDragData();

    void initialize(DRAGINFO *di);
    void reset(bool isAccepted);
    void reset() { reset(false); }

    void setDropped(bool d) { dropped = d; }
    bool isDropped() const { return dropped; }

    DRAGINFO *info() const { return di; }

    bool hasFormat_sys(const QString &mimeType);
    QStringList formats_sys();
    QVariant retrieveData_sys(const QString &mimeType,
                              QVariant::Type preferredType);

private:
    friend class QDragManager;

    void initWorkers();

    bool initialized : 1;
    bool dropped : 1;
    bool gotWorkers : 1;

    DRAGINFO *di;
    QList<QPMMime::DropWorker*> workers;

    QWidget *lastDragOverWidget; // last target widget
    USHORT lastDragOverOp; // last DM_DRAGOVER operation

    USHORT supportedOps; // operations supported by all items
    bool sourceAllowsOp : 1; // does source allow requested operation

    USHORT lastDropReply; // last reply to DM_DRAGOVER
    USHORT lastOpRequest; // last op requested in DM_DRAGOVER

    Qt::DropAction lastProposedAction; // last proposed action
    QRect lastAnswerRect; // last accepted rectangle from the target
};

QPMDragData::QPMDragData()
    : initialized(false), dropped(false)
    , gotWorkers(false), di(NULL)
    , lastDragOverWidget(0), lastDragOverOp(0)
    , supportedOps(0), sourceAllowsOp(false)
    , lastDropReply(DOR_NEVERDROP), lastOpRequest(DO_UNKNOWN)
    , lastProposedAction(Qt::CopyAction)
{
}

QPMDragData::~QPMDragData()
{
    reset();
}

void QPMDragData::initialize(DRAGINFO *info)
{
    Q_ASSERT(info);
    if (initialized || !info)
        return;

    initialized = true;
    di = info;
}

void QPMDragData::initWorkers()
{
    Q_ASSERT(initialized);
    if (!initialized || gotWorkers)
        return;

    gotWorkers = true;

    // go through all converters and collect DropWorkers to use
    foreach (QPMMime *mime, QPMMime::all()) {
        QPMMime::DropWorker *wrk = mime->dropWorkerFor(di);
        if (wrk) {
            if (wrk->isExclusive()) {
                // ignore all other workers if some of them identified itself
                // as exclusive
                workers.clear();
                workers.append(wrk);
                break;
            }
            // ensure there are no duplicateseed
            if (!workers.contains(wrk))
                workers.append(wrk);
        }
    }

    DEBUG(() << "QPMDragData:" << workers.count() << "drop workers for DRAGINFO" << di);

    // init all workers
    foreach (QPMMime::DropWorker *w, workers) {
        w->nfo = di;
        w->init();
    }
}

void QPMDragData::reset(bool isAccepted)
{
    if (!initialized)
        return;

    // cleanup all workers
    foreach (QPMMime::DropWorker *w, workers) {
        w->cleanup(isAccepted);
        w->nfo = 0;
    }

    workers.clear();
    di = 0;
    initialized = dropped = gotWorkers = false;
}

bool QPMDragData::hasFormat_sys(const QString &mimeType)
{
    if (!gotWorkers)
        initWorkers();

    foreach (QPMMime::DropWorker *w, workers)
        if (w->hasFormat(mimeType))
            return true;

    return false;
}

QStringList QPMDragData::formats_sys()
{
    QStringList mimes;

    if (!gotWorkers)
        initWorkers();

    foreach (QPMMime::DropWorker *w, workers)
        mimes << w->formats();

    return mimes;
}

QVariant QPMDragData::retrieveData_sys(const QString &mimeType,
                                       QVariant::Type preferredType)
{
    QVariant result;

    // we may only do data transfer after DM_DROP is sent. Return shortly.
    if (!isDropped())
        return result;

    if (!gotWorkers)
        initWorkers();

    foreach (QPMMime::DropWorker *w, workers) {
        if (w->hasFormat(mimeType)) {
            result = w->retrieveData(mimeType, preferredType);
            break;
        }
    }

    return result;
}

static Qt::DropActions toQDragDropActions(USHORT ops)
{
    Qt::DropActions actions = Qt::IgnoreAction;
    if (ops & DO_LINKABLE)
        actions |= Qt::LinkAction;
    if (ops & DO_COPYABLE)
        actions |= Qt::CopyAction;
    if (ops & DO_MOVEABLE)
        actions |= Qt::MoveAction;
    return actions;
}

static Qt::DropAction toQDragDropAction(USHORT op)
{
    if (op == DO_LINK)
        return Qt::LinkAction;
    if (op == DO_COPY)
        return Qt::CopyAction;
    if (op == DO_MOVE)
        return Qt::MoveAction;
    return Qt::IgnoreAction;
}

static USHORT toPmDragDropOp(Qt::DropActions action)
{
    if (action & Qt::LinkAction)
        return DO_LINK;
    if (action & Qt::CopyAction)
        return DO_COPY;
    if (action & Qt::MoveAction)
        return DO_MOVE;
    return DO_UNKNOWN;
}

static USHORT toPmDragDropOps(Qt::DropActions actions)
{
    USHORT op = DO_UNKNOWN;
    if (actions & Qt::LinkAction)
        op |= DO_LINKABLE;
    if (actions & Qt::CopyAction)
        op |= DO_COPYABLE;
    if (actions & Qt::MoveAction)
        op |= DO_MOVEABLE;
    return op;
}

void QDragManager::init_sys()
{
    inDrag = false;

    Q_ASSERT(dropData);
    Q_ASSERT(!dropData->d);
    dropData->d = new QPMDragData();
}

void QDragManager::uninit_sys()
{
    Q_ASSERT(dropData);
    Q_ASSERT(dropData->d);
    delete dropData->d;
    dropData->d = 0;
}

void QDragManager::sendDropEvent(QWidget *receiver, QEvent *event)
{
    Q_ASSERT(!inDrag);
    inDrag = true;
    QApplication::sendEvent(receiver, event);
    // make sure that all issued update requests are handled before we
    // return from the DM_DRAGOVER/DM_DRAGLEAVE/DM_DROP message; otherwise
    // we'll get screen corruption due to unexpected screen updates while
    // dragging
    QApplication::sendPostedEvents(0, QEvent::UpdateRequest);
    inDrag = false;
}

/*!
 *  \internal
 *  Direct manipulation (Drag & Drop) event handler
 */
MRESULT QDragManager::dispatchDragAndDrop(QWidget *widget, const QMSG &qmsg)
{
    Q_ASSERT(widget);

    BOOL ok = FALSE;

    QDragManager *manager = QDragManager::self();
    QPMDragData *dragData = manager->dropData->d;
    Q_ASSERT(dragData);

    // @todo use dragData->lastAnswerRect to compress DM_DRAGOVER events

    switch (qmsg.msg) {
        case DM_DRAGOVER: {
            DEBUG(("DM_DRAGOVER: hwnd %lX di %p x %d y %d", qmsg.hwnd, qmsg.mp1,
                   SHORT1FROMMP(qmsg.mp2), SHORT2FROMMP(qmsg.mp2)));

            DRAGINFO *info = (DRAGINFO *)qmsg.mp1;
            ok = DrgAccessDraginfo(info);
            Q_ASSERT(ok && info);
            if (!ok || !info)
                return MRFROM2SHORT(DOR_NEVERDROP, 0);

            // flip y coordinate
            QPoint pnt(info->xDrop, info->yDrop);
            pnt.setY(QApplication::desktop()->height() - (pnt.y() + 1));
            pnt = widget->mapFromGlobal(pnt);

            QWidget *dragOverWidget = widget->childAt(pnt);
            if (!dragOverWidget)
                dragOverWidget = widget;

            bool first = dragData->lastDragOverWidget != dragOverWidget;
            if (first) {
                // the first DM_DRAGOVER message
                if (dragData->lastDragOverWidget != 0) {
                    // send drag leave to the old widget
                    dragData->reset();
                    QPointer<QWidget> dragOverWidgetGuard(dragOverWidget);
                    QDragLeaveEvent dle;
                    sendDropEvent(dragData->lastDragOverWidget, &dle);
                    if (!dragOverWidgetGuard) {
                        dragOverWidget = widget->childAt(pnt);
                        if (!dragOverWidget)
                            dragOverWidget = widget;
                    }
                }
                dragData->lastDragOverWidget = dragOverWidget;
                dragData->lastDragOverOp = 0;
                dragData->supportedOps = DO_COPYABLE | DO_MOVEABLE | DO_LINKABLE;
                dragData->sourceAllowsOp = false;
                dragData->lastDropReply = DOR_NEVERDROP;
                dragData->lastOpRequest = DO_UNKNOWN;
                dragData->lastProposedAction =
                    manager->defaultAction(toQDragDropActions(dragData->supportedOps),
                                           Qt::NoModifier);
                dragData->lastAnswerRect = QRect();
                // ensure drag data is reset (just in case of a wrong msg flow...)
                dragData->reset();
            }

            if (!dragOverWidget->acceptDrops()) {
                // We don't reply with DOR_NEVERDROP here because in this
                // case PM will stop sending DM_DRAGOVER to the given HWND but
                // we may have another non-native child (that has the same HWND)
                // at a different position that accepts drops
                DrgFreeDraginfo(info);
                return MRFROM2SHORT(DOR_NODROP, 0);
            }

            USHORT dropReply = DOR_DROP;

            if (first) {
                // determine the set of operations supported by *all* items
                // (this implies that DRAGITEM::fsSupportedOps is a bit field)
                ULONG itemCount = DrgQueryDragitemCount(info);
                for (ULONG i = 0; i < itemCount; ++ i) {
                    PDRAGITEM item = DrgQueryDragitemPtr(info, i);
                    Q_ASSERT(item);
                    if (!item) {
                        dropReply = DOR_NEVERDROP;
                        break;
                    }
                    dragData->supportedOps &= item->fsSupportedOps;
                }
                if (dropReply != DOR_NEVERDROP) {
                    Q_ASSERT(itemCount);
                    if (!itemCount || !dragData->supportedOps) {
                        // items don't have even a single common operation...
                        dropReply = DOR_NEVERDROP;
                    }
                }
            }

            if (dropReply != DOR_NEVERDROP) {

                if (first || dragData->lastDragOverOp != info->usOperation) {
                    // the current drop operation was changed by a modifier key
                    dragData->lastDragOverOp = info->usOperation;
                    USHORT realOp = info->usOperation;
                    if (realOp == DO_DEFAULT || realOp == DO_UNKNOWN) {
                        // the default operation is requested
                        realOp = toPmDragDropOp(dragData->lastProposedAction);
                    }
                    dragData->sourceAllowsOp =
                        ((dragData->supportedOps & DO_MOVEABLE) && realOp == DO_MOVE) ||
                        ((dragData->supportedOps & DO_COPYABLE) && realOp == DO_COPY) ||
                        ((dragData->supportedOps & DO_LINKABLE) && realOp == DO_LINK);
                }

                // Note that if sourceAllowsOp = false here, we have to deliver
                // events anyway (stealing them from Qt would be confusing), but
                // we will silently ignore any accept commands and always reject
                // the drop. Other platforms seem to do the same.

                QMimeData *data = manager->source() ? manager->dragPrivate()->data : manager->dropData;

                Qt::DropAction action = toQDragDropAction(dragData->lastOpRequest);

                if (first) {
                    dragData->initialize(info);
                    QDragEnterEvent dee(pnt,
                                        toQDragDropActions(dragData->supportedOps),
                                        data, QApplication::mouseButtons(),
                                        QApplication::keyboardModifiers());
                    sendDropEvent(dragOverWidget, &dee);
                    // if not allowed or not accepted, always reply DOR_NODROP
                    // to have DM_DRAGOVER delivered to us again for a new test
                    if (dragData->sourceAllowsOp && dee.isAccepted()) {
                        dropReply = DOR_DROP;
                        action = dee.dropAction();
                        dragData->lastProposedAction = dee.proposedAction();
                        dragData->lastAnswerRect = dee.answerRect();
                    } else {
                        dropReply = DOR_NODROP;
                    }
                }

                // note: according to the Qt documentation, the move event
                // is sent immediately after the enter event, do so (only if
                // the target accepts it)
                if (!first || dropReply == DOR_DROP) {
                    QDragMoveEvent dme(pnt,
                                       toQDragDropActions(dragData->supportedOps),
                                       data, QApplication::mouseButtons(),
                                       QApplication::keyboardModifiers());
                    // accept by default, since enter event was accepted.
                    dme.setDropAction(action);
                    dme.accept();
                    sendDropEvent(dragOverWidget, &dme);
                    if (dragData->sourceAllowsOp && dme.isAccepted()) {
                        dropReply = DOR_DROP;
                        action = dme.dropAction();
                        dragData->lastProposedAction = dme.proposedAction();
                        dragData->lastAnswerRect = dme.answerRect();
                    } else {
                        dropReply = DOR_NODROP;
                    }
                }

                dragData->lastDropReply = dropReply;
                dragData->lastOpRequest = toPmDragDropOp(action);
            }

            DrgFreeDraginfo(info);

            return MRFROM2SHORT(dropReply, dragData->lastOpRequest);
        }
        case DM_DRAGLEAVE: {
            DEBUG(("DM_DRAGLEAVE: hwnd %lX di %p", qmsg.hwnd, qmsg.mp1));

            // Odin32 apps produce incorrect message flow, ignore
            Q_ASSERT(dragData->lastDragOverWidget != 0);
            if (dragData->lastDragOverWidget == 0)
                return 0;

            QDragLeaveEvent de;
            sendDropEvent(dragData->lastDragOverWidget, &de);

            dragData->lastDragOverWidget = 0;
            dragData->reset();

            return 0;
        }
        case DM_DROP: {
            DEBUG(("DM_DROP: hwnd %lX di %p", qmsg.hwnd, qmsg.mp1));

            // Odin32 apps produce incorrect message flow, ignore
            Q_ASSERT(dragData->lastDragOverWidget != 0);
            if (dragData->lastDragOverWidget == 0)
                return 0;

            // Odin32 apps send DM_DROP even if we replied DOR_NEVERDROP or
            // DOR_NODROP, simulate DM_DRAGLEAVE
            Q_ASSERT(dragData->lastDropReply == DOR_DROP);
            if (dragData->lastDropReply != DOR_DROP) {
                QMSG qmsg2 = qmsg;
                qmsg2.msg = DM_DRAGLEAVE;
                dispatchDragAndDrop(widget, qmsg2);
                return 0;
            }

            DRAGINFO *info = (DRAGINFO *)qmsg.mp1;
            ok = DrgAccessDraginfo(info);
            Q_ASSERT(ok && info);
            if (!ok || !info)
                return 0;

            // flip y coordinate
            QPoint pnt(info->xDrop, info->yDrop);
            pnt.setY(QApplication::desktop()->height() - (pnt.y() + 1));
            pnt = widget->mapFromGlobal(pnt);

            Q_ASSERT(dragData->lastOpRequest == info->usOperation);

            Q_ASSERT(dragData->lastDragOverWidget->acceptDrops());
            if (!dragData->lastDragOverWidget->acceptDrops()) {
                DrgDeleteDraginfoStrHandles(info);
                DrgFreeDraginfo(info);
                return 0;
            }

            QMimeData *data = manager->source() ? manager->dragPrivate()->data : manager->dropData;

            Qt::DropAction action = toQDragDropAction(dragData->lastOpRequest);

            dragData->setDropped(true);

            QDropEvent de(pnt, action, data, QApplication::mouseButtons(),
                          QApplication::keyboardModifiers());
            if (dragData->lastDropReply == DOR_DROP)
                de.setDropAction(action);

            sendDropEvent(dragData->lastDragOverWidget, &de);

            if (dragData->lastDropReply == DOR_DROP)
                de.accept();

            dragData->lastDragOverWidget = 0;
            dragData->reset(de.isAccepted());

            ULONG targetReply = de.isAccepted() ?
                DMFL_TARGETSUCCESSFUL : DMFL_TARGETFAIL;

            if (de.isAccepted() && de.dropAction() == Qt::TargetMoveAction) {
                // Qt::TargetMoveAction means that the target will move the
                // object (e.g. copy it to itself and delete from the source).
                // In this case, we always send DMFL_TARGETFAIL to the source to
                // prevent it from doing the same on its side.
                targetReply = DMFL_TARGETFAIL;
            }

            // send DM_ENDCONVERSATION for every item
            ULONG itemCount = DrgQueryDragitemCount(info);
            for (ULONG i = 0; i < itemCount; ++ i) {
                PDRAGITEM item = DrgQueryDragitemPtr(info, i);
                Q_ASSERT(item);
                if (!item)
                    continue;
                // it is possible that this item required DM_RENDERPREPARE but
                // returned false in reply to it (so hwndItem may be NULL)
                if (!item->hwndItem)
                    continue;
                DrgSendTransferMsg(item->hwndItem, DM_ENDCONVERSATION,
                                   MPFROMLONG(item->ulItemID),
                                   MPFROMLONG(targetReply));
            }

            DrgDeleteDraginfoStrHandles(info);
            DrgFreeDraginfo(info);

            return 0;
        }
        default:
            break;
    }

    return WinDefWindowProc(qmsg.hwnd, qmsg.msg, qmsg.mp1, qmsg.mp2);
}

//---------------------------------------------------------------------
//                    QDropData
//---------------------------------------------------------------------

bool QDropData::hasFormat_sys(const QString &mimeType) const
{
    Q_ASSERT(d);
    if (d)
        return d->hasFormat_sys(mimeType);
    return false;
}

QStringList QDropData::formats_sys() const
{
    QStringList fmts;
    Q_ASSERT(d);
    if (d)
        fmts = d->formats_sys();
    return fmts;
}

QVariant QDropData::retrieveData_sys(const QString &mimeType, QVariant::Type type) const
{
    QVariant result;
    Q_ASSERT(d);
    if (d)
        result = d->retrieveData_sys(mimeType, type);
    return result;
}

//---------------------------------------------------------------------
//                    QPMCoopDragWorker
//---------------------------------------------------------------------

class QPMCoopDragWorker : public QPMMime::DragWorker, public QPMObjectWindow
{
public:
    QPMCoopDragWorker() : info(0) {}
    bool collectWorkers(QDrag *o);

    // DragWorker interface
    void init();
    bool cleanup(bool isCancelled);
    bool isExclusive() const { return true; }
    ULONG itemCount() const { return 0; }
    HWND hwnd() const;
    DRAGINFO *createDragInfo(const QString &targetName, USHORT supportedOps);

    // QPMObjectWindow interface
    MRESULT message(ULONG msg, MPARAM mp1, MPARAM mp2);
    
private:
    QList<DragWorker*> workers;
    // todo check workers!
    //QPtrList<DragWorker> workers;
    DRAGINFO *info;
};

bool QPMCoopDragWorker::collectWorkers(QDrag *o)
{
    Q_ASSERT(o);
    if (!o)
        return false;

    bool gotExcl = false; // got isExclusive() worker?
    bool skipExcl = false; // skip isExclusive() or itemCount() > 1 workers?
    ULONG coopLevel = 0; // itemCount() level for !isExclusive() workers 

    bool gotExclForMime = false;

    // go through all formats and all converters to collect DragWorkers
    QMimeData *mimeData = o->mimeData();
    foreach (const QString &fmt, mimeData->formats()) {
        DEBUG(() << "QPMCoopDragWorker: Searching for worker for mime" << fmt);
        foreach (QPMMime *mime, QPMMime::all()) {
            DragWorker *wrk = mime->dragWorkerFor(fmt, mimeData);
            if (!wrk)
                continue;
            DEBUG(() << "QPMCoopDragWorker: Got worker" << wrk
                     << "(isExclusive" << wrk->isExclusive() << ", "
                     << "itemCount" << wrk->itemCount() << ") from convertor"
                     << mime->convertorName() << " (gotExclForMime"
                     << gotExclForMime << ", " << "gotExcl" << gotExcl
                     << ", skipExcl" << skipExcl << ", coopLevel"
                     << coopLevel << ")");
            if (wrk->isExclusive()) {
                if (!skipExcl && !gotExclForMime) {
                    gotExclForMime = true;
                    if (!gotExcl) {
                        gotExcl = true;
                        workers.append(wrk);
                    } else {
                        // skip everything exclusive unless it's exactly the
                        // same worker
                        skipExcl = !workers.contains(wrk);
                    }
                }
                // continue to search for a fall-back cooperative 1-item worker
                // (like QPMMimeAnyMime) for the case if this worker quits
                // the game
                continue;
            }
            ULONG itemCnt = wrk->itemCount();
            if (itemCnt == 0) {
                DEBUG(() << "QPMCoopDragWorker: Cooperative DragWorker"
                         << wrk << "for mime " << fmt << " has "
                         << "itemCount = 0!");
                continue;
            }
            if (itemCnt > 1) {
                // coop workers with item count > 1 are also considered exclusive
                // here, because may not co-exist with 1-item workers that should
                // always be able to contribute
                if (!gotExcl && !skipExcl && !gotExclForMime) {
                    gotExclForMime = true;
                    workers.append(wrk);
                    if (!coopLevel)
                        coopLevel = itemCnt;
                    // only those for the same number of items can proceed
                    if (itemCnt != coopLevel)
                        skipExcl = true;
                }
                // continue to search for a fall-back cooperative 1-item worker
                // (like QPMMimeAnyMime) for the case if this worker quits
                // the game
                continue;
            }
            workers.append(wrk);
            // Don't search for other workrers for the same mime type --
            // we've already got a drag worker for it and adding another
            // one would just introduce mime type duplicates on the drop
            // target's side (where the first encountered drop worker
            // for the given RMF would be used for actual data transfer
            // anyway). See also QClipboard::setData(). 
            break;
        }
        if (gotExclForMime) {
            // ensure we have a fall-back coop (should be the last added item)
            DragWorker *w = workers.last();
            if (w->isExclusive() || w->itemCount() > 1) {
                DEBUG(() << "QPMCoopDragWorker: DragWorker" << w
                         << "for" << fmt << "(isExclusive" << w->isExclusive()
                         << ", itemCount" << w->itemCount()
                         <<") has no fall-back cooperative 1-item worker!");
                workers.removeLast();
            }
            gotExclForMime = false;
        } else {
            // got a regular (non-fall-back) 1-item coop, skip evreything else
            skipExcl = true;
        }
    }

    // remove either all exclusive workers or all their fall-back workers
    // (depending on skipExcl) and remove duplicates 
    for (QList<DragWorker*>::iterator it = workers.begin();
         it <= workers.end();) {
        DragWorker *wrk = *it;
        bool excl = wrk->isExclusive() || wrk->itemCount() > 1;
        if (skipExcl == excl || workers.count(wrk) > 1) {
            it = workers.erase(it);
        } else {
            ++it;
        }
    }

#if defined(QDND_DEBUG)            
    foreach (DragWorker *wrk, workers) {
        DEBUG(() << "QPMCoopDragWorker: Will use worker" << wrk
                 << "(isExclusive" << wrk->isExclusive()
                 << ", itemCount" << wrk->itemCount() << ")"); 
    }
#endif

    Q_ASSERT(workers.count() > 0);
    return workers.count() > 0;
}

HWND QPMCoopDragWorker::hwnd() const
{
    DragWorker *firstWorker = workers.first();
    Q_ASSERT(firstWorker);
    if (!firstWorker)
        return 0;
    
    if (firstWorker->isExclusive() && firstWorker->itemCount() == 0) {
        // this is a super exclusive worker that will do everything on its own
        return firstWorker->hwnd();
    }
    
    return QPMObjectWindow::hwnd();
}

void QPMCoopDragWorker::init()
{
    Q_ASSERT(source());
    foreach(DragWorker *wrk, workers) {
        wrk->src = source();
        wrk->init();
    }
}

bool QPMCoopDragWorker::cleanup(bool isCancelled)
{
    bool moveDisallowed = false;
    
    foreach(DragWorker *wrk, workers) {
        // disallow the Move operation if at least one worker asked so
        moveDisallowed |= wrk->cleanup(isCancelled);
        wrk->src = 0;
    }
    workers.clear();
    info = 0;
    return moveDisallowed;
}

DRAGINFO *QPMCoopDragWorker::createDragInfo(const QString &targetName,
                                            USHORT supportedOps)
{
    Q_ASSERT(!info);
    if (info)
        return 0;

    DragWorker *firstWorker = workers.first();
    Q_ASSERT(firstWorker);
    if (!firstWorker)
        return 0;

    ULONG itemCnt = firstWorker->itemCount();

    if (firstWorker->isExclusive() && itemCnt == 0) {
        // this is a super exclusive worker that will do everything on its own
        DEBUG(() << "QPMCoopDragWorker: Will redirect to super worker"
                 << firstWorker);
        return firstWorker->createDragInfo(targetName, supportedOps);
    }
    
    // note that all workers at this place require the same amount of items
    // (guaranteed by collectWorkers())
    
    DEBUG(() << "QPMCoopDragWorker: itemCnt" << itemCnt);

    info = DrgAllocDraginfo(itemCnt);
    Q_ASSERT(info);
    if (!info)
        return 0;
    
    // collect all mechanism/format pairs
    QByteArray allFormats;
    foreach (DragWorker *wrk, workers) {
        QByteArray formats = wrk->composeFormatString();
        Q_ASSERT(!formats.isNull());
        if (!formats.isNull()) {
            if (allFormats.isNull())
                allFormats = formats;
            else {
                allFormats += ",";
                allFormats += formats;
            }
        }
    }

    DEBUG(() << "QPMCoopDragWorker: allFormats" << allFormats);

    static ULONG itemID = 0;
    
    const char *type = 0;
    const char *ext = 0;
    firstWorker->defaultFileType(type, ext);

    bool ok = true;
    for (ULONG i = 0; i < itemCnt; ++i) {
        DRAGITEM *item = DrgQueryDragitemPtr(info, i);
        Q_ASSERT(item);
        if (!item) {
            ok = false;
            break;
        }

        QString name;
        if (itemCnt == 1)
            name = targetName;
        else
            name = QString(QLatin1String("%1 %2")).arg(targetName).arg(i + 1);
        
        if (ext) {
            name += QLatin1Char('.');
            name += QFile::decodeName(QByteArray(ext));
        }

        DEBUG(() << "QPMCoopDragWorker: item" << i << ": type" << type
                 << " name" << name);

        // Note 1: DRAGITEM::hstrType is actually ignored by WPS,
        // only the target extension matters.
        
        // Note 2: We're not required to fill in the hwndItem field because we
        // use the DC_PREPARE flag (to indicate it will be filled later, after
        // DM_RENDERPREPARE); however, Mozilla refuses to render if hwndItem
        // is initially 0. Set it to our HWND instead (we'll issue a warning if
        // DM_RENDER or DM_ENDCONVERSATION is erroneously sent to us) 

        item->hwndItem = hwnd();
        item->ulItemID = itemID ++;
        item->hstrType = DrgAddStrHandle(type ? type : DRT_UNKNOWN);
        item->hstrRMF = DrgAddStrHandle(allFormats);
        item->hstrContainerName = 0;
        item->hstrSourceName = 0;
        item->hstrTargetName = DrgAddStrHandle(QFile::encodeName(name));
        item->cxOffset = 0;
        item->cyOffset = 0;
        item->fsControl = DC_PREPARE; // require DM_RENDERPREPARE from target
        item->fsSupportedOps = supportedOps;
    }
   
    if (!ok) {
        DrgFreeDraginfo(info);
        info = 0;
    }
    
    return info;
}

MRESULT QPMCoopDragWorker::message(ULONG msg, MPARAM mp1, MPARAM mp2)
{
    if (msg == DM_RENDERPREPARE) {
        if (!info) {
            qWarning("Drop target sent DM_RENDERPREPARE after the DnD session "
                     "is over!");
            // free the given DRAGTRANSFER structure to avoud memory leak 
            DRAGTRANSFER *xfer = (DRAGTRANSFER *)mp1;
            if (xfer)
                qt_DrgFreeDragtransfer(xfer);
            return (MRESULT)FALSE;
        }
    
        DRAGTRANSFER *xfer = (DRAGTRANSFER *)mp1;
        Q_ASSERT(xfer && xfer->pditem);
        if (!xfer || !xfer->pditem)
            return (MRESULT)FALSE;
        
        // find the item's index (ordinal number)
        ULONG itemCnt = DrgQueryDragitemCount(info);
        ULONG index = 0;
        for (; index <  itemCnt; ++index)
            if (DrgQueryDragitemPtr(info, index) == xfer->pditem)
                break;
        
        Q_ASSERT(index < itemCnt);
        if (index >= itemCnt)
            return (MRESULT)FALSE;

        DEBUG(() << "QPMCoopDragWorker: Got DM_RENDERPREPARE to"
                 << QPMMime::queryHSTR(xfer->hstrSelectedRMF) << "for item"
                 << index << "(id" << xfer->pditem->ulItemID << ")");
        
        QByteArray drm, drf;
        if (!QPMMime::parseRMF(xfer->hstrSelectedRMF, drm, drf)) {
            Q_ASSERT(false);
            return (MRESULT)FALSE;
        }

        DragWorker *wrk = 0;
        foreach(wrk, workers)
            if (wrk->prepare(drm, drf, xfer->pditem, index))
                break;
        if (!wrk) {
            DEBUG(() << "QPMCoopDragWorker: No suitable worker found");
            return (MRESULT)FALSE;
        }

        xfer->pditem->hwndItem = wrk->hwnd();
        Q_ASSERT(xfer->pditem->hwndItem);
        return (MRESULT)(xfer->pditem->hwndItem ? TRUE : FALSE);
    }
    
    if (msg == DM_RENDER || msg == DM_ENDCONVERSATION) {
        qWarning("Drop target sent DM_RENDER or DM_ENDCONVERSATION to the "
                 "drag source window instead of the drag item window!");
        if (msg == DM_RENDER) {
            // free the given DRAGTRANSFER structure to avoud memory leak 
            DRAGTRANSFER *xfer = (DRAGTRANSFER *)mp1;
            if (xfer)
                qt_DrgFreeDragtransfer(xfer);
        }
    }
    
    return (MRESULT)FALSE;
}

//---------------------------------------------------------------------
//                    QDragManager
//---------------------------------------------------------------------

Qt::DropAction QDragManager::drag(QDrag *o)

{
    DEBUG(() << "QDragManager::drag");

    if (object == o || !o || !o->d_func()->source)
        return Qt::IgnoreAction;

    if (object) {
        cancel();
        qApp->removeEventFilter(this);
        beingCancelled = false;
    }

    // detect a mouse button to end dragging
    LONG vkTerminate = 0;
    {
        ULONG msg = WinQuerySysValue(HWND_DESKTOP, SV_BEGINDRAG) & 0xFFFF;
        switch(msg) {
            case WM_BUTTON1MOTIONSTART: vkTerminate = VK_BUTTON1; break; 
            case WM_BUTTON2MOTIONSTART: vkTerminate = VK_BUTTON2; break; 
            case WM_BUTTON3MOTIONSTART: vkTerminate = VK_BUTTON3; break; 
        }

        if (WinGetKeyState(HWND_DESKTOP, vkTerminate) & 0x8000) {
            // prefer the default button if it is pressed
        } else if (WinGetKeyState(HWND_DESKTOP, VK_BUTTON2) & 0x8000) {
            vkTerminate = VK_BUTTON2;
        } else if (WinGetKeyState(HWND_DESKTOP, VK_BUTTON1) & 0x8000) {
            vkTerminate = VK_BUTTON1;
        } else if (WinGetKeyState(HWND_DESKTOP, VK_BUTTON3) & 0x8000) {
            vkTerminate = VK_BUTTON3;
        } else {
            vkTerminate = 0;
        }
    }

    if (!vkTerminate) {
        DEBUG(() << "QDragManager::drag: No valid mouse button pressed, "
                    "dragging cancelled!");
        o->deleteLater();
        return Qt::IgnoreAction;
    }

    USHORT supportedOps = toPmDragDropOps(dragPrivate()->possible_actions);

    static QPMCoopDragWorker dragWorker;
    
    bool ok = dragWorker.collectWorkers(o);
    Q_ASSERT(ok);
    Q_ASSERT(dragWorker.hwnd());
    if (!ok || !dragWorker.hwnd()) {
        o->deleteLater();
        return Qt::IgnoreAction;
    }

    dragWorker.src = o->mimeData();
    dragWorker.init();
    DRAGINFO *info = dragWorker.createDragInfo(o->objectName(), supportedOps);

    Q_ASSERT(info);
    if (!info) {
        dragWorker.cleanup(true /* isCancelled */);
        dragWorker.src = 0;
        o->deleteLater();
        return Qt::IgnoreAction;
    }

    object = o;

    DEBUG(() << "QDragManager::drag: actions"
             << dragActionsToString(dragPrivate()->possible_actions));

    dragPrivate()->target = 0;

#ifndef QT_NO_ACCESSIBILITY
    QAccessible::updateAccessibility(this, 0, QAccessible::DragDropStart);
#endif

    // @todo custom drag pixmap?

    DRAGIMAGE img;
    img.cb = sizeof(DRAGIMAGE);
    img.hImage = WinQuerySysPointer(HWND_DESKTOP, SPTR_FILE, FALSE);
    img.fl = DRG_ICON;
    img.cxOffset = 0;
    img.cyOffset = 0;

    // the mouse is most likely captured by Qt at this point, uncapture it
    // or DrgDrag() will definitely fail
    WinSetCapture(HWND_DESKTOP, 0);

    HWND target = DrgDrag(dragWorker.hwnd(), info, &img, 1, vkTerminate,
                          (PVOID)0x80000000L); // don't lock the desktop PS

    DEBUG(("QDragManager::drag: DrgDrag() returned %08lX (error 0x%08lX)",
            target, WinGetLastError(0)));

    // we won't get any mouse release event, so manually adjust qApp state
    qt_pmMouseButtonUp();

    bool moveDisallowed = dragWorker.cleanup(beingCancelled || target == 0);
    dragWorker.src = 0;
    
    moveDisallowed |= beingCancelled || target == 0 ||
                      info->usOperation != DO_MOVE;

    DEBUG(() << "QDragManager::drag: moveDisallowed" << moveDisallowed);

    Qt::DropAction ret = Qt::IgnoreAction;
    if (target != 0) {
        ret = toQDragDropAction(info->usOperation);
        if (moveDisallowed && info->usOperation == DO_MOVE)
            ret = Qt::TargetMoveAction;
    }

    DEBUG(() << "QDragManager::drag: result" << dragActionsToString(ret));

    if (target == 0)
        DrgDeleteDraginfoStrHandles(info);
    DrgFreeDraginfo(info);

    if (!beingCancelled) {
        dragPrivate()->target = QWidget::find(target);
        cancel(); // this will delete o (object)
    }

    beingCancelled = false;

#ifndef QT_NO_ACCESSIBILITY
    QAccessible::updateAccessibility(this, 0, QAccessible::DragDropEnd);
#endif

    return ret;
}

void QDragManager::cancel(bool deleteSource)
{
    // Note: the only place where this function is called with
    // deleteSource = false so far is QDrag::~QDrag()

    Q_ASSERT(object && !beingCancelled);
    if (!object || beingCancelled)
        return;
    
    beingCancelled = true;

    object->setMimeData(0);

    if (deleteSource)
        object->deleteLater();
    object = 0;

#ifndef QT_NO_CURSOR
    // insert cancel code here ######## todo

    if (restoreCursor) {
        QApplication::restoreOverrideCursor();
        restoreCursor = false;
    }
#endif
#ifndef QT_NO_ACCESSIBILITY
    QAccessible::updateAccessibility(this, 0, QAccessible::DragDropEnd);
#endif
}

void QDragManager::updatePixmap()
{
    // not used in PM implementation
}

bool QDragManager::eventFilter(QObject *, QEvent *)
{
    // not used in PM implementation
    return false;
}

void QDragManager::timerEvent(QTimerEvent*)
{
    // not used in PM implementation
}

void QDragManager::move(const QPoint &)
{
    // not used in PM implementation
}

void QDragManager::drop()
{
    // not used in PM implementation
}

QT_END_NAMESPACE

#endif // QT_NO_DRAGANDDROP
