/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qapplication.h"

#include "qapplication_p.h"
#include "qevent.h"
#include "qpainter.h"
#include "qwidget.h"
#include "qbuffer.h"
#include "qdatastream.h"
#include "qcursor.h"
#include "qdesktopwidget.h"
#include "qdnd_p.h"
#include "qdebug.h"

#include "qt_os2.h"

#define QDND_DEBUG // in pair with qmime_pm.cpp

#ifdef QDND_DEBUG
#   define DEBUG(a) qDebug a
#else
#   define DEBUG(a) do {} while(0)
#endif

QT_BEGIN_NAMESPACE

#if !defined(QT_NO_DRAGANDDROP) && !defined(QT_NO_CLIPBOARD)

/** \internal
 *  Data for QDragEnterEvent/QDragMoveEvent/QPMDropEvent.
 */
class QPMDragData
{
public:
    QPMDragData();
    ~QPMDragData();

    void initialize(DRAGINFO *di);
    void reset(bool isAccepted);
    void reset() { reset(false); }

    void setDropped(bool d) { dropped = d; }
    bool isDropped() const { return dropped; }

    DRAGINFO *info() const { return di; }

    bool hasFormat_sys(const QString &mimeType);
    QStringList formats_sys();
    QVariant retrieveData_sys(const QString &mimeType, QVariant::Type type);

private:

    void initWorkers();

    bool initialized : 1;
    bool dropped : 1;
    bool gotWorkers : 1;

    DRAGINFO *di;
    QList<QPMMime::DropWorker*> workers;
};

QPMDragData::QPMDragData()
    : initialized(false), dropped(false)
    , gotWorkers(false), di(NULL)
{
    QDragManager *manager = QDragManager::self();
    Q_ASSERT(!manager->dropData->d);
    manager->dropData->d = this;
}

QPMDragData::~QPMDragData()
{
    reset();

    QDragManager *manager = QDragManager::self();
    if (manager) {
        Q_ASSERT(manager->dropData->d == this);
        manager->dropData->d = 0;
    }
}

void QPMDragData::initialize(DRAGINFO *info)
{
    Q_ASSERT(info);
    if (initialized || !info)
        return;

    initialized = true;
    di = info;
}

void QPMDragData::initWorkers()
{
    Q_ASSERT(initialized);
    if (!initialized || gotWorkers)
        return;

    gotWorkers = true;

    // go through all converters and collect DropWorkers to use
    foreach (QPMMime *mime, QPMMime::all()) {
        QPMMime::DropWorker *wrk = mime->dropWorkerFor(di);
        if (wrk) {
            if (wrk->isExclusive()) {
                // ignore all other workers if some of them identified itself
                // as exclusive
                workers.clear();
                workers.append(wrk);
                break;
            }
            // ensure there are no duplicateseed
            if (!workers.contains(wrk))
                workers.append(wrk);
        }
    }

    DEBUG(() << "QPMDragData:" << workers.count() << "drop workers for DRAGINFO" << di);

    // init all workers
    foreach (QPMMime::DropWorker *w, workers) {
        w->nfo = di;
        w->init();
    }
}

void QPMDragData::reset(bool isAccepted)
{
    if (!initialized)
        return;

    // cleanup all workers
    foreach (QPMMime::DropWorker *w, workers) {
        w->cleanup(isAccepted);
        w->nfo = 0;
    }

    workers.clear();
    di = 0;
    initialized = dropped = gotWorkers = false;
}

bool QPMDragData::hasFormat_sys(const QString &mimeType)
{
    if (!gotWorkers)
        initWorkers();

    foreach (QPMMime::DropWorker *w, workers)
        if (w->hasFormat(mimeType))
            return true;

    return false;
}

QStringList QPMDragData::formats_sys()
{
    QStringList mimes;

    if (!gotWorkers)
        initWorkers();

    foreach (QPMMime::DropWorker *w, workers)
        mimes << w->formats();

    return mimes;
}

QVariant QPMDragData::retrieveData_sys(const QString &mimeType,
                                       QVariant::Type type)
{
    QVariant result;

    // we may only do data transfer after DM_DROP is sent. Return shortly.
    if (!isDropped())
        return result;

    if (!gotWorkers)
        initWorkers();

    foreach (QPMMime::DropWorker *w, workers) {
        if (w->hasFormat(mimeType)) {
            result = w->retrieveData(mimeType, type);
            break;
        }
    }

    return result;
}

static Qt::DropActions toQDragDropActions(USHORT ops)
{
    Qt::DropActions actions = Qt::IgnoreAction;
    if (ops & DO_LINKABLE)
        actions |= Qt::LinkAction;
    if (ops & DO_COPYABLE)
        actions |= Qt::CopyAction;
    if (ops & DO_MOVEABLE)
        actions |= Qt::MoveAction;
    return actions;
}

static Qt::DropAction toQDragDropAction(USHORT op)
{
    if (op == DO_LINK)
        return Qt::LinkAction;
    if (op == DO_COPY)
        return Qt::CopyAction;
    if (op == DO_MOVE)
        return Qt::MoveAction;
    return Qt::IgnoreAction;
}

static USHORT toPmDragDropOp(Qt::DropActions action)
{
    if (action & Qt::LinkAction)
        return DO_LINK;
    if (action & Qt::CopyAction)
        return DO_COPY;
    if (action & Qt::MoveAction)
        return DO_MOVE;
    return DO_UNKNOWN;
}

/*!
 *  \internal
 *  Direct manipulation (Drag & Drop) event handler
 */
MRESULT qt_dispatchDragAndDrop(QWidget *widget, const QMSG &qmsg)
{
    // @todo maybe delete last*Op variables

    static HWND lastDragOverHwnd = 0; // last target window
    static USHORT lastDragOverOp = 0; // last DM_DRAGOVER operation

    static USHORT supportedOps = 0; // operations supported by all items

    static USHORT lastDropReply = DOR_NEVERDROP; // last reply to DM_DRAGOVER
    static USHORT lastOpRequest = DO_UNKNOWN; // last op requested in DM_DRAGOVER

    static Qt::DropAction lastProposedAction = Qt::IgnoreAction; // last proposed action
    static QRect lastAnswerRect; // last accepted rectangle from the target
    // @todo use lastAnswerRect to compress DM_DRAGOVER events

    static QPMDragData dragData;

    Q_ASSERT(widget);

    BOOL ok = FALSE;

    switch (qmsg.msg) {
        case DM_DRAGOVER: {
            DEBUG(("DM_DRAGOVER: hwnd %lX di %p x %d y %d", qmsg.hwnd, qmsg.mp1,
                   SHORT1FROMMP(qmsg.mp2), SHORT2FROMMP(qmsg.mp2)));

            bool first = lastDragOverHwnd != qmsg.hwnd;
            if (first) {
                // the first DM_DRAGOVER message
                lastDragOverHwnd = qmsg.hwnd;
                lastDropReply = DOR_NEVERDROP;
                lastOpRequest = DO_UNKNOWN;
                lastProposedAction = Qt::IgnoreAction;
                lastAnswerRect = QRect();
                supportedOps = DO_COPYABLE | DO_MOVEABLE | DO_LINKABLE;
                // ensure drag data is reset (just in case of a wrong msg flow...)
                dragData.reset();
            }

            Q_ASSERT(first || widget->acceptDrops());
            if (!widget->acceptDrops()) {
                if (!first) {
                    // Odin32 apps are dramatically bogus, they continue to send
                    // DM_DRAGOVER even if we reply DOR_NEVERDROP. Simulate
                    // DM_DRAGLEAVE
                    lastDragOverHwnd = 0;
                    dragData.reset();
                }
                return MRFROM2SHORT(DOR_NEVERDROP, 0);
            }

            DRAGINFO *info = (DRAGINFO *)qmsg.mp1;
            ok = DrgAccessDraginfo(info);
            Q_ASSERT(ok && info);
            if (!ok || !info)
                return MRFROM2SHORT(DOR_NEVERDROP, 0);

            USHORT dropReply = DOR_DROP;

            if (first) {
                // determine the set of operations supported by *all* items
                // (this implies that DRAGITEM::fsSupportedOps is a bit field)
                ULONG itemCount = DrgQueryDragitemCount(info);
                for (ULONG i = 0; i < itemCount; ++ i) {
                    PDRAGITEM item = DrgQueryDragitemPtr(info, i);
                    Q_ASSERT(item);
                    if (!item) {
                        dropReply = DOR_NEVERDROP;
                        break;
                    }
                    supportedOps &= item->fsSupportedOps;
                }
                if (dropReply != DOR_NEVERDROP) {
                    Q_ASSERT(itemCount);
                    if (!itemCount || !supportedOps) {
                        // items don't have even a single common operation...
                        dropReply = DOR_NEVERDROP;
                    }
                }
            }

            if (dropReply != DOR_NEVERDROP) {

                bool sourceAllowsOp = false;

                if (first || lastDragOverOp != info->usOperation) {
                    // the current drop operation was changed by a modifier key
                    lastDragOverOp = info->usOperation;
                    USHORT realOp = info->usOperation;
                    if (realOp == DO_DEFAULT || realOp == DO_UNKNOWN) {
                        // the default operation is requested
                        realOp = toPmDragDropOp(lastProposedAction);
                    }
                    sourceAllowsOp =
                        ((supportedOps & DO_MOVEABLE) && realOp == DO_MOVE) ||
                        ((supportedOps & DO_COPYABLE) && realOp == DO_COPY) ||
                        ((supportedOps & DO_LINKABLE) && realOp == DO_LINK);
                }

                // Note that if sourceAllowsOp = false here, we have to deliver
                // events anyway (stealing them from Qt would be confusing), but
                // we will silently ignore any accept commands and always reject
                // the drop. Other platforms seem to do the same.

                // flip y coordinate
                QPoint pnt(info->xDrop, info->yDrop);
                pnt.setY(QApplication::desktop()->height() - (pnt.y() + 1));
                pnt = widget->mapFromGlobal(pnt);

                QDragManager *manager = QDragManager::self();
                QMimeData *data = manager->source() ? manager->dragPrivate()->data : manager->dropData;

                Qt::DropAction action = toQDragDropAction(lastOpRequest);

                if (first) {
                    dragData.initialize(info);
                    QDragEnterEvent dee(pnt, toQDragDropActions(supportedOps), data,
                                        QApplication::mouseButtons(),
                                        QApplication::keyboardModifiers());
                    QApplication::sendEvent(widget, &dee);
                    // if not allowed or not accepted, always reply DOR_NODROP
                    // to have DM_DRAGOVER delivered to us again in any case
                    if (sourceAllowsOp && dee.isAccepted()) {
                        dropReply = DOR_DROP;
                        action = dee.dropAction();
                        lastAnswerRect = dee.answerRect();
                    } else {
                        dropReply = DOR_NODROP;
                    }
                }

                // note: according to the Qt documentation, the move event
                // is sent immediately after the enter event, do so (only if
                // the target accepts it)
                if (!first || dropReply == DOR_DROP) {
                    QDragEnterEvent dme(pnt, toQDragDropActions(supportedOps), data,
                                        QApplication::mouseButtons(),
                                        QApplication::keyboardModifiers());
                    // accept by default, since enter event was accepted.
                    dme.setDropAction(action);
                    dme.accept();
                    QApplication::sendEvent(widget, &dme);
                    if (sourceAllowsOp && dme.isAccepted()) {
                        dropReply = DOR_DROP;
                        action = dme.dropAction();
                    } else {
                        dropReply = DOR_NODROP;
                    }
                }

                lastDropReply = dropReply;
                lastOpRequest = toPmDragDropOp(action);
            }

            DrgFreeDraginfo(info);

            return MRFROM2SHORT(dropReply, lastOpRequest);
        }
        case DM_DRAGLEAVE: {
            DEBUG(("DM_DRAGLEAVE: hwnd %lX di %p", qmsg.hwnd, qmsg.mp1));

            // Odin32 apps produce incorrect message flow, ignore
            Q_ASSERT(lastDragOverHwnd != 0);
            if (lastDragOverHwnd == 0)
                return 0;

            lastDragOverHwnd = 0;
            dragData.reset();

            if (!widget->acceptDrops())
                return 0;

            QDragLeaveEvent de;
            QApplication::sendEvent(widget, &de);
            return 0;
        }
        case DM_DROP: {
            DEBUG(("DM_DROP: hwnd %lX di %p", qmsg.hwnd, qmsg.mp1));

            // Odin32 apps produce incorrect message flow, ignore
            Q_ASSERT(lastDragOverHwnd != 0);
            if (lastDragOverHwnd == 0)
                return 0;

            // Odin32 apps send DM_DROP even if we replied DOR_NEVERDROP or
            // DOR_NODROP, simulate DM_DRAGLEAVE
            Q_ASSERT(lastDropReply == DM_DROP);
            if (lastDropReply != DM_DROP) {
                WinSendMsg(qmsg.hwnd, DM_DRAGLEAVE, 0, 0);
                return 0;
            }

            lastDragOverHwnd = 0;

            Q_ASSERT(widget->acceptDrops());
            if (!widget->acceptDrops())
                return 0;

            DRAGINFO *info = (DRAGINFO *)qmsg.mp1;
            ok = DrgAccessDraginfo(info);
            Q_ASSERT(ok && info);
            if (!ok || !info)
                return MRFROM2SHORT(DOR_NEVERDROP, 0);

            Q_ASSERT(lastOpRequest == info->usOperation);

            // flip y coordinate
            QPoint pnt(info->xDrop, info->yDrop);
            pnt.setY(QApplication::desktop()->height() - (pnt.y() + 1));
            pnt = widget->mapFromGlobal(pnt);

            QDragManager *manager = QDragManager::self();
            QMimeData *data = manager->source() ? manager->dragPrivate()->data : manager->dropData;

            Qt::DropAction action = toQDragDropAction(lastOpRequest);

            dragData.setDropped(true);

            QDropEvent de(pnt, action, data, QApplication::mouseButtons(),
                          QApplication::keyboardModifiers());
            if (lastDropReply == DOR_DROP)
                de.setDropAction(action);

            QApplication::sendEvent(widget, &de);

            if (lastDropReply == DOR_DROP)
                de.accept();

            dragData.reset(de.isAccepted());

            ULONG targetReply = de.isAccepted() ?
                DMFL_TARGETSUCCESSFUL : DMFL_TARGETFAIL;

            if (de.isAccepted() && de.dropAction() == Qt::TargetMoveAction) {
                // Qt::TargetMoveAction means that the target will move the
                // object (e.g. copy it to itself and delete from the source).
                // In this case, we always send DMFL_TARGETFAIL to the source to
                // prevent it from doing the same on its side.
                targetReply = DMFL_TARGETFAIL;
            }

            // send DM_ENDCONVERSATION for every item
            ULONG itemCount = DrgQueryDragitemCount(info);
            for (ULONG i = 0; i < itemCount; ++ i) {
                PDRAGITEM item = DrgQueryDragitemPtr(info, i);
                Q_ASSERT(item);
                if (!item)
                    continue;
                // it is possible that this item required DM_RENDERPREPARE but
                // returned false in reply to it (so hwndItem may be NULL)
                if (!item->hwndItem)
                    continue;
                DrgSendTransferMsg(item->hwndItem, DM_ENDCONVERSATION,
                                   MPFROMLONG(item->ulItemID),
                                   MPFROMLONG(targetReply));
            }

            DrgDeleteDraginfoStrHandles(info);
            DrgFreeDraginfo(info);

            return 0;
        }
        default:
            break;
    }

    return WinDefWindowProc(qmsg.hwnd, qmsg.msg, qmsg.mp1, qmsg.mp2);
}

//---------------------------------------------------------------------
//                    QDropData
//---------------------------------------------------------------------

bool QDropData::hasFormat_sys(const QString &mimeType) const
{
    Q_ASSERT(d);
    if (d)
        return d->hasFormat_sys(mimeType);
    return false;
}

QStringList QDropData::formats_sys() const
{
    QStringList fmts;
    Q_ASSERT(d);
    if (d)
        fmts = d->formats_sys();
    return fmts;
}

QVariant QDropData::retrieveData_sys(const QString &mimeType, QVariant::Type type) const
{
    QVariant result;
    Q_ASSERT(d);
    if (d)
        result = d->retrieveData_sys(mimeType, type);
    return result;
}

Qt::DropAction QDragManager::drag(QDrag *o)

{
    DEBUG(() << "QDragManager::drag");

    // @todo implement
    return Qt::IgnoreAction;
}

void QDragManager::cancel(bool /* deleteSource */)
{
    // @todo implement
}

void QDragManager::updatePixmap()
{
    // not used in PM implementation
}

bool QDragManager::eventFilter(QObject *, QEvent *)
{
    // not used in PM implementation
    return false;
}

void QDragManager::timerEvent(QTimerEvent*)
{
    // not used in PM implementation
}

void QDragManager::move(const QPoint &)
{
    // not used in PM implementation
}

void QDragManager::drop()
{
    // not used in PM implementation
}

QT_END_NAMESPACE

#endif // QT_NO_DRAGANDDROP
