/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qmime.h"

#include "qimagereader.h"
#include "qimagewriter.h"
#include "qdatastream.h"
#include "qbuffer.h"
#include "qt_os2.h"
#include "qapplication_p.h"
#include "qtextcodec.h"
#include "qregexp.h"
#include "qalgorithms.h"
#include "qmap.h"
#include "qdnd_p.h"
#include "qurl.h"
#include "qvariant.h"
#include "qtextdocument.h"
#include "qdir.h"

#define QMIME_DEBUG

QT_BEGIN_NAMESPACE

class QPMMimeList
{
public:
    QPMMimeList();
    ~QPMMimeList();
    void addMime(QPMMime *mime);
    void removeMime(QPMMime *mime);
    QList<QPMMime*> mimes();

private:
    void init();
    bool initialized;
    QList<QPMMime*> list;
};

Q_GLOBAL_STATIC(QPMMimeList, theMimeList);


/*!
    \class QPMMime
    \brief The QMPMime class maps open-standard MIME to OS/2 PM Clipboard
    formats.
    \ingroup io
    \ingroup draganddrop
    \ingroup misc

    Qt's drag-and-drop and clipboard facilities use the MIME standard.
    On X11, this maps trivially to the Xdnd protocol, but on OS/2
    although some applications use MIME types to describe clipboard
    formats, others use arbitrary non-standardized naming conventions,
    or unnamed built-in formats of the Presentation Manager.

    By instantiating subclasses of QPMMime that provide conversions between OS/2
    PM Clipboard and MIME formats, you can convert proprietary clipboard formats
    to MIME formats.

    Qt has predefined support for the following PM Clipboard formats:

    \table
    \header \o PM Format \o Equivalent MIME type
    \row \o \c CF_TEXT        \o \c text/plain
    \row \o \c CF_BITMAP      \o \c{image/xyz}, where \c xyz is
                                 a \l{QImageWriter::supportedImageFormats()}{Qt image format}
    \endtable

    An example use of this class would be to map the PM Metafile
    clipboard format (\c CF_METAFILE) to and from the MIME type
    \c{image/x-metafile}. This conversion might simply be adding or removing a
    header, or even just passing on the data. See \l{Drag and Drop} for more
    information on choosing and definition MIME types.
*/

/*!
Constructs a new conversion object, adding it to the globally accessed
list of available converters.
*/
QPMMime::QPMMime()
{
    theMimeList()->addMime(this);
}

/*!
Destroys a conversion object, removing it from the global
list of available converters.
*/
QPMMime::~QPMMime()
{
    theMimeList()->removeMime(this);
}

/*!
    Registers the MIME type \a mime, and returns an ID number
    identifying the format on OS/2. Intended to be used by QPMMime
    implementations for registering custom clipboard formats they use.
*/
// static
ULONG QPMMime::registerMimeType(const QString &mime)
{
    ULONG cf = WinAddAtom(WinQuerySystemAtomTable(), mime.toLocal8Bit());
    if (!cf) {
#ifndef QT_NO_DEBUG
        qWarning("QPMMime: WinAddAtom failed with %lX",
                 WinGetLastError(NULLHANDLE));
#endif
        return 0;
    }

    return cf;
}

/*!
    Allocates a block of shared memory of the given size and returns the address
    of this block. This memory block may be then filled with data and returned
    by convertFromMimeData() as the value of the CFI_POINTER type.
*/
// static
ULONG QPMMime::allocateMemory(size_t size)
{
    if (size == 0)
        return 0;

    ULONG data = 0;

    // allocate giveable memory for the array + dword for its size
    APIRET arc = DosAllocSharedMem((PVOID *)&data, NULL, size + 4,
                                   PAG_WRITE  | PAG_COMMIT | OBJ_GIVEABLE);
    if (arc != NO_ERROR) {
#ifndef QT_NO_DEBUG
        qWarning("QPMMime::allocateMemory: DosAllocSharedMem failed with %lu", arc);
#endif
        return 0;
    }

    /// @todo I think we don't need it any more
#if 0
    // get the size rounded to the page boundary (4K)
    ULONG sz = ~0, flags = 0;
    arc = DosQueryMem((PVOID)(data + size - 1), &sz, &flags);
    if (arc != NO_ERROR) {
#ifndef QT_NO_DEBUG
        qWarning("QPMMime::allocateMemory: DosQueryMem failed with %lu", arc);
#endif
        DosFreeMem((PVOID)data);
        return 0;
    }
    sz += (size - 1);
    // store the exact size to the last dword of the last page
    *(((ULONG *)(data + sz)) - 1) = size;
#endif

    return data;
}

/*!
    Frees memory allocated by allocateMemory(). Normally, not used because the
    CFI_POINTER memory blocks are owned by the system after
    convertFromMimeData() returns.
*/
// static
void QPMMime::freeMemory(ULONG addr)
{
    DosFreeMem((PVOID)addr);
}

/*!
    \fn QList<ULONG> QPMMime::formatsForMimeData(const QMimeData *mimeData) const

    Returns a list of ULONG values representing the different OS/2 PM
    clipboard formats that can be provided for the \a mimeData, in order of
    precedence (the most suitable format goes first), or an empty list if
    neither of the mime types provided by \a mimeData is supported by this
    converter.

    All subclasses must reimplement this pure virtual function.
*/

/*!
    \fn bool QPMMime::convertFromMimeData(const QMimeData *mimeData, ULONG format,
                                          ULONG &flags, ULONG *data) const

    Converts the \a mimeData to the specified \a format.

    If \a data is not NULL, a handle to the converted data should then be placed
    in a variable pointed to by \a data and with the necessary flags describing
    the handle returned in the \a flags variable.

    The following flags describing the data type are recognized:

    \table
    \row \o \c CFI_POINTER        \o \a data is a pointer to a block of memory
                                      allocated with QPMMime::allocateMemory()
    \row \o \c CFI_HANDLE         \o \a data is a handle to the appropriate
                                      PM resource
    \endtable

    If \a data is NULL then a delayed conversion is requested by the caller.
    The implementation should return the appropriate flags in the \a flags
    variable and may perform the real data conversion later when this method is
    called again with \a data being non-NULL.

    Return true if the conversion was successful.

    All subclasses must reimplement this pure virtual function.
*/

/*!
    \fn QList<MimeCFPair> QPMMime::mimesForFormats(const QList<ULONG> &formats) const

    Returns a list of mime types that will be created form the specified \a list
    of \a formats, in order of precedence (the most suitable mime type comes
    first), or an empty list if neither of the \a formats is supported by this
    converter. Note that each pair in the returned list consists of the mime
    type name and the corresponding format identifier.

    All subclasses must reimplement this pure virtual function.
*/

/*!
    \fn QVariant QPMMime::convertFromFormat(ULONG format, ULONG flags, ULONG data,
                                            const QString &mimeType,
                                            QVariant::Type preferredType) const

    Returns a QVariant containing the converted from the \a data in the
    specified \a format with the given \a flags to the requested \a mimeType. If
    possible the QVariant should be of the \a preferredType to avoid needless
    conversions.

    All subclasses must reimplement this pure virtual function.
*/

// static
QList<QPMMime::Match> QPMMime::allConvertersFromFormats(const QList<ULONG> &formats)
{
    QList<Match> matches;

    QList<QPMMime*> mimes = theMimeList()->mimes();
    for (int i = mimes.size()-1; i >= 0; --i) {
        QList<MimeCFPair> fmts = mimes[i]->mimesForFormats(formats);
        int priority = 0;
        foreach (MimeCFPair fmt, fmts) {
            ++priority;
            QList<Match>::iterator it = matches.begin();
            for (; it != matches.end(); ++it) {
                Match &match = *it;
                if (match.mime == fmt.first) {
                    // replace if priority is higher, ignore otherwise
                    if (priority < match.priority) {
                        match.converter = mimes[i];
                        match.format = fmt.second;
                        match.priority = priority;
                    }
                    break;
                }
            }
            if (it == matches.end()) {
                matches += Match(mimes[i], fmt.first, fmt.second, priority);
            }
        }
    }

    return matches;
}

// static
QList<QPMMime::Match> QPMMime::allConvertersFromMimeData(const QMimeData *mimeData)
{
    QList<Match> matches;

    QList<QPMMime*> mimes = theMimeList()->mimes();
    for (int i = mimes.size()-1; i >= 0; --i) {
        QList<ULONG> cfs = mimes[i]->formatsForMimeData(mimeData);
        int priority = 0;
        foreach (ULONG cf, cfs) {
            ++priority;
            QList<Match>::iterator it = matches.begin();
            for (; it != matches.end(); ++it) {
                Match &match = *it;
                if (match.format == cf) {
                    // replace if priority is higher, ignore otherwise
                    if (priority < match.priority) {
                        match.converter = mimes[i];
                        match.priority = priority;
                    }
                    break;
                }
            }
            if (it == matches.end()) {
                matches += Match(mimes[i], cf, priority);
            }
        }
    }

    return matches;
}

////////////////////////////////////////////////////////////////////////////////

class QPMMimeText : public QPMMime
{
public:
    QPMMimeText();

    QList<ULONG> formatsForMimeData(const QMimeData *mimeData) const;
    bool convertFromMimeData(const QMimeData *mimeData, ULONG format,
                             ULONG &flags, ULONG *data) const;

    QList<MimeCFPair> mimesForFormats(const QList<ULONG> &formats) const;
    QVariant convertFromFormat(ULONG format, ULONG flags, ULONG data,
                               const QString &mimeType,
                               QVariant::Type preferredType) const;

    const ULONG CF_TextUnicode;
};

QPMMimeText::QPMMimeText()
    // "text/unicode" is what Mozilla uses to for unicode, so Qt apps will
    // be able to interchange unicode text with Mozilla apps
    : CF_TextUnicode (registerMimeType(QLatin1String("text/unicode")))
{
}

QList<ULONG> QPMMimeText::formatsForMimeData(const QMimeData *mimeData) const
{
    QList<ULONG> cfs;
    if (mimeData->hasText())
        cfs << CF_TEXT << CF_TextUnicode;
    return cfs;
}

// text/plain is defined as using CRLF, but so many programs don't,
// and programmers just look for '\n' in strings.
// OS/2 really needs CRLF, so we ensure it here.
bool QPMMimeText::convertFromMimeData(const QMimeData *mimeData, ULONG format,
                                      ULONG &flags, ULONG *data) const
{
    if (!mimeData->hasText())
        return false;

    flags = CFI_POINTER;

    if (data == NULL)
        return true; // delayed rendering, nothing to do

    QByteArray r;

    if (format == CF_TEXT) {
        QByteArray str = mimeData->text().toLocal8Bit();
        // Anticipate required space for CRLFs at 1/40
        int maxsize = str.size()+str.size()/40+1;
        r.fill('\0', maxsize);
        char *o = r.data();
        const char *d = str.data();
        const int s = str.size();
        bool cr = false;
        int j = 0;
        for (int i = 0; i < s; i++) {
            char c = d[i];
            if (c == '\r')
                cr = true;
            else {
                if (c == '\n') {
                    if (!cr)
                        o[j++] = '\r';
                }
                cr = false;
            }
            o[j++] = c;
            if (j+1 >= maxsize) {
                maxsize += maxsize/4;
                r.resize(maxsize);
                o = r.data();
            }
        }
        if (j < r.size())
            o[j] = '\0';
    } else if (format == CF_TextUnicode) {
        QString str = mimeData->text();
        const QChar *u = str.unicode();
        QString res;
        const int s = str.length();
        int maxsize = s + s/40 + 3;
        res.resize(maxsize);
        int ri = 0;
        bool cr = false;
        for (int i = 0; i < s; ++i) {
            if (*u == QLatin1Char('\r'))
                cr = true;
            else {
                if (*u == QLatin1Char('\n') && !cr)
                    res[ri++] = QLatin1Char('\r');
                cr = false;
            }
            res[ri++] = *u;
            if (ri+3 >= maxsize) {
                maxsize += maxsize/4;
                res.resize(maxsize);
            }
            ++u;
        }
        res.truncate(ri);
        const int byteLength = res.length()*2;
        r.fill('\0', byteLength + 2);
        memcpy(r.data(), res.unicode(), byteLength);
        r[byteLength] = 0;
        r[byteLength+1] = 0;
    } else{
        return false;
    }

    *data = QPMMime::allocateMemory(r.size());
    if (!*data)
        return false;

    memcpy((void *)*data, r.data(), r.size());
    return true;
}

QList<QPMMime::MimeCFPair> QPMMimeText::mimesForFormats(const QList<ULONG> &formats) const
{
    QList<MimeCFPair> mimes;
    foreach(ULONG cf, formats) {
        // prefer unicode over local8Bit
        if (cf == CF_TextUnicode)
            mimes.prepend(qMakePair(QString(QLatin1String("text/plain")), cf));
        if (cf == CF_TEXT)
            mimes.append(qMakePair(QString(QLatin1String("text/plain")), cf));
    }
    return mimes;
}

QVariant QPMMimeText::convertFromFormat(ULONG format, ULONG flags, ULONG data,
                                        const QString &mimeType,
                                        QVariant::Type preferredType) const
{
    QVariant ret;

    // @todo why is it startsWith? the rest of the mime specification (encoding,
    // etc) isn't taken into account... Anyway, copied the logic from Windows.
    if (!mimeType.startsWith("text/plain"))
        return ret;
    if (!(flags & CFI_POINTER) || !data)
        return ret;

    QString str;

    if (format == CF_TEXT) {
        const char *d = (const char *)data;
        QByteArray r("");
        if (*d) {
            const int s = qstrlen(d);
            r.fill('\0', s);
            char *o = r.data();
            int j = 0;
            for (int i = 0; i < s; i++) {
                char c = d[i];
                if (c != '\r')
                    o[j++] = c;
            }
        }
        str = QString::fromLocal8Bit(r);
    } else if (format == CF_TextUnicode) {
        str = QString::fromUtf16((const unsigned short *)data);
        str.replace(QLatin1String("\r\n"), QLatin1String("\n"));
    }

    if (preferredType == QVariant::String)
        ret = str;
    else
        ret = str.toUtf8();

    return ret;
}

////////////////////////////////////////////////////////////////////////////////

QPMMimeList::QPMMimeList()
    : initialized(false)
{
}

QPMMimeList::~QPMMimeList()
{
    while (list.size())
        delete list.first();
}


void QPMMimeList::init()
{
    if (!initialized) {
        initialized = true;
        new QPMMimeText;
    }
}

void QPMMimeList::addMime(QPMMime *mime)
{
    init();
    list.append(mime);
}

void QPMMimeList::removeMime(QPMMime *mime)
{
    init();
    list.removeAll(mime);
}

QList<QPMMime*> QPMMimeList::mimes()
{
    init();
    return list;
}

QT_END_NAMESPACE
