/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qclipboard.h"

#ifndef QT_NO_CLIPBOARD

#include "qapplication.h"
#include "qapplication_p.h"
#include "qeventloop.h"
#include "qwidget.h"
#include "qevent.h"
#include "qmime.h"
#include "qdnd_p.h"
#include "qclipboard_p.h"

#include "qt_os2.h"
#include "private/qpmobjectwindow_pm_p.h"

#define QCLIPBOARD_DEBUG

#ifdef QCLIPBOARD_DEBUG
#include "qdebug.h"
#endif

QT_BEGIN_NAMESPACE

////////////////////////////////////////////////////////////////////////////////

class QClipboardWatcher : public QInternalMimeData
{
public:
    QClipboardWatcher() : isDirty(true) {}

    bool hasFormat_sys(const QString &mimetype) const;
    QStringList formats_sys() const;
    QVariant retrieveData_sys(const QString &mimetype, QVariant::Type preferredType) const;

private:

    void peekData() const;

    mutable QList<ULONG> formats;
    mutable QList<QPMMime::Match> matches;
    mutable bool isDirty;
};

void QClipboardWatcher::peekData() const
{
    if (!isDirty)
        return;

    if (!WinOpenClipbrd(NULLHANDLE)) {
#ifndef QT_NO_DEBUG
        qWarning("QClipboardWatcher::peekData: WinOpenClipbrd "
                 "failed with 0x%lX", WinGetLastError(NULLHANDLE));
#endif
        return;
    }

    formats.clear();
    ULONG cf = 0;
    while ((cf = WinEnumClipbrdFmts(NULLHANDLE, cf)))
        formats << cf;

    WinCloseClipbrd(NULLHANDLE);

    matches = QPMMime::allConvertersFromFormats(formats);
    isDirty = false;
}

bool QClipboardWatcher::hasFormat_sys(const QString &mime) const
{
    peekData();
    if (isDirty)
        return false; // peekData() failed

    foreach (QPMMime::Match match, matches)
        if (match.mime == mime)
            return true;

    return false;
}

QStringList QClipboardWatcher::formats_sys() const
{
    QStringList fmts;

    peekData();
    if (isDirty)
        return fmts; // peekData() failed

    foreach (QPMMime::Match match, matches)
        fmts << match.mime;

    return fmts;
}

QVariant QClipboardWatcher::retrieveData_sys(const QString &mime,
                                             QVariant::Type type) const
{
    QVariant result;

    peekData();
    if (isDirty)
        return result; // peekData() failed

    foreach (QPMMime::Match match, matches) {
        if (match.mime == mime) {
            ULONG flags;
            if (WinQueryClipbrdFmtInfo(NULLHANDLE, match.format, &flags)) {
                ULONG data = WinQueryClipbrdData(NULLHANDLE, match.format);
                result = match.converter->convertFromFormat(match.format, flags,
                                                            data, match.mime, type);
            }
            return result;
        }
    }

    return result;
}

////////////////////////////////////////////////////////////////////////////////

class QClipboardData : public QPMObjectWindow
{
public:
    QClipboardData();
    ~QClipboardData();

    void setSource(QMimeData *s);

    void setAsClipboardViewer();
    bool ownsClipboard();
    void renderAllFormats(bool isDelayed);

    static QClipboardData *instance()
    {
        if (instancePtr == 0) {
            instancePtr = new QClipboardData;
        }
        Q_ASSERT(instancePtr);
        return instancePtr;
    }

    static void deleteInstance()
    {
        delete instancePtr;
        instancePtr = 0;
    }

private:
    bool openClipboard();
    void closeClipboard();
    bool setClipboard(QPMMime *converter, ULONG format, bool isDelayed);
    void renderFormat(ULONG format);

    MRESULT message(ULONG msg, MPARAM mp1, MPARAM mp2);

    QMimeData *src;
    QList<QPMMime::Match> matches;
    HWND prevClipboardViewer;

    bool ignore_WM_DESTROYCLIPBOARD;

    static QClipboardData *instancePtr;
};

// static
QClipboardData *QClipboardData::instancePtr = 0;

QClipboardData::QClipboardData()
    : src(0), prevClipboardViewer(NULLHANDLE)
    , ignore_WM_DESTROYCLIPBOARD(false)
{
}

QClipboardData::~QClipboardData()
{
    renderAllFormats(false);
    setSource(0);

    // make sure we are not the clipboard viewer any more
    if (hwnd() == WinQueryClipbrdViewer(NULLHANDLE))
        WinSetClipbrdViewer(NULLHANDLE, prevClipboardViewer);
}

void QClipboardData::setSource(QMimeData *s)
{
    if (s == src)
        return;
    delete src;
    src = s;

    // build the list of all mime <-> cf matches
    matches.clear();
    if (src)
        matches = QPMMime::allConvertersFromMimeData(src);
}

void QClipboardData::setAsClipboardViewer()
{
    HWND clipboardViewer = WinQueryClipbrdViewer(NULLHANDLE);
    if (hwnd() != clipboardViewer) {
        prevClipboardViewer = clipboardViewer;
        BOOL ok = WinSetClipbrdViewer(NULLHANDLE, hwnd());
        Q_UNUSED(ok);
#ifndef QT_NO_DEBUG
        if (!ok)
            qWarning("QClipboardData::setAsClipboardViewer: WinSetClipbrdViewer "
                     " failed with 0x%lX", WinGetLastError(NULLHANDLE));
#endif
    }
}

bool QClipboardData::ownsClipboard()
{
    return src && hwnd() == WinQueryClipbrdOwner(NULLHANDLE);
}

bool QClipboardData::openClipboard()
{
    if (!WinOpenClipbrd(NULLHANDLE)) {
#ifndef QT_NO_DEBUG
        qWarning("QClipboardData::openClipboard: WinOpenClipbrd "
                 "failed with 0x%lX", WinGetLastError(NULLHANDLE));
#endif
        return false;
    }
    return true;
}

void QClipboardData::closeClipboard()
{
    WinCloseClipbrd(NULLHANDLE);
}

bool QClipboardData::setClipboard(QPMMime *converter, ULONG format,
                                  bool isDelayed)
{
    Q_ASSERT(src);
    if (!src)
        return false;

    bool ok;
    ULONG flags = 0, data = 0;

    if (isDelayed) {
        // setup delayed rendering of clipboard data
        ok = converter->convertFromMimeData(src, format, flags, 0);
        if (ok) {
            WinSetClipbrdOwner(NULLHANDLE, hwnd());
            WinSetClipbrdData(NULLHANDLE, 0, format, flags);
        }
    } else {
        // render now
        ok = converter->convertFromMimeData(src, format, flags, &data);
        if (ok)
            WinSetClipbrdData(NULLHANDLE, data, format, flags);
    }
#ifdef QCLIPBOARD_DEBUG
    qDebug("QClipboardData::setClipboard: convert to CF 0x%lX, flags 0x%lX,"
           "data 0x%lX, delayed %d, ok %d", format, flags, data, isDelayed, ok);
#endif

    return ok;
}

void QClipboardData::renderFormat(ULONG format)
{
#ifdef QCLIPBOARD_DEBUG
    qDebug("QClipboardData::renderFormat: CF 0x%lX", format);
#endif

    if (!src)
        return;

    if (!openClipboard())
        return;

    foreach(QPMMime::Match match, matches) {
        if (match.format == format) {
            setClipboard(match.converter, match.format, false);
            break;
        }
    }

    closeClipboard();
}

void QClipboardData::renderAllFormats(bool isDelayed)
{
#ifdef QCLIPBOARD_DEBUG
    qDebug() << "QClipboardData::renderAllFormats: isDelayed" << isDelayed;
#endif

    if (!openClipboard())
        return;

    // delete the clipboard contents before we render everything to make sure
    // nothing is left there from another owner
    ignore_WM_DESTROYCLIPBOARD = true;
    BOOL ok = WinEmptyClipbrd(NULLHANDLE);
    ignore_WM_DESTROYCLIPBOARD = false;
    if (!ok) {
#ifndef QT_NO_DEBUG
        qWarning("QClipboardData::renderAllFormats: WinEmptyClipbrd "
                 "failed with 0x%lX", WinGetLastError(NULLHANDLE));
#endif
        return;
    }

    if (src) {
#ifdef QCLIPBOARD_DEBUG
        qDebug() << "QClipboardData::renderAllFormats: mimes" << src->formats();
#endif
        foreach(QPMMime::Match match, matches)
            setClipboard(match.converter, match.format, isDelayed);
    }

    closeClipboard();
}

MRESULT QClipboardData::message(ULONG msg, MPARAM mp1, MPARAM mp2)
{
#ifdef QCLIPBOARD_DEBUG
    qDebug("QClipboardData::message: msg %08lX, mp1 %p, mp2 %p",
           msg, mp1, mp2);
#endif

    switch (msg) {

        case WM_DRAWCLIPBOARD: {
            // ask QClipboard to emit changed() signals
            QClipboardEvent e(reinterpret_cast<QEventPrivate *>(1));
            qt_sendSpontaneousEvent(QApplication::clipboard(), &e);

            if (hwnd() != WinQueryClipbrdOwner(NULLHANDLE) && src) {
                // we no longer the clipboard, clean up the clipboard object
                setSource(0);
            }
            // PM doesn't inform the previous clipboard viewer if another
            // app changes it (nor does it support viewer chains in some other
            // way). The best we can do is to propagate the message to the
            // previous clipboard viewer ourselves (though there is no guarantee
            // that all non-Qt apps will do the same).
            if (prevClipboardViewer) {
                // propagate the message to the previous clipboard viewer
                BOOL ok = WinPostMsg(prevClipboardViewer, msg, mp1, mp2);
                if (!ok)
                    prevClipboardViewer = NULLHANDLE;
            }
            break;
        }

        case WM_DESTROYCLIPBOARD:
            if (!ignore_WM_DESTROYCLIPBOARD)
                setSource(0);
            break;

        case WM_RENDERFMT:
            renderFormat((ULONG)mp1);
            break;

        case WM_RENDERALLFMTS:
            renderAllFormats(false);
            break;

        default:
            break;
    }

    return (MRESULT)TRUE;
}

////////////////////////////////////////////////////////////////////////////////

QClipboard::~QClipboard()
{
    QClipboardData::deleteInstance();
}

void QClipboard::setMimeData(QMimeData *src, Mode mode)
{
#ifdef QCLIPBOARD_DEBUG
    qDebug() << "QClipboard::setMimeData: src" << src << "mode" << mode;
#endif

    if (mode != Clipboard) {
        delete src;
        return;
    }

    QClipboardData *d = QClipboardData::instance();
    d->setSource(src);

    if (!src)
        return; // nothing to do

    // use delayed rendering only if the application runs the event loop
    bool runsEventLoop = d_func()->threadData->loopLevel != 0;

    d->renderAllFormats(runsEventLoop);
}

void QClipboard::clear(Mode mode)
{
    setMimeData(0, Clipboard);
}

bool QClipboard::event(QEvent *e)
{
    if (e->type() != QEvent::Clipboard)
        return QObject::event(e);

    if (!((QClipboardEvent*)e)->data()) {
        // this is sent by QApplication to render all formats at app shut down
        QClipboardData::instance()->renderAllFormats(false);
    } else {
        // this is sent by QClipboardData to notify about clipboard data change
        emitChanged(QClipboard::Clipboard);
    }

    return true;
}

void QClipboard::connectNotify(const char *signal)
{
    if (qstrcmp(signal, SIGNAL(dataChanged())) == 0) {
        // ensure we are up and running (by instantiating QClipboardData and
        // setting it as the clipboard viewer to receive notifications on
        // clipboard contents chages) but block signals so the dataChange signal
        // is not emitted while being connected to.
        bool blocked = blockSignals(true);
        QClipboardData::instance()->setAsClipboardViewer();
        blockSignals(blocked);
    }
}

const QMimeData *QClipboard::mimeData(Mode mode) const
{
    // @todo implement
    return 0;
}

bool QClipboard::supportsMode(Mode mode) const
{
    return (mode == Clipboard);
}

bool QClipboard::ownsMode(Mode mode) const
{
    if (mode == Clipboard) {
        return QClipboardData::instance()->ownsClipboard();
    }
    return false;
}

void QClipboard::ownerDestroyed()
{
    // not used
}

QT_END_NAMESPACE

#endif // QT_NO_CLIPBOARD
