/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qclipboard.h"

#ifndef QT_NO_CLIPBOARD

#include "qapplication.h"
#include "qapplication_p.h"
#include "qeventloop.h"
#include "qwidget.h"
#include "qevent.h"
#include "qmime.h"
#include "qdnd_p.h"

#include "qt_os2.h"
#include "private/qpmobjectwindow_pm_p.h"

#define QCLIPBOARD_DEBUG

#ifdef QCLIPBOARD_DEBUG
#include "qdebug.h"
#endif

QT_BEGIN_NAMESPACE

////////////////////////////////////////////////////////////////////////////////

class QClipboardWatcher : public QInternalMimeData
{
public:
    QClipboardWatcher() : isDirty(true) {}

    bool hasFormat_sys(const QString &mimetype) const;
    QStringList formats_sys() const;
    QVariant retrieveData_sys(const QString &mimetype, QVariant::Type preferredType) const;

private:

    void peekData() const;

    mutable QList<ULONG> formats;
    mutable QList<QPMMime::Match> matches;
    mutable bool isDirty;
};

void QClipboardWatcher::peekData() const
{
    if (!isDirty)
        return;

    if (!WinOpenClipbrd(NULLHANDLE)) {
#ifndef QT_NO_DEBUG
        qWarning("QClipboardWatcher: WinOpenClipbrd failed with 0x%lX",
                 WinGetLastError(NULLHANDLE));
#endif
        return;
    }

    formats.clear();
    ULONG cf = 0;
    while ((cf = WinEnumClipbrdFmts(NULLHANDLE, cf)))
        formats << cf;

    WinCloseClipbrd(NULLHANDLE);

    matches = QPMMime::allConvertersFromFormats(formats);
    isDirty = false;
}

bool QClipboardWatcher::hasFormat_sys(const QString &mime) const
{
    peekData();
    if (isDirty)
        return false; // peekData() failed

    foreach (QPMMime::Match match, matches)
        if (match.mime == mime)
            return true;

    return false;
}

QStringList QClipboardWatcher::formats_sys() const
{
    QStringList fmts;

    peekData();
    if (isDirty)
        return fmts; // peekData() failed

    foreach (QPMMime::Match match, matches)
        fmts << match.mime;

    return fmts;
}

QVariant QClipboardWatcher::retrieveData_sys(const QString &mime,
                                             QVariant::Type type) const
{
    QVariant result;

    peekData();
    if (isDirty)
        return result; // peekData() failed

    foreach (QPMMime::Match match, matches) {
        if (match.mime == mime) {
            ULONG flags;
            if (WinQueryClipbrdFmtInfo(NULLHANDLE, match.format, &flags)) {
                ULONG data = WinQueryClipbrdData(NULLHANDLE, match.format);
                result = match.converter->convertFromFormat(match.format, flags,
                                                            data, match.mime, type);
            }
            return result;
        }
    }

    return result;
}

////////////////////////////////////////////////////////////////////////////////

class QClipboardData : public QPMObjectWindow
{
public:
    QClipboardData();
    ~QClipboardData();

    void setSource(QMimeData *s)
    {
        if (s == src)
            return;
        delete src;
        src = s;
    }

    QMimeData *source()
    {
        return src;
    }

    bool setClipboard(QPMMime *converter, ULONG format, bool isDelayed);

    MRESULT message(ULONG msg, MPARAM mp1, MPARAM mp2);

private:
    QMimeData *src;
};

QClipboardData::QClipboardData() : src(0)
{
}

QClipboardData::~QClipboardData()
{
    delete src;
}

bool QClipboardData::setClipboard(QPMMime *converter, ULONG format,
                                  bool isDelayed)
{
    Q_ASSERT(src);
    if (!src)
        return false;

    bool ok;
    ULONG flags = 0, data = 0;

    if (isDelayed) {
        // setup delayed rendering of clipboard data
        ok = converter->convertFromMimeData(src, format, flags, 0);
        if (ok) {
            WinSetClipbrdOwner(NULLHANDLE, hwnd());
            WinSetClipbrdData(NULLHANDLE, 0, format, flags);
        }
    } else {
        // render now
        ok = converter->convertFromMimeData(src, format, flags, &data);
        if (ok)
            WinSetClipbrdData(NULLHANDLE, data, format, flags);
    }
#ifdef QCLIPBOARD_DEBUG
    qDebug("QClipboardData::setClipboard: convert to CF 0x%lX, flags 0x%lX,"
           "data 0x%lX, ok %d", format, flags, data, ok);
#endif

    return ok;
}

MRESULT QClipboardData::message(ULONG msg, MPARAM mp1, MPARAM mp2)
{
    return 0;
}

static QClipboardData *ptrClipboardData = 0;

static QClipboardData *clipboardData()
{
    if (ptrClipboardData == 0) {
        ptrClipboardData = new QClipboardData;
    }
    return ptrClipboardData;
}

static void cleanupClipboardData()
{
    delete ptrClipboardData;
    ptrClipboardData = 0;
}

////////////////////////////////////////////////////////////////////////////////

static bool ignore_WM_DESTROYCLIPBOARD = FALSE;

QClipboard::~QClipboard()
{
    cleanupClipboardData();
}

void QClipboard::setMimeData(QMimeData *src, Mode mode)
{
    if (mode != Clipboard) {
        delete src;
        return;
    }

    if (!WinOpenClipbrd(NULLHANDLE)) {
#ifndef QT_NO_DEBUG
        qWarning("QClipboard::setMimeData: WinOpenClipbrd failed with 0x%lX",
                 WinGetLastError(NULLHANDLE));
#endif
        delete src;
        return;
    }

    QClipboardData *d = clipboardData();
    d->setSource(src);

    ignore_WM_DESTROYCLIPBOARD = TRUE;
    BOOL ok = WinEmptyClipbrd(NULLHANDLE);
    ignore_WM_DESTROYCLIPBOARD = FALSE;
#ifndef QT_NO_DEBUG
    if (!ok)
        qWarning("QClipboard::setMimeData: WinEmptyClipbrd failed with 0x%lX",
                 WinGetLastError(NULLHANDLE));
#else
    Q_UNUSED(ok);
#endif

    if (!src)
        return; // nothing to do

    bool runsEventLoop = QCoreApplication::instance() &&
                         QCoreApplication::instance()->d_func()->in_exec;

    QStringList formats = src->formats();
    foreach(QString mime, formats) {
#ifdef QCLIPBOARD_DEBUG
        qDebug() << "QClipboard::setMimeData: src mime" << mime;
#endif
        QList<QPMMime::Match> matches = QPMMime::allConvertersFromMimeData(src);
        foreach(QPMMime::Match match, matches) {
            d->setClipboard(match.converter, match.format, !runsEventLoop);
        }
    }

    WinCloseClipbrd(NULLHANDLE);
}

void QClipboard::clear(Mode mode)
{
    // @todo implement
}

bool QClipboard::event(QEvent *e)
{
    // @todo implement
    return false;
}

void QClipboard::connectNotify(const char *signal)
{
    // @todo implement
}

const QMimeData *QClipboard::mimeData(Mode mode) const
{
    // @todo implement
    return 0;
}

bool QClipboard::supportsMode(Mode mode) const
{
    return (mode == Clipboard);
}

bool QClipboard::ownsMode(Mode mode) const
{
    // @todo implement
    return false;
}

void QClipboard::ownerDestroyed()
{
}

QT_END_NAMESPACE

#endif // QT_NO_CLIPBOARD
