/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qmime.h"

#include "qimagereader.h"
#include "qimagewriter.h"
#include "qdatastream.h"
#include "qbuffer.h"
#include "qt_os2.h"
#include "qapplication_p.h"
#include "qtextcodec.h"
#include "qregexp.h"
#include "qalgorithms.h"
#include "qmap.h"
#include "qdnd_p.h"
#include "qurl.h"
#include "qvariant.h"
#include "qtextdocument.h"
#include "qdir.h"

#define QMIME_DEBUG

QT_BEGIN_NAMESPACE

class QPMMimeList
{
public:
    QPMMimeList();
    ~QPMMimeList();
    void addMime(QPMMime *mime);
    void removeMime(QPMMime *mime);
    QList<QPMMime*> mimes();

private:
    void init();
    bool initialized;
    QList<QPMMime*> list;
};

Q_GLOBAL_STATIC(QPMMimeList, theMimeList);


/*!
    \class QPMMime
    \brief The QMPMime class maps open-standard MIME to OS/2 PM Clipboard
    formats.
    \ingroup io
    \ingroup draganddrop
    \ingroup misc

    Qt's drag-and-drop and clipboard facilities use the MIME standard.
    On X11, this maps trivially to the Xdnd protocol, but on OS/2
    although some applications use MIME types to describe clipboard
    formats, others use arbitrary non-standardized naming conventions,
    or unnamed built-in formats of the Presentation Manager.

    By instantiating subclasses of QPMMime that provide conversions between OS/2
    PM Clipboard and MIME formats, you can convert proprietary clipboard formats
    to MIME formats.

    Qt has predefined support for the following PM Clipboard formats:

    \table
    \header \o PM Format \o Equivalent MIME type
    \row \o \c CF_TEXT        \o \c text/plain
    \row \o \c CF_BITMAP      \o \c{image/xyz}, where \c xyz is
                                 a \l{QImageWriter::supportedImageFormats()}{Qt image format}
    \endtable

    An example use of this class would be to map the PM Metafile
    clipboard format (\c CF_METAFILE) to and from the MIME type
    \c{image/x-metafile}. This conversion might simply be adding or removing a
    header, or even just passing on the data. See \l{Drag and Drop} for more
    information on choosing and definition MIME types.

    You can check if a MIME type is convertible using canConvertFromMime() and
    can perform conversions with convertToMime() and convertFromMime().
*/

/*!
Constructs a new conversion object, adding it to the globally accessed
list of available converters.
*/
QPMMime::QPMMime()
{
    theMimeList()->addMime(this);
}

/*!
Destroys a conversion object, removing it from the global
list of available converters.
*/
QPMMime::~QPMMime()
{
    theMimeList()->removeMime(this);
}

/*!
    Registers the MIME type \a mime, and returns an ID number
    identifying the format on OS/2.
*/
ULONG QPMMime::registerMimeType(const QString &mime)
{
    QString atom = QLatin1String("mime:") + mime;

    ULONG cf = WinAddAtom(WinQuerySystemAtomTable(), atom.toLocal8Bit());
    if (!cf) {
#ifndef QT_NO_DEBUG
        qWarning("QPMMime: WinAddAtom failed with %lX",
                 WinGetLastError(NULLHANDLE));
#endif
        return 0;
    }

    return cf;
}

/*!
    \fn bool QPMMime::canConvertFromMime(ULONG format, const QMimeData *mimeData) const

    Returns true if the converter can convert from the \a mimeData to
    the specified \a format.

    All subclasses must reimplement this pure virtual function.
*/

/*!
    \fn bool QPMMime::convertFromMime(ULONG format, const QMimeData *mimeData,
                                      ULONG &flags, ULONG &data) const

    Convert the \a mimeData to the specified \a format.
    The converted data should then be placed in the \a data variable with the
    necessary flags returned in the \a flags variable.

    Return true if the conversion was successful.

    All subclasses must reimplement this pure virtual function.
*/

/*!
    \fn QVector<ULONG> QPMMime::formatsForMime(const QString &mimeType, const QMimeData *mimeData) const

    Returns a QVector of ULONG values representing the different OS/2 PM clipboard
    formats that can be provided for the \a mimeType from the \a mimeData.

    All subclasses must reimplement this pure virtual function.
*/

/*!
    \fn bool QPMMime::canConvertToMime(const QString &mimeType, ULONG format) const

    Returns true if the converter can convert to the \a mimeType from
    the specified \a format.

    All subclasses must reimplement this pure virtual function.
*/

/*!
    \fn QVariant QPMMime::convertToMime(const QString &mimeType, QVariant::Type preferredType,
                                        ULONG format, ULONG flags, ULONG data) const

    Returns a QVariant containing the converted data for \a mimeType from the
    \a data in the specified \a format and \a flags. If possible the QVariant
    should be of the \a preferredType to avoid needless conversions.

    All subclasses must reimplement this pure virtual function.
*/

/*!
    \fn QString QPMMime::mimeForFormat(ULONG format) const

    Returns the mime type that will be created form the specified \a format, or an
    empty string if this converter does not support \a format.

    All subclasses must reimplement this pure virtual function.
*/

// static
QPMMime *QPMMime::converterToMime(const QString &mimeType, ULONG format)
{
    QList<QPMMime*> mimes = theMimeList()->mimes();
    for (int i = mimes.size()-1; i >= 0; --i) {
        if (mimes.at(i)->canConvertToMime(mimeType, format))
            return mimes.at(i);
    }
    return 0;
}

// static
QStringList QPMMime::allMimesForFormats(const QVector<ULONG> &formats)
{
    QList<QPMMime*> mimes = theMimeList()->mimes();
    QStringList fmts;

    foreach(ULONG cf, formats) {
#ifdef QMIME_DEBUG
        HATOMTBL tbl = WinQuerySystemAtomTable();
        ULONG len = WinQueryAtomLength(tbl, cf);
        QByteArray atom(len, 0);
        WinQueryAtomName(tbl, cf, atom.data(), atom.size() + 1 /* '\0' */);
        qDebug("QPMMime::allMimesForFormats() CF %lu (%s)", cf, atom.constData());
#endif
        for (int i = mimes.size()-1; i >= 0; --i) {
            QString format = mimes.at(i)->mimeForFormat(cf);
            if (!format.isEmpty() && !fmts.contains(format)) {
                fmts += format;
            }
        }
    }

    return fmts;
}

// static
QPMMime *QPMMime::converterFromMime(ULONG format, const QMimeData *mimeData)
{
    QList<QPMMime*> mimes = theMimeList()->mimes();
    for (int i = mimes.size()-1; i >= 0; --i) {
        if (mimes.at(i)->canConvertFromMime(format, mimeData))
            return mimes.at(i);
    }
    return 0;
}

// static
QVector<ULONG> QPMMime::allFormatsForMime(const QMimeData *mimeData)
{
    QList<QPMMime*> mimes = theMimeList()->mimes();
    QVector<ULONG> cfs(4);
    QStringList formats = QInternalMimeData::formatsHelper(mimeData);
    for (int f = 0; f < formats.size(); ++f) {
        for (int i = mimes.size()-1; i >= 0; --i)
            cfs += mimes.at(i)->formatsForMime(formats.at(f), mimeData);
    }
    return cfs;
}

////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////

QPMMimeList::QPMMimeList()
    : initialized(false)
{
}

QPMMimeList::~QPMMimeList()
{
    while (list.size())
        delete list.first();
}


void QPMMimeList::init()
{
    if (!initialized) {
        initialized = true;
        // @todo new QPMMimeXXX;
    }
}

void QPMMimeList::addMime(QPMMime *mime)
{
    init();
    list.append(mime);
}

void QPMMimeList::removeMime(QPMMime *mime)
{
    init();
    list.removeAll(mime);
}

QList<QPMMime*> QPMMimeList::mimes()
{
    init();
    return list;
}

QT_END_NAMESPACE
