/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qabstractfileengine.h"

#include "qfontengine_pm_p.h"

#include "qsettings.h"
#include "qfileinfo.h"
#include "qdatetime.h"
#include "qhash.h"

#include <ft2build.h>
#include FT_FREETYPE_H
#include FT_TYPES_H
#include FT_TRUETYPE_TABLES_H
#include FT_LCD_FILTER_H

QT_BEGIN_NAMESPACE

struct FaceData
{
    int index;
    QString familyName;
    QtFontStyle::Key styleKey;
    QList<QFontDatabase::WritingSystem> systems;
    bool fixedPitch;
    bool smoothScalable;
    QList<unsigned short> pixelSizes;
};

static QDataStream &operator<<(QDataStream &data, const QFontDatabase::WritingSystem &ws)
{
    data << (int)ws;
    return data;
}

static QDataStream &operator>>(QDataStream &data, QFontDatabase::WritingSystem &ws)
{
    data >> (int&)ws;
    return data;
}

static QDataStream &operator<<(QDataStream &data, const FaceData &cached)
{
    data << cached.familyName;
    data << cached.styleKey.style << cached.styleKey.weight
         << cached.styleKey.stretch;
    data << cached.systems;
    data << cached.fixedPitch << cached.smoothScalable;
    data << cached.pixelSizes;
    return data;
}

static QDataStream &operator>>(QDataStream &data, FaceData &cached)
{
    data >> cached.familyName;
    uint style;
    int weight, stretch;
    data >> style; cached.styleKey.style = style;
    data >> weight; cached.styleKey.weight = weight;
    data >> stretch; cached.styleKey.stretch = stretch;
    data >> cached.systems;
    data >> cached.fixedPitch >> cached.smoothScalable;
    data >> cached.pixelSizes;
    return data;
}

struct FileData
{
    FileData() : seen(false) {}
    FileData(const QFileInfo &fi, bool s) : fileInfo(fi), seen(s) {}

    QFileInfo fileInfo;
    bool seen;
};

typedef QHash<QString, FileData> FontFileHash;
static FontFileHash knownFontFiles;
static bool knownFontFilesInitialized = false;

static void populateDatabase(const QString& fam)
{
    QFontDatabasePrivate *db = privateDb();
    if (!db)
        return;

    QtFontFamily *family = 0;
    if(!fam.isEmpty()) {
        family = db->family(fam);
        if(family)
            return;
    } else if (db->count) {
        return;
    }

    // we don't recognize foundries on OS/2, use an empty one
    const QString foundryName;

#ifdef QFONTDATABASE_DEBUG
    QTime timer;
    timer.start();
#endif

    QSettings fontCache(QSettings::UserScope, QLatin1String("Trolltech"));
    fontCache.beginGroup(QLatin1String("Qt/Fonts/Cache 1.0"));

    if (!knownFontFilesInitialized) {
        // get the initial list of know font files from the cache (necessary to
        // detect deleted font files)
        knownFontFilesInitialized = true;
        QStringList files = fontCache.childGroups();
        foreach(QString file, files) {
            file.replace("|", "/");
            knownFontFiles.insert(file, FileData());
            // note that QFileInfo is empty so the file will be considered as
            // NEW which is necessary for the font to get into the database
        }
    } else {
        // reset the 'seen' flag
        for (FontFileHash::iterator it = knownFontFiles.begin();
             it != knownFontFiles.end(); ++it)
            it.value().seen = false;
    }

    QList<QFileInfo> fontFiles;

    ULONG bufSize = 0;
    BOOL ok = PrfQueryProfileSize(HINI_USERPROFILE, "PM_Fonts", 0, &bufSize);
    Q_ASSERT(ok);
    if (ok) {
        char *buffer = new char[bufSize + 1 /*terminating NULL*/];
        Q_ASSERT(buffer);
        if (buffer) {
            ULONG bufLen = PrfQueryProfileString(HINI_USERPROFILE, "PM_Fonts", 0, 0,
                                                 buffer, bufSize);
            if (bufLen) {
                char *key = buffer;
                while (*key) {
                    ULONG keySize = 0;
                    ok = PrfQueryProfileSize(HINI_USERPROFILE, "PM_Fonts", key,
                                             &keySize);
                    if (ok) {
                        QByteArray file(keySize, 0);
                        ULONG keyLen =
                            PrfQueryProfileString(HINI_USERPROFILE, "PM_Fonts", key, 0,
                                                  file.data(), file.size());
                        file.truncate(keyLen - 1 /*terminating NULL*/);
                        if (!file.isEmpty()) {
                            // FreeType doesn't understand .OFM but understands .PFB
                            if (file.toUpper().endsWith(".OFM")) {
                                file.chop(4);
                                file.append(".PFB");
                            }
                            QFileInfo fileInfo(QFile::decodeName(file));
                            QString fileName = fileInfo.canonicalFilePath().toLower();
                            if (!fileName.isEmpty()) { // file may have been deleted
                                fileInfo.setFile(fileName);
                                // check the in-process file name cache
                                FileData &cached = knownFontFiles[fileName];
                                if (cached.fileInfo.filePath().isEmpty() ||
                                    cached.fileInfo.lastModified() != fileInfo.lastModified() ||
                                    cached.fileInfo.size() != fileInfo.size()) {
                                    // no cache entry or outdated, process it
                                    cached.fileInfo = fileInfo;
                                    cached.seen = true;
                                    fontFiles << fileInfo;
                                    FD_DEBUG("populateDatabase: NEW/UPDATED font file %s",
                                             qPrintable(fileName));
                                } else {
                                    // just set the 'seen' flag and skip this font
                                    // (it's already in the database)
                                    knownFontFiles[fileName].seen = true;
                                    FD_DEBUG("populateDatabase: UNCHANGED font file %s",
                                             qPrintable(fileName));
                                }
                            }
                        }
                    }
                    key += strlen(key) + 1;
                }
            }
            delete buffer;
        }
    }

    extern FT_Library qt_getFreetype(); // qfontengine_ft.cpp
    FT_Library lib = qt_getFreetype();

    // go through each font file and get available faces
    foreach(const QFileInfo &fileInfo, fontFiles) {
        QString fileKey = fileInfo.absoluteFilePath().toLower();
        QByteArray file = QFile::encodeName(fileKey);

        // QSettings uses / for splitting into groups, suppress it
        fileKey.replace("/", "|");

        QList<FaceData> cachedFaces;

        // first, look up the cached data
        fontCache.beginGroup(fileKey);

        if (fontCache.value(QLatin1String("DateTime")).toDateTime() != fileInfo.lastModified() ||
            fontCache.value(QLatin1String("Size")).toUInt() != fileInfo.size()) {
            // the cache is outdated or doesn't exist, query the font file

            FT_Long numFaces = 0;
            FT_Face face;

            FT_Error rc = FT_New_Face(lib, file, -1, &face);
            if (rc == 0) {
                numFaces = face->num_faces;
                FT_Done_Face(face);
            } else {
                // invalid/unsupported font file, numFaces is left 0 so that
                // only DateTime and Size will be cached indicating that this
                // file is not recognized
            }

            FD_DEBUG("populateDatabase: Font file %s: FT error %d, has %ld faces",
                     file.constData(), (int) rc, numFaces);

            // go throuhg each face
            for (FT_Long idx = 0; idx < numFaces; ++idx) {
                rc = FT_New_Face(lib, file, idx, &face);
                if (rc != 0)
                    continue;

                FaceData cached;

                cached.index = idx;
                cached.familyName = QString::fromLatin1(face->family_name);

                // familyName may contain extra spaces (at least this is true for
                // TNR.PFB that is reported as "Times New Roman ". Trim them.
                cached.familyName = cached.familyName.trimmed();

                cached.styleKey.style = face->style_flags & FT_STYLE_FLAG_ITALIC ?
                    QFont::StyleItalic : QFont::StyleNormal;

                TT_OS2 *os2_table = 0;
                if (face->face_flags & FT_FACE_FLAG_SFNT) {
                    os2_table = (TT_OS2 *)FT_Get_Sfnt_Table(face, ft_sfnt_os2);
                }
                if (os2_table) {
                    // map weight and width values
                    if (os2_table->usWeightClass < 400)
                        cached.styleKey.weight = QFont::Light;
                    else if (os2_table->usWeightClass < 600)
                        cached.styleKey.weight = QFont::Normal;
                    else if (os2_table->usWeightClass < 700)
                        cached.styleKey.weight = QFont::DemiBold;
                    else if (os2_table->usWeightClass < 800)
                        cached.styleKey.weight = QFont::Bold;
                    else
                        cached.styleKey.weight = QFont::Black;
FD_DEBUG("os2_table->usWeightClass %u", os2_table->usWeightClass);

                    switch (os2_table->usWidthClass) {
                        case 1: cached.styleKey.stretch = QFont::UltraCondensed; break;
                        case 2: cached.styleKey.stretch = QFont::ExtraCondensed; break;
                        case 3: cached.styleKey.stretch = QFont::Condensed; break;
                        case 4: cached.styleKey.stretch = QFont::SemiCondensed; break;
                        case 5: cached.styleKey.stretch = QFont::Unstretched; break;
                        case 6: cached.styleKey.stretch = QFont::SemiExpanded; break;
                        case 7: cached.styleKey.stretch = QFont::Expanded; break;
                        case 8: cached.styleKey.stretch = QFont::ExtraExpanded; break;
                        case 9: cached.styleKey.stretch = QFont::UltraExpanded; break;
                        default: cached.styleKey.stretch = QFont::Unstretched; break;
                    }

                    quint32 unicodeRange[4] = {
                        os2_table->ulUnicodeRange1, os2_table->ulUnicodeRange2,
                        os2_table->ulUnicodeRange3, os2_table->ulUnicodeRange4
                    };
                    quint32 codePageRange[2] = {
                        os2_table->ulCodePageRange1, os2_table->ulCodePageRange2
                    };
                    cached.systems =
                        determineWritingSystemsFromTrueTypeBits(unicodeRange, codePageRange);
FD_DEBUG() << "cached.systems" << cached.systems;
                } else {
                    // we've only got simple weight information and no stretch
                    cached.styleKey.weight = face->style_flags & FT_STYLE_FLAG_BOLD ?
                        QFont::Bold : QFont::Normal;
                    cached.styleKey.stretch = QFont::Unstretched;
                }

                cached.fixedPitch = face->face_flags & FT_FACE_FLAG_FIXED_WIDTH;

                cached.smoothScalable = face->face_flags & FT_FACE_FLAG_SCALABLE;

                // the font may both be scalable and contain fixed size bitmaps
                if (face->face_flags & FT_FACE_FLAG_FIXED_SIZES) {
                    for (FT_Int i = 0; i < face->num_fixed_sizes; ++i) {
                        cached.pixelSizes << face->available_sizes[i].height;
                    }
                }

                cachedFaces << cached;

                FT_Done_Face(face);
            }

            // store the data into the cache
            fontCache.setValue(QLatin1String("DateTime"), fileInfo.lastModified());
            fontCache.setValue(QLatin1String("Size"), fileInfo.size());
            foreach(FaceData cached, cachedFaces) {
                QByteArray rawData;
                QDataStream data(&rawData, QIODevice::WriteOnly);
                data << cached;

                QString face = QString::number(cached.index);
                fontCache.beginGroup(face);
                fontCache.setValue(QLatin1String("Info"), rawData);
                fontCache.endGroup();
            }
        } else {
            // take the face data from the cache

            QStringList faces = fontCache.childGroups();

            FD_DEBUG("populateDatabase: Font file %s: IN CACHE, has %d faces",
                     file.constData(), faces.count());

            foreach(QString face, faces) {
                bool ok = false;
                FaceData cached;
                cached.index = face.toInt(&ok);
                if (!ok || cached.index < 0) // not a valid index
                    continue;

                fontCache.beginGroup(face);
                QByteArray rawData =
                    fontCache.value(QLatin1String("Info")).toByteArray();
                QDataStream data(rawData);
                data >> cached;
                fontCache.endGroup();

                cachedFaces << cached;
            }
        }

        fontCache.endGroup();

        // go throuhg each cached face and add it to the database
        foreach(FaceData cached, cachedFaces) {

            QtFontFamily *family = privateDb()->family(cached.familyName, true);

            // @todo is it possible that the same family is both fixed and not?
            Q_ASSERT(!family->fixedPitch || cached.fixedPitch);
            family->fixedPitch = cached.fixedPitch;

            if (cached.systems.isEmpty()) {
                // it was hard or impossible to determine the actual writing system
                // of the font (as in case of OS/2 bitmap and PFB fonts for which it is
                // usually simply reported that they support standard/system codepages).
                // Pretend that we support all writing systems to not miss the one.
                //
                // @todo find a proper way to detect actual supported scripts to make
                // sure these fonts are not matched for scripts they don't support.
                for (int ws = 0; ws < QFontDatabase::WritingSystemsCount; ++ws)
                    family->writingSystems[ws] = QtFontFamily::Supported;
            } else {
                for (int i = 0; i < cached.systems.count(); ++i)
                    family->writingSystems[cached.systems.at(i)] = QtFontFamily::Supported;
            }

            QtFontFoundry *foundry = family->foundry(foundryName, true);
            QtFontStyle *style = foundry->style(cached.styleKey, true);

            // so far, all recognized fonts are antialiased
            style->antialiased = true;

            if (cached.smoothScalable && !style->smoothScalable) {
                // add new scalable style only if it hasn't been already added --
                // the first one of two duplicate (in Qt terms) non-bitmap font
                // styles wins.
                style->smoothScalable = true;
                QtFontSize *size =
                    style->pixelSize(SMOOTH_SCALABLE, true);
                size->fileName = file;
                size->fileIndex = cached.index;
                size->systems = cached.systems;
            }

            foreach(unsigned short pixelSize, cached.pixelSizes) {
                QtFontSize *size = style->pixelSize(pixelSize, true);
                // the first bitmap style with a given pixel and point size wins
                if (!size->fileName.isEmpty())
                    continue;
                size->fileName = file;
                size->fileIndex = cached.index;
                size->systems = cached.systems;
            }
        }
    }

    // go through the known file list to detect what files have been removed
    for (FontFileHash::iterator it = knownFontFiles.begin();
         it != knownFontFiles.end();) {
        if (!it.value().seen) {
            FD_DEBUG("populateDatabase: DELETED font file %s",
                     qPrintable(it.key()));
            // remove from the both caches
            QString fileKey = it.key();
            fileKey.replace("/", "|");
            fontCache.remove(fileKey);
            it = knownFontFiles.erase(it);
            // @todo should we remove all references to this file from the
            // font database? My concern is that this font may be in use by Qt
            // and its glyphs may be still cached when file deletion happens
        } else {
            ++it;
        }
    }

#ifdef QFONTDATABASE_DEBUG
    FD_DEBUG("populateDatabase: took %d ms", timer.elapsed());
#endif
}

static void initializeDb()
{
    QFontDatabasePrivate *db = privateDb();
    if (!db || db->count)
        return;

    populateDatabase(QString());

#ifdef QFONTDATABASE_DEBUG
    // print the database
    qDebug("initializeDb:");
    for (int f = 0; f < db->count; f++) {
        QtFontFamily *family = db->families[f];
        qDebug("    %s: %p", qPrintable(family->name), family);
        populateDatabase(family->name);
#if 1
        qDebug("        writing systems supported:");
        QStringList systems;
        for (int ws = 0; ws < QFontDatabase::WritingSystemsCount; ++ws)
            if (family->writingSystems[ws] & QtFontFamily::Supported)
                systems << QFontDatabase::writingSystemName((QFontDatabase::WritingSystem)ws);
        qDebug() << "            " << systems;
        for (int fd = 0; fd < family->count; fd++) {
            QtFontFoundry *foundry = family->foundries[fd];
            qDebug("        %s", foundry->name.isEmpty() ? "(empty foundry)" :
                   qPrintable(foundry->name));
            for (int s = 0; s < foundry->count; s++) {
                QtFontStyle *style = foundry->styles[s];
                qDebug("            style: style=%d weight=%d smooth=%d",  style->key.style,
                       style->key.weight, style->smoothScalable);
                for(int i = 0; i < style->count; ++i) {
                    if (style->pixelSizes[i].pixelSize == SMOOTH_SCALABLE)
                        qDebug("                smooth %s:%d",
                               style->pixelSizes[i].fileName.constData(),
                               style->pixelSizes[i].fileIndex);
                    else
                        qDebug("                %d px %s:%d", style->pixelSizes[i].pixelSize,
                               style->pixelSizes[i].fileName.constData(),
                               style->pixelSizes[i].fileIndex);
                }
            }
        }
#endif
    }
#endif // QFONTDATABASE_DEBUG
}

static inline void load(const QString &family = QString(), int = -1)
{
    populateDatabase(family);
}

static void registerFont(QFontDatabasePrivate::ApplicationFont *fnt)
{
    // @todo implement
}

static QFontDef fontDescToFontDef(const QFontDef &req, const QtFontDesc &desc)
{
    static LONG dpi = -1;
    if (dpi == -1) {
        // PM cannot change resolutions on the fly so cache it
        int hps = qt_display_ps();
        DevQueryCaps(GpiQueryDevice(hps), CAPS_HORIZONTAL_FONT_RES, 1, &dpi);
    }

    QFontDef fontDef;

    fontDef.family = desc.family->name;

    if (desc.size->pixelSize == SMOOTH_SCALABLE) {
        // scalable font matched, calculate the missing size (points or pixels)
        fontDef.pointSize = req.pointSize;
        fontDef.pixelSize = req.pixelSize;
        if (req.pointSize < 0) {
            fontDef.pointSize = req.pixelSize * 72. / dpi;
        } else if (req.pixelSize == -1) {
            fontDef.pixelSize = qRound(req.pointSize * dpi / 72.);
        }
    } else {
        // non-scalable font matched, calculate both point and pixel size
        fontDef.pixelSize = desc.size->pixelSize;
        fontDef.pointSize = desc.size->pixelSize * 72. / dpi;
    }

    fontDef.styleStrategy = req.styleStrategy;
    fontDef.styleHint = req.styleHint;

    fontDef.weight = desc.style->key.weight;
    fontDef.fixedPitch = desc.family->fixedPitch;
    fontDef.style = desc.style->key.style;
    fontDef.stretch = desc.style->key.stretch;

    return fontDef;
}

static QFontEngine *loadEngine(const QFontDef &req, const QtFontDesc &desc)
{
    // @todo all these fixed so far; make configurable through the Registry
    // on per-family basis
    QFontEnginePMFT::HintStyle hintStyle = QFontEnginePMFT::HintFull;
    bool autoHint = true;
    QFontEngineFT::SubpixelAntialiasingType subPixel = QFontEngineFT::Subpixel_None;
    int lcdFilter = FT_LCD_FILTER_DEFAULT;
    bool useEmbeddedBitmap = true;

    QFontEngine::FaceId faceId;
    faceId.filename = desc.size->fileName;
    faceId.index = desc.size->fileIndex;

    QFontEngineFT *fe = new QFontEnginePMFT(fontDescToFontDef(req, desc), faceId,
                                            desc.style->antialiased, hintStyle,
                                            autoHint, subPixel, lcdFilter,
                                            useEmbeddedBitmap);
    Q_ASSERT(fe);
    if (fe && fe->invalid()) {
        FM_DEBUG("   --> invalid!\n");
        delete fe;
        fe = 0;
    }
    return fe;
}

static QFontEngine *loadPM(const QFontPrivate *d, int script, const QFontDef &req)
{
    // list of families to try
    QStringList families = familyList(req);

    const char *styleHint = qt_fontFamilyFromStyleHint(d->request);
    if (styleHint)
        families << QLatin1String(styleHint);

    // add the default family
    QString defaultFamily = QApplication::font().family();
    if (! families.contains(defaultFamily))
        families << defaultFamily;

    // add QFont::defaultFamily() to the list, for compatibility with
    // previous versions
    families << QApplication::font().defaultFamily();

    // null family means find the first font matching the specified script
    families << QString();

    QtFontDesc desc;
    QFontEngine *fe = 0;
    QList<int> blacklistedFamilies;

    while (!fe) {
        for (int i = 0; i < families.size(); ++i) {
            QString family, foundry;
            parseFontName(families.at(i), foundry, family);
            FM_DEBUG("loadPM: >>>>>>>>>>>>>> trying to match '%s'", qPrintable(family));
            QT_PREPEND_NAMESPACE(match)(script, req, family, foundry, -1, &desc, blacklistedFamilies);
            if (desc.family)
                break;
        }
        if (!desc.family)
            break;
        FM_DEBUG("loadPM: ============== matched '%s'", qPrintable(desc.family->name));
        fe = loadEngine(req, desc);
        if (!fe)
            blacklistedFamilies.append(desc.familyIndex);
    }
    return fe;
}

void QFontDatabase::load(const QFontPrivate *d, int script)
{
    Q_ASSERT(script >= 0 && script < QUnicodeTables::ScriptCount);

    // normalize the request to get better caching
    QFontDef req = d->request;
    if (req.pixelSize <= 0)
        req.pixelSize = qMax(1, qRound(req.pointSize * d->dpi / 72.));
    req.pointSize = 0;
    if (req.weight == 0)
        req.weight = QFont::Normal;
    if (req.stretch == 0)
        req.stretch = 100;

    // @todo a hack to substitute "WarpSans" with "Workplace Sans". Remove this
    // when we start supporting OS/2 bitmap fonts.
    if (req.family == QLatin1String("WarpSans"))
        req.family = QLatin1String("Workplace Sans");

    QFontCache::Key key(req, d->rawMode ? QUnicodeTables::Common : script, d->screen);
    if (!d->engineData)
        getEngineData(d, key);

    // the cached engineData could have already loaded the engine we want
    if (d->engineData->engines[script])
        return;

    // set it to the actual pointsize, so QFontInfo will do the right thing
    req.pointSize = req.pixelSize * 72. / d->dpi;

    QFontEngine *fe = QFontCache::instance()->findEngine(key);

    if (!fe) {
        if (qt_enable_test_font && req.family == QLatin1String("__Qt__Box__Engine__")) {
            fe = new QTestFontEngine(req.pixelSize);
            fe->fontDef = req;
        } else {
            QMutexLocker locker(fontDatabaseMutex());
            if (!privateDb()->count)
                initializeDb();
            fe = loadPM(d, script, req);
        }
        if (!fe) {
            fe = new QFontEngineBox(req.pixelSize);
            fe->fontDef = QFontDef();
        }
    }
    if (fe->symbol || (d->request.styleStrategy & QFont::NoFontMerging)) {
        for (int i = 0; i < QUnicodeTables::ScriptCount; ++i) {
            if (!d->engineData->engines[i]) {
                d->engineData->engines[i] = fe;
                fe->ref.ref();
            }
        }
    } else {
        d->engineData->engines[script] = fe;
        fe->ref.ref();
    }
    QFontCache::instance()->insertEngine(key, fe);
}

bool QFontDatabase::removeApplicationFont(int handle)
{
    // @todo implement
    return false;
}

bool QFontDatabase::removeAllApplicationFonts()
{
    // @todo implement
    return false;
}

bool QFontDatabase::supportsThreadedFontRendering()
{
    // @todo implement
    return false;
}

QT_END_NAMESPACE
