/*
 * Extended system tray widget for XCenter/eCenter
 *
 * Implementation
 *
 * Made by netlabs.org
 *
 * Author: Dmitry A. Kuminov
 *
 * This software is public domain.
 *
 * WITHOUT ANY WARRANTY..., AT YOUR OWN RISC... ETC.
 *
 */

/*
 * This code is based on the code template for a minimal XCenter widget
 * from the XWorkplace project (c) by Ulrich Moeller.
 */

#pragma strings(readonly)

/*
 *  Suggested #include order:
 *  1)  os2.h
 *  2)  C library headers
 *  3)  setup.h (code generation and debugging options)
 *  4)  headers in helpers\
 *  5)  at least one SOM implementation header (*.ih)
 *  6)  dlgids.h, headers in shared\ (as needed)
 *  7)  headers in implementation dirs (e.g. filesys\, as needed)
 *  8)  #pragma hdrstop and then more SOM headers which crash with precompiled headers
 */

#define INCL_BASE
#define INCL_PM
#include <os2.h>

// C library headers
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

// generic headers
// If this file were part of the XWorkplace sources, we'd now include
// the generic "setup.h" file, which has common set up code for every
// single XWorkplace code file. But it's not, so we won't include that.
// #include "setup.h"                      // code generation and debugging options

// headers in /helpers
// This would be the place to include headers from the "XWorkplace helpers".
// But since we do a minimal sample here, we can't include those helpful
// routines... again, see the src\widgets in the XWorkplace source code
// for how these functions can be imported from XFLDR.DLL to avoid duplicate
// code.
// #include "helpers\dosh.h"               // Control Program helper routines
// #include "helpers\gpih.h"               // GPI helper routines
// #include "helpers\prfh.h"               // INI file helper routines;
                                        // this include is required for some
                                        // of the structures in shared\center.h
// #include "helpers\winh.h"               // PM helper routines
// #include "helpers\xstring.h"            // extended string helpers

// XWorkplace implementation headers
// If this file were part of the XCenter sources, we'd now include
// "center.h" from the "include\shared" directory. Since we're not
// part of the XCenter sources here, we include that file from the
// "toolkit" directory in the binary release. That file is identical
// to "include\shared\center.h" in the XWorkplace sources.
#include "shared\center.h"              // public XCenter interfaces

#include "xsystray.h"

#pragma hdrstop                     // VAC++ keeps crashing otherwise

// primitive debug logging to a file
#if 0
static void __LOG_WORKER(const char *fmt, ...)
{
    static FILE *f = NULL;
    if (f == NULL)
    {
        f = fopen("c:\\xsystray.dbg", "a");
        setbuf(f, NULL);
    }
    if (f != NULL)
    {
        va_list vl;
        va_start(vl, fmt);
        vfprintf(f, fmt, vl);
        va_end(vl);
    }
}
#define LOG(m)  do { __LOG_WORKER m; } while(0)
#define LOGF(m) do { __LOG_WORKER("%s: ", __FUNCTION__); __LOG_WORKER m; } while(0)
#else
#define LOG(m)  do {} while (0)
#define LOGF(m) do {} while (0)
#endif

/* ******************************************************************
 *
 *   Private definitions
 *
 ********************************************************************/

/*
 *@@ ICONDATA:
 *      Per-icon data.
 */

typedef struct _ICONDATA
{
    HWND        hwnd;
                // associated window
    ULONG       ulId;
                // icon ID
    HPOINTER    hIcon;
                // icon handle
    ULONG       ulMsgId;
                // message ID for notifications
    PSZ         pszToolTip;
                // icon tooltip (NULL if none)

} ICONDATA, *PICONDATA;

/*
 *@@ SYSTRAYDATA:
 *      Global system tray data.
 */

typedef struct
{
    HWND        hwndServer;
                // systtem tray server window handle
    LONG        lIconWidth;
                // system icon width in px
    LONG        lIconHeight;
                // system icon height in px
    LONG        lIconPad;
                // padding around each icon in px
    PICONDATA   pIcons;
                // array of icons currently shown in the system tray
    size_t      cIcons;
                // number of icons in the pIcons array
    size_t      cIconsMax;
                // maximum number of icons pIcons can fit

} SYSTRAYDATA, *PSYSTRAYDATA;

#define ICONARRAY_GROW  4
        // number of element the icon array is increased by when there is no
        // space for the newly added icon

static ULONG QWL_USER_SERVER_DATA = 0;
             // offset to the PXCENTERWIDGET pointer in the widget data array

#define TID_CHECKALIVE          1
         // timer that checks if windows associated with icons are still alive
#define TID_CHECKALIVE_TIMEOUT  2000 // ms
         // how often to perform alive checks

/* ******************************************************************
 *
 *   XCenter widget class definition
 *
 ********************************************************************/

/*
 *      This contains the name of the PM window class and
 *      the XCENTERWIDGETCLASS definition(s) for the widget
 *      class(es) in this DLL.
 *
 *      The address of this structure (or an array of these
 *      structures, if there were several widget classes in
 *      this plugin) is returned by the "init" export
 *      (WgtInitModule).
 */

static const XCENTERWIDGETCLASS G_WidgetClasses[] =
{
    {
        WNDCLASS_WIDGET_XSYSTRAY,   // PM window class name
        0,                          // additional flag, not used here
        INTCLASS_WIDGET_XSYSTRAY,   // internal widget class name
        HUMANSTR_WIDGET_XSYSTRAY,   // widget class name displayed to user
        WGTF_UNIQUEGLOBAL,          // widget class flags
        NULL                        // no settings dialog
    }
};

/* ******************************************************************
 *
 *   Function imports from XFLDR.DLL
 *
 ********************************************************************/

/*
 *      To reduce the size of the widget DLL, it can
 *      be compiled with the VAC subsystem libraries.
 *      In addition, instead of linking frequently
 *      used helpers against the DLL again, you can
 *      import them from XFLDR.DLL, whose module handle
 *      is given to you in the INITMODULE export.
 *
 *      Note that importing functions from XFLDR.DLL
 *      is _not_ a requirement. We can't do this in
 *      this minimal sample anyway without having access
 *      to the full XWorkplace source code.
 *
 *      If you want to know how you can import the useful
 *      functions from XFLDR.DLL to use them in your widget
 *      plugin, again, see src\widgets in the XWorkplace sources.
 *      The actual imports would then be made by WgtInitModule.
 */

// @todo the function declarations should become not necessary when we move into
// XWorkplace. Let's hope the prototypes will not change until then. Let's also
// pray that XFLDRxxx.DLL has the _Optlink calling convention for exports and
// not the default __cdecl (which happens if it builds with GCC according to the
// XWPENTRY definition in the current xwphelpers sources)

#define XWPENTRY _Optlink

typedef VOID XWPENTRY GPIHDRAW3DFRAME(HPS hps,
                                      PRECTL prcl,
                                      USHORT usWidth,
                                      LONG lColorLeft,
                                      LONG lColorRight);
typedef GPIHDRAW3DFRAME *PGPIHDRAW3DFRAME;
PGPIHDRAW3DFRAME pgpihDraw3DFrame = NULL;

typedef struct _RESOLVEFUNCTION
{
    const char  *pcszFunctionName;
    PFN         *ppFuncAddress;
} RESOLVEFUNCTION, *PRESOLVEFUNCTION;
RESOLVEFUNCTION G_aImports[] =
{
    { "gpihDraw3DFrame", (PFN*)&pgpihDraw3DFrame },
};

/* ******************************************************************
 *
 *   Private widget instance data
 *
 ********************************************************************/

// None presently. The samples in src\widgets in the XWorkplace
// sources cleanly separate setup string data from other widget
// instance data to allow for easier manipulation with settings
// dialogs. We have skipped this for the minimal sample.

/* ******************************************************************
 *
 *   Widget setup management
 *
 ********************************************************************/

// None presently. See above.

/* ******************************************************************
 *
 *   Widget settings dialog
 *
 ********************************************************************/

// None currently. To see how a setup dialog can be done,
// see the "window list" widget in the XWorkplace sources
// (src\widgets\w_winlist.c).

/* ******************************************************************
 *
 *   Callbacks stored in XCENTERWIDGETCLASS
 *
 ********************************************************************/

// If you implement a settings dialog, you must write a
// "show settings dlg" function and store its function pointer
// in XCENTERWIDGETCLASS.

/* ******************************************************************
 *
 *   Helper methods
 *
 ********************************************************************/

/*
 *@@ FreeIconData:
 *      Frees all members of the ICONDATA structure and resets them to 0.
 */

static
VOID FreeIconData(PICONDATA pData)
{
    pData->hwnd = NULLHANDLE;
    pData->ulId = 0;
    if (pData->hIcon != NULLHANDLE)
    {
        WinDestroyPointer(pData->hIcon);
        pData->hIcon = NULLHANDLE;

    }
    pData->ulMsgId = 0;
    if (pData->pszToolTip)
    {
        free(pData->pszToolTip);
        pData->pszToolTip = NULL;
    }
}

/*
 *@@ FindIconData:
 *      Searches for the icon with the given identity. Returns NULL if not
 *      found.
 */

static
PICONDATA FindIconData(PSYSTRAYDATA pSysTrayData,
                       HWND hwnd,       // in: associated window handle
                       ULONG ulId,      // in: icon ID
                       size_t *pIdx)    // out: index of the icon in the icon array
                                        // (optional, may be NULL)
{
    size_t i;
    for (i = 0; i < pSysTrayData->cIcons; ++i)
    {
        if (pSysTrayData->pIcons[i].hwnd == hwnd &&
            pSysTrayData->pIcons[i].ulId == ulId)
        {
            if (pIdx)
                *pIdx = i;
            return &pSysTrayData->pIcons[i];
        }
    }

    if (pIdx)
        *pIdx = i;
    return NULL;
}

/*
 *@@ DrawPointer:
 *      Draws a pointer in a presentation space.
 */

static
BOOL DrawPointer(HPS hps, LONG lx, LONG ly, HPOINTER hptrPointer, BOOL bMini)
{
    return WinDrawPointer(hps, lx, ly, hptrPointer, bMini ? DP_MINI : DP_NORMAL);
    // @todo: for icons with real alpha, do manual alpha blending
}

/* ******************************************************************
 *
 *   PM window class implementation
 *
 ********************************************************************/

/*
 *      This code has the actual PM window class.
 *
 */

/*
 *@@ MwgtControl:
 *      implementation for WM_CONTROL in fnwpXSysTray.
 *
 *      The XCenter communicates with widgets thru
 *      WM_CONTROL messages. At the very least, the
 *      widget should respond to XN_QUERYSIZE because
 *      otherwise it will be given some dumb default
 *      size.
 */

static
BOOL WgtControl(PXCENTERWIDGET pWidget,
                MPARAM mp1,
                MPARAM mp2)
{
    PSYSTRAYDATA pSysTrayData = (PSYSTRAYDATA)pWidget->pUser;
    BOOL brc = FALSE;

    USHORT  usID = SHORT1FROMMP(mp1),
            usNotifyCode = SHORT2FROMMP(mp1);

    // is this from the XCenter client?
    if (usID == ID_XCENTER_CLIENT)
    {
        // yes:

        switch (usNotifyCode)
        {
            /*
             * XN_QUERYSIZE:
             *      XCenter wants to know our size.
             */

            case XN_QUERYSIZE:
            {
                PSIZEL pszl = (PSIZEL)mp2;
                LONG pad2 = pSysTrayData->lIconPad * 2;
                pszl->cx = (pSysTrayData->lIconWidth + pad2) * pSysTrayData->cIcons; // desired width
                pszl->cy = pSysTrayData->lIconHeight + pad2; // desired minimum height
                brc = TRUE;
            }
            break;

        } // end switch (usNotifyCode)
    } // end if (usID == ID_XCENTER_CLIENT)

    return brc;
}

/*
 *@@ WgtPaint:
 *      implementation for WM_PAINT in fnwpXSysTray.
 *
 *      This really does nothing, except painting a
 *      3D rectangle and printing a question mark.
 */

static
VOID WgtPaint(HWND hwnd,
              PXCENTERWIDGET pWidget)
{
    PSYSTRAYDATA pSysTrayData = (PSYSTRAYDATA)pWidget->pUser;
    HPS hps;
    RECTL rclPaint;

    if ((hps = WinBeginPaint(hwnd, NULLHANDLE, &rclPaint)))
    {
        SWP     swp;
        RECTL   rcl;
        BOOL    bLeftToRight;
        LONG    x, y, lTotalWidth;
        size_t  i;

        WinQueryWindowPos(hwnd, &swp);
        WinQueryWindowRect(pWidget->pGlobals->hwndClient, &rcl);

        // correct the paint area
        // @todo find out why it exceeds the window bounds
        if (rclPaint.xLeft < 0)
            rclPaint.xLeft = 0;
        if (rclPaint.xRight > swp.cx)
            rclPaint.xRight = swp.cx;
        if (rclPaint.yBottom < 0)
            rclPaint.yBottom = 0;
        if (rclPaint.yTop > swp.cy)
            rclPaint.yTop = swp.cy;

        LOGF(("rclPaint %d,%d-%d,%d\n",
              rclPaint.xLeft, rclPaint.yBottom, rclPaint.xRight, rclPaint.yTop));

        // switch HPS to RGB mode
        GpiCreateLogColorTable(hps, 0, LCOLF_RGB, 0, 0, NULL);

        // draw icons left to right if our center is closer to the left edge
        // of XCenter and right to left otherwise
        bLeftToRight = swp.x + swp.cx / 2 < (rcl.xRight / 2);

        WinFillRect(hps, &rclPaint,
                    WinQuerySysColor(HWND_DESKTOP, SYSCLR_DIALOGBACKGROUND, 0));

        if ((pWidget->pGlobals->flDisplayStyle & XCS_SUNKBORDERS))
        {
            rcl.xLeft = 0;
            rcl.yBottom = 0;
            rcl.xRight = swp.cx - 1;
            rcl.yTop = swp.cy - 1;
            pgpihDraw3DFrame(hps, &rcl, 1,
                             pWidget->pGlobals->lcol3DDark,
                             pWidget->pGlobals->lcol3DLight);
        }

        // always center the icon vertically (we may be given more height than
        // we requested)
        y = (swp.cy - pSysTrayData->lIconWidth) / 2;

        if (bLeftToRight)
            x = pSysTrayData->lIconPad;
        else
            x = swp.cx - pSysTrayData->lIconPad - pSysTrayData->lIconWidth;

        lTotalWidth = pSysTrayData->lIconWidth + pSysTrayData->lIconPad * 2;

        // where to start from?
        if (bLeftToRight)
        {
            i = rclPaint.xLeft / lTotalWidth;
            x = pSysTrayData->lIconPad + i * lTotalWidth;
        }
        else
        {
            i = (swp.cx - rclPaint.xRight) / lTotalWidth;
            x = swp.cx - (i + 1) * lTotalWidth + pSysTrayData->lIconPad;
        }

        // draw as many icons as we can / need
        for (i = 0; i < pSysTrayData->cIcons; ++i)
        {
            if (x >= rclPaint.xRight)
                break;

            DrawPointer(hps, x, y, pSysTrayData->pIcons[i].hIcon, DP_MINI);
            if (bLeftToRight)
                x += lTotalWidth;
            else
                x -= lTotalWidth;
        }

        WinEndPaint(hps);
    }
}

/*
 *@@ fnwpXSysTray:
 *      window procedure for the Extended system tray widget class.
 *
 *      There are a few rules which widget window procs
 *      must follow. See XCENTERWIDGETCLASS in center.h
 *      for details.
 *
 *      Other than that, this is a regular window procedure
 *      which follows the basic rules for a PM window class.
 */

static
MRESULT EXPENTRY fnwpXSysTray(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;
    // get widget data from QWL_USER (stored there by WM_CREATE)
    PXCENTERWIDGET pWidget = (PXCENTERWIDGET)WinQueryWindowPtr(hwnd, QWL_USER);
                    // this ptr is valid after WM_CREATE

    switch (msg)
    {
        /*
         * WM_CREATE:
         *      as with all widgets, we receive a pointer to the
         *      XCENTERWIDGET in mp1, which was created for us.
         *
         *      The first thing the widget MUST do on WM_CREATE
         *      is to store the XCENTERWIDGET pointer (from mp1)
         *      in the QWL_USER window word by calling:
         *
         *          WinSetWindowPtr(hwnd, QWL_USER, mp1);
         *
         *      We could use XCENTERWIDGET.pUser for allocating
         *      another private memory block for our own stuff,
         *      for example to be able to store fonts and colors.
         *      We ain't doing this in the minimal sample.
         */

        case WM_CREATE:
        {
            PSYSTRAYDATA pSysTrayData = NULL;

            mrc = (MPARAM)TRUE; // being pessimistic gives more compact code

            WinSetWindowPtr(hwnd, QWL_USER, mp1);
            if (    (!(pWidget = (PXCENTERWIDGET)mp1))
                 || (!pWidget->pfnwpDefWidgetProc)
               )
                // shouldn't happen... stop window creation!!
                break;

            pSysTrayData = malloc(sizeof(*pSysTrayData));
            if (pSysTrayData == NULL)
                break;

            // initialize the SYSTRAYDATA structure
            pSysTrayData->lIconWidth = WinQuerySysValue(HWND_DESKTOP, SV_CXICON) / 2;
            pSysTrayData->lIconHeight = WinQuerySysValue(HWND_DESKTOP, SV_CYICON) / 2;
            pSysTrayData->lIconPad = pSysTrayData->lIconHeight / 8;
            pSysTrayData->cIconsMax = ICONARRAY_GROW;
            pSysTrayData->pIcons = malloc(sizeof(*pSysTrayData->pIcons) *
                                          pSysTrayData->cIconsMax);
            if (pSysTrayData->pIcons == NULL)
            {
                free(pSysTrayData);
                break;
            }
            pSysTrayData->cIcons = 0;

            // create the "server" window (note that we pass the XCENTERWIDGET
            // pointer on to it)
            pSysTrayData->hwndServer =
                WinCreateWindow(HWND_DESKTOP, WNDCLASS_WIDGET_XSYSTRAY_SERVER,
                                NULL, WS_MINIMIZED,
                                0, 0, 0, 0,
                                HWND_DESKTOP, HWND_BOTTOM,
                                0, mp1, NULL);
            if (pSysTrayData->hwndServer == NULLHANDLE)
            {
                free(pSysTrayData->pIcons);
                free(pSysTrayData);
                break;
            }

            pWidget->pUser = pSysTrayData;

            mrc = FALSE; // confirm success
        }
        break;

        /*
         * WM_CONTROL:
         *      process notifications/queries from the XCenter.
         */

        case WM_CONTROL:
            mrc = (MPARAM)WgtControl(pWidget, mp1, mp2);
        break;

        /*
         * WM_PAINT:
         *      well, paint the widget.
         */

        case WM_PAINT:
            WgtPaint(hwnd, pWidget);
        break;

        /*
         * WM_PRESPARAMCHANGED:
         *      A well-behaved widget would intercept
         *      this and store fonts and colors.
         */

        /* case WM_PRESPARAMCHANGED:
        break; */

        /*
         * WM_DESTROY:
         *      clean up. This _must_ be passed on to
         *      ctrDefWidgetProc.
         */

        case WM_DESTROY:
        {
            // free all system tray data
            PSYSTRAYDATA pSysTrayData = (PSYSTRAYDATA)pWidget->pUser;
            size_t i;
            for (i = 0; i < pSysTrayData->cIcons; ++i)
                FreeIconData(&pSysTrayData->pIcons[i]);
            free(pSysTrayData->pIcons);
            free(pSysTrayData);
            pWidget->pUser = NULL;
            // We _MUST_ pass this on, or the default widget proc
            // cannot clean up.
            mrc = pWidget->pfnwpDefWidgetProc(hwnd, msg, mp1, mp2);
        }
        break;

        default:
            mrc = pWidget->pfnwpDefWidgetProc(hwnd, msg, mp1, mp2);

    } // end switch(msg)

    return mrc;
}

/*
 *@@ WgtXSysTrayUpdateAfterIconAddRemove:
 *      Name says it all.
 */

static
VOID WgtXSysTrayUpdateAfterIconAddRemove(PXCENTERWIDGET pWidget)
{
    PSYSTRAYDATA pSysTrayData = (PSYSTRAYDATA)pWidget->pUser;

    if (pSysTrayData->cIcons == 1)
    {
        // start a timer to perform "is window alive" checks
        WinStartTimer(pWidget->habWidget, pSysTrayData->hwndServer,
                      TID_CHECKALIVE,
                      TID_CHECKALIVE_TIMEOUT);
    }
    else
    if (pSysTrayData->cIcons == 0)
    {
        // stop the check alive timer
        WinStopTimer(pWidget->habWidget, pSysTrayData->hwndServer,
                     TID_CHECKALIVE);
    }

    // ask XCenter to take our new size into account
    // (this will also invalidate us)
    WinPostMsg(pWidget->pGlobals->hwndClient,
               XCM_REFORMAT,
               (MPARAM)XFMF_GETWIDGETSIZES,
               0);
}

/*
 *@@ WgtXSysTrayControl:
 *      implementation for WM_XST_CONTROL in fnwpXSysTrayServer.
 *
 *      Serves as an entry point for all client-side API requests to the
 *      Extended system tray.
 *
 *      Note that this message is being sent from another process which is
 *      awaiting for an answer, so it must return as far as possible (by
 *      rescheduling any potentially long term operation for another cycle).
 */

static
BOOL WgtXSysTrayControl(HWND hwnd, PXCENTERWIDGET pWidget,
                        PSYSTRAYCTLDATA pCtlData)
{
    BOOL brc = FALSE;
    PSYSTRAYDATA pSysTrayData = (PSYSTRAYDATA)pWidget->pUser;

    switch (pCtlData->ulCommand)
    {
        case SYSTRAYCMD_GETVERSION:
        {
            LOGF(("SYSTRAYCMD_GETVERSION\n"));

            pCtlData->bAcknowledged = TRUE;
            pCtlData->u.version.ulMajor = XSYSTRAY_VERSION_MAJOR;
            pCtlData->u.version.ulMajor = XSYSTRAY_VERSION_MINOR;
            pCtlData->u.version.ulRevision = XSYSTRAY_VERSION_REVISION;
            brc = TRUE;
        }
        break;

        case SYSTRAYCMD_ADDICON:
        {
            POINTERINFO Info;
            HPOINTER hIcon = NULLHANDLE;
            size_t i;
            PICONDATA pData;

            LOGF(("SYSTRAYCMD_ADDICON\n"));
            LOGF((" hwnd  %x\n", pCtlData->hwndSender));
            LOGF((" ulId  %ld\n", pCtlData->u.icon.ulId));
            LOGF((" hIcon %x\n", pCtlData->u.icon.hIcon));

            pCtlData->bAcknowledged = TRUE;

            // make a private copy of the provided icon (it will get lost after
            // we return from this message)
            brc = WinQueryPointerInfo(pCtlData->u.icon.hIcon, &Info);
            if (!brc)
                break;
            hIcon = WinCreatePointerIndirect(HWND_DESKTOP, &Info);
            if (hIcon == NULLHANDLE)
            {
                brc = FALSE;
                break;
            }

            pData = FindIconData(pSysTrayData, pCtlData->hwndSender,
                                 pCtlData->u.icon.ulId, &i);
            if (pData)
            {
                LOGF((" Updating hIcon %x\n", hIcon));

                WinDestroyPointer(pData->hIcon);
                pData->hIcon = hIcon;
                pData->ulMsgId = pCtlData->u.icon.ulMsgId;

                // we didn't change the number of icons so simply invalidate
                WinInvalidateRect(pWidget->hwndWidget, NULL, FALSE);
            }
            else
            {
                LOGF((" Adding new hIcon %x\n", hIcon));

                if (pSysTrayData->cIcons == pSysTrayData->cIconsMax)
                {
                    PICONDATA pNewIcons;

                    LOGF((" Allocating more memory (new icon count is %u)!\n",
                          pSysTrayData->cIcons + 1));

                    pNewIcons = realloc(pSysTrayData->pIcons,
                                        sizeof(*pSysTrayData->pIcons) *
                                        pSysTrayData->cIconsMax + ICONARRAY_GROW);
                    if (pNewIcons == NULL)
                    {
                        WinDestroyPointer(hIcon);
                        break;
                    }

                    pSysTrayData->pIcons = pNewIcons;
                    pSysTrayData->cIconsMax += ICONARRAY_GROW;
                }

                i = pSysTrayData->cIcons;
                ++pSysTrayData->cIcons;

                pData = &pSysTrayData->pIcons[i];
                pData->hwnd = pCtlData->hwndSender;
                pData->ulId = pCtlData->u.icon.ulId;
                pData->hIcon = hIcon;
                pData->ulMsgId = pCtlData->u.icon.ulMsgId;

                WgtXSysTrayUpdateAfterIconAddRemove(pWidget);
            }

            brc = TRUE;
        }
        break;

        case SYSTRAYCMD_REMOVEICON:
        {
            size_t i;
            PICONDATA pData;

            LOGF(("SYSTRAYCMD_REMOVEICON\n"));
            LOGF((" hwnd  %x\n", pCtlData->hwndSender));
            LOGF((" ulId  %ld\n", pCtlData->u.icon.ulId));

            pCtlData->bAcknowledged = TRUE;

            pData = FindIconData(pSysTrayData, pCtlData->hwndSender,
                                 pCtlData->u.icon.ulId, &i);
            if (pData)
            {
                LOGF((" Removing hIcon %x\n", pData->hIcon));

                FreeIconData(pData);

                --pSysTrayData->cIcons;
                if (pSysTrayData->cIcons > 0)
                {
                    memcpy(&pSysTrayData->pIcons[i],
                           &pSysTrayData->pIcons[i + 1],
                           sizeof(*pSysTrayData->pIcons) * (pSysTrayData->cIcons - i));
                }

                WgtXSysTrayUpdateAfterIconAddRemove(pWidget);

                brc = TRUE;
            }
            else
                LOGF((" Icon not found!\n"));
        }
        break;

        default:
            break;
    }

    LOGF(("return %d (WinGetLastError is %x)\n",
          brc, WinGetLastError(pWidget->habWidget)));

    return brc;
}

/*
 *@@ WgtXSysTrayTimer:
 *      implementation for WM_TIMER in fnwpXSysTrayServer.
 */

static
VOID WgtXSysTrayTimer(HWND hwnd, PXCENTERWIDGET pWidget,
                      USHORT usTimerId)
{
    PSYSTRAYDATA pSysTrayData = (PSYSTRAYDATA)pWidget->pUser;

    if (usTimerId == TID_CHECKALIVE)
    {
        // check if windows associated with the icons are still alive
        // and remove those icons whose windows are invalid
        BOOL bAnyDead = FALSE;
        size_t i;
        for (i = 0; i < pSysTrayData->cIcons; ++i)
        {
            if (!WinIsWindow(pWidget->habWidget, pSysTrayData->pIcons[i].hwnd))
            {
                LOGF(("Removing icon of dead window!\n"));
                LOGF((" hwnd  %x\n", pSysTrayData->pIcons[i].hwnd));
                LOGF((" ulId  %ld\n", pSysTrayData->pIcons[i].ulId));
                LOGF((" hIcon %x\n", pSysTrayData->pIcons[i].hIcon));

                bAnyDead = TRUE;
                FreeIconData(&pSysTrayData->pIcons[i]);
                // hwnd is NULLHANDLE here
            }
        }

        if (bAnyDead)
        {
            // compact the icon array
            i = 0;
            while (i < pSysTrayData->cIcons)
            {
                if (pSysTrayData->pIcons[i].hwnd == NULLHANDLE)
                {
                    --pSysTrayData->cIcons;
                    if (pSysTrayData->cIcons > 0)
                    {
                        memcpy(&pSysTrayData->pIcons[i],
                               &pSysTrayData->pIcons[i + 1],
                               sizeof(*pSysTrayData->pIcons) * (pSysTrayData->cIcons - i));
                    }
                }
                else
                    ++i;
            }

            WgtXSysTrayUpdateAfterIconAddRemove(pWidget);
        }
    }
}

/*
 *@@ fnwpXSysTrayServer:
 *      window procedure for the Extended system tray server window class.
 *
 *      A separate "server" class is necessary because we need a CS_FRAME
 *      top-level window for DDE (which we need to support to be backward
 *      compatible with the System tray wdget from the SysTray/WPS package) and
 *      also to make ourselves discoverable for the client-side implementation
 *      of our new extended API (which queries the class of each top-level
 *      window to find the system tray server).
 */

static
MRESULT EXPENTRY fnwpXSysTrayServer(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;
    // get widget data from QWL_USER_SERVER_DATA (stored there by WM_CREATE)
    PXCENTERWIDGET pWidget =
        (PXCENTERWIDGET)WinQueryWindowPtr(hwnd, QWL_USER_SERVER_DATA);
                    // this ptr is valid after WM_CREATE

    switch (msg)
    {
        case WM_CREATE:
            WinSetWindowPtr(hwnd, QWL_USER_SERVER_DATA, mp1);
        break;

        /*
         * WM_XST_CONTROL:
         *      This is the message sent to us by the clinet-side implementation
         *      of the API to request some function. mp1 points to a
         *      SYSTRAYCTLDATA structure.
         */

        case WM_XST_CONTROL:
            mrc = (MRESULT)WgtXSysTrayControl(hwnd, pWidget,
                                              (PSYSTRAYCTLDATA)mp1);
        break;

        /*
         * WM_TIMER:
         *      timer event.
         */

        case WM_TIMER:
            WgtXSysTrayTimer(hwnd, pWidget, SHORT1FROMMP(mp1));
        break;

        default:
            mrc = WinDefWindowProc(hwnd, msg, mp1, mp2);
    } // end switch(msg)

    return mrc;
}

/* ******************************************************************
 *
 *   Exported procedures
 *
 ********************************************************************/

/*
 *@@ WgtInitModule:
 *      required export with ordinal 1, which must tell
 *      the XCenter how many widgets this DLL provides,
 *      and give the XCenter an array of XCENTERWIDGETCLASS
 *      structures describing the widgets.
 *
 *      With this call, you are given the module handle of
 *      XFLDR.DLL. For convenience, and if you have the full
 *      XWorkplace source code, you could resolve imports
 *      for some useful functions which are exported thru
 *      src\shared\xwp.def. We don't do this here.
 *
 *      This function must also register the PM window classes
 *      which are specified in the XCENTERWIDGETCLASS array
 *      entries. For this, you are given a HAB which you
 *      should pass to WinRegisterClass. For the window
 *      class style (4th param to WinRegisterClass),
 *      you should specify
 *
 +          CS_PARENTCLIP | CS_SIZEREDRAW | CS_SYNCPAINT
 *
 *      Your widget window _will_ be resized by the XCenter,
 *      even if you're not planning it to be.
 *
 *      This function only gets called _once_ when the widget
 *      DLL has been successfully loaded by the XCenter. If
 *      there are several instances of a widget running (in
 *      the same or in several XCenters), this function does
 *      not get called again. However, since the XCenter unloads
 *      the widget DLLs again if they are no longer referenced
 *      by any XCenter, this might get called again when the
 *      DLL is re-loaded.
 *
 *      There will ever be only one load occurence of the DLL.
 *      The XCenter manages sharing the DLL between several
 *      XCenters. As a result, it doesn't matter if the DLL
 *      has INITINSTANCE etc. set or not.
 *
 *      If this returns 0, this is considered an error, and the
 *      DLL will be unloaded again immediately.
 *
 *      If this returns any value > 0, *ppaClasses must be
 *      set to a static array (best placed in the DLL's
 *      global data) of XCENTERWIDGETCLASS structures,
 *      which must have as many entries as the return value.
 */

ULONG EXPENTRY WgtInitModule(HAB hab,               // XCenter's anchor block
                             HMODULE hmodPlugin, // module handle of the widget DLL
                             HMODULE hmodXFLDR,     // XFLDR.DLL module handle
                             PCXCENTERWIDGETCLASS *ppaClasses,
                             PSZ pszErrorMsg)       // if 0 is returned, 500 bytes of error msg
{
    ULONG       ulrc = 0, ul = 0;
    CLASSINFO   ClassInfo;

    LOGF(("hmodPlugin %x\n", hmodPlugin));

    do
    {
        // resolve imports from XFLDR.DLL (this is basically
        // a copy of the doshResolveImports code, but we can't
        // use that before resolving...)
        for (ul = 0;
             ul < sizeof(G_aImports) / sizeof(G_aImports[0]);
             ul++)
        {
            APIRET arc;
            if ((arc = DosQueryProcAddr(hmodXFLDR,
                                        0,               // ordinal, ignored
                                        (PSZ)G_aImports[ul].pcszFunctionName,
                                        G_aImports[ul].ppFuncAddress))
                        != NO_ERROR)
            {
                snprintf(pszErrorMsg, 500,
                         "Import %s failed with %ld.",
                         G_aImports[ul].pcszFunctionName, arc);
                break;
            }
        }
        if (ul < sizeof(G_aImports) / sizeof(G_aImports[0]))
            break;

        // register our PM window class
        if (!WinRegisterClass(hab,
                             WNDCLASS_WIDGET_XSYSTRAY,
                             fnwpXSysTray,
                             CS_PARENTCLIP | CS_SIZEREDRAW | CS_SYNCPAINT,
                             sizeof(PVOID))
                                // extra memory to reserve for QWL_USER
            )
        {
            snprintf(pszErrorMsg, 500,
                     "WinRegisterClass(%s) failed with %lX.",
                     WNDCLASS_WIDGET_XSYSTRAY, WinGetLastError(hab));
            break;
        }

        // get the window data size for the WC_FRAME class (any window class
        // that specifies CS_FRAME must have at least this number, otherise
        // WinRegisterClass returns 0x1003
        if (!WinQueryClassInfo(hab, (PSZ)WC_FRAME, &ClassInfo))
            break;
        QWL_USER_SERVER_DATA = ClassInfo.cbWindowData;

        if (!WinRegisterClass(hab,
                              WNDCLASS_WIDGET_XSYSTRAY_SERVER,
                              fnwpXSysTrayServer,
                              CS_FRAME,
                              QWL_USER_SERVER_DATA + sizeof(PVOID))
                                // extra memory to reserve for QWL_USER
            )
        {
            // error registering class: report error then
            snprintf(pszErrorMsg, 500,
                     "WinRegisterClass(%s) failed with %lX",
                     WNDCLASS_WIDGET_XSYSTRAY_SERVER, WinGetLastError(hab));
            break;
        }

        // no error:
        // return widget classes array
        *ppaClasses = G_WidgetClasses;

        // return no. of classes in this DLL (one here):
        ulrc = sizeof(G_WidgetClasses) / sizeof(G_WidgetClasses[0]);
    }
    while (0);

    LOGF(("pszErrorMsg '%s'\n", pszErrorMsg));
    LOGF(("ulrc %d\n", ulrc));

    return ulrc;
}

/*
 *@@ WgtUnInitModule:
 *      optional export with ordinal 2, which can clean
 *      up global widget class data.
 *
 *      This gets called by the XCenter right before
 *      a widget DLL gets unloaded. Note that this
 *      gets called even if the "init module" export
 *      returned 0 (meaning an error) and the DLL
 *      gets unloaded right away.
 */

VOID EXPENTRY WgtUnInitModule(VOID)
{
}

/*
 *@@ WgtQueryVersion:
 *      this new export with ordinal 3 can return the
 *      XWorkplace version number which is required
 *      for this widget to run. For example, if this
 *      returns 0.9.10, this widget will not run on
 *      earlier XWorkplace versions.
 *
 *      NOTE: This export was mainly added because the
 *      prototype for the "Init" export was changed
 *      with V0.9.9. If this returns 0.9.9, it is
 *      assumed that the INIT export understands
 *      the new FNWGTINITMODULE_099 format (see center.h).
 */

VOID EXPENTRY WgtQueryVersion(PULONG pulMajor,
                              PULONG pulMinor,
                              PULONG pulRevision)
{
    *pulMajor = 0;
    *pulMinor = 9;
    *pulRevision = 9;
}

