/*
 * Extended system tray widget for XCenter/eCenter
 *
 * Public API implementation
 *
 * Made by netlabs.org
 *
 * Author: Dmitry A. Kuminov
 *
 * This software is public domain.
 *
 * WITHOUT ANY WARRANTY..., AT YOUR OWN RISC... ETC.
 *
 */

#define INCL_DOSERRORS
#define INCL_WINWINDOWMGR
#define INCL_WINATOM
#include <os2.h>

#include "xsystray_api.h"
#include "xsystray.h"

#include <string.h>
#include <sys/builtin.h>        // atomics
#include <InnotekLIBC/thread.h> // TLS

static HWND G_hwndSysTray = NULLHANDLE;
static int G_itlsSysTrayCtlData = -1;

static HWND FindSysTrayWindow()
{
    char buf[sizeof(WNDCLASS_WIDGET_XSYSTRAY_SERVER) + 1];
    HWND hwnd;
    HENUM henum = WinBeginEnumWindows(HWND_DESKTOP);
    while ((hwnd = WinGetNextWindow(henum)) != NULLHANDLE)
    {
        LONG len = WinQueryClassName(hwnd, sizeof(buf), buf);
        buf[len] = '\0';
        if (strcmp(WNDCLASS_WIDGET_XSYSTRAY_SERVER, buf) == 0)
            break;
    }
    WinEndEnumWindows(henum);

    return hwnd;
}

static BOOL SendSysTrayCtlMsg(PSYSTRAYCTLDATA pData)
{
    APIRET arc;
    PID pid;
    TID tid;
    MRESULT mrc;

    BOOL bTriedFind = FALSE;

    do
    {
        if (G_hwndSysTray == NULLHANDLE)
        {
            bTriedFind = TRUE;
            HWND hwnd = FindSysTrayWindow();
            __atomic_cmpxchg32((uint32_t *)&G_hwndSysTray, hwnd, NULLHANDLE);
            if (G_hwndSysTray == NULLHANDLE)
                break;
        }

        if (bTriedFind)
        {
            arc = ERROR_INVALID_HANDLE;
            if (WinQueryWindowProcess(G_hwndSysTray, &pid, &tid))
                arc = DosGiveSharedMem(__libc_TLSGet(G_itlsSysTrayCtlData),
                                       pid, PAG_READ | PAG_WRITE);
            if (arc != NO_ERROR)
                break;
        }

        pData->bAcknowledged = FALSE;

        mrc = WinSendMsg(G_hwndSysTray, WM_XST_CONTROL, pData, NULL);
        if (mrc == (MRESULT)TRUE && pData->bAcknowledged)
            return TRUE;

        // if we failed to send the message, it may mean that XCenter was restarted
        // or the system tray was re-enabled. Try to get a new handle (only if we
        // didn't already do it in this call)
        if (!bTriedFind)
        {
            G_hwndSysTray = NULLHANDLE;
            continue;
        }

        break;
    }
    while (TRUE);

    return FALSE;
}

static PSYSTRAYCTLDATA GetSysTrayCtlDataPtr()
{
    APIRET arc;

    // allocate a thread local storage entry if not done so
    if (G_itlsSysTrayCtlData == -1)
    {
        // @todo does XWorkplace have its own TLS? Not? Use
        // DosAllocThreadLocalMemory() directly then (though it's not nice due
        // to the lack of space in that area)
        int itls = __libc_TLSAlloc();
        if (!__atomic_cmpxchg32(&G_itlsSysTrayCtlData, itls, -1))
        {
            // another thread has already got an entry, discard our try
            if (itls != -1)
                __libc_TLSFree(itls);
        }

        if (G_itlsSysTrayCtlData == -1)
            return NULL;
    }

    // allocate a SYSTRAYCTLDATA struct for this thread if not done so
    PSYSTRAYCTLDATA pData = __libc_TLSGet(G_itlsSysTrayCtlData);
    if (!pData)
    {
        arc = DosAllocSharedMem((PVOID)&pData, NULL, sizeof(SYSTRAYCTLDATA),
                                PAG_COMMIT | PAG_READ | PAG_WRITE | OBJ_GIVEABLE);
        if (arc != NO_ERROR)
            return NULL;

        __libc_TLSSet(G_itlsSysTrayCtlData, pData);

        // note that we don't ever free the allocated block since our API doesn't
        // have a concept of initialization/termination and therefore it's fine
        // if the memory stays allocated until application termination
    }

    return pData;
}

/*
 *@@ xstQuerySysTrayVersion:
 *
 *      Returns the version of the Extended system tray in the variables pointed
 *      to by arguments. Any argument may be NULL in which case the
 *      corresponding component of the version is not returned.
 *
 *      Returns TRUE on success and FALSE if the Extended system tray is not
 *      installed or not operational.
 *
 *      NOTE: When the Extended system tray is started up or gets enabled after
 *      being temporarily disabled, it sends a message with the ID returned by
 *      xstGetSysTrayCreatedMsgId() to all top-level WC_FRAME windows on the
 *      desktop to let them add tray icons if they need.
 */

BOOL xstQuerySysTrayVersion(PULONG pulMajor,    // out: major version number
                            PULONG pulMinor,    // out: minor version number
                            PULONG pulRevision) // out: revision number
{
    PSYSTRAYCTLDATA pData = GetSysTrayCtlDataPtr();
    if (!pData)
        return FALSE;

    pData->ulCommand = SYSTRAYCMD_GETVERSION;
    pData->hwndSender = NULLHANDLE;

    BOOL brc = SendSysTrayCtlMsg(pData);
    if (brc)
    {
        if (pulMajor)
            *pulMajor = pData->u.version.ulMajor;
        if (pulMinor)
            *pulMinor = pData->u.version.ulMinor;
        if (pulRevision)
            *pulRevision = pData->u.version.ulRevision;
    }

    return brc;
}

/*
 *@@ xstAddSysTrayIcon:
 *
 *      Adds an icon for the given window handle to the system tray. The icon ID
 *      is used to distinguish between several icons for the same window handle.
 *      If the icon with the specified ID already exists in the system tray, it
 *      will be replaced.
 *
 *      Returns TRUE on success and FALSE otherwise.
 *
 *      The specified window handle receives notification messages about icon
 *      events using the message ID specified by the ulMsgId parameter. The
 *      layout of the message parameters is as follows:
 *
 *          param1
 *              USHORT  usID        icon ID
 *              USHORT  usCode      notify code, one of XST_IN_ constants
 *
 *          param2
 *              PVOID   pData       notify code specific data (see below)
 *
 *      The following notify codes are currently recognized:
 *
 *          XST_IN_MOUSE:
 *              Mouse event in the icon area. Currently, only mouse click
 *              messages are recognized. param2 is a pointer to the XSTMOUSEMSG
 *              structure containing full mouse message details.
 */

BOOL xstAddSysTrayIcon(HWND hwnd,       // in: window handle associated with the icon
                       ULONG ulId,      // in: icon ID to add
                       HPOINTER hIcon,  // in: icon handle
                       ULONG ulMsgId,   // in: message ID for notifications
                       ULONG ulFlags)   // in: flags (not currently used, must be 0)
{
    PSYSTRAYCTLDATA pData = GetSysTrayCtlDataPtr();
    if (!pData)
        return FALSE;

    pData->ulCommand = SYSTRAYCMD_ADDICON;
    pData->hwndSender = hwnd;
    pData->u.icon.ulId = ulId;
    pData->u.icon.hIcon = hIcon;
    pData->u.icon.ulMsgId = ulMsgId;

    return SendSysTrayCtlMsg(pData);
}

/*
 *@@ xstRemoveSysTrayIcon:
 *
 *      Removes the icon previously added by xstAddSysTrayIcon() from the system
 *      tray.
 *
 *      Returns TRUE on success and FALSE otherwise.
 */

BOOL xstRemoveSysTrayIcon(HWND hwnd,    // in: window handle associated with the icon
                          ULONG ulId)   // in: icon ID to remove
{
    PSYSTRAYCTLDATA pData = GetSysTrayCtlDataPtr();
    if (!pData)
        return FALSE;

    pData->ulCommand = SYSTRAYCMD_REMOVEICON;
    pData->hwndSender = hwnd;
    pData->u.icon.ulId = ulId;

    return SendSysTrayCtlMsg(pData);
}

/*
 *@@ xstSetSysTrayIconToolTip:
 *
 *      Sets the tooltip text for the given icon in the system tray. This text
 *      is shown when the mouse pointer is held still over the icon area for
 *      some time.
 *
 *      If pszText is NULL, the tooltip text is reset and will not be shown next
 *      time. The old tooltip is hidden if it is being currently shown.
 *
 *      Returns TRUE on success and FALSE otherwise.
 *
 *      NOTE: The maximum tooltip text length (including terminating null) is
 *      limited to a value returned by xstGetSysTrayMaxTextLen(). If the
 *      supplied string is longer, it will be truncated.
 */

BOOL xstSetSysTrayIconToolTip(HWND hwnd,    // in: window handle associated with the icon
                              ULONG ulId,   // in: icon ID to set the tooltip for
                              PSZ pszText)  // in: tooltip text
{
    PSYSTRAYCTLDATA pData = GetSysTrayCtlDataPtr();
    if (!pData)
        return FALSE;

    pData->ulCommand = SYSTRAYCMD_SETTOOLTIP;
    pData->hwndSender = hwnd;
    pData->u.tooltip.ulId = ulId;

    if (pszText == NULL)
        pData->u.tooltip.szText[0] = '\0';
    else
    {
        strncpy(pData->u.tooltip.szText, pszText,
                sizeof(pData->u.tooltip.szText) - 1);
        // be on the safe side
        pData->u.tooltip.szText[sizeof(pData->u.tooltip.szText) - 1] = '\0';
    }

    return SendSysTrayCtlMsg(pData);
}

BOOL xstShowSysTrayIconBalloon(HWND hwnd, ULONG ulId, PSZ pszTitle, PSZ pszText,
                               ULONG ulFlags, ULONG ulTimeout)
{
    // @todo implement
    return FALSE;
}

BOOL xstHideSysTrayIconBalloon(HWND hwnd, ULONG ulId)
{
    // @todo implement
    return FALSE;
}

/*
 *@@ xstQuerySysTrayIconRect:
 *
 *      Obtains a rectangle occupied by the given icon (in screen coordinates,
 *      top right corner exclusive).
 *
 *      Returns TRUE on success and FALSE otherwise.
 */
BOOL xstQuerySysTrayIconRect(HWND hwnd, ULONG ulId, PRECTL prclRect)
{
    // @todo implement
    return FALSE;
}

/*
 *@@ xstGetSysTrayCreatedMsgId:
 *
 *      Returns the ID of the message that is sent by the Extended system tray
 *      to all top-level WC_FRAME windows on the desktop to let them add tray
 *      icons if they need.
 *
 *      NOTE: The returned value never changes until reboot so it is a good
 *      practice to cache it instead of calling this function each time from the
 *      window procedure of every involved window.
 */

ULONG xstGetSysTrayCreatedMsgId()
{
    static ULONG WM_XST_CREATED = 0;
    if (WM_XST_CREATED == 0)
        WM_XST_CREATED = WinAddAtom(WinQuerySystemAtomTable(),
                                    "ExtendedSysTray.WM_XST_CREATED");
    return WM_XST_CREATED;
}

/*
 *@@ xstGetSysTrayMaxTextLen:
 *
 *      Returns the maximum length of the text (in bytes, including the
 *      terminating null) that can be shown in the tooltop of the icon in the
 *      system tray. You can use the returned value to determine the maximum
 *      length of the string passed as pszText to xstSetSysTrayIconToolTip().
 *
 *      The returned value also defines the maximum length of both the title and
 *      the text (including terminating nulls) of the icon's balloon for the
 *      xstShowSysTrayIconBalloon() call.
 *
 *      Returns TRUE on success and FALSE otherwise.
 *
 *      NOTE: The returned value never changes until reboot so it is a good
 *      practice to cache it instead of calling this function each time.
 */

ULONG xstGetSysTrayMaxTextLen()
{
    return sizeof(((PSYSTRAYCTLDATA)0)->u.tooltip.szText);
}

