/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qabstractfileengine.h"

#include "qfontengine_pm_p.h"

#include <ft2build.h>
#include FT_FREETYPE_H
#include FT_TYPES_H
#include FT_TRUETYPE_TABLES_H

QT_BEGIN_NAMESPACE

static void populateDatabase(const QString& fam)
{
    QFontDatabasePrivate *db = privateDb();
    if (!db)
        return;

    QtFontFamily *family = 0;
    if(!fam.isEmpty()) {
        family = db->family(fam);
        if(family)
            return;
    } else if (db->count) {
        return;
    }

    // we don't recognize foundries on OS/2, use an empty one
    const QString foundryName;

    QList<QByteArray> fontFiles;

    ULONG bufSize = 0;
    BOOL ok = PrfQueryProfileSize(HINI_USERPROFILE, "PM_Fonts", 0, &bufSize);
    Q_ASSERT(ok);
    if (ok) {
        char *buffer = new char[bufSize + 1 /*terminating NULL*/];
        Q_ASSERT(buffer);
        if (buffer) {
            ULONG bufLen = PrfQueryProfileString(HINI_USERPROFILE, "PM_Fonts", 0, 0,
                                                 buffer, bufSize);
            if (bufLen) {
                char *key = buffer;
                while (*key) {
                    ULONG keySize = 0;
                    ok = PrfQueryProfileSize(HINI_USERPROFILE, "PM_Fonts", key,
                                             &keySize);
                    if (ok) {
                        QByteArray file(keySize, 0);
                        ULONG keyLen =
                            PrfQueryProfileString(HINI_USERPROFILE, "PM_Fonts", key, 0,
                                                  file.data(), file.size());
                        file.truncate(keyLen - 1 /*terminating NULL*/);
                        if (!file.isEmpty()) {
                            // FreeType doesn't understand .OFM but understands .PFB
                            if (file.toUpper().endsWith(".OFM")) {
                                file.chop(4);
                                file.append(".PFB");
                            }
                            fontFiles << file;
                        }
                    }
                    key += strlen(key) + 1;
                }
            }
            delete buffer;
        }
    }

    extern FT_Library qt_getFreetype(); // qfontengine_ft.cpp
    FT_Library lib = qt_getFreetype();

    foreach(const QByteArray &file, fontFiles) {
        FT_Face face;
        FT_Error rc = FT_New_Face(lib, file, -1, &face);

        FD_DEBUG("populateDatabase: Font file %s: FT error %d, has %ld faces",
                 file.constData(), (int) rc, rc ? -1 : face->num_faces);
        if (rc != 0)
            continue;

        FT_Long numFaces = face->num_faces;
        FT_Done_Face(face);

        // go throuhg each face
        for (FT_Long idx = 0; idx < numFaces; ++idx) {
            rc = FT_New_Face(lib, file, idx, &face);
            if (rc != 0)
                continue;

            QString familyName = QString::fromLatin1(face->family_name);

            // familyName may contain extra spaces (at least this is true for
            // TNR.PFB that is reported as "Times New Roman ". Trim them.
            familyName = familyName.trimmed();

            QtFontStyle::Key styleKey;

            styleKey.style = face->style_flags & FT_STYLE_FLAG_ITALIC ?
                QFont::StyleItalic : QFont::StyleNormal;

            QList<QFontDatabase::WritingSystem> systems;

            TT_OS2 *os2_table = 0;
            if (face->face_flags & FT_FACE_FLAG_SFNT) {
                os2_table = (TT_OS2 *)FT_Get_Sfnt_Table(face, ft_sfnt_os2);
            }
            if (os2_table) {
                // map weight and width values
                if (os2_table->usWeightClass < 400)
                    styleKey.weight = QFont::Light;
                else if (os2_table->usWeightClass < 600)
                    styleKey.weight = QFont::Normal;
                else if (os2_table->usWeightClass < 700)
                    styleKey.weight = QFont::DemiBold;
                else if (os2_table->usWeightClass < 800)
                    styleKey.weight = QFont::Bold;
                else
                    styleKey.weight = QFont::Black;

                switch (os2_table->usWidthClass) {
                    case 1: styleKey.stretch = QFont::UltraCondensed; break;
                    case 2: styleKey.stretch = QFont::ExtraCondensed; break;
                    case 3: styleKey.stretch = QFont::Condensed; break;
                    case 4: styleKey.stretch = QFont::SemiCondensed; break;
                    case 5: styleKey.stretch = QFont::Unstretched; break;
                    case 6: styleKey.stretch = QFont::SemiExpanded; break;
                    case 7: styleKey.stretch = QFont::Expanded; break;
                    case 8: styleKey.stretch = QFont::ExtraExpanded; break;
                    case 9: styleKey.stretch = QFont::UltraExpanded; break;
                    default: styleKey.stretch = QFont::Unstretched; break;
                }

                quint32 unicodeRange[4] = {
                    os2_table->ulUnicodeRange1, os2_table->ulUnicodeRange2,
                    os2_table->ulUnicodeRange3, os2_table->ulUnicodeRange4
                };
                quint32 codePageRange[2] = {
                    os2_table->ulCodePageRange1, os2_table->ulCodePageRange2
                };
                systems = determineWritingSystemsFromTrueTypeBits(unicodeRange, codePageRange);
            } else {
                // we've only got simple weight information and no stretch
                styleKey.weight = face->style_flags & FT_STYLE_FLAG_BOLD ?
                    QFont::Bold : QFont::Normal;
                styleKey.stretch = QFont::Unstretched;
            }

            QtFontFamily *family = privateDb()->family(familyName, true);

            // @todo is it possible that the same family is both fixed and not?
            Q_ASSERT(!family->fixedPitch || face->face_flags & FT_FACE_FLAG_FIXED_WIDTH);
            family->fixedPitch = face->face_flags & FT_FACE_FLAG_FIXED_WIDTH;

            if (systems.isEmpty()) {
                // it was hard or impossible to determine the actual writing system
                // of the font (as in case of OS/2 bitmap and PFB fonts for which it is
                // usually simply reported that they support standard/system codepages).
                // Pretend that we support all writing systems to not miss the one.
                // @todo find a way to detect it properly to make sure these fonts
                // are not matched for scripts they don't support
                for (int ws = 0; ws < QFontDatabase::WritingSystemsCount; ++ws)
                    family->writingSystems[ws] = QtFontFamily::Supported;
            } else {
                for (int i = 0; i < systems.count(); ++i)
                    family->writingSystems[systems.at(i)] = QtFontFamily::Supported;
            }

            QtFontFoundry *foundry = family->foundry(foundryName, true);
            QtFontStyle *style = foundry->style(styleKey, true);

            // so far, all recognized fonts are antialiased
            style->antialiased = true;

            if ((face->face_flags & FT_FACE_FLAG_SCALABLE) &&
                !style->smoothScalable) {
                // add new scalable style only if it hasn't been already added --
                // the first one of two duplicate (in Qt terms) non-bitmap font
                // styles wins.
                style->smoothScalable = true;
                QtFontSize *size =
                    style->pixelSize(SMOOTH_SCALABLE, true);
                size->fileName = file;
                size->fileIndex = idx;
            }

            // the font may both be scalable and contain fixed size bitmaps
            if (face->face_flags & FT_FACE_FLAG_FIXED_SIZES) {
                for (FT_Int i = 0; i < face->num_fixed_sizes; ++i) {
                    QtFontSize *size =
                        style->pixelSize(face->available_sizes[i].height, true);
                    // the first bitmap style with a given pixel and point size wins
                    if (!size->fileName.isEmpty())
                        continue;
                    size->fileName = file;
                    size->fileIndex = idx;
                }
            }

            FT_Done_Face(face);
        }
    }
}

static void initializeDb()
{
    QFontDatabasePrivate *db = privateDb();
    if (!db || db->count)
        return;

    populateDatabase(QString());

#ifdef QFONTDATABASE_DEBUG
    // print the database
    qDebug("initializeDb:");
    for (int f = 0; f < db->count; f++) {
        QtFontFamily *family = db->families[f];
        qDebug("    %s: %p", qPrintable(family->name), family);
        populateDatabase(family->name);
#if 1
        qDebug("        writing systems supported:");
        QStringList systems;
        for (int ws = 0; ws < QFontDatabase::WritingSystemsCount; ++ws)
            if (family->writingSystems[ws] & QtFontFamily::Supported)
                systems << QFontDatabase::writingSystemName((QFontDatabase::WritingSystem)ws);
        qDebug() << "            " << systems;
        for (int fd = 0; fd < family->count; fd++) {
            QtFontFoundry *foundry = family->foundries[fd];
            qDebug("        %s", foundry->name.isEmpty() ? "(empty foundry)" :
                   qPrintable(foundry->name));
            for (int s = 0; s < foundry->count; s++) {
                QtFontStyle *style = foundry->styles[s];
                qDebug("            style: style=%d weight=%d smooth=%d",  style->key.style,
                       style->key.weight, style->smoothScalable);
                for(int i = 0; i < style->count; ++i) {
                    if (style->pixelSizes[i].pixelSize == SMOOTH_SCALABLE)
                        qDebug("                smooth %s:%d",
                               style->pixelSizes[i].fileName.constData(),
                               style->pixelSizes[i].fileIndex);
                    else
                        qDebug("                %d px %s:%d", style->pixelSizes[i].pixelSize,
                               style->pixelSizes[i].fileName.constData(),
                               style->pixelSizes[i].fileIndex);
                }
            }
        }
#endif
    }
#endif // QFONTDATABASE_DEBUG
}

static inline void load(const QString &family = QString(), int = -1)
{
    populateDatabase(family);
}

static void registerFont(QFontDatabasePrivate::ApplicationFont *fnt)
{
    // @todo implement
}

static QFontDef fontDescToFontDef(const QFontDef &req, const QtFontDesc &desc)
{
    static LONG dpi = -1;
    if (dpi == -1) {
        // PM cannot change resolutions on the fly so cache it
        int hps = qt_display_ps();
        DevQueryCaps(GpiQueryDevice(hps), CAPS_HORIZONTAL_FONT_RES, 1, &dpi);
    }

    QFontDef fontDef;

    fontDef.family = desc.family->name;

    if (desc.size->pixelSize == SMOOTH_SCALABLE) {
        // scalable font matched, calculate the missing size (points or pixels)
        fontDef.pointSize = req.pointSize;
        fontDef.pixelSize = req.pixelSize;
        if (req.pointSize < 0) {
            fontDef.pointSize = req.pixelSize * 72. / dpi;
        } else if (req.pixelSize == -1) {
            fontDef.pixelSize = qRound(req.pointSize * dpi / 72.);
        }
    } else {
        // non-scalable font matched, calculate both point and pixel size
        fontDef.pixelSize = desc.size->pixelSize;
        fontDef.pointSize = desc.size->pixelSize * 72. / dpi;
    }

    fontDef.styleStrategy = req.styleStrategy;
    fontDef.styleHint = req.styleHint;

    fontDef.weight = desc.style->key.weight;
    fontDef.fixedPitch = desc.family->fixedPitch;
    fontDef.style = desc.style->key.style;
    fontDef.stretch = desc.style->key.stretch;

    return fontDef;
}

static QFontEngine *loadEngine(const QFontDef &req, const QtFontDesc &desc)
{
    QFontEngineFT *fe = new QFontEnginePMFT(fontDescToFontDef(req, desc));
    Q_ASSERT(fe);
    if (fe) {
        QFontEngine::FaceId faceId;
        faceId.filename = desc.size->fileName;
        faceId.index = desc.size->fileIndex;
        fe->init(faceId, desc.style->antialiased);
        if (fe->invalid()) {
            FM_DEBUG("   --> invalid!\n");
            delete fe;
            fe = 0;
        }
    }
    return fe;
}

static QFontEngine *loadPM(const QFontPrivate *d, int script, const QFontDef &req)
{
    // list of families to try
    QStringList families = familyList(req);

    const char *styleHint = qt_fontFamilyFromStyleHint(d->request);
    if (styleHint)
        families << QLatin1String(styleHint);

    // add the default family
    QString defaultFamily = QApplication::font().family();
    if (! families.contains(defaultFamily))
        families << defaultFamily;

    // add QFont::defaultFamily() to the list, for compatibility with
    // previous versions
    families << QApplication::font().defaultFamily();

    // null family means find the first font matching the specified script
    families << QString();

    QtFontDesc desc;
    QFontEngine *fe = 0;
    QList<int> blacklistedFamilies;

    while (!fe) {
        for (int i = 0; i < families.size(); ++i) {
            QString family, foundry;
            parseFontName(families.at(i), foundry, family);
            FM_DEBUG("loadPM: >>>>>>>>>>>>>> trying to match '%s'", qPrintable(family));
            QT_PREPEND_NAMESPACE(match)(script, req, family, foundry, -1, &desc, blacklistedFamilies);
            if (desc.family)
                break;
        }
        if (!desc.family)
            break;
        fe = loadEngine(req, desc);
        if (!fe)
            blacklistedFamilies.append(desc.familyIndex);
    }
    return fe;
}

void QFontDatabase::load(const QFontPrivate *d, int script)
{
    Q_ASSERT(script >= 0 && script < QUnicodeTables::ScriptCount);

    // normalize the request to get better caching
    QFontDef req = d->request;
    if (req.pixelSize <= 0)
        req.pixelSize = qMax(1, qRound(req.pointSize * d->dpi / 72.));
    req.pointSize = 0;
    if (req.weight == 0)
        req.weight = QFont::Normal;
    if (req.stretch == 0)
        req.stretch = 100;

    QFontCache::Key key(req, d->rawMode ? QUnicodeTables::Common : script, d->screen);
    if (!d->engineData)
        getEngineData(d, key);

    // the cached engineData could have already loaded the engine we want
    if (d->engineData->engines[script])
        return;

    // set it to the actual pointsize, so QFontInfo will do the right thing
    req.pointSize = req.pixelSize * 72. / d->dpi;

    QFontEngine *fe = QFontCache::instance()->findEngine(key);

    if (!fe) {
        if (qt_enable_test_font && req.family == QLatin1String("__Qt__Box__Engine__")) {
            fe = new QTestFontEngine(req.pixelSize);
            fe->fontDef = req;
        } else {
            QMutexLocker locker(fontDatabaseMutex());
            if (!privateDb()->count)
                initializeDb();
            fe = loadPM(d, script, req);
        }
        if (!fe) {
            fe = new QFontEngineBox(req.pixelSize);
            fe->fontDef = QFontDef();
        }
    }
    if (fe->symbol || (d->request.styleStrategy & QFont::NoFontMerging)) {
        for (int i = 0; i < QUnicodeTables::ScriptCount; ++i) {
            if (!d->engineData->engines[i]) {
                d->engineData->engines[i] = fe;
                fe->ref.ref();
            }
        }
    } else {
        d->engineData->engines[script] = fe;
        fe->ref.ref();
    }
    QFontCache::instance()->insertEngine(key, fe);
}

bool QFontDatabase::removeApplicationFont(int handle)
{
    // @todo implement
    return false;
}

bool QFontDatabase::removeAllApplicationFonts()
{
    // @todo implement
    return false;
}

bool QFontDatabase::supportsThreadedFontRendering()
{
    // @todo implement
    return false;
}

QT_END_NAMESPACE
