/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qabstractfileengine.h"

#include "qfontengine_pm_p.h"

QT_BEGIN_NAMESPACE

static void populateDatabase(const QString& fam)
{
    QFontDatabasePrivate *db = privateDb();
    if (!db)
        return;

    QtFontFamily *family = 0;
    if(!fam.isEmpty()) {
        family = db->family(fam);
        if(family)
            return;
    } else if (db->count) {
        return;
    }

    // we don't recognize foundries on OS/2, use an empty one
    const QString foundryName;

    int hps = qt_display_ps();

    LONG cFonts = 0;
    cFonts = GpiQueryFonts(hps, QF_PUBLIC, NULL, &cFonts, 0, NULL);
    PFONTMETRICS afm = new FONTMETRICS[cFonts];
    GpiQueryFonts(hps, QF_PUBLIC, NULL, &cFonts, sizeof(FONTMETRICS), afm);

    LONG info[2];
    DevQueryCaps(GpiQueryDevice(hps), CAPS_HORIZONTAL_FONT_RES, 2, info);
    LONG xRes = info[0], yRes = info[1];

    for (PFONTMETRICS fm = afm ; cFonts ; cFonts--, fm++) {
        // ignore the default (lMatch = 0) GPI font, since it is always
        // present with non-zero lMatch in the list
        if (!fm->lMatch)
            continue;

        QString familyName = QString::fromLocal8Bit(fm->szFamilyname);
        bool italic = fm->fsSelection & FM_SEL_ITALIC;
        bool fixed = fm->fsType & FM_TYPE_FIXED;
        bool scalable = fm->fsDefn & FM_DEFN_OUTLINE;
        USHORT weight = fm->usWeightClass;
        USHORT width = fm->usWidthClass;

        // ignore bitmap fonts that do not match the current device resolution
        if (!scalable && (fm->sXDeviceRes != xRes || fm->sYDeviceRes != yRes))
            continue;

        // the "@family" fonts are just the same as "family". Ignore them.
        if (familyName[0] == '@')
            continue;

        QtFontStyle::Key styleKey;
        styleKey.style = italic ? QFont::StyleItalic : QFont::StyleNormal;

        if (weight < 4)
            styleKey.weight = QFont::Light;
        else if (weight < 6)
            styleKey.weight = QFont::Normal;
        else if (weight < 7)
            styleKey.weight = QFont::DemiBold;
        else if (weight < 8)
            styleKey.weight = QFont::Bold;
        else
            styleKey.weight = QFont::Black;

        switch (width) {
            case 1: styleKey.stretch = QFont::UltraCondensed; break;
            case 2: styleKey.stretch = QFont::ExtraCondensed; break;
            case 3: styleKey.stretch = QFont::Condensed; break;
            case 4: styleKey.stretch = QFont::SemiCondensed; break;
            case 5: styleKey.stretch = QFont::Unstretched; break;
            case 6: styleKey.stretch = QFont::SemiExpanded; break;
            case 7: styleKey.stretch = QFont::Expanded; break;
            case 8: styleKey.stretch = QFont::ExtraExpanded; break;
            case 9: styleKey.stretch = QFont::UltraExpanded; break;
            default: styleKey.stretch = QFont::Unstretched; break;
        }

        // @todo why?
        familyName.replace('-', ' ');

        QtFontFamily *family = privateDb()->family(familyName, true);
        // @todo is it possible that the same family is both fixed and not?
        family->fixedPitch = fixed;

        QtFontFoundry *foundry = family->foundry(foundryName, true);
        QtFontStyle *style = foundry->style(styleKey, true);

        // add new scalable style only if it hasn't been already added --
        // the first one of two duplicate (in Qt terms) non-bitmap font
        // styles wins.
        if (scalable && style->smoothScalable)
            continue;

        if (style->faceName[0]) {
            // if a duplicate (having the same style in Qt terms) bitmap
            // font is encountered but it has the different facename, ignore
            // it (we have only one facename field per style -- it should be
            // the same for all sizes otherwise we will not be able to create
            // a font with some sizes later).
            if (strcmp(style->faceName, fm->szFacename))
                continue;
        } else {
            strcpy(style->faceName, fm->szFacename);
        }

        if (scalable) {
            style->smoothScalable = TRUE;
            QtFontSize *size =
                style->pixelSize(SMOOTH_SCALABLE, TRUE, fm->sNominalPointSize);
            size->lMatch = fm->lMatch;
        } else {
            QtFontSize *size =
                style->pixelSize(fm->lEmHeight, TRUE, fm->sNominalPointSize);
            // the first bitmap style with a given pixel and point size wins
            if (size->lMatch)
                continue;
            size->lMatch = fm->lMatch;
        }
    }

    delete[] afm;
}

static void initializeDb()
{
    QFontDatabasePrivate *db = privateDb();
    if (!db || db->count)
        return;

    populateDatabase(QString());

#ifdef QFONTDATABASE_DEBUG
    // print the database
    for (int f = 0; f < db->count; f++) {
        QtFontFamily *family = db->families[f];
        qDebug("    %s: %p", qPrintable(family->name), family);
        populateDatabase(family->name);

#if 1
        qDebug("        scripts supported:");
        for (int i = 0; i < QUnicodeTables::ScriptCount; i++)
            if(family->writingSystems[i] & QtFontFamily::Supported)
                qDebug("            %d", i);
        for (int fd = 0; fd < family->count; fd++) {
            QtFontFoundry *foundry = family->foundries[fd];
            qDebug("        %s", qPrintable(foundry->name));
            for (int s = 0; s < foundry->count; s++) {
                QtFontStyle *style = foundry->styles[s];
                qDebug("            style: style=%d weight=%d smooth=%d",  style->key.style,
                       style->key.weight, style->smoothScalable);
                if(!style->smoothScalable) {
                    for(int i = 0; i < style->count; ++i) {
                        qDebug("                %d", style->pixelSizes[i].pixelSize);
                    }
                }
            }
        }
#endif
    }
#endif // QFONTDATABASE_DEBUG
}

static inline void load(const QString &family = QString(), int = -1)
{
    populateDatabase(family);
}

static void registerFont(QFontDatabasePrivate::ApplicationFont *fnt)
{
    // @todo implement
}

static QFontEngine *loadPM(const QFontPrivate *d, int script, const QFontDef &req)
{
    // @todo initializeDb() and stuff, get the engine
    QFontDef fd = req;
    QFontEngine *fe = new QFontEnginePMFT(fd);
    return fe;
}

void QFontDatabase::load(const QFontPrivate *d, int script)
{
    Q_ASSERT(script >= 0 && script < QUnicodeTables::ScriptCount);

    // normalize the request to get better caching
    QFontDef req = d->request;
    if (req.pixelSize <= 0)
        req.pixelSize = qMax(1, qRound(req.pointSize * d->dpi / 72.));
    req.pointSize = 0;
    if (req.weight == 0)
        req.weight = QFont::Normal;
    if (req.stretch == 0)
        req.stretch = 100;

    QFontCache::Key key(req, d->rawMode ? QUnicodeTables::Common : script, d->screen);
    if (!d->engineData)
        getEngineData(d, key);

    // the cached engineData could have already loaded the engine we want
    if (d->engineData->engines[script])
        return;

    // set it to the actual pointsize, so QFontInfo will do the right thing
    req.pointSize = req.pixelSize * 72. / d->dpi;

    QFontEngine *fe = QFontCache::instance()->findEngine(key);

    if (!fe) {
        if (qt_enable_test_font && req.family == QLatin1String("__Qt__Box__Engine__")) {
            fe = new QTestFontEngine(req.pixelSize);
            fe->fontDef = req;
        } else {
            QMutexLocker locker(fontDatabaseMutex());
            if (!privateDb()->count)
                initializeDb();
            fe = loadPM(d, script, req);
        }
        if (!fe) {
            fe = new QFontEngineBox(req.pixelSize);
            fe->fontDef = QFontDef();
        }
    }
    if (fe->symbol || (d->request.styleStrategy & QFont::NoFontMerging)) {
        for (int i = 0; i < QUnicodeTables::ScriptCount; ++i) {
            if (!d->engineData->engines[i]) {
                d->engineData->engines[i] = fe;
                fe->ref.ref();
            }
        }
    } else {
        d->engineData->engines[script] = fe;
        fe->ref.ref();
    }
    QFontCache::instance()->insertEngine(key, fe);
}

bool QFontDatabase::removeApplicationFont(int handle)
{
    // @todo implement
    return false;
}

bool QFontDatabase::removeAllApplicationFonts()
{
    // @todo implement
    return false;
}

bool QFontDatabase::supportsThreadedFontRendering()
{
    // @todo implement
    return false;
}

QT_END_NAMESPACE
