/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

//#define QPROCESS_DEBUG
#include "qdebug.h"

#ifndef QT_NO_PROCESS

#if defined QPROCESS_DEBUG
#include "qstring.h"
#include <ctype.h>

#include <qthread.h>

/*
    Returns a human readable representation of the first \a len
    characters in \a data.
*/
QT_BEGIN_NAMESPACE
static QByteArray qt_prettyDebug(const char *data, int len, int maxSize)
{
    if (!data) return "(null)";
    QByteArray out;
    for (int i = 0; i < len; ++i) {
        char c = data[i];
        if (isprint(c)) {
            out += c;
        } else switch (c) {
        case '\n': out += "\\n"; break;
        case '\r': out += "\\r"; break;
        case '\t': out += "\\t"; break;
        default:
            QString tmp;
            tmp.sprintf("\\%o", c);
            out += tmp.toLatin1();
        }
    }

    if (len < maxSize)
        out += "...";

    return out;
}
QT_END_NAMESPACE
#endif

#include "qplatformdefs.h"

#include "qprocess.h"
#include "qprocess_p.h"

#include <QtCore/qt_os2.h>

#include <private/qcoreapplication_p.h>
#include <private/qthread_p.h>
#include <qdatetime.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qhash.h>
#include <qmutex.h>
#include <qdir.h>

#define HF_STDIN        HFILE(0)
#define HF_STDOUT       HFILE(1)
#define HF_STDERR       HFILE(2)

QT_BEGIN_NAMESPACE

enum
{
    PIPE_SIZE_STDIN = 65520, // max
    PIPE_SIZE_STDOUT = 65520, // max
    PIPE_SIZE_STDERR = 4096,
};

class QProcessManager : public QThread
{
public:

    enum { InvalidProcKey = USHORT(~0), MaxProcKey = (USHORT(~0) >> 2) };

    static void addRef();
    static void release();

    static USHORT addProcess(QProcess *process);
    static void removeProcess(USHORT procKey);

private:

    QProcessManager();
    ~QProcessManager();

    void installSigHandler();
    void uninstallSigHandler();
    void run();

    int refcnt;
    bool finish;

    HEV eventSem;
    QAtomicInt eventSemGuard;
    QAtomicInt deathFlag;

    typedef QHash<USHORT, QProcess *> ProcessList;
    ProcessList processes;
    USHORT lastProcKey;

    void (*sa_old_sigchld_handler)(int);

    static void sa_sigchld_handler(int signum);

    static USHORT toPipeKey(USHORT procKey, QProcessPrivate::PipeType type) {
        Q_ASSERT((procKey << 2) >> 2 == procKey);
        return (procKey << 2) | type;
    }
    static USHORT toProcKey(USHORT key) { return key >> 2; }
    static QProcessPrivate::PipeType toPipeType(USHORT key) {
        return QProcessPrivate::PipeType(key & 0x3);
    }

    static QProcessManager *instance;
    static QMutex mutex;
};

// static
QProcessManager *QProcessManager::instance = 0;
QMutex QProcessManager::mutex;

// static
void QProcessManager::sa_sigchld_handler(int signum)
{
#if defined (QPROCESS_DEBUG)
    fprintf(stderr, "*** SIGCHLD\n");
#endif

    Q_ASSERT(instance);

    // eventSemGuard is used as follows:
    // 0 - QProcessManager is not operational
    // 1 - QProcessManager is being run, eventSem is fine
    // 2 - another signal handler is in action

    if (!instance->eventSemGuard.testAndSetAcquire(1, 2))
        return;

    // set deathEvent to 1 and post the semaphore if not already done so
    if (instance->deathFlag.testAndSetRelaxed(0, 1)) {
        APIRET rc = DosPostEventSem(instance->eventSem);
        Q_ASSERT(rc == NO_ERROR || rc == ERROR_ALREADY_POSTED);
    }

    instance->eventSemGuard.testAndSetRelease(2, 1);

    if (instance->sa_old_sigchld_handler &&
        instance->sa_old_sigchld_handler != SIG_IGN)
        instance->sa_old_sigchld_handler(signum);
}

// static
void QProcessManager::addRef()
{
    QMutexLocker locker(&mutex);

    if (instance == 0) {
        instance = new QProcessManager();
    }

    ++instance->refcnt;
}

// static
void QProcessManager::release()
{
    QMutexLocker locker(&mutex);

    Q_ASSERT(instance);

    if (--instance->refcnt == 0) {
        // make sure we don't globally exist anymore before leaving the lock
        QProcessManager *instance = QProcessManager::instance;
        QProcessManager::instance = 0;
        // disable the signal handler and stop the thread if necessary
        if (instance->isRunning()) {
            Q_ASSERT(instance->eventSemGuard == 1 || instance->eventSemGuard == 2);
            while (!instance->eventSemGuard.testAndSetRelease(1, 0))
                DosSleep(0);
            instance->uninstallSigHandler();
            // stop the thread
            instance->finish = true;
            locker.unlock();
            DosPostEventSem(instance->eventSem);
            instance->wait();
        }
        delete instance;
    }
}

// static
USHORT QProcessManager::addProcess(QProcess *process)
{
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessManager::addProcess(%p)", process);
#endif

    QMutexLocker locker(&mutex);
    Q_ASSERT(instance);

    // lazily enable SIGCHLD handler and start the worker
    if (instance->eventSemGuard.testAndSetAcquire(0, 1)) {
        instance->installSigHandler();
        instance->start();
    }

    USHORT procKey = instance->lastProcKey + 1;
    if (procKey > MaxProcKey) {
        // limit reached, find an unused number
        procKey = 0;
        while (++procKey <= MaxProcKey &&
               instance->processes.contains(procKey));
        Q_ASSERT(procKey <= MaxProcKey);
        if (procKey > MaxProcKey) {
            // oops, no more free keys!
            process->setErrorString(QString("Internal error: Too many processes"));
            return InvalidProcKey;
        }
    } else {
        instance->lastProcKey = procKey;
    }

    // attach the semahpore to the pipes of the process
    APIRET arc = NO_ERROR;
    QProcessPrivate *d = process->d_func();
    if (d->stdinChannel.type == QProcessPrivate::Channel::Normal &&
        d->stdinChannel.pipe.server != HPIPE(~0)) {
        arc = DosSetNPipeSem(d->stdinChannel.pipe.server, (HSEM)instance->eventSem,
                             toPipeKey(procKey, QProcessPrivate::InPipe));
    }
    if (arc == NO_ERROR &&
        d->stdoutChannel.type == QProcessPrivate::Channel::Normal &&
        d->stdoutChannel.pipe.server != HPIPE(~0)) {
        arc = DosSetNPipeSem(d->stdoutChannel.pipe.server, (HSEM)instance->eventSem,
                             toPipeKey(procKey, QProcessPrivate::OutPipe));
    }
    if (arc == NO_ERROR &&
        d->stderrChannel.type == QProcessPrivate::Channel::Normal &&
        d->stderrChannel.pipe.server != HPIPE(~0)) {
        arc = DosSetNPipeSem(d->stderrChannel.pipe.server, (HSEM)instance->eventSem,
                             toPipeKey(procKey, QProcessPrivate::ErrPipe));
    }
    if (arc != NO_ERROR) {
        process->setErrorString(QProcess::tr("Internal error: DOS error %1")
                                .arg(arc));
        if (procKey == instance->lastProcKey)
            --instance->lastProcKey;
        return InvalidProcKey;
    }

    instance->processes[procKey] = process;

    return procKey;
}

// static
void QProcessManager::removeProcess(USHORT procKey)
{
    QMutexLocker locker(&mutex);
    Q_ASSERT(instance);

    Q_ASSERT(instance->processes.contains(procKey));
    QProcess *process = instance->processes[procKey];

#if defined (QPROCESS_DEBUG)
    qDebug("QProcessManager::removeProcess(%p)", process);
#endif

    // to guarantee that the given procKey may be reused, we must close all
    // pipes in order to ensure that we won't get late NPSS_CLOSE for the
    // removed process with the key that may be already associated with a new one
    QProcessPrivate *d = process->d_func();
    d->destroyPipe(d->stdinChannel.pipe);
    d->destroyPipe(d->stdoutChannel.pipe);
    d->destroyPipe(d->stderrChannel.pipe);

    instance->processes.remove(procKey);

    // small optimization: released the highest key
    if (procKey == instance->lastProcKey)
        --instance->lastProcKey;
}

QProcessManager::QProcessManager()
    : refcnt(0), finish(false), eventSem(NULLHANDLE), sa_old_sigchld_handler(0)
{
#if defined (QPROCESS_DEBUG)
    qDebug() << "QProcessManager::QProcessManager()";
#endif

    APIRET rc = DosCreateEventSem(NULL, &eventSem,
                                  DC_SEM_SHARED | DCE_AUTORESET | DCE_POSTONE,
                                  FALSE);
    Q_ASSERT(rc == NO_ERROR);

    lastProcKey = InvalidProcKey;
}

QProcessManager::~QProcessManager()
{
#if defined (QPROCESS_DEBUG)
    qDebug() << "QProcessManager::~QProcessManager()";
#endif

    Q_ASSERT(!refcnt);
    Q_ASSERT(!processes.size());

    DosCloseEventSem(eventSem);
}

void QProcessManager::installSigHandler()
{
    // set up the SIGCHLD handler which calls posts a semaphore wenever
    // our child dies
    struct sigaction oldAction;
    struct sigaction action;
    memset(&action, 0, sizeof(action));
    action.sa_handler = sa_sigchld_handler;
    action.sa_flags = SA_NOCLDSTOP;
    ::sigaction(SIGCHLD, &action, &oldAction);
    if (oldAction.sa_handler != sa_sigchld_handler)
    sa_old_sigchld_handler = oldAction.sa_handler;
}

void QProcessManager::uninstallSigHandler()
{
    struct sigaction oldAction;
    struct sigaction action;
    memset(&action, 0, sizeof(action));
    action.sa_handler = sa_old_sigchld_handler;
    action.sa_flags = SA_NOCLDSTOP;
    ::sigaction(SIGCHLD, &action, &oldAction);
    if (oldAction.sa_handler != sa_sigchld_handler) {
        ::sigaction(SIGCHLD, &oldAction, 0);
    }
}

void QProcessManager::run()
{
#if defined (QPROCESS_DEBUG)
    qDebug() << "QProcessManager::run() BEGIN";
#endif

    // Note: the rationale behind using a worker thread so far is that
    // calling complex functions from a signal handler is not really a good
    // idea unless there is a 100% guarantee that they are reentrant. So, the
    // handler only posts a semaphore (I *hope* DosPostEventSem is reentrant)
    // and all complex work is done here asynchronously.

    QMutexLocker locker(&mutex);
    APIRET arc;

    // an array for 1 process is initially enough
    QVector<PIPESEMSTATE> pipeStates(3 * 2 + 1);

    do {
        locker.unlock();
        qDosNI(arc = DosWaitEventSem(eventSem, SEM_INDEFINITE_WAIT));
        locker.relock();

        if (finish)
            break;

        if (instance->deathFlag.testAndSetRelaxed(1, 0)) {
#if defined (QPROCESS_DEBUG)
            qDebug() << "QProcessManager::run(): child death signaled";
#endif
            foreach (QProcess *proc, processes) {
                QProcessPrivate::WaitMode mode = (QProcessPrivate::WaitMode)
                    proc->d_func()->waitMode.fetchAndStoreRelaxed(QProcessPrivate::SigChild);
                switch (mode) {
                case QProcessPrivate::Semaphore:
                    DosPostEventSem(proc->d_func()->waitSem);
                    break;
                case QProcessPrivate::Async:
                case QProcessPrivate::SigChild:
                    QMetaObject::invokeMethod(proc, "_q_processDied", Qt::QueuedConnection);
                    break;
                }
            }
        }

#if defined (QPROCESS_DEBUG)
        qDebug() << "QProcessManager::run(): checking pipes";
#endif

        // make sure the state array is big enough. The best size for sz pipes
        // is sz * 2 (to be able to store both NPSS_RDATA/NPSS_WSPACE and
        // NPSS_CLOSE for every pipe) + one for NPSS_EOI
        int bestSize = processes.size() * 3 * 2 + 1;
        if (pipeStates.size() < bestSize)
            pipeStates.resize(bestSize);

        arc = DosQueryNPipeSemState((HSEM)eventSem, pipeStates.data(),
                                    pipeStates.size() * sizeof(PIPESEMSTATE));
        if (arc != NO_ERROR) {
            qWarning("QProcessManager::run: DosQueryNPipeSemState returned %lu", arc);
            continue;
        }

        // In the returned information array, CLOSE and READ records for the
        // same pipe key may be mixed. We need CLOSE messages to be posted after
        // READ messages, so we do two passes.

        int pass = 0;
        for (int i = 0; pass < 2; ++i) {
            BYTE status = pipeStates[i].fStatus;
            if (status == NPSS_EOI) {
                ++ pass;
                i = -1;
                continue;
            }
            if (pass == 0 && status != NPSS_RDATA && status != NPSS_WSPACE)
                continue;
            if (pass == 1 && status != NPSS_CLOSE)
                continue;
#if defined(QPROCESS_DEBUG)
            qDebug(" %d/%d: fStatus %u fFlag f%02X usKey %04hX usAvail %hu",
                   pass, i, (uint) pipeStates[i].fStatus,
                   (uint) pipeStates[i].fFlag, pipeStates[i].usKey,
                   pipeStates[i].usAvail);
#endif
            int procKey = toProcKey(pipeStates[i].usKey);
            QProcessPrivate::PipeType type = toPipeType(pipeStates[i].usKey);

            QProcess *proc = processes[procKey];
            Q_ASSERT(proc);
#if defined(QPROCESS_DEBUG)
            qDebug("  process %p", proc);
#endif
            QProcessPrivate *d = proc->d_func();

            if (status == NPSS_CLOSE) {
                // nothing to do but notify the object; it should close the
                // pipe if it sees that it has been notified but there is no
                // data to read, or if it fails to write to a (closed) pipe
            } else {
                // update the counter
                if (d->pipeData[type].newBytes.fetchAndStoreRelaxed(pipeStates[i].usAvail) != 0) {
                    // the object didn't process the previous notification yet;
                    // there is no point to send a new one
                    continue;
                }
            }

            // signal the process object
            QProcessPrivate::WaitMode mode = (QProcessPrivate::WaitMode)
                proc->d_func()->waitMode.fetchAndStoreRelaxed(QProcessPrivate::SigChild);
            switch (mode) {
            case QProcessPrivate::Semaphore:
                DosPostEventSem(proc->d_func()->waitSem);
                break;
            case QProcessPrivate::Async:
            case QProcessPrivate::SigChild:
                const char *method;
                switch(type) {
                case QProcessPrivate::InPipe:
                    Q_ASSERT(status == NPSS_CLOSE || status == NPSS_WSPACE);
                    method = "_q_canWrite";
                    break;
                case QProcessPrivate::OutPipe:
                    Q_ASSERT(status == NPSS_CLOSE || status == NPSS_RDATA);
                    method = "_q_canReadStandardOutput";
                    break;
                case QProcessPrivate::ErrPipe:
                    Q_ASSERT(status == NPSS_CLOSE || status == NPSS_RDATA);
                    method = "_q_canReadStandardError";
                    break;
                }
                QMetaObject::invokeMethod(proc, method, Qt::QueuedConnection);
                break;
            }
        }
    } while (true);

#if defined (QPROCESS_DEBUG)
    qDebug() << "QProcessManager::run() END";
#endif
}


void QProcessPrivate::init()
{
    waitMode = Async;
    waitSem = NULLHANDLE;

    pipeData[InPipe].bytesLeft = 0;
    pipeData[OutPipe].bytesLeft = 0;
    pipeData[ErrPipe].bytesLeft = 0;

    procKey = QProcessManager::InvalidProcKey;

    QProcessManager::addRef();
}

void QProcessPrivate::uninit()
{
    QProcessManager::release();

    if (waitSem != NULLHANDLE)
        DosCloseEventSem(waitSem);
}

void QProcessPrivate::ensureWaitSem()
{
    if (waitSem == NULLHANDLE) {
        APIRET rc = DosCreateEventSem(0, &waitSem, DCE_AUTORESET, FALSE);
        Q_ASSERT(rc == NO_ERROR);
    }
}

bool QProcessPrivate::createPipe(PipeType type, Channel::Pipe &pipe,
                                 const char *name /*= 0*/)
{
    APIRET rc;
    char pathBuf[CCHMAXPATH];

    pipe.server = HPIPE(~0);
    pipe.client = HFILE(~0);

    // we need the process identifier to guarantee pipe name unicity
    PPIB ppib = NULL;
    DosGetInfoBlocks(NULL, &ppib);

    switch (type) {
    case InPipe:
        // create our end of the pipe
        sprintf(pathBuf, "\\pipe\\Qt4\\%08lX\\QProcess\\%p\\%s",
                ppib->pib_ulpid, this->q_func(), name ? name : "Stdin");
        rc = DosCreateNPipe(pathBuf, &pipe.server,
                            NP_ACCESS_OUTBOUND | NP_NOINHERIT,
                            NP_NOWAIT | NP_TYPE_BYTE | 1,
                            PIPE_SIZE_STDIN, 0, 0);
        if (rc == NO_ERROR) {
            DosConnectNPipe(pipe.server);
            // ensure the other end blocks (vital for process->process redirections)
            DosSetNPHState(pipe.server, NP_WAIT);
            // open the client end of the pipe
            ULONG action = 0;
            rc = DosOpen(pathBuf, &pipe.client, &action, 0, FILE_NORMAL, FILE_OPEN,
                         OPEN_ACCESS_READONLY | OPEN_SHARE_DENYREADWRITE |
                         OPEN_FLAGS_NOINHERIT, (PEAOP2)NULL);
        }
        break;
    case OutPipe:
    case ErrPipe:
        // create our end of the pipe
        sprintf(pathBuf, "\\pipe\\Qt4\\%08lX\\QProcess\\%p\\%s",
                ppib->pib_ulpid, this->q_func(),
                name ? name : type == OutPipe ? "Stdout" : "Stderr");
        rc = DosCreateNPipe(pathBuf, &pipe.server,
                            NP_ACCESS_INBOUND | NP_NOINHERIT,
                            NP_NOWAIT | NP_TYPE_BYTE | 1,
                            0, type == OutPipe ? PIPE_SIZE_STDOUT : PIPE_SIZE_STDERR, 0);
        if (rc == NO_ERROR) {
            DosConnectNPipe(pipe.server);
            // ensure the other end blocks (vital for process->process redirections)
            DosSetNPHState(pipe.server, NP_WAIT);
            // open the client end of the pipe
            ULONG action = 0;
            rc = DosOpen(pathBuf, &pipe.client, &action, 0, FILE_NORMAL, FILE_OPEN,
                         OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYREADWRITE |
                         OPEN_FLAGS_NOINHERIT, (PEAOP2)NULL);
        }
        break;
    }

    if (rc != NO_ERROR) {
        qWarning("QProcessPrivate::createPipe: %s(%s) returned %lu",
                 pipe.server == HPIPE(~0) ? "DosCreateNPipe" : "DosOpen",
                 pathBuf, rc);
    }

    return rc == NO_ERROR;
}

void QProcessPrivate::destroyPipe(Channel::Pipe &pipe)
{
    if (pipe.client != ~HFILE(~0)) {
        DosClose(pipe.client);
        pipe.client = HFILE(~0);
    }
    if (pipe.server != HPIPE(~0)) {
        DosDisConnectNPipe(pipe.server);
        DosClose(pipe.server);
        pipe.server = HPIPE(~0);
    }
}

/*
    Create the pipes to a QProcessPrivate::Channel.

    This function must be called in order: stdin, stdout, stderr
*/
bool QProcessPrivate::createChannel(Channel &channel)
{
    Q_Q(QProcess);

    channel.pipe.server = HPIPE(~0);
    channel.pipe.client = HFILE(~0);

    if (&channel == &stderrChannel && processChannelMode == QProcess::MergedChannels) {
        return true;
    }

    if (channel.type == Channel::Normal) {
        // we're piping this channel to our own process
        PipeType type;
        if (&channel == &stdinChannel)
            type = InPipe;
        else if (&channel == &stdoutChannel)
            type = OutPipe;
        else if (&channel == &stderrChannel)
            type = ErrPipe;
        else
            Q_ASSERT(false);
        return createPipe(type, channel.pipe);
    } else if (channel.type == Channel::Redirect) {
        // we're redirecting the channel to/from a file
        QByteArray fname = QFile::encodeName(channel.file);

        APIRET rc;
        if (&channel == &stdinChannel) {
            // try to open in read-only mode
            ULONG action = 0;
            rc = DosOpen(fname, &channel.pipe.client, &action, 0, FILE_NORMAL, FILE_OPEN,
                         OPEN_ACCESS_READONLY | OPEN_FLAGS_NOINHERIT, (PEAOP2)NULL);

            if (rc == NO_ERROR)
                return true; // success

            q->setErrorString(QProcess::tr("Could not open input redirection for reading"));
        } else {
            int mode = FILE_CREATE;
            if (channel.append)
                mode |= FILE_OPEN;
            else
                mode |= FILE_TRUNCATE;
            ULONG action = 0;
            rc = DosOpen(fname, &channel.pipe.client, &action, 0, FILE_NORMAL, mode,
                         OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYWRITE |
                         OPEN_FLAGS_NOINHERIT, (PEAOP2)NULL);

            if (rc == NO_ERROR && channel.append) {
                ULONG actual = 0;
                rc = DosSetFilePtr(channel.pipe.client, 0, FILE_END, &actual);
            }
            if (rc == NO_ERROR)
                return true; // success

            q->setErrorString(QProcess::tr("Could not open output redirection for writing"));
        }

        // could not open file
        processError = QProcess::FailedToStart;
        emit q->error(processError);
        cleanup();
        return false;
    } else {
        Q_ASSERT_X(channel.process, "QProcess::start", "Internal error");

        // the first process started is the server, the second one is the client.
        // the server stores handles to both ends in its channel.pipe member;
        // the other part will pickup the handle of the client end from there.

        if (channel.type == Channel::PipeSource) {
            // we are the source
            Q_ASSERT(&channel == &stdoutChannel);
            Q_ASSERT(channel.process->stdinChannel.process == this &&
                     channel.process->stdinChannel.type == Channel::PipeSink);
            if (channel.process->stdinChannel.pipe.server != HPIPE(~0)) {
                // the other process has already started and became the server
            } else {
                // note: InPipe, since the type is relative to the other side
                createPipe(InPipe, channel.pipe, "Source");
            }
        } else {
            // we are the sink (and the server)
            Q_ASSERT(channel.type == Channel::PipeSink);
            Q_ASSERT(&channel == &stdinChannel);
            Q_ASSERT(channel.process->stdoutChannel.process == this &&
                     channel.process->stdoutChannel.type == Channel::PipeSource);
            if (channel.process->stdoutChannel.pipe.server != HPIPE(~0)) {
                // the other process has already started and became the server
            } else {
                // note: OutPipe, since the type is relative to the other side
                createPipe(OutPipe, channel.pipe, "Sink");
            }
        }

        return true;
    }
}

static int qt_startProcess(const QString &program, const QStringList &arguments,
                           const QString &workingDirectory,
                           const QStringList *environment, bool detached = false)
{
    // @todo P_DETACH isn't supported by kLIBC atm, do something else
    int mode = detached ? P_DETACH : P_NOWAIT;

    // Create argument list with right number of elements, and set the final
    // one to 0.
    char **argv = new char *[arguments.count() + 2];
    argv[arguments.count() + 1] = 0;

    // Encode the program name.
    QByteArray encodedProgramName = QFile::encodeName(program);

    // Add the program name to the argument list.
    argv[0] = encodedProgramName.data();

    // Add every argument to the list
    for (int i = 0; i < arguments.count(); ++i) {
        QString arg = arguments.at(i);
        argv[i + 1] = qstrdup(arg.toLocal8Bit().constData());
    }

    // Duplicate the environment.
    int envc = 0;
    char **envv;
    if (environment && environment->count()) {
        bool seenPATH = false;
        bool seenCOMSPEC = false;

        envv = new char *[environment->count() + 1 + 2 /* may be PATH + COMSPEC */];
        for (; envc < environment->count(); ++envc) {
            QString item = environment->at(envc);
            envv[envc] = qstrdup(item.toLocal8Bit().constData());
            if (!seenPATH)
                seenPATH = !qstrncmp(envv[envc], "PATH=", 4);
            if (!seenCOMSPEC)
                seenCOMSPEC = !qstrncmp(envv[envc], "COMSPEC=", 8);
        }
        if (!seenPATH) {
            // inherit PATH if missing (for convenience)
            // (note that BEGINLIBPATH and ENDLIBPATH, if any, are automatically
            // inherited, while LIBPATH is always a global setting)
            QByteArray path = qgetenv("PATH");
            path.prepend("PATH=");
            envv[envc++] = qstrdup(path);
        }
        // inherit COMSPEC if missing (to let the child start .cmd and .bat)
        if (!seenCOMSPEC) {
            QByteArray comspec = qgetenv("COMSPEC");
            comspec.prepend("COMSPEC=");
            envv[envc++] = qstrdup(comspec);
        }
        envv[envc] = 0;
    } else {
        // inherit the parent environment
        envv = environ;
    }

    // Set the working directory if it's non-empty
    QString curDir;
    if (!workingDirectory.isEmpty()) {
        curDir = QDir::currentPath();
        QDir::setCurrent(workingDirectory);
    }

    // start the program
#ifdef QPROCESS_DEBUG
    qDebug("Trying spawnvpe(%d, \"%s\")",
           mode, encodedProgramName.constData());
#endif
    int pid = spawnvpe(mode, encodedProgramName, argv, envv);

    // if spawnvpe() couldn't find the executable, try .CMD and .BAT
    // extensions (in order of CMD.EXE precedence); spawnvpe() knows how to
    // locate and run them (i.e. using CMD.EXE /c)
    if (pid == -1 && errno == ENOENT) {
        encodedProgramName += ".cmd";
        argv[0] = encodedProgramName.data();
#ifdef QPROCESS_DEBUG
            qDebug("Trying spawnvpe(%d, \"%s\")",
                   mode, encodedProgramName.constData());
#endif
        pid = spawnvpe(mode, encodedProgramName, argv, envv);
        if ( pid == -1 && errno == ENOENT ) {
            qstrcpy(encodedProgramName.data() + encodedProgramName.length() - 4, ".bat");
            argv[0] = encodedProgramName.data();
#ifdef QPROCESS_DEBUG
            qDebug("Trying spawnvpe(%d, \"%s\")",
                   mode, encodedProgramName.constData());
#endif
            pid = spawnvpe(mode, encodedProgramName, argv, envv);
        }
    }

    // Clean up duplicated memory.
    for (int i = 1 /* 0 is encodedProgramName */; i <= arguments.count(); ++i)
        delete [] argv[i];
    delete [] argv;
    if (envv != environ) {
        for (int i = 0; i < envc; ++i)
            delete [] envv[i];
        delete [] envv;
    }

    // restore the current directory
    QDir::setCurrent(curDir);

    return pid;
}

void QProcessPrivate::startProcess()
{
    Q_Q(QProcess);

#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::startProcess()");
#endif

    // Initialize pipes
    if (!createChannel(stdinChannel))
        return;
    if (processChannelMode != QProcess::ForwardedChannels) {
        if (!createChannel(stdoutChannel) || !createChannel(stderrChannel))
            return;
    }

    procKey = QProcessManager::addProcess(q);

    if (procKey == QProcessManager::InvalidProcKey) {
        // setErrorString() is called inside addProcess()
        Q_ASSERT(!q->errorString().isEmpty());
        processError = QProcess::FailedToStart;
        emit q->error(processError);
        cleanup();
        return;
    }

    // Start the process (platform dependent)

    q->setProcessState(QProcess::Starting);

    APIRET arc;

    QString error;
    HFILE tmpStdin = HFILE(~0), tmpStdout = HFILE(~0), tmpStderr = HFILE(~0);
    HFILE realStdin = HF_STDIN, realStdout = HF_STDOUT, realStderr = HF_STDERR;

    do {
        // save & copy the stdin handle
        if ((arc = DosDupHandle(realStdin, &tmpStdin)) == NO_ERROR) {
            HFILE handle = stdinChannel.pipe.client;
            if (stdinChannel.type == Channel::PipeSink) {
                // process -> process redirection
                if (stdinChannel.pipe.server != HPIPE(~0)) {
                    // we are the server
                    handle = (HFILE)stdinChannel.pipe.server;
                } else {
                    // we are the client, use the server's variable
                    handle = stdinChannel.process->stdoutChannel.pipe.client;
                }
                Q_ASSERT(handle != HFILE(~0));
            }
            arc = DosDupHandle(handle, &realStdin);
        }
        if (arc != NO_ERROR) {
#if defined (QPROCESS_DEBUG)
            qDebug("QProcessPrivate::startProcess: DosDupHandle for STDIN "
                   "returned %lu", arc);
#endif
            break;
        }
        // save & copy the stdout and stderr handles if asked to
        if (processChannelMode != QProcess::ForwardedChannels) {
            // save & copy the stdout handle
            if ((arc = DosDupHandle(realStdout, &tmpStdout)) == NO_ERROR) {
                HFILE handle = stdoutChannel.pipe.client;
                if (stdoutChannel.type == Channel::PipeSource) {
                    // process -> process redirection
                    if (stdoutChannel.pipe.server != HPIPE(~0)) {
                        // we are the server
                        handle = (HFILE)stdoutChannel.pipe.server;
                    } else {
                        // we are the client, use the server's variable
                        handle = stdoutChannel.process->stdinChannel.pipe.client;
                    }
                    Q_ASSERT(handle != HFILE(~0));
                }
                arc = DosDupHandle(handle, &realStdout);
            }
            if (arc != NO_ERROR) {
#if defined (QPROCESS_DEBUG)
                qDebug("QProcessPrivate::startProcess: DosDupHandle for STDOUT "
                       "returned %lu", arc);
#endif
                break;
            }
            if ((arc = DosDupHandle(realStderr, &tmpStderr)) == NO_ERROR) {
                // merge stdout and stderr if asked to
                if (processChannelMode == QProcess::MergedChannels) {
                    arc = DosDupHandle(realStdout, &realStderr);
                } else {
                    // qDebug() uses STDERR so only redirect if !QPROCESS_DEBUG
#if !defined (QPROCESS_DEBUG)
                    arc = DosDupHandle(stderrChannel.pipe.client, &realStderr);
#endif
                }
            }
            if (arc != NO_ERROR)
                break;
        }

    } while (false);

    int pid = -1;
    if (arc == NO_ERROR)
        pid = qt_startProcess(program, arguments, workingDirectory,
                              &environment);

    // cancel STDIN/OUT/ERR redirections
    if (tmpStdin != HFILE(~0)) {
        DosDupHandle(tmpStdin, &realStdin);
        DosClose(tmpStdin);
    }
    if (tmpStdout != HFILE(~0)) {
        DosDupHandle(tmpStdout, &realStdout);
        DosClose(tmpStdout);
    }
    if ( tmpStderr != HFILE(~0)) {
        DosDupHandle(tmpStderr, &realStderr);
        DosClose(tmpStderr);
    }

    if (arc != NO_ERROR || pid == -1) {
        // Cleanup, report error and return
        q->setProcessState(QProcess::NotRunning);
        processError = QProcess::FailedToStart;
        if (arc != NO_ERROR) {
            // handle duplication failed
            q->setErrorString(QProcess::tr("Process failed to start: %1")
                                           .arg(QString("DOS error %1").arg(arc)));
        } else {
#if defined (QPROCESS_DEBUG)
            qDebug("spawnvpe failed: %s", qPrintable(qt_error_string(errno)));
#endif
            q->setErrorString(QProcess::tr("Process failed to start: %1")
                                           .arg(qPrintable(qt_error_string(errno))));
        }
        emit q->error(processError);
        QProcessManager::removeProcess(procKey);
        procKey = QProcessManager::InvalidProcKey;
        cleanup();
        return;
    }

    this->pid = Q_PID(pid);

    // close the client ends inherited by the started process (it's necessary to
    // make sure that the started process owns the only handle to the client end
    // and when it closes this handle the other party will notice it and e.g.
    // stop waiting for new data).

    if (stdinChannel.type == Channel::PipeSink) {
        // process -> process redirection
        if (stdinChannel.pipe.server != HPIPE(~0)) {
            // we are the server, leave the handle for the other party
        } else {
            // we are the client, close the handle
            DosClose(stdinChannel.process->stdoutChannel.pipe.client);
            stdinChannel.process->stdoutChannel.pipe.client = HFILE(~0);
        }
    } else {
        if (stdinChannel.pipe.client != HFILE(~0)) {
            DosClose(stdinChannel.pipe.client);
            stdinChannel.pipe.client = HFILE(~0);
        }
    }
    if (stdoutChannel.type == Channel::PipeSource) {
        // process -> process redirection
        if (stdoutChannel.pipe.server != HPIPE(~0)) {
            // we are the server, leave the handle for the other party
        } else {
            // we are the client, close the handle
            DosClose(stdoutChannel.process->stdinChannel.pipe.client);
            stdoutChannel.process->stdinChannel.pipe.client = HFILE(~0);
        }
    } else {
        if (stdoutChannel.pipe.client != HFILE(~0)) {
            DosClose(stdoutChannel.pipe.client);
            stdoutChannel.pipe.client = HFILE(~0);
        }
    }
    if (stderrChannel.pipe.client != HFILE(~0)) {
        DosClose(stderrChannel.pipe.client);
        stderrChannel.pipe.client = HFILE(~0);
    }

    // give the process a chance to start ...
    DosSleep(100);

    _q_startupNotification();
}

bool QProcessPrivate::processStarted()
{
    // we don't actually wait for any notification from the child process
    // assuming it has been started as long as spawnvpe() returns success
    return processState == QProcess::Starting;
}

qint64 QProcessPrivate::bytesAvailableFromStdout() const
{
    QProcessPrivate* that = const_cast<QProcessPrivate*>(this);

    int newBytes = 0;
    if (dying) {
        // we are dying and won't get notifications from QProcessManager
        // anymore, look manually if there's stiil something in the pipe
        APIRET arc;
        ULONG state;
        AVAILDATA avail;
        if (that->stdoutChannel.type == QProcessPrivate::Channel::Normal &&
            that->stdoutChannel.pipe.server != HPIPE(~0)) {
            arc = DosPeekNPipe(that->stdoutChannel.pipe.server, 0, 0, 0, &avail, &state);
            Q_ASSERT(arc == NO_ERROR || arc == ERROR_INVALID_PARAMETER);
            // note that even if ERROR_INVALID_PARAMETER, it seems to return the
            // correct values in avail and state (undocumented)
            newBytes = avail.cbpipe;
            that->pipeData[OutPipe].newBytes = 0;
        }
    } else {
        // grab new bytes from QProcessManager (if any)
        newBytes = that->pipeData[OutPipe].newBytes.fetchAndStoreRelaxed(0);
    }

    that->pipeData[OutPipe].bytesLeft += newBytes;

#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::bytesAvailableFromStdout() == %lld",
           that->pipeData[OutPipe].bytesLeft);
#endif
    return that->pipeData[OutPipe].bytesLeft;
}

qint64 QProcessPrivate::bytesAvailableFromStderr() const
{
    QProcessPrivate* that = const_cast<QProcessPrivate*>(this);

    int newBytes = 0;
    if (dying) {
        // we are dying and won't get notifications from QProcessManager
        // anymore, look manually if there's stiil something in the pipe
        APIRET arc;
        ULONG state;
        AVAILDATA avail;
        if (that->stderrChannel.type == QProcessPrivate::Channel::Normal &&
            that->stderrChannel.pipe.server != HPIPE(~0)) {
            arc = DosPeekNPipe(that->stderrChannel.pipe.server, 0, 0, 0, &avail, &state);
            Q_ASSERT(arc == NO_ERROR || arc == ERROR_INVALID_PARAMETER);
            // note that even if ERROR_INVALID_PARAMETER, it seems to return the
            // correct values in avail and state (undocumented)
            newBytes = avail.cbpipe;
            that->pipeData[ErrPipe].newBytes = 0;
        }
    } else {
        // grab new bytes from QProcessManager (if any)
        newBytes = that->pipeData[ErrPipe].newBytes.fetchAndStoreRelaxed(0);
    }

#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::bytesAvailableFromStderr() == %lld",
           that->pipeData[ErrPipe].bytesLeft);
#endif
    return that->pipeData[ErrPipe].bytesLeft;
}

qint64 QProcessPrivate::bytesAvailableInStdin() const
{
    QProcessPrivate* that = const_cast<QProcessPrivate*>(this);

    // grab new bytes from QProcessManager (if any)
    int newBytes = that->pipeData[InPipe].newBytes.fetchAndStoreRelaxed(0);
    that->pipeData[InPipe].bytesLeft += newBytes;

#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::bytesAvailableInStdin() == %lld",
           that->pipeData[InPipe].bytesLeft);
#endif
    return that->pipeData[InPipe].bytesLeft;
}

qint64 QProcessPrivate::readFromStdout(char *data, qint64 maxlen)
{
    ULONG actual = 0;
    APIRET arc = DosRead(stdoutChannel.pipe.server, data, maxlen, &actual);

    qint64 bytesRead = -1;
    if (arc == NO_ERROR) {
        bytesRead = (qint64)actual;
        // update our counter
        Q_ASSERT(pipeData[OutPipe].bytesLeft >= bytesRead);
        pipeData[OutPipe].bytesLeft -= bytesRead;
    }

#if defined QPROCESS_DEBUG
    qDebug("QProcessPrivate::readFromStdout(%p \"%s\", %lld) == %lld",
           data, qt_prettyDebug(data, bytesRead, 16).constData(), maxlen, bytesRead);
#endif
    return bytesRead;
}

qint64 QProcessPrivate::readFromStderr(char *data, qint64 maxlen)
{
    ULONG actual = 0;
    APIRET arc = DosRead(stderrChannel.pipe.server, data, maxlen, &actual);

    qint64 bytesRead = -1;
    if (arc == NO_ERROR) {
        bytesRead = (qint64)actual;
        // update our counter
        Q_ASSERT(pipeData[ErrPipe].bytesLeft >= bytesRead);
        pipeData[ErrPipe].bytesLeft -= bytesRead;
    }

#if defined QPROCESS_DEBUG
    qDebug("QProcessPrivate::readFromStderr(%p \"%s\", %lld) == %lld",
           data, qt_prettyDebug(data, bytesRead, 16).constData(), maxlen, bytesRead);
#endif
    return bytesRead;
}

qint64 QProcessPrivate::writeToStdin(const char *data, qint64 maxlen)
{
    ULONG actual = 0;
    APIRET arc = DosWrite(stdinChannel.pipe.server, data, maxlen, &actual);

    qint64 written = -1;
    if (arc == NO_ERROR) {
        written = (qint64)actual;
        // update our counter
        Q_ASSERT(pipeData[InPipe].bytesLeft >= written);
        pipeData[InPipe].bytesLeft -= written;
    }

#if defined QPROCESS_DEBUG
    qDebug("QProcessPrivate::writeToStdin(%p \"%s\", %lld) == %lld",
           data, qt_prettyDebug(data, maxlen, 16).constData(), maxlen, written);
#endif
    return written;
}

void QProcessPrivate::terminateProcess()
{
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::terminateProcess()");
#endif
    if (pid) {
        HSWITCH hswitch = WinQuerySwitchHandle(NULL, pid);
        if (hswitch != NULLHANDLE) {
            SWCNTRL swcntrl = { 0 };
            APIRET rc = WinQuerySwitchEntry(hswitch,  &swcntrl);
            // WinQuerySwitchEntry will return a switch entry of the parent
            // process if the specfied one doesn't have a separate session
            // (running a plain CMD.EXE is an example); ignore this case.
            if (rc == NO_ERROR && swcntrl.idProcess == pid)
            {
                // first, ensure that the Close action is enabled in the main frame
                // window (otherwise WM_SYSCOMMAND/SC_CLOSE will be ignored)
                HWND hwndSysMenu = WinWindowFromID(swcntrl.hwnd, FID_SYSMENU);
                if (hwndSysMenu) {
                    WinPostMsg(hwndSysMenu, MM_SETITEMATTR,
                               MPFROM2SHORT(SC_CLOSE, TRUE),
                               MPFROM2SHORT(MIA_DISABLED, 0));
                }
                WinPostMsg(swcntrl.hwnd, WM_SYSCOMMAND,
                           MPFROM2SHORT(SC_CLOSE, CMDSRC_OTHER),
                           MPFROMLONG(FALSE));
            }
        }
    }
}

void QProcessPrivate::killProcess()
{
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::killProcess()");
#endif
    if (pid)
        DosKillProcess(DKP_PROCESS, pid);
}

/*
   Returns the difference between msecs and elapsed. If msecs is -1,
   however, -1 is returned.
*/
static int qt_timeout_value(int msecs, int elapsed)
{
    if (msecs == -1)
        return -1;

    int timeout = msecs - elapsed;
    return timeout < 0 ? 0 : timeout;
}

bool QProcessPrivate::waitForStarted(int msecs)
{
    Q_Q(QProcess);

    if (processState == QProcess::Running)
        return true;

    if (processError == QProcess::FailedToStart)
        return false;

    processError = QProcess::Timedout;
    q->setErrorString(QProcess::tr("Process operation timed out"));
    return false;
}

bool QProcessPrivate::waitForReadyRead(int msecs)
{
    Q_Q(QProcess);
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::waitForReadyRead(%d)", msecs);
#endif

    QTime stopWatch;
    stopWatch.start();

    APIRET arc;
    ensureWaitSem();

    forever {
        bool timedOut = false, failed = false;
        if (waitMode.testAndSetRelaxed(Async, Semaphore)) {
            int timeout = qt_timeout_value(msecs, stopWatch.elapsed());
            qDosNI(arc = DosWaitEventSem(waitSem, (ULONG)timeout));
            if (arc == ERROR_TIMEOUT) {
                timedOut = true;
            } else if (arc != NO_ERROR) {
                Q_ASSERT(arc == NO_ERROR);
                failed = true;
            } else {
                bool readyReadEmitted = false;
                if (pipeData[OutPipe].newBytes.fetchAndAddRelaxed(0)) {
                    bool canRead = _q_canReadStandardOutput();
                    if (processChannel == QProcess::StandardOutput && canRead)
                        readyReadEmitted = true;
                }
                if (pipeData[ErrPipe].newBytes.fetchAndAddRelaxed(0)) {
                    bool canRead = _q_canReadStandardError();
                    if (processChannel == QProcess::StandardError && canRead)
                        readyReadEmitted = true;
                }
                if (readyReadEmitted) {
                    waitMode.fetchAndStoreRelaxed(Async);
                    return true;
                }

                if (pipeData[InPipe].newBytes.fetchAndAddRelaxed(0))
                    _q_canWrite();

                if (!pid)
                    failed = true;
            }
        } else {
            // we've got SIGCHLD, proceeed to _q_processDied()
        }

        waitMode.fetchAndStoreRelaxed(Async);

        if (timedOut || failed) {
            if (timedOut) {
                processError = QProcess::Timedout;
                q->setErrorString(QProcess::tr("Process operation timed out"));
            }
            break;
        }

        if (_q_processDied())
            return false;
    }
    return false;
}

bool QProcessPrivate::waitForBytesWritten(int msecs)
{
    Q_Q(QProcess);
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::waitForBytesWritten(%d)", msecs);
#endif

    QTime stopWatch;
    stopWatch.start();

    APIRET arc;
    ensureWaitSem();

    while (!writeBuffer.isEmpty()) {
        bool timedOut = false, failed = false;
        if (waitMode.testAndSetRelaxed(Async, Semaphore)) {
            int timeout = qt_timeout_value(msecs, stopWatch.elapsed());
            qDosNI(arc = DosWaitEventSem(waitSem, (ULONG)timeout));
            if (arc == ERROR_TIMEOUT) {
                timedOut = true;
            } else if (arc != NO_ERROR) {
                Q_ASSERT(arc == NO_ERROR);
                failed = true;
            } else {
                if (pipeData[InPipe].newBytes.fetchAndAddRelaxed(0)) {
                    waitMode.fetchAndStoreRelaxed(Async);
                    return _q_canWrite();
                }

                if (pipeData[OutPipe].newBytes.fetchAndAddRelaxed(0))
                    _q_canReadStandardOutput();

                if (pipeData[ErrPipe].newBytes.fetchAndAddRelaxed(0))
                    _q_canReadStandardError();

                if (!pid)
                    failed = true;
            }
        } else {
            // we've got SIGCHLD, proceeed to _q_processDied()
        }

        waitMode.fetchAndStoreRelaxed(Async);

        if (timedOut || failed) {
            if (timedOut) {
                processError = QProcess::Timedout;
                q->setErrorString(QProcess::tr("Process operation timed out"));
            }
            break;
        }

        if (_q_processDied())
            return false;
    }

    return false;
}

bool QProcessPrivate::waitForFinished(int msecs)
{
    Q_Q(QProcess);
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::waitForFinished(%d)", msecs);
#endif

    QTime stopWatch;
    stopWatch.start();

    APIRET arc;
    ensureWaitSem();

    forever {
        bool timedOut = false, failed = false;
        if (waitMode.testAndSetRelaxed(Async, Semaphore)) {
            int timeout = qt_timeout_value(msecs, stopWatch.elapsed());
            qDosNI(arc = DosWaitEventSem(waitSem, (ULONG)timeout));
            if (arc == ERROR_TIMEOUT) {
                timedOut = true;
            } else if (arc != NO_ERROR) {
                Q_ASSERT(arc == NO_ERROR);
                failed = true;
            } else {
                if (pipeData[InPipe].newBytes.fetchAndAddRelaxed(0))
                    _q_canWrite();

                if (pipeData[OutPipe].newBytes.fetchAndAddRelaxed(0))
                    _q_canReadStandardOutput();

                if (pipeData[ErrPipe].newBytes.fetchAndAddRelaxed(0))
                    _q_canReadStandardError();

                if (!pid) {
                    waitMode.fetchAndStoreRelaxed(Async);
                    return true;
                }
            }
        } else {
            // we've got SIGCHLD, proceeed to _q_processDied()
        }

        waitMode.fetchAndStoreRelaxed(Async);

        if (timedOut || failed) {
            if (timedOut) {
                processError = QProcess::Timedout;
                q->setErrorString(QProcess::tr("Process operation timed out"));
            }
            break;
        }

        if (_q_processDied())
            return true;
    }
    return false;
}

bool QProcessPrivate::waitForWrite(int msecs)
{
    // ### this function isn't actually used in OS/2 and Unix code paths

    APIRET arc;
    ensureWaitSem();

    bool ret = false;
    if (waitMode.testAndSetRelaxed(Async, Semaphore)) {
        qDosNI(arc = DosWaitEventSem(waitSem, (ULONG)msecs));
        if (arc == NO_ERROR) {
            ret = pipeData[InPipe].newBytes.fetchAndAddRelaxed(0);
        }
    }
    waitMode.fetchAndStoreRelaxed(Async);
    return ret;
}

void QProcessPrivate::findExitCode()
{
    // note: this method is unconditionally called from QProcess destructor
    // to make sure the process manager removes the watcher even in such a rare
    // case when the process is still running after killing it and waiting
    // for termination (in which case the child termination code path that
    // normally calls findExitCode() won't be walked)

    if (procKey != QProcessManager::InvalidProcKey) {
        QProcessManager::removeProcess(procKey);
        procKey = QProcessManager::InvalidProcKey;
    }
}

bool QProcessPrivate::waitForDeadChild()
{
    // check if our process is dead
    int exitStatus;
    pid_t waitResult = waitpid(pid, &exitStatus, WNOHANG);
    if (waitResult > 0) {
        crashed = !WIFEXITED(exitStatus);
        exitCode = WEXITSTATUS(exitStatus);
#if defined QPROCESS_DEBUG
        qDebug() << "QProcessPrivate::waitForDeadChild() dead with exitCode"
                 << exitCode << ", crashed?" << crashed;
#endif
        return true;
    }
#if defined QPROCESS_DEBUG
    qDebug() << "QProcessPrivate::waitForDeadChild() not dead!";
    if (waitResult == -1)
        qDebug() << "QProcessPrivate::waitForDeadChild()" << strerror(errno);
#endif
    return false;
}

void QProcessPrivate::_q_notified()
{
}

/*! \internal
 */
bool QProcessPrivate::startDetached(const QString &program, const QStringList &arguments,
                                    const QString &workingDirectory, qint64 *pid)
{
    int startedPid = qt_startProcess(program, arguments, workingDirectory,
                                     NULL, true /* detached */);

    if (startedPid == -1)
        return false;

    *pid = startedPid;
    return true;
}

QT_END_NAMESPACE

#endif // QT_NO_PROCESS
