/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

//#define QPROCESS_DEBUG
#include "qdebug.h"

#ifndef QT_NO_PROCESS

#if defined QPROCESS_DEBUG
#include "qstring.h"
#include <ctype.h>

/*
    Returns a human readable representation of the first \a len
    characters in \a data.
*/
QT_BEGIN_NAMESPACE
static QByteArray qt_prettyDebug(const char *data, int len, int maxSize)
{
    if (!data) return "(null)";
    QByteArray out;
    for (int i = 0; i < len; ++i) {
        char c = data[i];
        if (isprint(c)) {
            out += c;
        } else switch (c) {
        case '\n': out += "\\n"; break;
        case '\r': out += "\\r"; break;
        case '\t': out += "\\t"; break;
        default:
            QString tmp;
            tmp.sprintf("\\%o", c);
            out += tmp.toLatin1();
        }
    }

    if (len < maxSize)
        out += "...";

    return out;
}
QT_END_NAMESPACE
#endif

#include "qplatformdefs.h"

#include "qprocess.h"
#include "qprocess_p.h"

#include <QtCore/qt_os2.h>

#include <private/qcoreapplication_p.h>
#include <private/qthread_p.h>
#include <qdatetime.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qlist.h>
#include <qmap.h>
#include <qmutex.h>
#include <qsemaphore.h>
#include <qsocketnotifier.h>
#include <qdir.h>

#include <errno.h>
#include <sys/filio.h>
#include <sys/socket.h> // socketpair

QT_BEGIN_NAMESPACE

static void qt_create_pipe(int *pipe)
{
    if (pipe[0] != -1)
        ::close(pipe[0]);
    if (pipe[1] != -1)
        ::close(pipe[1]);
    // Note: with kLIBC 0.6.3 and earlier (GCC 3.3.5 CSD3), fds created with
    // pipe() cannot be used in select(), so use socketpair() instead
    if (::socketpair(AF_OS2, SOCK_STREAM, 0, pipe) != 0) {
        qWarning("QProcessPrivate::createPipe: ::socketpair() failed with '%s'",
                 qPrintable(qt_error_string(errno)));
    }
    ::fcntl(pipe[0], F_SETFD, FD_CLOEXEC);
    ::fcntl(pipe[1], F_SETFD, FD_CLOEXEC);
}

void QProcessPrivate::destroyPipe(int *pipe)
{
    if (pipe[1] != -1) {
        ::close(pipe[1]);
        pipe[1] = -1;
    }
    if (pipe[0] != -1) {
        ::close(pipe[0]);
        pipe[0] = -1;
    }
}

/*
    Create the pipes to a QProcessPrivate::Channel.

    This function must be called in order: stdin, stdout, stderr
*/
bool QProcessPrivate::createChannel(Channel &channel)
{
    Q_Q(QProcess);

    if (&channel == &stderrChannel && processChannelMode == QProcess::MergedChannels) {
        channel.pipe[0] = -1;
        channel.pipe[1] = -1;
        return true;
    }

    if (channel.type == Channel::Normal) {
        // we're piping this channel to our own process
        qt_create_pipe(channel.pipe);

        // create the socket notifiers
        if (threadData->eventDispatcher) {
            if (&channel == &stdinChannel) {
                channel.notifier = new QSocketNotifier(channel.pipe[1],
                                                       QSocketNotifier::Write, q);
                channel.notifier->setEnabled(false);
                QObject::connect(channel.notifier, SIGNAL(activated(int)),
                                 q, SLOT(_q_canWrite()));
            } else {
                channel.notifier = new QSocketNotifier(channel.pipe[0],
                                                       QSocketNotifier::Read, q);
                const char *receiver;
                if (&channel == &stdoutChannel)
                    receiver = SLOT(_q_canReadStandardOutput());
                else
                    receiver = SLOT(_q_canReadStandardError());
                QObject::connect(channel.notifier, SIGNAL(activated(int)),
                                 q, receiver);
            }
        }

        return true;
    } else if (channel.type == Channel::Redirect) {
        // we're redirecting the channel to/from a file
        QByteArray fname = QFile::encodeName(channel.file);

        if (&channel == &stdinChannel) {
            // try to open in read-only mode
            channel.pipe[1] = -1;
            if ( (channel.pipe[0] = ::open(fname, O_RDONLY)) != -1)
                return true;    // success

            q->setErrorString(QProcess::tr("Could not open input redirection for reading"));
        } else {
            int mode = O_WRONLY | O_CREAT;
            if (channel.append)
                mode |= O_APPEND;
            else
                mode |= O_TRUNC;

            channel.pipe[0] = -1;
            if ( (channel.pipe[1] = ::open(fname, mode, 0666)) != -1)
                return true; // success

            q->setErrorString(QProcess::tr("Could not open output redirection for writing"));
        }

        // could not open file
        processError = QProcess::FailedToStart;
        emit q->error(processError);
        cleanup();
        return false;
    } else {
        Q_ASSERT_X(channel.process, "QProcess::start", "Internal error");

        Channel *source;
        Channel *sink;

        if (channel.type == Channel::PipeSource) {
            // we are the source
            source = &channel;
            sink = &channel.process->stdinChannel;

            Q_ASSERT(source == &stdoutChannel);
            Q_ASSERT(sink->process == this && sink->type == Channel::PipeSink);
        } else {
            // we are the sink;
            source = &channel.process->stdoutChannel;
            sink = &channel;

            Q_ASSERT(sink == &stdinChannel);
            Q_ASSERT(source->process == this && source->type == Channel::PipeSource);
        }

        if (source->pipe[1] != INVALID_Q_PIPE || sink->pipe[0] != INVALID_Q_PIPE) {
            // already created, do nothing
            return true;
        } else {
            Q_ASSERT(source->pipe[0] == INVALID_Q_PIPE && source->pipe[1] == INVALID_Q_PIPE);
            Q_ASSERT(sink->pipe[0] == INVALID_Q_PIPE && sink->pipe[1] == INVALID_Q_PIPE);

            Q_PIPE pipe[2] = { -1, -1 };
            qt_create_pipe(pipe);
            sink->pipe[0] = pipe[0];
            source->pipe[1] = pipe[1];

            return true;
        }
    }
}

static int qt_startProcess(const QString &program, const QStringList &arguments,
                           const QString &workingDirectory,
                           const QStringList *environment, bool detached = false)
{
    // @todo P_DETACH isn't supported by kLIBC atm, do something else
    int mode = detached ? P_DETACH : P_NOWAIT;

    // Create argument list with right number of elements, and set the final
    // one to 0.
    char **argv = new char *[arguments.count() + 2];
    argv[arguments.count() + 1] = 0;

    // Encode the program name.
    QByteArray encodedProgramName = QFile::encodeName(program);

    // Add the program name to the argument list.
    argv[0] = encodedProgramName.data();

    // Add every argument to the list
    for (int i = 0; i < arguments.count(); ++i) {
        QString arg = arguments.at(i);
        argv[i + 1] = qstrdup(arg.toLocal8Bit().constData());
    }

    // Duplicate the environment.
    int envc = 0;
    char **envv;
    if (environment && environment->count()) {
        bool seenPATH = false;
        bool seenCOMSPEC = false;

        envv = new char *[environment->count() + 1 + 2 /* may be PATH + COMSPEC */];
        for (; envc < environment->count(); ++envc) {
            QString item = environment->at(envc);
            envv[envc] = qstrdup(item.toLocal8Bit().constData());
            if (!seenPATH)
                seenPATH = !qstrncmp(envv[envc], "PATH=", 4);
            if (!seenCOMSPEC)
                seenCOMSPEC = !qstrncmp(envv[envc], "COMSPEC=", 8);
        }
        if (!seenPATH) {
            // inherit PATH if missing (for convenience)
            // (note that BEGINLIBPATH and ENDLIBPATH, if any, are automatically
            // inherited, while LIBPATH is always a global setting)
            QByteArray path = qgetenv("PATH");
            path.prepend("PATH=");
            envv[envc++] = qstrdup(path);
        }
        // inherit COMSPEC if missing (to let the child start .cmd and .bat)
        if (!seenCOMSPEC) {
            QByteArray comspec = qgetenv("COMSPEC");
            comspec.prepend("COMSPEC=");
            envv[envc++] = qstrdup(comspec);
        }
        envv[envc] = 0;
    } else {
        // inherit the parent environment
        envv = environ;
    }

    // Set the working directory if it's non-empty
    QString curDir;
    if (!workingDirectory.isEmpty()) {
        curDir = QDir::currentPath();
        QDir::setCurrent(workingDirectory);
    }

    // start the program
#ifdef QPROCESS_DEBUG
    qDebug("Trying spawnvpe(%d, \"%s\")",
           mode, encodedProgramName.constData());
#endif
    int pid = spawnvpe(mode, encodedProgramName, argv, envv);

    // if spawnvpe() couldn't find the executable, try .CMD and .BAT
    // extensions (in order of CMD.EXE precedence); spawnvpe() knows how to
    // locate and run them (i.e. using CMD.EXE /c)
    if (pid == -1 && errno == ENOENT) {
        encodedProgramName += ".cmd";
        argv[0] = encodedProgramName.data();
#ifdef QPROCESS_DEBUG
            qDebug("Trying spawnvpe(%d, \"%s\")",
                   mode, encodedProgramName.constData());
#endif
        pid = spawnvpe(mode, encodedProgramName, argv, envv);
        if ( pid == -1 && errno == ENOENT ) {
            qstrcpy(encodedProgramName.data() + encodedProgramName.length() - 4, ".bat");
            argv[0] = encodedProgramName.data();
#ifdef QPROCESS_DEBUG
            qDebug("Trying spawnvpe(%d, \"%s\")",
                   mode, encodedProgramName.constData());
#endif
            pid = spawnvpe(mode, encodedProgramName, argv, envv);
        }
    }

    // Clean up duplicated memory.
    for (int i = 1 /* 0 is encodedProgramName */; i <= arguments.count(); ++i)
        delete [] argv[i];
    delete [] argv;
    if (envv != environ) {
        for (int i = 0; i < envc; ++i)
            delete [] envv[i];
        delete [] envv;
    }

    // restore the current directory
    QDir::setCurrent(curDir);

    return pid;
}

void QProcessPrivate::startProcess()
{
    Q_Q(QProcess);

#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::startProcess()");
#endif

    // Initialize pipes
    if (!createChannel(stdinChannel) ||
        !createChannel(stdoutChannel) ||
        !createChannel(stderrChannel))
        return;

    // Start the process (platform dependent)
    q->setProcessState(QProcess::Starting);

    // save & copy the stdin socket
    int stdin_copy = ::dup(fileno(stdin));
    ::dup2(stdinChannel.pipe[0], fileno(stdin));

    // save & copy the stdout and stderr if asked to
    int stdout_copy = -1, stderr_copy = -1;
    if (processChannelMode != QProcess::ForwardedChannels) {
        stdout_copy = ::dup(fileno(stdout));
        ::dup2(stdoutChannel.pipe[1], fileno(stdout));

        // merge stdout and stderr if asked to
        stderr_copy = ::dup(fileno(stderr));
        if (processChannelMode == QProcess::MergedChannels) {
            ::dup2(fileno(stdout), fileno(stderr));
        } else {
#if !defined (QPROCESS_DEBUG)
            // don't redirect it as qDebug() uses it
            ::dup2(stderrChannel.pipe[1], fileno(stderr));
#endif
        }
    }

    int pid = qt_startProcess(program, arguments, workingDirectory,
                              &environment);

    // restore stdin/stdout/stderr
    if (stdin_copy != -1) {
        ::dup2(stdin_copy, fileno(stdin));
        ::close(stdin_copy);
    }
    if (stdout_copy != -1) {
        ::dup2(stdout_copy, fileno(stdout));
        ::close(stdout_copy);
    }
    if (stderr_copy != -1) {
        ::dup2(stderr_copy, fileno(stderr));
        ::close(stderr_copy);
    }

    if (pid == -1) {
        // Cleanup, report error and return
#if defined (QPROCESS_DEBUG)
        qDebug("spawnvpe failed: %s", qPrintable(qt_error_string(errno)));
#endif
        q->setProcessState(QProcess::NotRunning);
        processError = QProcess::FailedToStart;
        q->setErrorString(QProcess::tr("Process failed to start: %1")
                                       .arg(qPrintable(qt_error_string(errno))));
        emit q->error(processError);
        cleanup();
        return;
    }

    this->pid = Q_PID(pid);

    // close the ends we don't use and make all pipes non-blocking
    if (stdinChannel.pipe[0] != -1) {
        ::close(stdinChannel.pipe[0]);
        stdinChannel.pipe[0] = -1;
    }
    if (stdinChannel.pipe[1] != -1)
        ::fcntl(stdinChannel.pipe[1], F_SETFL, ::fcntl(stdinChannel.pipe[1], F_GETFL) | O_NONBLOCK);

    if (stdoutChannel.pipe[1] != -1) {
        ::close(stdoutChannel.pipe[1]);
        stdoutChannel.pipe[1] = -1;
    }
    if (stdoutChannel.pipe[0] != -1)
        ::fcntl(stdoutChannel.pipe[0], F_SETFL, ::fcntl(stdoutChannel.pipe[0], F_GETFL) | O_NONBLOCK);

    if (stderrChannel.pipe[1] != -1) {
        ::close(stderrChannel.pipe[1]);
        stderrChannel.pipe[1] = -1;
    }
    if (stderrChannel.pipe[0] != -1)
        ::fcntl(stderrChannel.pipe[0], F_SETFL, ::fcntl(stderrChannel.pipe[0], F_GETFL) | O_NONBLOCK);

    q->setProcessState(QProcess::Running);
    // User can call kill()/terminate() from the stateChanged() slot
    // so check before proceeding
    if (!pid)
        return;

    // give the process a chance to start ...
    DosSleep(100);
    _q_startupNotification();
}

bool QProcessPrivate::processStarted()
{
    return processState == QProcess::Running;
}

qint64 QProcessPrivate::bytesAvailableFromStdout() const
{
    size_t nbytes = 0;
    qint64 available = 0;
    if (::ioctl(stdoutChannel.pipe[0], FIONREAD, (char *) &nbytes) >= 0)
        available = (qint64) *((int *) &nbytes);
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::bytesAvailableFromStdout() == %lld", available);
#endif
    return available;
}

qint64 QProcessPrivate::bytesAvailableFromStderr() const
{
    size_t nbytes = 0;
    qint64 available = 0;
    if (::ioctl(stderrChannel.pipe[0], FIONREAD, (char *) &nbytes) >= 0)
        available = (qint64) *((int *) &nbytes);
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::bytesAvailableFromStderr() == %lld", available);
#endif
    return available;
}

qint64 QProcessPrivate::readFromStdout(char *data, qint64 maxlen)
{
    qint64 bytesRead = ::read(stdoutChannel.pipe[0], data, maxlen);
#if defined QPROCESS_DEBUG
    qDebug("QProcessPrivate::readFromStdout(%p \"%s\", %lld) == %lld",
           data, qt_prettyDebug(data, bytesRead, 16).constData(), maxlen, bytesRead);
#endif
    return bytesRead;
}

qint64 QProcessPrivate::readFromStderr(char *data, qint64 maxlen)
{
    qint64 bytesRead = ::read(stderrChannel.pipe[0], data, maxlen);
#if defined QPROCESS_DEBUG
    qDebug("QProcessPrivate::readFromStderr(%p \"%s\", %lld) == %lld",
           data, qt_prettyDebug(data, bytesRead, 16).constData(), maxlen, bytesRead);
#endif
    return bytesRead;
}

qint64 QProcessPrivate::writeToStdin(const char *data, qint64 maxlen)
{
    qint64 written = ::write(stdinChannel.pipe[1], data, maxlen);
#if defined QPROCESS_DEBUG
    qDebug("QProcessPrivate::writeToStdin(%p \"%s\", %lld) == %lld",
           data, qt_prettyDebug(data, maxlen, 16).constData(), maxlen, written);
#endif
    return written;
}

void QProcessPrivate::terminateProcess()
{
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::terminateProcess()");
#endif
    if (pid) {
        HSWITCH hswitch = WinQuerySwitchHandle(NULL, pid);
        if (hswitch != NULLHANDLE) {
            SWCNTRL swcntrl = { 0 };
            APIRET rc = WinQuerySwitchEntry(hswitch,  &swcntrl);
            // WinQuerySwitchEntry will return a switch entry of the parent
            // process if the specfied one doesn't have a separate session
            // (running a plain CMD.EXE is an example); ignore this case.
            if (rc == NO_ERROR && swcntrl.idProcess == pid)
            {
                // first, ensure that the Close action is enabled in the main frame
                // window (otherwise WM_SYSCOMMAND/SC_CLOSE will be ignored)
                HWND hwndSysMenu = WinWindowFromID(swcntrl.hwnd, FID_SYSMENU);
                if (hwndSysMenu) {
                    WinPostMsg(hwndSysMenu, MM_SETITEMATTR,
                               MPFROM2SHORT(SC_CLOSE, TRUE),
                               MPFROM2SHORT(MIA_DISABLED, 0));
                }
                WinPostMsg(swcntrl.hwnd, WM_SYSCOMMAND,
                           MPFROM2SHORT(SC_CLOSE, CMDSRC_OTHER),
                           MPFROMLONG(FALSE));
            }
        }
    }
}

void QProcessPrivate::killProcess()
{
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::killProcess()");
#endif
    if (pid)
        DosKillProcess(DKP_PROCESS, pid);
}

/*
   Returns the difference between msecs and elapsed. If msecs is -1,
   however, -1 is returned.
*/
static int qt_timeout_value(int msecs, int elapsed)
{
    if (msecs == -1)
        return -1;

    int timeout = msecs - elapsed;
    return timeout < 0 ? 0 : timeout;
}

static int qt_select(int maxfd, fd_set *fdread, fd_set *fdwrite, int timeout)
{
    struct timeval tv;
    tv.tv_sec = timeout / 1000;
    tv.tv_usec = (timeout % 1000) * 1000;

    int ret = select(maxfd, fdread, fdwrite, 0, timeout < 0 ? 0 : &tv);
    return ret;
}

bool QProcessPrivate::waitForStarted(int msecs)
{
    Q_Q(QProcess);

    if (processStarted())
        return true;

    if (processError == QProcess::FailedToStart)
        return false;

    processError = QProcess::Timedout;
    q->setErrorString(QProcess::tr("Process operation timed out"));
    return false;
}

bool QProcessPrivate::waitForReadyRead(int msecs)
{
    Q_Q(QProcess);
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::waitForReadyRead(%d)", msecs);
#endif

    QTime stopWatch;
    stopWatch.start();

    forever {
        fd_set fdread;
        fd_set fdwrite;

        FD_ZERO(&fdread);
        FD_ZERO(&fdwrite);

        int maxFd = 0;
        if (stdoutChannel.pipe[0] != -1) {
            FD_SET(stdoutChannel.pipe[0], &fdread);
            maxFd = qMax(maxFd, stdoutChannel.pipe[0]);
        }
        if (stderrChannel.pipe[0] != -1) {
            FD_SET(stderrChannel.pipe[0], &fdread);
            maxFd = qMax(maxFd, stderrChannel.pipe[0]);
        }

        if (!writeBuffer.isEmpty() && stdinChannel.pipe[1] != -1) {
            FD_SET(stdinChannel.pipe[1], &fdwrite);
            maxFd = qMax(maxFd, stdinChannel.pipe[1]);
        }

        int timeout = qt_timeout_value(msecs, stopWatch.elapsed());
        int ret;
        if (maxFd >= 0) {
            ret = qt_select(maxFd + 1, &fdread, &fdwrite, timeout);
        } else {
            int exitStatus;
            ret = waitpid(pid, &exitStatus, 0);
        }
        if (ret < 0) {
            break;
        }
        if (ret == 0) {
            processError = QProcess::Timedout;
            q->setErrorString(QProcess::tr("Process operation timed out"));
            return false;
        }

        bool readyReadEmitted = false;
    	if (stdoutChannel.pipe[0] != -1 && FD_ISSET(stdoutChannel.pipe[0], &fdread)) {
    	    bool canRead = _q_canReadStandardOutput();
            if (processChannel == QProcess::StandardOutput && canRead)
                readyReadEmitted = true;
    	}
    	if (stderrChannel.pipe[0] != -1 && FD_ISSET(stderrChannel.pipe[0], &fdread)) {
    	    bool canRead = _q_canReadStandardError();
            if (processChannel == QProcess::StandardError && canRead)
                readyReadEmitted = true;
    	}
        if (readyReadEmitted)
            return true;

    	if (stdinChannel.pipe[1] != -1 && FD_ISSET(stdinChannel.pipe[1], &fdwrite))
    	    _q_canWrite();

        if (!pid)
            return false;
        if (_q_processDied())
            return false;
    }
    return false;
}

bool QProcessPrivate::waitForBytesWritten(int msecs)
{
    Q_Q(QProcess);
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::waitForBytesWritten(%d)", msecs);
#endif

    QTime stopWatch;
    stopWatch.start();

    while (!writeBuffer.isEmpty()) {
        fd_set fdread;
        fd_set fdwrite;

        FD_ZERO(&fdread);
        FD_ZERO(&fdwrite);

        int maxFd = 0;
        if (stdoutChannel.pipe[0] != -1) {
            FD_SET(stdoutChannel.pipe[0], &fdread);
            maxFd = qMax(maxFd, stdoutChannel.pipe[0]);
        }
        if (stderrChannel.pipe[0] != -1) {
            FD_SET(stderrChannel.pipe[0], &fdread);
            maxFd = qMax(maxFd, stderrChannel.pipe[0]);
        }

        if (!writeBuffer.isEmpty() && stdinChannel.pipe[1] != -1) {
            FD_SET(stdinChannel.pipe[1], &fdwrite);
            maxFd = qMax(maxFd, stdinChannel.pipe[1]);
        }

        int timeout = qt_timeout_value(msecs, stopWatch.elapsed());
        int ret;
        if (maxFd >= 0) {
            ret = qt_select(maxFd + 1, &fdread, &fdwrite, timeout);
        } else {
            int exitStatus;
            ret = waitpid(pid, &exitStatus, 0);
        }
        if (ret < 0) {
            break;
        }
        if (ret == 0) {
            processError = QProcess::Timedout;
            q->setErrorString(QProcess::tr("Process operation timed out"));
            return false;
        }

    	if (stdinChannel.pipe[1] != -1 && FD_ISSET(stdinChannel.pipe[1], &fdwrite))
    	    return _q_canWrite();

    	if (stdoutChannel.pipe[0] != -1 && FD_ISSET(stdoutChannel.pipe[0], &fdread))
    	    _q_canReadStandardOutput();

    	if (stderrChannel.pipe[0] != -1 && FD_ISSET(stderrChannel.pipe[0], &fdread))
    	    _q_canReadStandardError();

        if (!pid)
            return false;
        if (_q_processDied())
            return false;
    }

    return false;
}

bool QProcessPrivate::waitForFinished(int msecs)
{
    Q_Q(QProcess);
#if defined (QPROCESS_DEBUG)
    qDebug("QProcessPrivate::waitForFinished(%d)", msecs);
#endif

    QTime stopWatch;
    stopWatch.start();

    forever {
        fd_set fdread;
        fd_set fdwrite;

        FD_ZERO(&fdread);
        FD_ZERO(&fdwrite);

        int maxFd = -1;
        if (stdoutChannel.pipe[0] != -1) {
            FD_SET(stdoutChannel.pipe[0], &fdread);
            maxFd = qMax(maxFd, stdoutChannel.pipe[0]);
        }
        if (stderrChannel.pipe[0] != -1) {
            FD_SET(stderrChannel.pipe[0], &fdread);
            maxFd = qMax(maxFd, stderrChannel.pipe[0]);
        }

        if (!writeBuffer.isEmpty() && stdinChannel.pipe[1] != -1) {
            FD_SET(stdinChannel.pipe[1], &fdwrite);
            maxFd = qMax(maxFd, stdinChannel.pipe[1]);
        }

        int timeout = qt_timeout_value(msecs, stopWatch.elapsed());
        int ret;
        if (maxFd >= 0) {
            ret = qt_select(maxFd + 1, &fdread, &fdwrite, timeout);
        } else {
            int exitStatus;
            ret = waitpid(pid, &exitStatus, 0);
        }
        if (ret < 0) {
            break;
        }
    	if (ret == 0) {
    	    processError = QProcess::Timedout;
    	    q->setErrorString(QProcess::tr("Process operation timed out"));
    	    return false;
    	}

    	if (stdinChannel.pipe[1] != -1 && FD_ISSET(stdinChannel.pipe[1], &fdwrite))
    	    _q_canWrite();

    	if (stdoutChannel.pipe[0] != -1 && FD_ISSET(stdoutChannel.pipe[0], &fdread))
    	    _q_canReadStandardOutput();

    	if (stderrChannel.pipe[0] != -1 && FD_ISSET(stderrChannel.pipe[0], &fdread))
    	    _q_canReadStandardError();

        if (!pid)
            return true;
        if (_q_processDied())
            return true;
    }
    return false;
}

bool QProcessPrivate::waitForWrite(int msecs)
{
    fd_set fdwrite;
    FD_ZERO(&fdwrite);
    FD_SET(stdinChannel.pipe[1], &fdwrite);

    int ret = qt_select(stdinChannel.pipe[1] + 1, 0, &fdwrite,
                        msecs < 0 ? 0 : msecs) == 1;
    return ret == 1;
}

void QProcessPrivate::findExitCode()
{
}

bool QProcessPrivate::waitForDeadChild()
{
    // check if our process is dead
    int exitStatus;
    pid_t waitResult = waitpid(pid, &exitStatus, WNOHANG);
    if (waitResult > 0) {
        crashed = !WIFEXITED(exitStatus);
        exitCode = WEXITSTATUS(exitStatus);
#if defined QPROCESS_DEBUG
        qDebug() << "QProcessPrivate::waitForDeadChild() dead with exitCode"
                 << exitCode << ", crashed?" << crashed;
#endif
        return true;
    }
#if defined QPROCESS_DEBUG
    qDebug() << "QProcessPrivate::waitForDeadChild() not dead!";
#endif
    return false;
}

void QProcessPrivate::_q_notified()
{
}

/*! \internal
 */
bool QProcessPrivate::startDetached(const QString &program, const QStringList &arguments,
                                    const QString &workingDirectory, qint64 *pid)
{
    int startedPid = qt_startProcess(program, arguments, workingDirectory,
                                     NULL, true /* detached */);

    if (startedPid == -1)
        return false;

    *pid = startedPid;
    return true;
}

QT_END_NAMESPACE

#endif // QT_NO_PROCESS
