/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** Copyright (C) 2009 netlabs.org. OS/2 parts.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qt_os2.h"

#include "qwidget.h"
#include "qwidget_p.h"

#include "qapplication.h"
#include "qdesktopwidget.h"

#include "private/qapplication_p.h"
#include "private/qbackingstore_p.h"

#include <qdebug.h>

#define QT_DEBUGWINCREATEDESTROY
#define QT_DEBUGWIDGETMASK

QT_BEGIN_NAMESPACE

static QWidget *mouseGrb    = 0;
static QCursor *mouseGrbCur = 0;
static QWidget *keyboardGrb = 0;

// defined in qapplication_pm.cpp
extern bool qt_nograb();
extern MRESULT EXPENTRY QtWndProc(HWND, ULONG, MPARAM, MPARAM);
extern PFNWP QtOldFrameProc;
extern MRESULT EXPENTRY QtFrameProc(HWND, ULONG, MPARAM, MPARAM);

typedef QSet<QString> WinClassNameHash;
Q_GLOBAL_STATIC(WinClassNameHash, winclassNames)

// register window class
static const QString qt_reg_winclass(QWidget *w)
{
    int flags = w->windowFlags();
    int type = flags & Qt::WindowType_Mask;

    QString cname;
    ULONG style = 0;

    if (type == Qt::Popup || type == Qt::ToolTip) {
        cname = QLatin1String("QPopup");
        style |= CS_SAVEBITS;
    } else if (w->isWindow()) {
        // this class is for top level widgets which are client HWNDs of a
        // WC_FRAME parent HWND internally maintaned for them
        cname = QLatin1String("QWindow");
        style |= CS_MOVENOTIFY;
    } else {
        cname = QLatin1String("QWidget");
    }

    if (winclassNames()->contains(cname)) // already registered in our list
        return cname;

    // QT_EXTRAWINDATASIZE is defined in qwindowdefs_pm.h
    WinRegisterClass(0, cname.toLatin1(), QtWndProc, style, QT_EXTRAWINDATASIZE);

    winclassNames()->insert(cname);
    return cname;

    // Note: there is no need to unregister private window classes registered by
    // this function -- it is done automatically upon process termination.
}

// Returns a QWidget pointer or 0 if there is no widget corresponding to the
// given HWND. As opposed to QWidget::find(), correctly handles WC_FRAME windows
// created for top level widgets. Used for debugging.
static QWidget *widgetFromId(HWND hwnd)
{
    char buf[10];
    if (WinQueryClassName(hwnd, sizeof(buf), buf)) {
        if (!strcmp(buf, "#1")) // WC_FRAME
            hwnd = WinWindowFromID(hwnd, FID_CLIENT);
        return QWidget::find(hwnd);
    }
    return 0;
}

// Returns a human readable widget name in the form "class/name".
// Used for debugging.
static QByteArray widgetName(QWidget *w)
{
    if (w)
        return QString()
            .sprintf("%s/%s", w->metaObject()->className(),
                     w->objectName().toLocal8Bit().constData()).toLocal8Bit();
    return QByteArray("<no-widget>");
}

/*!
    \internal

    Extended version of WinQueryClipRegion(). If the given window has a clip
    region, the given region will receive a copy of the clip region clipped to
    the current window rectangle. If there is no clip region, the given region
    will contain only the window rectangle on return.
 */
void qt_WinQueryClipRegionOrRect(HWND hwnd, HRGN hrgn)
{
    RECTL rcl;
    WinQueryWindowRect(hwnd, &rcl);

    HPS hps = qt_display_ps();
    GpiSetRegion(hps, hrgn, 1, &rcl);
    if (WinQueryClipRegion(hwnd, 0) != QCRGN_NO_CLIP_REGION) {
        HRGN hrgnTemp = GpiCreateRegion(hps, 0, NULL);
        WinQueryClipRegion(hwnd, hrgnTemp);
        GpiCombineRegion(hps, hrgn, hrgnTemp, hrgn, CRGN_AND);
        GpiDestroyRegion(hps, hrgnTemp);
    }
}

/*!
    \internal

    Extended version of WinInvalidateRegion(): invalidates the specified region
    of the given window and regions of children from \a hwndFrom to \a hwndTo
    if they intersect with the invalid region. If either of child window
    handles is NULLHANDLE, children are not invalidated at all. Also, HWND_TOP
    can be used as \a hwndFrom, HWND_BOTTOM \a can be used as \a hwndTo.
 */
static BOOL qt_WinInvalidateRegionEx(HWND hwnd, HRGN hrgn,
                                     HWND hwndFrom, HWND hwndTo)
{
#if defined(QT_DEBUGWIDGETMASK)
    qDebug("qt_WinInvalidateRegionEx: hwnd=%08lX [%s] "
           "hwndFrom=%08lX hwndTo=%08lX", hwnd,
           widgetName(widgetFromId(hwnd)).constData(),
           hwndFrom, hwndTo);
#endif

    if (hwndFrom == HWND_TOP)
        hwndFrom = WinQueryWindow(hwnd, QW_TOP);
    if (hwndTo == HWND_BOTTOM)
        hwndTo = WinQueryWindow(hwnd, QW_BOTTOM);

    if (hwndFrom == 0 || hwndTo == 0)
        return WinInvalidateRegion(hwnd, hrgn, FALSE);

    if (WinQueryWindow(hwndFrom, QW_PARENT) != hwnd ||
        WinQueryWindow(hwndTo, QW_PARENT) != hwnd)
        return FALSE;

    HPS hps = qt_display_ps();

    SWP swp;
    HWND child = hwndFrom;
    HRGN hrgnChild = GpiCreateRegion(hps, 0, NULL);
    HRGN hrgnInv = GpiCreateRegion(hps, 0, NULL);
    GpiCombineRegion(hps, hrgnInv, hrgn, 0, CRGN_COPY);

    LONG cmplx = RGN_RECT;

    while (child) {
        WinQueryWindowPos(child, &swp);
#if defined(QT_DEBUGWIDGETMASK)
        qDebug(" child=%08lX [fl=%08lX] [%s]", child, swp.fl,
               widgetName(widgetFromId(child)).constData());
#endif
        // proceed only if not hidden
        if (swp.fl & SWP_SHOW) {
            // get sibling's bounds (clip region or rect)
            qt_WinQueryClipRegionOrRect(child, hrgnChild);
            // translate the region to the parent's coordinate system
            POINTL ptl = { swp.x, swp.y };
            GpiOffsetRegion(hps, hrgnChild, &ptl);
            // intersect the child's region with the invalid one
            // and invalidate if not NULL
            cmplx = GpiCombineRegion(hps, hrgnChild, hrgnChild, hrgnInv,
                                      CRGN_AND);
            if (cmplx != RGN_NULL) {
                POINTL ptl2 = { -swp.x, -swp.y };
                GpiOffsetRegion(hps, hrgnChild, &ptl2);
                WinInvalidateRegion(child, hrgnChild, TRUE);
                GpiOffsetRegion(hps, hrgnChild, &ptl);
                // substract the invalidated area from the widget's region
                // (no need to invalidate it any more)
                cmplx = GpiCombineRegion(hps, hrgnInv, hrgnInv, hrgnChild,
                                          CRGN_DIFF);
#if defined(QT_DEBUGWIDGETMASK)
                qDebug("  processed");
#endif
                // finish if nothing left
                if (cmplx == RGN_NULL)
                    break;
            }
        }
        // iterate to the next window (below)
        if (child == hwndTo)
            break;
        child = WinQueryWindow(child, QW_NEXT);
    }

    BOOL ok = (cmplx == RGN_NULL) || (child == hwndTo);

    if (ok) {
        // invalidate what's left invalid after substracting children
        WinInvalidateRegion(hwnd, hrgnInv, FALSE);
    }

    GpiDestroyRegion(hps, hrgnInv);
    GpiDestroyRegion(hps, hrgnChild);

    return ok;
}

/** \internal flags for qt_WinProcessWindowObstacles() */
enum {
    PWO_Children = 0x01,
    PWO_Sibings = 0x02,
    PWO_Ancestors = 0x04,
    PWO_Screen = 0x08,
    PWO_TopLevel = 0x80000000,
    // PWO_Default is suitable in most cases (for simple paint operations)
    PWO_Default = PWO_Children | PWO_Sibings | PWO_Ancestors | PWO_Screen,
};

/*!
    \internal

    Helper function to collect all relative windows intersecting with the
    given window and placed above it in z-order.

    \param hwnd  window in question
    \param prcl  rectangle (in window coordinates) to limit processing to
                 (if null, the whole window rectange is used)
    \param hrgn  region where to combine all obstacles
                 (if 0, obstacles are directly validated instead of collecting)
    \param op    region operation perfomed when combining obstacles (CRGN_*)
    \param flags flags defining the scope (PWO_* ORed together)

    \return complexity of the combined region (only when \a hrgn is not 0)
 */
static LONG qt_WinProcessWindowObstacles(HWND hwnd, PRECTL prcl, HRGN hrgn,
                                         LONG op, LONG flags = PWO_Default)
{
    Q_ASSERT(hwnd);

    HPS displayPS = qt_display_ps();

#if defined(QT_DEBUGWIDGETMASK)
    qDebug("qt_WinProcessWindowObstacles: hwnd=%08lX [%s], prcl=%p "
           "hrgn=%08lX, op=%ld flags=%08lX", hwnd,
           widgetName(widgetFromId(hwnd)).constData(),
           prcl, hrgn, op, flags);
#endif

    SWP swpSelf;
    WinQueryWindowPos(hwnd, &swpSelf);

    RECTL rclSelf = { 0, 0, swpSelf.cx, swpSelf.cy };
    if (prcl)
        rclSelf = *prcl;

    HRGN whrgn = GpiCreateRegion(displayPS, 0, NULL);

    LONG cmplx = RGN_NULL;
    HWND relative;
    SWP swp;

    // first, process areas placed outside the screen bounds
    if (flags & PWO_Screen) {
        RECTL rclScr = { 0, 0, QApplication::desktop()->width(),
                               QApplication::desktop()->height() };
        WinMapWindowPoints(HWND_DESKTOP, hwnd, (PPOINTL) &rclScr, 2);
        // rough check of whether some window part is outside bounds
        if (rclSelf.xLeft < rclScr.xLeft ||
            rclSelf.yBottom < rclScr.yBottom ||
            rclSelf.xRight > rclScr.xRight ||
            rclSelf.yTop > rclScr.yTop) {
            GpiSetRegion(displayPS, whrgn, 1, &rclSelf);
            HRGN hrgnScr = GpiCreateRegion(displayPS, 1, &rclScr);
            // substract the screen region from this window's region
            // to get parts placed outside
            GpiCombineRegion(displayPS, whrgn, whrgn, hrgnScr, CRGN_DIFF);
            GpiDestroyRegion(displayPS, hrgnScr);
            // process the region
            if (hrgn != NULLHANDLE) {
                cmplx = GpiCombineRegion(displayPS, hrgn, hrgn, whrgn, op);
            } else {
                WinValidateRegion(hwnd, whrgn, FALSE);
            }
#if defined(QT_DEBUGWIDGETMASK)
            qDebug(" collected areas outside screen bounds");
#endif
         }
    }

    // next, go through all children (in z-order)
    if (flags & PWO_Children) {
        relative = WinQueryWindow(hwnd, QW_BOTTOM);
        if (relative != NULLHANDLE) {
            for (; relative != HWND_TOP; relative = swp.hwndInsertBehind) {
                WinQueryWindowPos(relative, &swp);
#if defined(QT_DEBUGWIDGETMASK)
                qDebug(" child=%08lX [fl=%08lX] [%s]", relative, swp.fl,
                       widgetName(widgetFromId(relative)).constData());
#endif
                // skip if hidden
                if (!(swp.fl & SWP_SHOW))
                    continue;
                // rough check for intersection
                if (swp.x >= rclSelf.xRight || swp.y >= rclSelf.yTop ||
                    swp.x + swp.cx <= rclSelf.xLeft ||
                    swp.y + swp.cy <= rclSelf.yBottom)
                    continue;
                // get the bounds (clip region or rect)
                qt_WinQueryClipRegionOrRect(relative, whrgn);
                // translate the region to this window's coordinate system
                POINTL ptl = { swp.x, swp.y };
                GpiOffsetRegion(displayPS, whrgn, &ptl);
                // process the region
                if (hrgn != NULLHANDLE) {
                    cmplx = GpiCombineRegion(displayPS, hrgn, hrgn, whrgn, op);
                } else {
                    WinValidateRegion(hwnd, whrgn, FALSE);
                }
#if defined(QT_DEBUGWIDGETMASK)
                qDebug("  collected");
#endif
            }
        }
    }

    HWND desktop = WinQueryDesktopWindow(0, 0);
    HWND parent = WinQueryWindow(hwnd, QW_PARENT);

    // next, go through all siblings placed above (in z-order),
    // but only if they are not top-level windows (that cannot be
    // non-rectangular and thus are always correctly clipped by the system)
    if ((flags & PWO_Sibings) && parent != desktop) {
        for (relative = swpSelf.hwndInsertBehind;
              relative != HWND_TOP; relative = swp.hwndInsertBehind) {
            WinQueryWindowPos(relative, &swp);
#if defined(QT_DEBUGWIDGETMASK)
            qDebug(" sibling=%08lX [fl=%08lX] [%s]", relative, swp.fl,
                   widgetName(widgetFromId(relative)).constData());
#endif
            // skip if hidden
            if (!(swp.fl & SWP_SHOW))
                continue;
            // rough check for intersection
            if (swp.x >= swpSelf.x + rclSelf.xRight ||
                swp.y >= swpSelf.y + rclSelf.yTop ||
                swp.x + swp.cx <= swpSelf.x + rclSelf.xLeft ||
                swp.y + swp.cy <= swpSelf.y + rclSelf.yBottom)
                continue;
            // get the bounds (clip region or rect)
            qt_WinQueryClipRegionOrRect(relative, whrgn);
            // translate the region to this window's coordinate system
            POINTL ptl = { swp.x - swpSelf.x, swp.y - swpSelf.y };
            GpiOffsetRegion(displayPS, whrgn, &ptl);
            // process the region
            if (hrgn != NULLHANDLE) {
                cmplx = GpiCombineRegion(displayPS, hrgn, hrgn, whrgn, op);
            } else {
                WinValidateRegion(hwnd, whrgn, FALSE);
            }
#if defined(QT_DEBUGWIDGETMASK)
            qDebug("  collected");
#endif
        }
    }

    // last, go through all siblings of our parent and its ancestors
    // placed above (in z-order)
    if (flags & PWO_Ancestors) {
        POINTL delta = { swpSelf.x, swpSelf.y };
        while (parent != desktop) {
            HWND grandParent = WinQueryWindow(parent, QW_PARENT);
            if (!(flags & PWO_TopLevel)) {
                // When PWO_TopLevel is not specified, top-level windows
                // (children of the desktop) are not processed. It makes sense
                // when qt_WinProcessWindowObstacles() is used to clip out
                // overlying windows during regular paint operations (WM_PAINT
                // processing or drawing in a window directly through
                // WinGetPS()): in this case, top-level windows are always
                // correctly clipped out by the system (because they cannot be
                // non-rectangular).
                if (grandParent == desktop)
                    break;
            }

            WinQueryWindowPos(parent, &swp);
#if defined(QT_DEBUGWIDGETMASK)
            qDebug(" parent=%08lX [fl=%08lX] [%s]", parent, swp.fl,
                   widgetName(widgetFromId(parent)).constData());
#endif
            delta.x += swp.x;
            delta.y += swp.y;
            for (relative = swp.hwndInsertBehind;
                 relative != HWND_TOP; relative = swp.hwndInsertBehind) {
                WinQueryWindowPos(relative, &swp);
#if defined(QT_DEBUGWIDGETMASK)
                qDebug(" ancestor=%08lX [fl=%08lX] [%s]", relative, swp.fl,
                       widgetName(widgetFromId(parent)).constData());
#endif
                // skip if hidden
                if (!(swp.fl & SWP_SHOW))
                    continue;
                // rough check for intersection
                if (swp.x - delta.x >= rclSelf.xRight ||
                    swp.y - delta.y >= rclSelf.yTop ||
                    swp.x - delta.x + swp.cx <= rclSelf.xLeft ||
                    swp.y - delta.y + swp.cy <= rclSelf.yBottom)
                    continue;
                // get the bounds (clip region or rect)
                qt_WinQueryClipRegionOrRect(relative, whrgn);
                // translate the region to this window's coordinate system
                POINTL ptl = { swp.x - delta.x, swp.y - delta.y };
                GpiOffsetRegion(displayPS, whrgn, &ptl);
                // process the region
                if (hrgn != NULLHANDLE) {
                    cmplx = GpiCombineRegion(displayPS, hrgn, hrgn, whrgn, op);
                } else {
                    WinValidateRegion(hwnd, whrgn, FALSE);
                }
#if defined(QT_DEBUGWIDGETMASK)
                qDebug("  collected");
#endif
            }
            parent = grandParent;
        }
    }

    GpiDestroyRegion(displayPS, whrgn);

    return cmplx;
}

/*!
    \internal

    Partial reimplementation of the WinSetWindowPos() API that obeys window clip
    regions. Currently supported flags are SWP_ZORDER, SWP_SHOW, SWP_HIDE and
    SWP_ACTIVATE. Other flags should not be used. Note that if any other flag is
    specified (alone or in addition to the supported ones), or if the given
    window is a top-level window, this function acts exactly like the original
    WinSetWindowPos() function.
 */
static BOOL qt_WinSetWindowPos(HWND hwnd, HWND hwndInsertBehind,
                               LONG x, LONG y, LONG cx, LONG cy,
                               ULONG fl)
{
#if defined(QT_DEBUGWIDGETMASK)
    qDebug("qt_WinSetWindowPos: hwnd=%08lX [%s] fl=%08lX", hwnd,
           widgetName(widgetFromId(hwnd)).constData(), fl);
#endif

    HWND desktop = WinQueryDesktopWindow(0, 0);

    Q_ASSERT(((fl & ~(SWP_ZORDER | SWP_SHOW | SWP_HIDE)) == 0) ||
             hwnd == desktop || WinQueryWindow(hwnd, QW_PARENT) == desktop);

    if ((fl & ~(SWP_ZORDER | SWP_SHOW | SWP_HIDE)) != 0 ||
         hwnd == desktop || WinQueryWindow(hwnd, QW_PARENT) == desktop) {
        return WinSetWindowPos(hwnd, hwndInsertBehind, x, y, cx, cy, fl);
    }

    SWP swpOld;
    WinQueryWindowPos(hwnd, &swpOld);

    // do some checks
    if ((fl & SWP_ZORDER) && swpOld.hwndInsertBehind == hwndInsertBehind)
        fl &= ~SWP_ZORDER;
    if ((fl & SWP_SHOW) && (swpOld.fl & SWP_SHOW))
        fl &= ~SWP_SHOW;
    if ((fl & SWP_HIDE) && (swpOld.fl & SWP_HIDE))
        fl &= ~SWP_HIDE;
    if ((fl & (SWP_SHOW | SWP_HIDE)) == (SWP_SHOW | SWP_HIDE))
        fl &= ~SWP_HIDE;

    BOOL rc = WinSetWindowPos(hwnd, hwndInsertBehind, x, y, cx, cy,
                              fl | SWP_NOREDRAW);
    if (rc == FALSE || (fl & SWP_NOREDRAW))
        return rc;

    SWP swpNew;
    WinQueryWindowPos(hwnd, &swpNew);

    if (swpOld.hwndInsertBehind == swpNew.hwndInsertBehind)
        fl &= ~SWP_ZORDER;

    if ((fl & (SWP_ZORDER | SWP_SHOW | SWP_HIDE)) == 0)
        return rc;

    HPS hps = qt_display_ps();
    HWND hwndParent = WinQueryWindow(hwnd, QW_PARENT);

    // get window bounds
    HRGN hrgnSelf = GpiCreateRegion(hps, 0, NULL);
    qt_WinQueryClipRegionOrRect(hwnd, hrgnSelf);

    if (fl & SWP_SHOW) {
        WinInvalidateRegion(hwnd, hrgnSelf, TRUE);
    } else if (fl & SWP_HIDE) {
        // translate the region to the parent coordinate system
        POINTL ptl = { swpNew.x, swpNew.y };
        GpiOffsetRegion(hps, hrgnSelf, &ptl);
        // invalidate the parent and children below this window
        qt_WinInvalidateRegionEx(hwndParent, hrgnSelf,
                                  WinQueryWindow(hwnd, QW_NEXT), HWND_BOTTOM);
    } else { // fl & SWP_ZORDER
        // below we assume that WinSetWindowPos() returns FALSE if
        // an incorrect (unrelated) hwndInsertBehind is passed when SWP_ZORDER
        // is set

        // first, detect whether we are moving up or down
        BOOL up;
        HWND hwndFrom, hwndTo;
        if (swpOld.hwndInsertBehind == HWND_TOP) {
            up = FALSE;
            hwndFrom = WinQueryWindow(hwndParent, QW_TOP);
            hwndTo = swpNew.hwndInsertBehind;
        } else {
            up = TRUE;
            for (HWND hwndAbove = hwnd;
                  (hwndAbove = WinQueryWindow(hwndAbove, QW_PREV)) != 0;) {
                if (hwndAbove == swpOld.hwndInsertBehind) {
                    up = FALSE;
                    break;
                }
            }
            if (up) {
                hwndFrom = swpOld.hwndInsertBehind;
                hwndTo = WinQueryWindow(hwnd, QW_NEXT);
            } else {
                hwndFrom = WinQueryWindow(swpOld.hwndInsertBehind, QW_NEXT);
                hwndTo = swpNew.hwndInsertBehind;
            }
        }
#if defined(QT_DEBUGWIDGETMASK)
        qDebug(" moving up? %ld", up);
        qDebug(" hwndFrom=%08lX [%s]", hwndFrom,
               widgetName(widgetFromId(hwndFrom)).constData());
        qDebug(" hwndTo=%08lX [%s]", hwndTo,
               widgetName(widgetFromId(hwndTo)).constData());
#endif

        SWP swp;
        HWND sibling = hwndFrom;
        HRGN hrgn = GpiCreateRegion(hps, 0, NULL);
        HRGN hrgnUpd = GpiCreateRegion(hps, 0, NULL);

        if (up) {
            // go upwards in z-order
            while (1) {
                WinQueryWindowPos(sibling, &swp);
#if defined(QT_DEBUGWIDGETMASK)
                qDebug(" sibling=%08lX [fl=%08lX] [%s]", sibling, swp.fl,
                       widgetName(widgetFromId(sibling)).constData());
#endif
                // proceed only if not hidden
                if (swp.fl & SWP_SHOW) {
                    // get sibling's bounds (clip region or rect)
                    qt_WinQueryClipRegionOrRect(sibling, hrgn);
                    // translate the region to this window's coordinate system
                    POINTL ptl = { swp.x - swpNew.x, swp.y - swpNew.y };
                    GpiOffsetRegion(hps, hrgn, &ptl);
                    // add to the region of siblings we're moving on top of
                    GpiCombineRegion(hps, hrgnUpd, hrgnUpd, hrgn, CRGN_OR);
#if defined(QT_DEBUGWIDGETMASK)
                    qDebug("  processed");
#endif
                }
                // iterate to the prev window (above)
                if (sibling == hwndTo)
                    break;
                sibling = swp.hwndInsertBehind;
            }
            // intersect the resulting region with the widget region and
            // invalidate
            GpiCombineRegion(hps, hrgnUpd, hrgnSelf, hrgnUpd, CRGN_AND);
            WinInvalidateRegion(hwnd, hrgnUpd, TRUE);
        } else {
            // go downwards in reverse z-order
            POINTL ptl = { 0, 0 };
            while (1) {
                WinQueryWindowPos(sibling, &swp);
#if defined(QT_DEBUGWIDGETMASK)
                qDebug(" sibling=%08lX [fl=%08lX] [%s]", sibling, swp.fl,
                       widgetName(widgetFromId(sibling)).constData());
#endif
                // proceed only if not hidden
                if (swp.fl & SWP_SHOW) {
                    // get sibling's bounds (clip region or rect)
                    qt_WinQueryClipRegionOrRect(sibling, hrgn);
                    // undo the previous translation and translate this window's
                    // region to the siblings's coordinate system
                    ptl.x += swpNew.x - swp.x;
                    ptl.y += swpNew.y - swp.y;
                    GpiOffsetRegion(hps, hrgnSelf, &ptl);
                    // intersect the sibling's region with the translated one
                    // and invalidate the sibling
                    GpiCombineRegion(hps, hrgnUpd, hrgnSelf, hrgn, CRGN_AND);
                    WinInvalidateRegion(sibling, hrgnUpd, TRUE);
                    // substract the invalidated area from the widget's region
                    // (no need to invalidate it any more)
                    GpiCombineRegion(hps, hrgnSelf, hrgnSelf, hrgnUpd, CRGN_DIFF);
                    // prepare the translation from the sibling's
                    // coordinates back to this window's coordinates
                    ptl.x = swp.x - swpNew.x;
                    ptl.y = swp.y - swpNew.y;
#if defined(QT_DEBUGWIDGETMASK)
                    qDebug("  processed");
#endif
                }
                // iterate to the next window (below)
                if (sibling == hwndTo)
                    break;
                sibling = WinQueryWindow(sibling, QW_NEXT);
            }
        }

        GpiDestroyRegion(hps, hrgnUpd);
        GpiDestroyRegion(hps, hrgn);
    }

    GpiDestroyRegion(hps, hrgnSelf);

    return TRUE;
}

/*!
 * \internal
 * For some unknown reason, PM sends WM_SAVEAPPLICATION to every window
 * being destroyed, which makes it indistinguishable from WM_SAVEAPPLICATION
 * sent to top level windows during system shutdown. We use our own version of
 * WinDestroyWindow() and a special flag (qt_about_to_destroy_wnd) to
 * distinguish it in qapplication_pm.cpp.
 */
static BOOL qt_WinDestroyWindow(HWND hwnd)
{
#if !defined(QT_NO_SESSIONMANAGER)
    qt_about_to_destroy_wnd = true;
#endif
    BOOL rc = WinDestroyWindow(hwnd);
#if !defined(QT_NO_SESSIONMANAGER)
    qt_about_to_destroy_wnd = false;
#endif
    return rc;
}

static PFNWP QtOldSysMenuProc;
static MRESULT EXPENTRY QtSysMenuProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    if (msg == WM_MENUEND) {
        // the pull-down menu is closed, always dismiss the system menu itself
        WinPostMsg(hwnd, MM_ENDMENUMODE, MPFROMSHORT(TRUE), 0);
    }
    return QtOldSysMenuProc(hwnd, msg, mp1, mp2);
}

static void removeSysMenuAccels(HWND frame)
{
    HWND sysMenu = WinWindowFromID(frame, FID_SYSMENU);
    if (!sysMenu)
        return;

    SHORT subId = SHORT1FROMMR(WinSendMsg(sysMenu, MM_ITEMIDFROMPOSITION, 0, 0));
    if (subId != MIT_ERROR) {
        MENUITEM item;
        WinSendMsg(sysMenu, MM_QUERYITEM, MPFROM2SHORT(subId, FALSE), MPFROMP(&item));
        HWND subMenu = item.hwndSubMenu;
        if (subMenu) {
            USHORT cnt = SHORT1FROMMR(WinSendMsg(subMenu, MM_QUERYITEMCOUNT, 0, 0));
            for (int i = 0; i < cnt; i++) {
                USHORT id = SHORT1FROMMR(
                    WinSendMsg(subMenu, MM_ITEMIDFROMPOSITION, MPFROMSHORT(i), 0));
                if (id == SC_TASKMANAGER || id == SC_CLOSE) {
                    // accels for these entries always work in Qt, skip them
                    continue;
                }
                USHORT len = SHORT1FROMMR(
                    WinSendMsg(subMenu, MM_QUERYITEMTEXTLENGTH, MPFROMSHORT(id), 0));
                if (len++) {
                    char *text = new char[len];
                    WinSendMsg(subMenu, MM_QUERYITEMTEXT,
                        MPFROM2SHORT(id, len), MPFROMP(text));
                    char *tab = strrchr(text, '\t');
                    if (tab) {
                        *tab = 0;
                        WinSendMsg(subMenu, MM_SETITEMTEXT,
                            MPFROMSHORT(id), MPFROMP(text));
                    }
                    delete[] text;
                }
            }
            // sublclass the system menu to leave the menu mode completely
            // when the user presses the ESC key. by default, pressing ESC
            // while the pull-down menu is showing brings us to the menu bar,
            // which is confusing in the case of the system menu, because
            // there is only one item on the menu bar, and we cannot see
            // that it is active when the frame window has an icon.
            PFNWP oldProc = WinSubclassWindow(sysMenu, QtSysMenuProc);
            // set QtOldSysMenuProc only once: it must be the same for
            // all FID_SYSMENU windows.
            if (!QtOldSysMenuProc)
                QtOldSysMenuProc = oldProc;
        }
    }
}

/*****************************************************************************
  QWidget member functions
 *****************************************************************************/

void QWidgetPrivate::create_sys(WId window, bool initializeWindow, bool destroyOldWindow)
{
    // @todo when window is not zero, it represents an existing (external)
    // window handle we should create a QWidget "wrapper" for to incorporate it
    // to the Qt widget hierarchy. This functionality isn't really necessary on
    // OS/2 at the moment, so it is not currently supported (and the window
    // argument is simply ignored). Note that destroyOldWindow only makes
    //  sense when window is != 0 so it is also ignored.

    Q_ASSERT(window == 0);
    Q_UNUSED(destroyOldWindow);

    Q_Q(QWidget);
    static int sw = -1, sh = -1;

    Qt::WindowType type = q->windowType();
    Qt::WindowFlags flags = data.window_flags;

    bool topLevel = q->isWindow();
    bool popup = (type == Qt::Popup);
    bool dialog = (type == Qt::Dialog
                   || type == Qt::Sheet
                   || (flags & Qt::MSWindowsFixedSizeDialogHint));
    bool desktop = (type == Qt::Desktop);
    bool tool = (type == Qt::Tool || type == Qt::Drawer);

    WId id;

    QByteArray className = qt_reg_winclass(q).toLatin1();

    // @todo WindowStaysOnTopHint is ignored for now (does nothing)
    if (popup)
        flags |= Qt::WindowStaysOnTopHint; // a popup stays on top

    if (sw < 0) { // get the screen size
        sw = WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN);
        sh = WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN);
    }

    if (desktop && !q->testAttribute(Qt::WA_DontShowOnScreen)) { // desktop widget
        popup = false; // force this flags off
        // @todo use WinGetMaxPositionto take into account such things as XCenter?
        data.crect.setRect(0, 0, sw, sh);
    }

    QByteArray title;
    ULONG style = 0;
    ULONG fId = 0, fStyle = 0, fcFlags = 0;

    if (!desktop) {
        // @todo testing for Qt::WA_PaintUnclipped is commented out because it
        // is said that it causes some problems on Win32 and we also saw the
        // problems with this line enabled in Qt3 on OS/2 in QT_PM_NO_WIDGETMASK
        // mode (terrible flicker in QFileDialog because QSplitter used there
        // sets WA_PaintUnclipped). This however doesn't make a big difference
        // now since we don't support QT_PM_NO_WIDGETMASK anymore (read below
        // about clipping) and is left here just to show where WA_PaintUnclipped
        // was originally intended to be used.
#if 0
        if (!testAttribute(Qt::WA_PaintUnclipped))
#endif
        {
            // We don't use WS_CLIPSIBLINGS and WS_CLIPCHILDREN because when these
            // styles are set and the child (sibling) window has a non-NULL clip region,
            // PM still continues to exclude the entire child's rectangle from the
            // parent window's update region, ignoring its clip region. As a result,
            // areas outside the clip region are left unpainted. Instead, we correct
            // the update region of the window ourselves, on every WM_PAINT event.
            // Note: for top-level widgets, we specify WS_CLIPSIBLINGS anyway to let
            // the system do correct clipping for us (qt_WinProcessWindowObstacles()
            // relies on this). It's ok because top-level widgets cannot be non-
            // rectangular and therefore don't require our magic clipping procedure.
            if (topLevel)
                style |= WS_CLIPSIBLINGS;
        }

        // for all top-level windows except popups we create a WC_FRAME
        // as a parent and owner.
    	if (topLevel && !popup) {
            if ((type == Qt::Window || dialog || tool)) {
                if (!(flags & Qt::FramelessWindowHint)) {
                    if (flags & Qt::MSWindowsFixedSizeDialogHint) {
                        fcFlags |= FCF_DLGBORDER;
                    } else if (tool) {
                        fcFlags |= FCF_BORDER;
                    } else {
                        fcFlags |= FCF_SIZEBORDER;
                    }
                }
                if (flags & Qt::WindowTitleHint)
                    fcFlags |= FCF_TITLEBAR;
                if (flags & Qt::WindowSystemMenuHint)
                    fcFlags |= FCF_SYSMENU | FCF_CLOSEBUTTON;
                if (flags & Qt::WindowMinimizeButtonHint)
                    fcFlags |= FCF_MINBUTTON;
                if (flags & Qt::WindowMaximizeButtonHint)
                    fcFlags |= FCF_MAXBUTTON;
            } else {
                fcFlags |= FCF_BORDER;
            }

            fStyle |= FS_NOMOVEWITHOWNER | FS_NOBYTEALIGN;
            fStyle |= WS_CLIPSIBLINGS | WS_CLIPCHILDREN;
    	}
    }

    if (flags & Qt::WindowTitleHint) {
        QString t = topLevel ? qAppName() : q->objectName();
        t = t.left(1).toUpper() + t.mid(1).toLower();
        title = t.toLocal8Bit();
    }

    // The Qt::WA_WState_Created flag is checked by translateConfigEvent() in
    // qapplication_pm.cpp. We switch it off temporarily to avoid move
    // and resize events during creation
    q->setAttribute(Qt::WA_WState_Created, false);

    if (desktop) { // desktop widget
        id = WinQueryDesktopWindow(0, 0);
        // @todo commented out on Win32 too
//      QWidget *otherDesktop = QWidget::find(id); // is there another desktop?
//      if (otherDesktop && otherDesktop->testWFlags(Qt::WPaintDesktop)) {
//          otherDesktop->d_func()->setWinId(0); // remove id from widget mapper
//          setWinId(id); // make sure otherDesktop is found first
//          otherDesktop->d_func()->setWinId(id);
//      } else {
            setWinId(id);
//      }
    } else if (topLevel) {
        // create top-level widget
        if (!popup) {
            HWND ownerw = NULLHANDLE;
            QWidget *parent = q->parentWidget();

            if (parent && !(parent->windowType() == Qt::Desktop))
                ownerw = parent->window()->d_func()->frameWinId();
            // create WC_FRAME
            FRAMECDATA fcData;
            fcData.cb = sizeof(FRAMECDATA);
            fcData.flCreateFlags = fcFlags;
            fcData.hmodResources = NULL;
            fcData.idResources = 0;
            // check whether a default icon is present in .EXE and use it if so
            ULONG sz = 0;
            if (DosQueryResourceSize(NULL, RT_POINTER, 1, &sz) == 0) {
                fcData.flCreateFlags |= FCF_ICON;
                fcData.idResources = 1;
            }
#if defined(QT_DEBUGWINCREATEDESTROY)
            qDebug("|Creating top level window (frame) [%s]:\n"
                   "|  owner = %08lX\n"
                   "|  title = '%s'\n"
                   "|  style = %08lX\n"
                   "|  fcFlags = %08lX",
                   widgetName(q).constData(), ownerw, title.constData(), fStyle,
                   fcFlags);
#endif
            fId = WinCreateWindow(HWND_DESKTOP, WC_FRAME, title, fStyle,
                                  0, 0, 0, 0, ownerw, HWND_TOP, 0,
                                  &fcData, NULL);
#if defined(QT_DEBUGWINCREATEDESTROY)
            qDebug("|  hwnd = %08lX", fId);
#endif
            if (fId == NULLHANDLE)
                qWarning("QWidget::create(): WinCreateWindow(WC_FRAME) "
                         "failed with 0x%08lX", WinGetLastError(0));

            PFNWP oldProc = WinSubclassWindow(fId, QtFrameProc);
            // remember QtOldFrameProc only once: it's the same for
            // all WC_FRAME windows.
            if (!QtOldFrameProc)
                QtOldFrameProc = oldProc;

            removeSysMenuAccels(fId);

            // create client window
#if defined(QT_DEBUGWINCREATEDESTROY)
            qDebug("|Creating top level window (client) [%s]:\n"
                   "|  owner & parent = %08lX\n"
                   "|  class = '%s'\n"
                   "|  title = '%s'\n"
                   "|  style = %08lX",
                   widgetName(q).constData(), fId, className.constData(),
                   title.constData(), style);
#endif
            // note that we place the client on top (HWND_TOP) to exclude other
            // frame controls from being analyzed in qt_WinProcessWindowObstacles
            id = WinCreateWindow(fId, className, title, style, 0, 0, 0, 0,
                                 fId, HWND_TOP, FID_CLIENT, NULL, NULL);
        } else {
#if defined(QT_DEBUGWINCREATEDESTROY)
            qDebug("|Creating top level window (popup) [%s]:\n"
                   "|  class = '%s'\n"
                   "|  title = '%s'\n"
                   "|  style = %08lX",
                   widgetName(q).constData(), className.constData(),
                   title.constData(), style);
#endif
            id = WinCreateWindow(HWND_DESKTOP, className, title, style,
                                 0, 0, 0, 0, NULLHANDLE, HWND_TOP, 0, NULL, NULL);
        }
#if defined(QT_DEBUGWINCREATEDESTROY)
        qDebug("|  hwnd = %08lX", id);
#endif
        if (id == NULLHANDLE)
            qWarning("QWidget::create(): WinCreateWindow "
                     "failed with 0x%08lX", WinGetLastError(0));
        setWinId(id);

        // it isn't mentioned in PMREF that PM is obliged to initialize window
        // data with zeroes (although seems to), so do it ourselves
        for (LONG i = 0; i <= (LONG) (QT_EXTRAWINDATASIZE - 4); i += 4)
            WinSetWindowULong(id, i, 0);

        SWP swp;
        // Get the default window position and size. Note that when the
        // FS_SHELLPOSITION flag is specified during WC_FRAME window
        // creation its size and position remains zero until it is shown
        // for the first time. So, we don't use FS_SHELLPOSITION but emulate
        // its functionality here.
        WinQueryTaskSizePos(0, 0, &swp);

        //update position & initial size of POPUP window
        const bool wasMoved = q->testAttribute(Qt::WA_Moved);
        const bool wasResized = q->testAttribute(Qt::WA_Resized);
        const bool isVisibleOnScreen = !q->testAttribute(Qt::WA_DontShowOnScreen);
        if (popup && initializeWindow && isVisibleOnScreen) {
            if (!q->testAttribute(Qt::WA_Resized)) {
                swp.cx = sw / 2;
                swp.cy = 4 * sh / 10;
            }
            if (!wasMoved) {
                swp.x = sw / 2 - swp.cx / 2;
                swp.y = sh / 2 - swp.cy / 2;
            }
        }

        ULONG fl = SWP_SIZE | SWP_MOVE;
        HWND insertBehind = NULLHANDLE;
        if ((flags & Qt::WindowStaysOnTopHint) || (type == Qt::ToolTip)) {
            insertBehind = HWND_TOP;
            fl |= SWP_ZORDER;
            if (flags & Qt::WindowStaysOnBottomHint)
                qWarning() << "QWidget: Incompatible window flags: the window "
                              "can't be on top and on bottom at the same time";
        } else if (flags & Qt::WindowStaysOnBottomHint) {
            insertBehind = HWND_BOTTOM;
            fl |= SWP_ZORDER;
        }
        WinSetWindowPos(popup ? id : fId, insertBehind,
                        swp.x, swp.y, swp.cx, swp.cy, fl);

        if (!popup) {
            QTLWExtra *top = topData();
            top->fId = fId;
            WinQueryWindowPos(fId, &swp);
            SWP cswp;
            WinQueryWindowPos(id, &cswp);
            // flip y coordinates
            swp.y = sh - (swp.y + swp.cy);
            cswp.y = swp.cy - (cswp.y + cswp.cy);
            // store frame dimensions
            QRect &fs = top->frameStrut;
            fs.setCoords(cswp.x, cswp.y, swp.cx - cswp.x - cswp.cx,
                                         swp.cy - cswp.y - cswp.cy);
            data.fstrut_dirty = false;
            if (wasMoved || wasResized) {
                // resize & move if necessary (we couldn't do it earlier
                // because we didn't know the frame dimensions yet)
                if (wasMoved) {
                    swp.x = data.crect.left() - fs.left();
                    swp.y = data.crect.top() - fs.top();
                }
                if (wasResized) {
                    swp.cx = data.crect.width() + fs.left() + fs.right();
                    swp.cy = data.crect.height() + fs.top() + fs.bottom();
                }
                // flip y coordinate
                swp.y = sh - (swp.y + swp.cy);
                WinSetWindowPos(fId, NULLHANDLE, swp.x, swp.y, swp.cx, swp.cy,
                                SWP_SIZE | SWP_MOVE);
            }
        }

        if (!popup || (initializeWindow && isVisibleOnScreen)) {
            // fetch the actual geometry
            WinQueryWindowPos(popup ? id : fId, &swp);
            // flip y coordinate
            swp.y = sh - (swp.y + swp.cy);
            if (popup) {
                data.crect.setRect(swp.x, swp.y, swp.cx, swp.cy);
            } else {
                const QRect &fs = topData()->frameStrut;
                data.crect.setRect(swp.x + fs.left(), swp.y + fs.top(),
                                   swp.cx - fs.left() - fs.right(),
                                   swp.cy - fs.top() - fs.bottom());
            }
        }
    } else if (q->testAttribute(Qt::WA_NativeWindow) || paintOnScreen()) {
        // create child widget
        Q_ASSERT(q->parentWidget());
        HWND parentw = q->parentWidget()->effectiveWinId();

#if defined(QT_DEBUGWINCREATEDESTROY)
        qDebug("|Creating child window [%s]:\n"
               "|  owner & parent = %08lX\n"
               "|  class = '%s'\n"
               "|  title = '%s'\n"
               "|  style = %08lX",
               widgetName(q).constData(), parentw, className.constData(),
               title.constData(), style);
#endif
        id = WinCreateWindow(parentw, className, title, style,
                             data.crect.left(),
                             // flip y coordinate
                             q->parentWidget()->height() - data.crect.bottom() - 1,
                             data.crect.width(), data.crect.height(),
                             parentw, HWND_TOP, 0, NULL, NULL);
#if defined(QT_DEBUGWINCREATEDESTROY)
        qDebug("|  hwnd = %08lX", id);
#endif
        if (id == NULLHANDLE)
            qWarning("QWidget::create(): WinCreateWindow "
                     "failed with 0x%08lX", WinGetLastError(0));
    	setWinId(id);
    }

    if (desktop) {
        q->setAttribute(Qt::WA_WState_Visible);
    }

    q->setAttribute(Qt::WA_WState_Created); // accept move/resize events

    hd = 0; // no presentation space

    if (extra && !extra->mask.isEmpty())
        setMask_sys(extra->mask);

    if (topLevel && (data.crect.width() == 0 || data.crect.height() == 0)) {
        q->setAttribute(Qt::WA_OutsideWSRange, true);
    }

    if (!topLevel && q->testAttribute(Qt::WA_NativeWindow) && q->testAttribute(Qt::WA_Mapped)) {
        Q_ASSERT(q->internalWinId() != NULLHANDLE);
        WinShowWindow(q->internalWinId(), TRUE);
    }
}

void QWidget::destroy(bool destroyWindow, bool destroySubWindows)
{
    Q_D(QWidget);
    if (!isWindow() && parentWidget())
        parentWidget()->d_func()->invalidateBuffer(geometry());
    d->deactivateWidgetCleanup();
    if (testAttribute(Qt::WA_WState_Created)) {
        setAttribute(Qt::WA_WState_Created, false);
        for(int i = 0; i < d->children.size(); ++i) { // destroy all widget children
            register QObject *obj = d->children.at(i);
            if (obj->isWidgetType())
                ((QWidget*)obj)->destroy(destroySubWindows,
                                         destroySubWindows);
        }
        if (mouseGrb == this)
            releaseMouse();
        if (keyboardGrb == this)
            releaseKeyboard();
        if (testAttribute(Qt::WA_ShowModal)) // just be sure we leave modal
            QApplicationPrivate::leaveModal(this);
        else if ((windowType() == Qt::Popup))
            qApp->d_func()->closePopup(this);
        if (destroyWindow && !(windowType() == Qt::Desktop) &&
            internalWinId() != NULLHANDLE) {
            HWND id = internalWinId();
            if (isWindow() && !(windowType() == Qt::Popup)) {
                // extra data including extra->topextra has been already
                // deleted at this point by deleteExtra() and therefore
                // calling frameWinId() is useless -- it will return the
                // client window handle. Use WinQueryWindow() instead.
                id = WinQueryWindow(id, QW_PARENT);
                Q_ASSERT(id != NULLHANDLE);
            }
#if defined(QT_DEBUGWINCREATEDESTROY)
            qDebug("|Destroying window [%s]:\n"
                   "|  hwnd = %08lX", widgetName(this).constData(), id);
#endif

            qt_WinDestroyWindow(id);
        }
        d->setWinId(0);
    }
}

void QWidgetPrivate::setParent_sys(QWidget *parent, Qt::WindowFlags f)
{
    // @todo implement
}

QPoint QWidget::mapToGlobal(const QPoint &pos) const
{
    // @todo implement
    return QPoint();
}

QPoint QWidget::mapFromGlobal(const QPoint &pos) const
{
    // @todo implement
    return QPoint();
}

void QWidgetPrivate::updateSystemBackground() {}

#ifndef QT_NO_CURSOR
void QWidgetPrivate::setCursor_sys(const QCursor &cursor)
{
    // @todo implement
}

void QWidgetPrivate::unsetCursor_sys()
{
    // @todo implement
}
#endif

void QWidgetPrivate::setWindowTitle_sys(const QString &caption)
{
    // @todo implement
}

void QWidgetPrivate::setWindowIcon_sys(bool forceReset)
{
    // @todo implement
}

void QWidgetPrivate::setWindowIconText_sys(const QString &iconText)
{
    Q_UNUSED(iconText);
}

QCursor *qt_grab_cursor()
{
    return mouseGrbCur;
}

void QWidget::grabMouse()
{
    // @todo implement
}

void QWidget::grabMouse(const QCursor &cursor)
{
    // @todo implement
}

void QWidget::releaseMouse()
{
    // @todo implement
}

void QWidget::grabKeyboard()
{
    // @todo implement
}

void QWidget::releaseKeyboard()
{
    // @todo implement
}

QWidget *QWidget::mouseGrabber()
{
    // @todo implement
    return 0;
}

QWidget *QWidget::keyboardGrabber()
{
    // @todo implement
    return 0;
}

void QWidget::activateWindow()
{
    // @todo implement
}

void QWidget::setWindowState(Qt::WindowStates newstate)
{
    // @todo implement
}

void QWidgetPrivate::hide_sys()
{
    Q_Q(QWidget);
    deactivateWidgetCleanup();
    Q_ASSERT(q->testAttribute(Qt::WA_WState_Created));
    if (q->windowFlags() != Qt::Desktop) {
        ULONG fl = SWP_HIDE;
        if (q->isWindow() && frameWinId() != NULLHANDLE) {
            if (!(q->windowFlags() & Qt::Popup))
                fl |= SWP_DEACTIVATE;
        }
        qt_WinSetWindowPos(frameWinId(), 0, 0, 0, 0, 0, fl);
        HSWITCH swEntry = maybeTopData() ? maybeTopData()->swEntry : NULLHANDLE;
        if (swEntry != NULLHANDLE) {
            SWCNTRL swc;
            WinQuerySwitchEntry(swEntry, &swc);
            swc.uchVisibility = SWL_INVISIBLE;
            WinChangeSwitchEntry(swEntry, &swc);
        }
    }
    if (q->isWindow()) {
        if (QWidgetBackingStore *bs = maybeBackingStore())
            bs->releaseBuffer();
    } else {
        invalidateBuffer(q->rect());
    }
    q->setAttribute(Qt::WA_Mapped, false);
}

void QWidgetPrivate::show_sys()
{
    Q_Q(QWidget);
#if defined(QT_NON_COMMERCIAL)
    QT_NC_SHOW_WINDOW
#endif
    if (q->testAttribute(Qt::WA_OutsideWSRange))
        return;
    q->setAttribute(Qt::WA_Mapped);
    Q_ASSERT(q->testAttribute(Qt::WA_WState_Created));

    if (q->testAttribute(Qt::WA_DontShowOnScreen)) {
        invalidateBuffer(q->rect());
        return;
    }

    if (frameWinId() != NULLHANDLE) {
        ULONG fl = SWP_SHOW;
        if (q->isWindow()) {
            if (q->isMinimized()) {
                fl = SWP_MINIMIZE;
            } else if (q->isMaximized()) {
                fl |= SWP_MAXIMIZE;
            } else {
                fl |= SWP_RESTORE;
            }

            if (!(q->testAttribute(Qt::WA_ShowWithoutActivating)
                  || (q->windowType() == Qt::Popup)
                  || (q->windowType() == Qt::ToolTip)
                  || (q->windowType() == Qt::Tool))) {
                fl |= SWP_ACTIVATE;
            }
        }

        qt_WinSetWindowPos(frameWinId(), 0, 0, 0, 0, 0, fl);

        if (q->isWindow()) {
            if (q->windowType() != Qt::Popup &&
                q->windowType() != Qt::Tool &&
                (q->windowType() != Qt::Dialog || !q->parentWidget())
            ) {
                HSWITCH &swEntry = topData()->swEntry;
                if (!swEntry) {
                    // lazily create a new window list entry
                    HWND id = frameWinId();
                    PID pid;
                    WinQueryWindowProcess(id, &pid, NULL);
                    SWCNTRL swc;
                    memset(&swc, 0, sizeof(SWCNTRL));
                    swc.hwnd = id;
                    swc.idProcess = pid;
                    swc.uchVisibility = SWL_VISIBLE;
                    swc.fbJump = SWL_JUMPABLE;
                    WinQueryWindowText(id, sizeof(swc.szSwtitle), swc.szSwtitle);
                    swEntry = WinAddSwitchEntry(&swc);
                } else {
                    SWCNTRL swc;
                    WinQuerySwitchEntry(swEntry, &swc);
                    swc.uchVisibility = SWL_VISIBLE;
                    WinChangeSwitchEntry(swEntry, &swc);
                }
            }

            ULONG wstyle = WinQueryWindowULong(frameWinId(), QWL_STYLE);
            if (wstyle & WS_MINIMIZED)
                data.window_state |= Qt::WindowMinimized;
            if (wstyle & WS_MAXIMIZED)
                data.window_state |= Qt::WindowMaximized;
        }
    }

    invalidateBuffer(q->rect());
}

void QWidgetPrivate::setFocus_sys()
{
    // @todo implement
}

void QWidgetPrivate::raise_sys()
{
    // @todo implement
}

void QWidgetPrivate::lower_sys()
{
    // @todo implement
}

void QWidgetPrivate::stackUnder_sys(QWidget* w)
{
    // @todo implement
}

/*
  Helper function for non-toplevel widgets. Helps to map Qt's 32bit
  coordinate system to Windpws's 16bit coordinate system.

  This code is duplicated from the X11 code, so any changes there
  should also (most likely) be reflected here.

  (In all comments below: s/X/PM/g)
 */

void QWidgetPrivate::setWSGeometry(bool dontShow)
{
    // @todo implement
}

void QWidgetPrivate::setGeometry_sys(int x, int y, int w, int h, bool isMove)
{
    // @todo implement
}

void QWidgetPrivate::setConstraints_sys()
{
    // @todo implement
}

void QWidgetPrivate::scroll_sys(int dx, int dy)
{
    // @todo implement
}

void QWidgetPrivate::scroll_sys(int dx, int dy, const QRect &r)
{
    // @todo implement
}

int QWidget::metric(PaintDeviceMetric m) const
{
    // @todo implement
    return 0;
}

void QWidgetPrivate::createSysExtra()
{
    // @todo implement
}

void QWidgetPrivate::deleteSysExtra()
{
    // @todo implement
}

void QWidgetPrivate::createTLSysExtra()
{
    extra->topextra->fId = NULLHANDLE;
    extra->topextra->swEntry = NULLHANDLE;
}

void QWidgetPrivate::deleteTLSysExtra()
{
    if (extra->topextra->swEntry != NULLHANDLE)
        WinRemoveSwitchEntry(extra->topextra->swEntry);
    // Note: extra->topextra->fId is cleaned up in QWidget::destroy()
}

void QWidgetPrivate::registerDropSite(bool on)
{
    // @todo implement
}

void QWidgetPrivate::setMask_sys(const QRegion &region)
{
    // @todo implement
}

void QWidgetPrivate::updateFrameStrut()
{
    Q_Q(QWidget);

    if (!q->testAttribute(Qt::WA_WState_Created))
        return;

    if (q->internalWinId() == NULLHANDLE) {
        data.fstrut_dirty = false;
        return;
    }

    QTLWExtra *top = maybeTopData();
    if (!top || top->fId == NULLHANDLE) {
        data.fstrut_dirty = false;
        return;
    }

    // this widget has WC_FRAME
    SWP swp;
    WinQueryWindowPos(top->fId, &swp);
    SWP cswp;
    WinQueryWindowPos(data.winid, &cswp);
    // flip y coordinates
    swp.y = QApplication::desktop()->height() - (swp.y + swp.cy);
    cswp.y = swp.cy - (cswp.y + cswp.cy);
    QRect &fs = top->frameStrut;
    fs.setCoords(cswp.x, cswp.y, swp.cx - cswp.x - cswp.cx,
                                 swp.cy - cswp.y - cswp.cy);
    // @todo do we really need to update crect here??
    //data.crect.setRect(swp.x + cswp.x, swp.y + cswp.y, cswp.cx, cswp.cy);

    data.fstrut_dirty = false;
}

void QWidgetPrivate::setWindowOpacity_sys(qreal level)
{
    // @todo implement
}

QPaintEngine *QWidget::paintEngine() const
{
    // @todo this is a place to return some direct on-screen PaintEngine once
    // we decide to support it

    // We set this bit which is checked in setAttribute for
    // Qt::WA_PaintOnScreen. We do this to allow these two scenarios:
    //
    // 1. Users accidentally set Qt::WA_PaintOnScreen on X and port to
    // OS/2 which would mean suddenly their widgets stop working.
    //
    // 2. Users set paint on screen and subclass paintEngine() to
    // return 0, in which case we have a "hole" in the backingstore
    // allowing use of GPI directly.
    //
    // 1 is WRONG, but to minimize silent failures, we have set this
    // bit to ignore the setAttribute call. 2. needs to be
    // supported because its our only means of embeddeding native
    // graphics stuff.
    const_cast<QWidgetPrivate *>(d_func())->noPaintOnScreen = 1;

    return 0;
}

QWindowSurface *QWidgetPrivate::createDefaultWindowSurface_sys()
{
    // @todo implement
    return 0;
}

void QWidgetPrivate::setModal_sys()
{
    // @todo implement
}

QT_END_NAMESPACE
