/****************************************************************************
** $Id$
**
** Implementation of QProcess class for OS/2
**
** Copyright (C) 1992-2001 Trolltech AS.  All rights reserved.
** Copyright (C) 2005 netlabs.org.  Further OS/2 Development.
**
** This file is part of the kernel module of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid Qt Enterprise Edition or Qt Professional Edition
** licenses may use this file in accordance with the Qt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include "qplatformdefs.h"
#include "qprocess.h"

#ifndef QT_NO_PROCESS

#include "qapplication.h"
#include "qptrqueue.h"
#include "qtimer.h"
#include "qregexp.h"
#include "private/qinternal_p.h"
#include "qt_os2.h"

#include <string.h>

//#define QT_QPROCESS_DEBUG

const HFILE HF_STDIN = 0;
const HFILE HF_STDOUT = 1;
const HFILE HF_STDERR = 2;
const HFILE HF_ANY = 0xFFFFFFFF; 

const PID PID_NULL = ~1; 

const ULONG PIPE_SIZE = 1024;

// just fake constants, for convenience
const int FD_PIPE_STDOUT = 1;
const int FD_PIPE_STDERR = 2;

#if defined(QT_QPROCESS_DEBUG)
#include <stdarg.h>
#endif

/***********************************************************************
 *
 * QProcessPrivate
 *
 **********************************************************************/
class QProcessPrivate
{
public:
    QProcessPrivate( QProcess *proc )
    {
	stdinBufRead = 0;
        pipeStdin = HF_ANY;
        pipeStdout = HF_ANY;
        pipeStderr = HF_ANY;
	exitValuesCalculated = FALSE;

	lookup = new QTimer( proc );
	qApp->connect( lookup, SIGNAL(timeout()),
		proc, SLOT(timeout()) );

	pid = PID_NULL;
    }

    ~QProcessPrivate()
    {
	reset();
    }

    void reset()
    {
	while ( !stdinBuf.isEmpty() ) {
	    delete stdinBuf.dequeue();
	}
	closeHandles();
	stdinBufRead = 0;
        pipeStdin = HF_ANY;
        pipeStdout = HF_ANY;
        pipeStderr = HF_ANY;
	exitValuesCalculated = FALSE;

	pid = PID_NULL;
    }

    void closeHandles()
    {
	if( pipeStdin != HF_ANY ) {
            DosDisConnectNPipe( pipeStdin );
            DosClose( pipeStdin );
            pipeStdin = HF_ANY;
	}
	if( pipeStdout != HF_ANY ) {
            // check if STDERR was redirected to STDOUT
            if ( pipeStderr == pipeStdout )
                pipeStderr = HF_ANY;
            DosDisConnectNPipe( pipeStdout );
            DosClose( pipeStdout );
            pipeStdout = HF_ANY;
	}
	if( pipeStderr != HF_ANY ) {
            DosDisConnectNPipe( pipeStderr );
            DosClose( pipeStderr );
            pipeStderr = HF_ANY;
	}
    }

    QMembuf bufStdout;
    QMembuf bufStderr;

    QPtrQueue <QByteArray> stdinBuf;

    HPIPE pipeStdin;
    HPIPE pipeStdout;
    HPIPE pipeStderr;

    PID pid;
    
    QTimer *lookup;

    uint stdinBufRead;

    bool exitValuesCalculated;
};


/***********************************************************************
 *
 * QProcess
 *
 **********************************************************************/
 
void QProcess::init()
{
    d = new QProcessPrivate( this );
    exitStat = 0;
    exitNormal = FALSE;
}

void QProcess::reset()
{
    d->reset();
    exitStat = 0;
    exitNormal = FALSE;
    d->bufStdout.clear();
    d->bufStderr.clear();
}

QMembuf* QProcess::membufStdout()
{
    if( d->pipeStdout != 0 )
	socketRead( FD_PIPE_STDOUT );
    return &d->bufStdout;
}

QMembuf* QProcess::membufStderr()
{
    if( d->pipeStderr != 0 )
	socketRead( FD_PIPE_STDERR );
    return &d->bufStderr;
}

QProcess::~QProcess()
{
    delete d;
}

#if defined(QT_QPROCESS_DEBUG)
static void DosPrintf( HFILE fd, const char *fmt, ... )
{
   static char buf[ 1024 ];
    
   va_list args;
   va_start( args, fmt );
   vsnprintf( buf, sizeof(buf), fmt, args );
   va_end ( args );
   
   ULONG written = 0;
   DosWrite( fd, buf, strlen( buf ), &written );
}
#endif

bool QProcess::start( QStringList *env )
{
#if defined(QT_QPROCESS_DEBUG)
    qDebug( "QProcess::start()" );
#endif
    reset();

    if ( _arguments.isEmpty() )
	return FALSE;

    // construct the arguments for DosExecPgm()

    QCString appName, appArgs;
    {
        QString args;
        for ( QStringList::ConstIterator it = _arguments.begin();
              it != _arguments.end(); ++ it
        ) {
            if ( it == _arguments.begin() ) {
                appName = (*it).local8Bit();
            } else {
                QString s = *it;
                // quote the argument when it contains spaces
                if ( s.contains( ' ' ) )
                    s = '"' + s + '"';
                if ( args.isNull() )
                    args = s;
                else
                    args += ' ' + s;
            }
        }
        appArgs = args.local8Bit();

#if defined(QT_QPROCESS_DEBUG)
        qDebug( "QProcess::start(): app [%s], args [%s]",
                appName.data(), appArgs.data() );
#endif
    }

    QByteArray envlist;
    if ( env != 0 ) {
        uint pos = 0;
        // inherit PATH if missing (for convenience)
        // (note that BEGINLIBPATH and ENDLIBPATH, if any, are automatically
        // inherited, while LIBPATH is always a global setting)
        {
            const char *path = getenv( "PATH" );
            if ( env->grep( QRegExp( "^PATH=", FALSE ) ).empty() && path ) {
                uint sz = strlen( path ) + 5 /* PATH= */ + 1;
                envlist.resize( envlist.size() + sz );
                sprintf( envlist.data() + pos, "PATH=%s", path ); 
                pos += sz;
            }
        }
        // inherit COMSPEC if missing (to let the child start .cmd and .bat)
        {
            const char *comspec = getenv( "COMSPEC" );
            if ( env->grep( QRegExp( "^COMSPEC=", FALSE ) ).empty() && comspec ) {
                uint sz = strlen( comspec ) + 8 /* COMSPEC= */ + 1;
                envlist.resize( envlist.size() + sz );
                sprintf( envlist.data() + pos, "COMSPEC=%s", comspec ); 
                pos += sz;
            }
        }
        // add the user environment
        for ( QStringList::ConstIterator it = env->begin();
              it != env->end(); ++ it
        ) {
            QCString var = (*it).local8Bit();
            uint sz = var.length() + 1;
            envlist.resize( envlist.size() + sz );
            memcpy( envlist.data() + pos, var.data(), sz );
            pos += sz;
        }
        // add the terminating 0
        envlist.resize( envlist.size() + 1 );
        envlist[ pos ++ ] = 0;
    }
    
    // search for the app executable

    const uint appNameLen = appName.length();
    const uint appArgsLen = appArgs.length();
    
    bool hasPath = appName[ strcspn( appName.data(), "/\\" ) ] != '\0';
    bool hasSpaces = strchr( appName.data(), ' ' ) != NULL;
    
    // list of executable file extensions, in order of CMD.EXE's precedence
    // (the first runs directly, the rest requires CMD.EXE) 
    const char *exeExts[] = { ".exe", ".cmd", ".bat" };
    
    // detect which exe extension do we have, if any
    int ext = -1;
    if ( appNameLen >= 4 ) {
        for ( uint i = 0; ext < 0 && i < sizeof(exeExts) / sizeof(exeExts[0]); ++ i )
            if ( stricmp( appName.data() + appNameLen - 4, exeExts[i] ) == 0 )
                ext = i;
    }

    QByteArray buf;
    QCString appNameFull;
    
    APIRET rc = 0;
    char pathBuf[ CCHMAXPATH ];
    
    // run through all possible exe extension combinations (+ no extension case)
    for ( uint i = 0; i <= sizeof(exeExts) / sizeof(exeExts[ 0 ]); ++ i ) {
        if ( i == 0 ) {
            // try the unmodified app name first if it contains a path spec
            // or has one of predefined exe extensions 
            if ( hasPath || ext >= 0 ) {
                if ( buf.size() < appNameLen + 1 )
                    buf.resize ( appNameLen + 1 );
                strcpy( buf.data(), appName.data() );
            } else
                continue;
        } else {
            ext = i - 1;
            const uint extLen = strlen( exeExts[ ext ] );
            uint sz = appNameLen + extLen + 1;
            if ( buf.size() < sz )
                buf.resize( sz );
            strcpy( buf.data(), appName.data() );
            strcpy( buf.data() + appNameLen, exeExts[ ext ] );
        }

#if defined(QT_QPROCESS_DEBUG)
        qDebug( "QProcess::start(): trying to find [%s]", buf.data() );
#endif
        if ( hasPath ) {
            uint lastSep = strlen( buf.data() );
            while ( lastSep-- && buf[ lastSep ] != '/' && buf[ lastSep ] != '\\' )
                ;
            buf[ lastSep ] = '\0';
            rc = DosSearchPath( 0, buf.data(), buf.data() + lastSep + 1,
                                pathBuf, sizeof(pathBuf) );
        } else {
            rc = DosSearchPath( SEARCH_IGNORENETERRS | SEARCH_ENVIRONMENT |
                                SEARCH_CUR_DIRECTORY, "PATH",
                                buf.data(), pathBuf, sizeof(pathBuf) );
        }
        
        if ( rc == NO_ERROR ) {
#if defined(QT_QPROCESS_DEBUG)
            qDebug( "QProcess::start(): found [%s]", pathBuf ); 
#endif
            appNameFull = pathBuf;
            break;
        }

        if ( rc != ERROR_FILE_NOT_FOUND && rc != ERROR_PATH_NOT_FOUND ) {
#if defined(QT_QPROCESS_DEBUG)
            qDebug( "QProcess::start(): found [%s], but got an error:", pathBuf );
            qSystemWarning( "", rc );
#endif
            break;
        }
    }
    
    if ( appNameFull.isNull() ) {
#if defined(QT_CHECK_STATE) || defined(QT_QPROCESS_DEBUG)
#if defined(QT_DEBUG)
        qWarning( "[%s]:", appName.data() ); 
#endif
        qSystemWarning( "Failed to find the executable!", rc );
#endif
        return FALSE;
    }

    // final preparation of arguments
    
    if ( ext <= 0 ) {
        // run directly (args = <appName>\0\<appArgs>\0\0)
        uint sz = appNameLen + 1 + appArgsLen + 2;
        if ( buf.size() < sz )
             buf.resize( sz );
        strcpy( buf.data(), appName.data() );
        strcpy( buf.data() + appNameLen + 1, appArgs.data() );
        buf[ sz - 1 ] = '\0';
    } else {
        // run via shell (args = <shell>\0"/c "<appName>" "<appArgs>\0\0)
        appNameFull = getenv( "COMSPEC" );
        if ( appNameFull.isNull() ) {
            ULONG bootDrv = 0;
            rc = DosQuerySysInfo( QSV_BOOT_DRIVE, QSV_BOOT_DRIVE,
                             &bootDrv, sizeof(bootDrv) );
            if ( rc == NO_ERROR && bootDrv >= 1 ) {
                appNameFull = "?:\\OS2\\CMD.EXE";
                appNameFull[ 0 ] = char( bootDrv ) + 'A' - 1;
            }
        }
        if ( appNameFull.isNull() ) {
#if defined(QT_QPROCESS_DEBUG)
            qDebug( "QProcess::start(): OS/2 command line interpreter is not found!" );
#endif        
            return FALSE;
        }
        const uint shellLen = strlen( appNameFull );
        uint sz = shellLen + 1 + 3 + appNameLen + 1 + appArgsLen + 2;
        if ( hasSpaces )
            sz += 2; // for quotes
        if ( !appArgsLen )
            sz -= 1; // no need for space
        if ( buf.size() < sz )
             buf.resize( sz );
        strcpy( buf.data(), appNameFull );
        sprintf( buf.data() + shellLen + 1, hasSpaces ? "/c \"%s\"" : "/c %s",
                 appName.data() );
        if ( appArgsLen )
            strcat( buf.data() + shellLen + 1, appArgs.data() );
        buf[ sz - 1 ] = '\0';
    }

#if defined(QT_QPROCESS_DEBUG)
    qDebug( "QProcess::start(): will start [%s] as [%s] [%s]",
            appNameFull.data(), buf.data(), buf.data() + strlen( buf.data() ) + 1 ); 
#endif
    
    // create STDIN/OUT/ERR redirection pipes
    
    HFILE tmpStdin = HF_ANY, tmpStdout = HF_ANY, tmpStderr = HF_ANY;
    HFILE realStdin = HF_STDIN, realStdout = HF_STDOUT, realStderr = HF_STDERR;

    // we need the process identifier to guarantee pipe name unicity
    PPIB ppib = NULL;
    DosGetInfoBlocks( NULL, &ppib );

    if ( comms & Stdin ) {
        HFILE clientStdin = HF_ANY;
        do {
            // create a Stdin pipe
            sprintf( pathBuf, "\\pipe\\Qt\\%08lX\\QProcess\\%p\\Stdin",
                     ppib->pib_ulpid, this );
            rc = DosCreateNPipe( pathBuf, &d->pipeStdin,
                                 NP_ACCESS_OUTBOUND, NP_NOWAIT | NP_TYPE_BYTE | 1,
                                 PIPE_SIZE, PIPE_SIZE, 0 );
            if ( rc != NO_ERROR )
                break;
            DosConnectNPipe( d->pipeStdin );
            // open a client end of the Stdout pipe
            ULONG action = 0;
            rc = DosOpen( pathBuf, &clientStdin, &action, 0, FILE_NORMAL, FILE_OPEN,
                          OPEN_ACCESS_READONLY | OPEN_SHARE_DENYREADWRITE | OPEN_FLAGS_NOINHERIT,
                          (PEAOP2) NULL);
            if ( rc != NO_ERROR )
                break;
            // save the real STDIN handle
            if ( (rc = DosDupHandle( HF_STDIN, &tmpStdin)) != NO_ERROR )
                break;
            // redirect the real STDIN handle to the client end of the pipe
            if ( (rc = DosDupHandle( clientStdin, &realStdin)) != NO_ERROR )
                break;
        } while( 0 );
        // close the client end anyway (no more necessary)
        if ( clientStdin != HF_ANY )
            DosClose ( clientStdin );
        // close all opened handles on error
        if ( rc  != NO_ERROR ) {
#if defined(QT_QPROCESS_DEBUG)
            DosPrintf( tmpStderr != HF_ANY ? tmpStderr : realStderr, 
                       "Failed to create a STDIN redirection (%s): SYS%04ld\n\r",
                       pathBuf, rc ); 
#endif
            if ( tmpStdin != HF_ANY )
                DosClose ( tmpStdin );
            d->closeHandles();
            return FALSE;
        }
    }
    if ( comms & Stdout ) {
        HFILE clientStdout = HF_ANY;
        do {
            // create a Stdout pipe
            sprintf( pathBuf, "\\pipe\\Qt\\%08lX\\QProcess\\%p\\Stdout",
                     ppib->pib_ulpid, this );
            rc = DosCreateNPipe( pathBuf, &d->pipeStdout,
                                 NP_ACCESS_INBOUND, NP_NOWAIT | NP_TYPE_BYTE | 1,
                                 PIPE_SIZE, PIPE_SIZE, 0 );
            if ( rc != NO_ERROR )
                break;
            DosConnectNPipe( d->pipeStdout );
            // open a client end of the Stdout pipe
            ULONG action = 0;
            rc = DosOpen( pathBuf, &clientStdout, &action, 0, FILE_NORMAL, FILE_OPEN,
                          OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYREADWRITE | OPEN_FLAGS_NOINHERIT,
                          (PEAOP2) NULL);
            if ( rc != NO_ERROR )
                break;
            // save the real STDOUT handle
            if ( (rc = DosDupHandle( HF_STDOUT, &tmpStdout )) != NO_ERROR )
                break;
            // redirect the real STDOUT handle to the client end of the pipe
            if ( (rc = DosDupHandle( clientStdout, &realStdout )) != NO_ERROR )
                break;
        } while( 0 );
        // close the client end anyway (no more necessary)
        if ( clientStdout != HF_ANY )
            DosClose ( clientStdout );
        // close all opened handles on error
        if ( rc != NO_ERROR ) {
#if defined(QT_QPROCESS_DEBUG)
            DosPrintf( tmpStderr != HF_ANY ? tmpStderr : realStderr, 
                       "Failed to create a STDOUT redirection (%s): SYS%04ld\n\r", 
                       pathBuf, rc ); 
#endif
            if ( tmpStdin != HF_ANY ) {
                DosDupHandle( tmpStdin, &realStdin );
                DosClose( tmpStdin );
            }
            if ( tmpStdout != HF_ANY )
                DosClose ( tmpStdout );
            d->closeHandles();
            return FALSE;
        }
    }
    if ( (comms & Stderr) && !(comms & DupStderr) ) {
        HFILE clientStderr = HF_ANY;
        do {
            // create a Stderr pipe
            sprintf( pathBuf, "\\pipe\\Qt\\%08lX\\QProcess\\%p\\Stderr",
                     ppib->pib_ulpid, this );
            rc = DosCreateNPipe( pathBuf, &d->pipeStderr,
                                 NP_ACCESS_INBOUND, NP_NOWAIT | NP_TYPE_BYTE | 1,
                                 PIPE_SIZE, PIPE_SIZE, 0 );
            if ( rc != NO_ERROR )
                break;
            DosConnectNPipe( d->pipeStderr );
            // open a client end of the Stderr pipe
            ULONG action = 0;
            rc = DosOpen( pathBuf, &clientStderr, &action, 0, FILE_NORMAL, FILE_OPEN,
                          OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYREADWRITE | OPEN_FLAGS_NOINHERIT,
                          (PEAOP2) NULL);
            if ( rc != NO_ERROR )
                break;
            // save the real STDERR handle
            if ( (rc = DosDupHandle( HF_STDERR, &tmpStderr )) != NO_ERROR )
                break;
            // redirect the real STDERR handle to the client end of the pipe
            if ( (rc = DosDupHandle( clientStderr, &realStderr )) != NO_ERROR )
                break;
        } while( 0 );
        // close the client end anyway (no more necessary)
        if ( clientStderr != HF_ANY )
            DosClose ( clientStderr );
        // close all opened handles on error
        if ( rc != NO_ERROR ) {
#if defined(QT_QPROCESS_DEBUG)
            DosPrintf( tmpStderr != HF_ANY ? tmpStderr : realStderr, 
                       "Failed to create a STDERR redirection (%s): SYS%04ld\n\r", 
                       pathBuf, rc ); 
#endif
            if ( tmpStdin != HF_ANY ) {
                DosDupHandle( tmpStdin, &realStdin );
                DosClose( tmpStdin );
            }
            if ( tmpStdout != HF_ANY ) {
                DosDupHandle( tmpStdout, &realStdout );
                DosClose( tmpStdout );
            }
            if ( tmpStderr != HF_ANY )
                DosClose ( tmpStderr );
            d->closeHandles();
            return FALSE;
        }
    }
    if ( comms & DupStderr ) {
	d->pipeStderr = d->pipeStdout;
        do {
            // save the real STDERR handle
            if ( (rc = DosDupHandle( HF_STDERR, &tmpStderr )) != NO_ERROR )
                break;
            // redirect STDERR to STDOUT
            if ( (rc = DosDupHandle( HF_STDOUT, &realStderr )) != NO_ERROR )
                break;
        } while( 0 );
        // close all opened handles on error
        if ( rc != NO_ERROR ) {
#if defined(QT_QPROCESS_DEBUG)
            DosPrintf( tmpStderr != HF_ANY ? tmpStderr : realStderr, 
                       "Failed to redirect STDERR to STDOUT: SYS%04ld\n\r", 
                       rc ); 
#endif
            if ( tmpStdin != HF_ANY ) {
                DosDupHandle( tmpStdin, &realStdin );
                DosClose( tmpStdin );
            }
            if ( tmpStdout != HF_ANY ) {
                DosDupHandle( tmpStdout, &realStdout );
                DosClose( tmpStdout );
            }
            if ( tmpStderr != HF_ANY )
                DosClose ( tmpStderr );
            d->closeHandles();
            return FALSE;
        }
    }

    // DosExecPgm()
    
    RESULTCODES resc = { 0 };
        
    rc = DosExecPgm( pathBuf, sizeof(pathBuf), EXEC_ASYNCRESULT,
                     buf.data(), envlist.data(), &resc, appNameFull.data() );
    
    // cancel STDIN/OUT/ERR redirections
    if ( tmpStdin != HF_ANY ) {
        DosDupHandle( tmpStdin, &realStdin );
        DosClose( tmpStdin );
    }
    if ( tmpStdout != HF_ANY ) {
        DosDupHandle( tmpStdout, &realStdout );
        DosClose( tmpStdout );
    }
    if ( tmpStderr != HF_ANY ) {
        DosDupHandle( tmpStderr, &realStderr );
        DosClose( tmpStderr );
    }
    
    if ( rc != NO_ERROR ) {
#if defined(QT_CHECK_STATE) || defined(QT_QPROCESS_DEBUG)
#if defined(QT_DEBUG)
        qWarning( "[%s] [%s]\n(%s):",
                  buf.data(), buf.data() + strlen( buf.data() ) + 1,
                  pathBuf ); 
#endif
        qSystemWarning( "Failed to start a new process!", rc );
#endif
        d->closeHandles();
        return FALSE;
    }
    
    // memporize PID of the started process
    d->pid = resc.codeTerminate;
    
    if ( ioRedirection || notifyOnExit ) {
	d->lookup->start( 100 );
    }

    return TRUE;
}

void QProcess::tryTerminate() const
{
    if ( d->pid == PID_NULL )
        return;

    HSWITCH hswitch = WinQuerySwitchHandle( NULL, d->pid );
    if ( hswitch != NULLHANDLE ) {
        SWCNTRL swcntrl = { 0 };
        APIRET rc = WinQuerySwitchEntry( hswitch,  &swcntrl );
        // WinQuerySwitchEntry will return a switch entry of the parent
        // process if the specfied one doesn't have a separate session
        // (running a plain CMD.EXE is an example); ignore this case. 
        if ( rc == NO_ERROR && swcntrl.idProcess == d->pid )
        {
            // first, ensure that the Close action is enabled in the main frame
            // window (otherwise WM_SYSCOMMAND/SC_CLOSE will be ignored)
            HWND hwndSysMenu = WinWindowFromID( swcntrl.hwnd, FID_SYSMENU );
            if (hwndSysMenu) {
                WinPostMsg( hwndSysMenu, MM_SETITEMATTR,
                            MPFROM2SHORT( SC_CLOSE, TRUE ),
                            MPFROM2SHORT( MIA_DISABLED, 0 ) );            
            }
            WinPostMsg( swcntrl.hwnd, WM_SYSCOMMAND,
                        MPFROM2SHORT( SC_CLOSE, CMDSRC_OTHER ),
                        MPFROMLONG( FALSE ) );
        }
    }
}

void QProcess::kill() const
{
    if ( d->pid != PID_NULL )
	DosKillProcess( DKP_PROCESS, d->pid );
}

bool QProcess::isRunning() const
{
    if ( d->pid == PID_NULL )
	return FALSE;

    PID pidEnded = PID_NULL;
    RESULTCODES resc = { 0 };
    APIRET rc = DosWaitChild( DCWA_PROCESS, DCWW_NOWAIT, &resc,
                              &pidEnded, d->pid );        
    
    if ( rc == ERROR_CHILD_NOT_COMPLETE )
        return TRUE;
    
    // there might be data to read
    QProcess *that = (QProcess *) this;
    that->socketRead( FD_PIPE_STDOUT ); // try stdout
    that->socketRead( FD_PIPE_STDERR ); // try stderr
    
    // compute the exit values
    if ( !d->exitValuesCalculated ) {
        that->exitNormal = resc.codeTerminate == TC_EXIT;
        that->exitStat = resc.codeResult;
        d->exitValuesCalculated = TRUE;
    }
    
    d->pid = PID_NULL;
    d->closeHandles();
    return FALSE;
}

bool QProcess::canReadLineStdout() const
{
    if( d->pipeStdout == HF_ANY )
	return d->bufStdout.size() != 0;

    QProcess *that = (QProcess *) this;
    return that->membufStdout()->scanNewline( 0 );
}

bool QProcess::canReadLineStderr() const
{
    if( d->pipeStderr == HF_ANY )
	return d->bufStderr.size() != 0;

    QProcess *that = (QProcess *) this;
    return that->membufStderr()->scanNewline( 0 );
}

void QProcess::writeToStdin( const QByteArray& buf )
{
    d->stdinBuf.enqueue( new QByteArray(buf) );
    socketWrite( 0 );
}

void QProcess::closeStdin( )
{
    if ( d->pipeStdin != HF_ANY ) {
        DosDisConnectNPipe( d->pipeStdin );
        DosClose( d->pipeStdin );
        d->pipeStdin = HF_ANY;
    }
}

void QProcess::socketRead( int fd )
{
    Q_ASSERT( fd == FD_PIPE_STDOUT || fd == FD_PIPE_STDERR );
    
    HPIPE pipe = HF_ANY;
    switch( fd ) {
        case FD_PIPE_STDOUT: pipe = d->pipeStdout; break; 
        case FD_PIPE_STDERR: pipe = d->pipeStderr; break;
    }
    
    if ( pipe == HF_ANY )
        return;
    
    // get the number of bytes that are waiting to be read
    ULONG read = 0;
    AVAILDATA avail = { 0 };
    ULONG status = 0;
    APIRET rc = DosPeekNPipe( pipe, NULL, 0, &read, &avail, &status );
    if ( rc != NO_ERROR ) {
#if defined(QT_QPROCESS_DEBUG)
        qSystemWarning( "Failed to peek from the pipe!", rc ); 
#endif
        return;
    }
    
    if ( avail.cbpipe > 0 ) {
	QMembuf *buffer;
        if ( fd == FD_PIPE_STDOUT )
            buffer = &d->bufStdout;
        else
            buffer = &d->bufStderr; 
        
	QByteArray *ba = new QByteArray( avail.cbpipe );
        
        rc = DosRead( pipe, ba->data(), avail.cbpipe, &read );
        if ( rc != NO_ERROR ) {
            delete ba;
#if defined(QT_QPROCESS_DEBUG)
            qSystemWarning( "Failed to read from the pipe!", rc ); 
#endif
            return;
        }
        
	if ( read < avail.cbpipe )
	    ba->resize( avail.cbpipe );

	buffer->append( ba );
        if ( fd == FD_PIPE_STDOUT )
            emit readyReadStdout();
        else
            emit readyReadStderr(); 
    }
}

void QProcess::socketWrite( int )
{
    if ( d->pipeStdin == HF_ANY )
        return;
    
    while ( !d->stdinBuf.isEmpty() && isRunning() ) {
        ULONG written = 0;
        APIRET rc = DosWrite( d->pipeStdin,
                              d->stdinBuf.head()->data() + d->stdinBufRead,
                              QMIN( PIPE_SIZE, d->stdinBuf.head()->size() - d->stdinBufRead ),
                              &written );
        if ( rc != NO_ERROR ) {
#if defined(QT_QPROCESS_DEBUG)
            qSystemWarning( "Failed to write to the pipe!", rc ); 
#endif
	    d->lookup->start( 100 );
	    return;
        }
        
	d->stdinBufRead += written;
	if ( d->stdinBufRead == d->stdinBuf.head()->size() ) {
	    d->stdinBufRead = 0;
	    delete d->stdinBuf.dequeue();
	    if ( wroteToStdinConnected && d->stdinBuf.isEmpty() )
		emit wroteToStdin();
	}
    }
}

void QProcess::flushStdin()
{
    socketWrite( 0 );
}

/*
  Use a timer for polling misc. stuff.
*/
void QProcess::timeout()
{
    // Disable the timer temporary since one of the slots that are connected to
    // the readyRead...(), etc. signals might trigger recursion if
    // processEvents() is called.
    d->lookup->stop();

    // try to write pending data to stdin
    if ( !d->stdinBuf.isEmpty() )
	socketWrite( 0 );

    if ( ioRedirection ) {
	socketRead( FD_PIPE_STDOUT ); // try stdout
	socketRead( FD_PIPE_STDERR ); // try stderr
    }

    if ( isRunning() ) {
	// enable timer again, if needed
	if ( !d->stdinBuf.isEmpty() || ioRedirection || notifyOnExit )
	    d->lookup->start( 100 );
    } else if ( notifyOnExit ) {
	emit processExited();
    }
}

/*
  Used by connectNotify() and disconnectNotify() to change the value of
  ioRedirection (and related behaviour)
*/
void QProcess::setIoRedirection( bool value )
{
    ioRedirection = value;
    if ( !ioRedirection && !notifyOnExit )
	d->lookup->stop();
    if ( ioRedirection ) {
	if ( isRunning() )
	    d->lookup->start( 100 );
    }
}

/*
  Used by connectNotify() and disconnectNotify() to change the value of
  notifyOnExit (and related behaviour)
*/
void QProcess::setNotifyOnExit( bool value )
{
    notifyOnExit = value;
    if ( !ioRedirection && !notifyOnExit )
	d->lookup->stop();
    if ( notifyOnExit ) {
	if ( isRunning() )
	    d->lookup->start( 100 );
    }
}

/*
  Used by connectNotify() and disconnectNotify() to change the value of
  wroteToStdinConnected (and related behaviour)
*/
void QProcess::setWroteStdinConnected( bool value )
{
    wroteToStdinConnected = value;
}

QProcess::PID QProcess::processIdentifier()
{
    return d->pid;
}

#endif // QT_NO_PROCESS
