/****************************************************************************
** $Id: qsysxcpt_pm.cpp 145 2006-10-25 23:30:27Z dmik $
**
** OS/2 System Exception handling routines
**
** Copyright (C) 2006 netlabs.org.
**
** This file is part of the kernel module of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid Qt Enterprise Edition or Qt Professional Edition
** licenses may use this file in accordance with the Qt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

/*
 *  The below code is partly based on the except.h and except.c sources
 *  from the xwphelpers package (which is a part of the xworkplace product, see
 *  http://www.xworkplace.org, http://xworkplace.netlabs.org/ for more info).
 *  XWorkplace is Copyright (C) 1999-2002 Ulrich Moeller.
 */

#if !defined(QT_OS2_NO_SYSEXCEPTIONS)
 
#include "qt_os2.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>

/// @todo (r=dmik) add locking to:
//  a) handle simultaneous independent exceptions from different threads
//  b) raise (user) exceptions on all other threads in order to write their
//     contexts to the trap file when a real exception happens on one thread

/*! \class QtOS2SysXcptMainHandler qt_os2.h

    @todo (r=dmik) describe...
    
    - must be instantiated on the stack of the main thread only
    - must be instantiated before a QAplication instance is created
      (does nothing otherwise)
*/

/* static */
bool QtOS2SysXcptMainHandler::installed = FALSE;
QtOS2SysXcptCallback QtOS2SysXcptMainHandler::callback = NULL;
ERR QtOS2SysXcptMainHandler::libcHandler = NULL;

/*!
    @todo (r=dmik) describe...        
*/
QtOS2SysXcptMainHandler::QtOS2SysXcptMainHandler( QtOS2SysXcptCallback cb )
{
    rec.prev_structure = NULL;
    rec.ExceptionHandler = NULL;
    
    PTIB ptib = NULL;
    DosGetInfoBlocks( &ptib, NULL );
    Q_ASSERT( ptib && ptib->tib_ptib2 );

    if ( ptib && ptib->tib_ptib2 )
    {
        // must be instantiated only on the main (first) thread 
        Q_ASSERT( ptib->tib_ptib2->tib2_ultid == 1 );
        if ( ptib->tib_ptib2->tib2_ultid == 1 )
        {
            // must not be already instantiated
            Q_ASSERT( installed == FALSE );
            Q_ASSERT( libcHandler == NULL );
            if ( installed == FALSE && libcHandler == NULL )
            {
                installed = TRUE;
                callback = cb;
                // install the exception handler for the main thread 
                rec.ExceptionHandler = handler;
                DosSetExceptionHandler( &rec );
            }
        }
    }
}

/*!
    @todo (r=dmik) describe...        
*/
QtOS2SysXcptMainHandler::~QtOS2SysXcptMainHandler()
{
    Q_ASSERT( rec.ExceptionHandler == handler || rec.ExceptionHandler == NULL );
    if ( rec.ExceptionHandler == handler )
    {
        // uninstall the exception handler for the main thread
        DosUnsetExceptionHandler( &rec );
        rec.ExceptionHandler = NULL;
        callback = NULL; 
        installed = FALSE;
    }
}

static void qt_excEscapeString( FILE *file, const char *pcszStr );

#define XcptPvt QtOS2SysXcptMainHandler::Private

class XcptPvt
{
public:
    static void callbackWriter( const char *msg );

    static inline int askCallback( QtOS2SysXcptReq req )
    {
        if ( callback )
            return callback( req, NULL, 0 );
        return FALSE; 
    }

    static inline int letCallback( QtOS2SysXcptReq req )
    {
        if ( callback )
            return callback( req, callbackWriter, 0 );
        return FALSE; 
    }
    
    static FILE *file;
};

/* static */
FILE *XcptPvt::file = NULL;

/* static */
void XcptPvt::callbackWriter( const char *msg )
{
    if ( file )
        qt_excEscapeString( file, msg );
}

/*! \internal
    Writes the given string with [<>&'"] characters escaped for XML.
*/
static void qt_excEscapeString( FILE *file, const char *pcszStr )
{
    const char *pcszChars = "<>&'\"";
    const char *aszEntities[] = { "&lt;", "&gt;", "&amp;", "&apos;", "&quot;" };

    const char *pcsz = pcszStr;
    const char *pcszRepl = NULL;
    size_t cbLen = 0;

    if ( !pcsz )
        return;

    while( *pcsz )
    {
        cbLen = strcspn( pcsz, pcszChars );
        fwrite( pcsz, 1, cbLen, file );
        pcsz += cbLen;
        if ( !*pcsz )
            break;
        cbLen = strchr( pcszChars, *pcsz ) - pcszChars;
        pcszRepl = aszEntities[cbLen];
        fwrite( pcszRepl, 1, strlen(pcszRepl), file );
        ++ pcsz;
    }
}

/*! \internal
    Writes the given error message.
*/
static void qt_excWriteErrorMsg( FILE *file, ULONG ulIndent, const char *pcszMsg,
                                 ... )
{
    char szBuf[1024];
    va_list args;
    va_start( args, pcszMsg );
    fprintf( file, "%*s<Error message=\"", ulIndent, "" );
    vsnprintf( szBuf, sizeof(szBuf), pcszMsg, args );
    szBuf[sizeof(szBuf) - 1] = '\0';
    qt_excEscapeString( file, szBuf );
    fprintf( file, "\"/>\n" );
    va_end( args );
}

/*! \internal
    Writes the register name, value and optionally memory flags
*/
static void qt_excWriteReg( FILE *file, const char *pszName, ULONG ulValue,
                            BOOL bQueryMem = TRUE )
{
    fprintf( file, "     <Register name=\"%s\" value=\"%08lX\"", 
             pszName, ulValue );

    if ( bQueryMem )
    {
        APIRET arc;
        ULONG ulCount = 4;
        ULONG ulFlags = 0;
        arc = DosQueryMem( (PVOID) ulValue, &ulCount, &ulFlags );
    
        if ( arc == NO_ERROR || arc == ERROR_INVALID_ADDRESS )
        {
            if ( arc == NO_ERROR )
            {
                fprintf( file, " flags=\"%08lX\"", ulFlags );
                if ( ulFlags & (PAG_COMMIT | PAG_READ) == (PAG_COMMIT | PAG_READ) )
                    fprintf( file, " word=\"%08lX\"/>\n", *(ULONG *) ulValue );
                else
                    fprintf( file, "/>\n" );
            }
            else
                fprintf( file, " flags=\"invalid\"/>\n" );
        }
        else
        {
            fprintf( file, ">\n" );
            qt_excWriteErrorMsg( file, 6, "DosQueryMem returned %lu"
                                 "and flags %08lX", arc, ulFlags );
            fprintf( file, "     </Register>\n" );
        }
    }
    else
        fprintf( file, "/>\n" );
}

/*! \internal
    Writes information about a signle stack frame.
*/
static void qt_excWriteStackFrame( FILE *file, ULONG ulPointer, ULONG ulAddress )
{
    APIRET  arc = NO_ERROR;
    HMODULE hMod = NULLHANDLE;
    char    szMod[CCHMAXPATH] = "unknown";
    ULONG   ulObject = 0,
            ulOffset = 0;

    if ( ulPointer )
        fprintf( file, "     <Frame pointer=\"%08lX\">\n", ulPointer );
    else
        fprintf( file, "     <Frame pointer=\"current\">\n" );

    fprintf( file, "      <Location address=\"%08lX\">\n", ulAddress );
    
    arc = DosQueryModFromEIP( &hMod, &ulObject,
                              sizeof(szMod), szMod, &ulOffset,
                              ulAddress );

    if (arc != NO_ERROR)
        qt_excWriteErrorMsg( file, 7, "%s: DosQueryModFromEIP returned %lu",
                             szMod, arc );
    else
    {
        DosQueryModuleName( hMod, sizeof(szMod), szMod );
        fprintf( file, "       <Module ID=\"%04lX\" name=\"", hMod );
        qt_excEscapeString( file, szMod ); 
        fprintf( file, "\" \n"
                       "               segment=\"%04lX\" offset=\"%08lX\"/>\n",
                 ulObject + 1, ulOffset ); 
/// @todo (r=dmik) use .DBG and .SYM files to get symbols
//  (see debug.h and debug.c from the xwphelpers package) 
//        dbgPrintStackFrame(file,
//                           szFullName,
//                           ulObject,
//                           ulOffset);
    }

    {
        enum { enmDelta = 8 };
        UCHAR *pch = (UCHAR *) ulAddress - enmDelta;
        UCHAR *pchEnd = (UCHAR *) ulAddress + enmDelta - 1;
        ULONG ulCount = enmDelta * 2;
        ULONG ulFlags = 0;
        APIRET arc = NO_ERROR;
        while ( 1 )
        {
            arc = DosQueryMem( (void *) (ULONG) pch, &ulCount, &ulFlags );
            if ( arc == NO_ERROR )
            {
                if ( ulCount >= enmDelta * 2 )
                    break;
                if ( pch + ulCount <= (UCHAR *) ulAddress )
                {   // ulAddress is outside the pch object
                    pch += ulCount;
                    ulCount = enmDelta * 2 - ulCount;
                }
                else
                {   // ulAddress is within the pch object
                    pchEnd = pch += ulCount;
                    break;
                }
            }
            else if ( arc == ERROR_INVALID_ADDRESS )
            {
                if ( ((ULONG) pch) & 0xFFFFF000 == ulAddress & 0xFFFFF000 )
                    break; // the same page, ulAddress inaccessible
                pch = (UCHAR *) (ulAddress & 0xFFFFF000);
            }
        }
        fprintf( file, "       <Dump address=\"%08lX\">\n        ", pch );
        if ( arc == NO_ERROR &&
             ulFlags & (PAG_COMMIT|PAG_READ) == (PAG_COMMIT | PAG_READ) )
        {
            for ( ; pch < pchEnd; ++pch )
                fprintf( file, "%02lX%c", (ULONG) *pch,
                         ulAddress - (ULONG) pch == 1 ? '-' : ' ' );
            fprintf( file, "\n" );
        }
        else
            qt_excWriteErrorMsg( file, 0, "%08lX: DosQueryMem returned %lu"
                                 "and flags %08lX", pch, arc, ulFlags );
        fprintf( file, "       </Dump>\n" );
    }
    
    fprintf( file, "      </Location>\n"
                   "     </Frame>\n" );
}

/*! \internal
    Walks the stack and writes information about stack frames.
*/
static void qt_excWriteStackFrames( FILE *file, PTIB ptib,
                                    PCONTEXTRECORD pContextRec )
{
    PULONG pulStackWord = 0;

    fprintf( file, "    <Frames>\n" );

    // first the trapping address itself
    qt_excWriteStackFrame( file, 0, pContextRec->ctx_RegEip );

    pulStackWord = (PULONG) pContextRec->ctx_RegEbp;
    
    while (    pulStackWord != 0
            && pulStackWord < (PULONG) ptib->tib_pstacklimit )
    {
        if ( ((ULONG) pulStackWord & 0x00000FFF) == 0x00000000 )
        {
            // we're on a page boundary: check access
            ULONG ulCount = 0x1000;
            ULONG ulFlags = 0;
            APIRET arc = DosQueryMem( (void *) pulStackWord,
                                      &ulCount, &ulFlags );
            if (    (arc != NO_ERROR)
                 || (   arc == NO_ERROR
                     && (ulFlags & (PAG_COMMIT|PAG_READ))
                         != (PAG_COMMIT|PAG_READ)) )
            {
                fprintf( file, "     <Frame pointer=\"%08lX\">\n",
                         (ULONG) pulStackWord );
                qt_excWriteErrorMsg( file, 6, "DosQueryMem returned %lu "
                                     "and flags %08lX", arc, ulFlags );
                fprintf( file, "     </Frame>\n" );
                pulStackWord += 0x1000;
                continue; // while
            }
        }

        qt_excWriteStackFrame( file, (ULONG) pulStackWord, *(pulStackWord + 1) );
        pulStackWord = (PULONG) *(pulStackWord);
    } // end while

    fprintf( file, "    </Frames>\n" );
}

/*! \internal
    Writes the thread information.
*/
static void qt_excWriteThreadInfo( FILE *file, PTIB ptib,
                                   PEXCEPTIONREPORTRECORD pReportRec,
                                   PCONTEXTRECORD pContextRec )
{
    ULONG       ul = 0;
    ULONG       ulOldPriority = ~0;

    // raise this thread's priority, because this
    // might take some time
    ulOldPriority = ptib->tib_ptib2->tib2_ulpri;
    DosSetPriority( PRTYS_THREAD, PRTYC_REGULAR, PRTYD_MAXIMUM, 0 );

    fprintf( file, "  <Thread ID=\"%04lX\" slot=\"%04lX\" "
                             "priority=\"%04lX\" "
                             "mc=\"%04lX\" mcf=\"%04lX\">\n",
             ptib->tib_ptib2->tib2_ultid, ptib->tib_ordinal, ulOldPriority,
             ptib->tib_ptib2->tib2_usMCCount, ptib->tib_ptib2->tib2_fMCForceFlag );

    // *** generic exception info

    fprintf( file,
             "   <Exception type=\"%08lX\" flags=\"%08lX\" address=\"%08lX\">\n",
             pReportRec->ExceptionNum, pReportRec->fHandlerFlags,
             (ULONG) pReportRec->ExceptionAddress );
    
    for ( ul = 0; ul < pReportRec->cParameters;  ++ul )
    {
        fprintf( file, "    <Param value=\"%08lX\"/>\n",
                 pReportRec->ExceptionInfo[ul] );
    }

    fprintf( file, "   </Exception>\n" );
    
    // *** registers

    fprintf( file, "   <CPU>\n"
                   "    <Registers>\n" );

    if ( pContextRec->ContextFlags & CONTEXT_SEGMENTS )
    {
        qt_excWriteReg( file, "DS", pContextRec->ctx_SegDs, FALSE );
        qt_excWriteReg( file, "ES", pContextRec->ctx_SegEs, FALSE );
        qt_excWriteReg( file, "FS", pContextRec->ctx_SegFs, FALSE );
        qt_excWriteReg( file, "GS", pContextRec->ctx_SegGs, FALSE );
    }
    
    if ( pContextRec->ContextFlags & CONTEXT_INTEGER )
    {
        qt_excWriteReg( file, "EAX", pContextRec->ctx_RegEax );
        qt_excWriteReg( file, "EBX", pContextRec->ctx_RegEbx );
        qt_excWriteReg( file, "ECX", pContextRec->ctx_RegEcx );
        qt_excWriteReg( file, "EDX", pContextRec->ctx_RegEdx );
        qt_excWriteReg( file, "ESI", pContextRec->ctx_RegEsi );
        qt_excWriteReg( file, "EDI", pContextRec->ctx_RegEdi );
    }

    if ( pContextRec->ContextFlags & CONTEXT_CONTROL )
    {
        qt_excWriteReg( file, "CS", pContextRec->ctx_SegCs, FALSE );
        qt_excWriteReg( file, "EIP", pContextRec->ctx_RegEip );
        qt_excWriteReg( file, "SS", pContextRec->ctx_SegSs, FALSE );
        qt_excWriteReg( file, "ESP", pContextRec->ctx_RegEsp );
        qt_excWriteReg( file, "EBP", pContextRec->ctx_RegEbp );
        qt_excWriteReg( file, "EFLAGS", pContextRec->ctx_EFlags, FALSE );
    }
    
    fprintf( file, "    </Registers>\n"
                   "   </CPU>\n" );

    // *** stack

    fprintf( file, "   <Stack base=\"%08lX\" limit=\"%08lX\">\n",
             (ULONG) ptib->tib_pstack,
             (ULONG) ptib->tib_pstacklimit );
    
    if ( pContextRec->ContextFlags & CONTEXT_CONTROL )
    {
        qt_excWriteStackFrames( file, ptib, pContextRec );
    }    
    
    fprintf( file, "   </Stack>\n"
                   "  </Thread>\n" );
    
    // reset old priority
    DosSetPriority( PRTYS_THREAD, (ulOldPriority & 0x0F00) >> 8,
                                  (UCHAR) ulOldPriority,
                                  0 );
}

/*! \internal
    Writes exception information to the log file.
*/
static void qt_excWriteException( FILE *file,
                                  const char *pszExeName,
                                  const char *pszExeBase,
                                  PPIB ppib, PTIB ptib,
                                  PEXCEPTIONREPORTRECORD pReportRec,
                                  PCONTEXTRECORD pContextRec )
{
    ULONG       aulBuf[3];

    // *** application info

    {
        fprintf( file, " <Application name=\"" );
        if ( XcptPvt::askCallback( QtOS2SysXcptReq_AppName ) == TRUE )
            XcptPvt::letCallback( QtOS2SysXcptReq_AppName );
        else
            qt_excEscapeString( file, pszExeBase );
        fprintf( file, "\" version=\"" );
        if ( XcptPvt::askCallback( QtOS2SysXcptReq_AppVer ) == TRUE )
            XcptPvt::letCallback( QtOS2SysXcptReq_AppVer );
        else
            fprintf( file, "unknown" );
        fprintf( file, "\">\n" );

        if ( XcptPvt::askCallback( QtOS2SysXcptReq_ReportTo ) == TRUE )
        {
            fprintf( file, "  <Report to=\"" );
            XcptPvt::letCallback( QtOS2SysXcptReq_ReportTo );
            if ( XcptPvt::askCallback( QtOS2SysXcptReq_ReportSubj ) == TRUE )
            {
                fprintf( file, "\" subject=\"" );
                XcptPvt::letCallback( QtOS2SysXcptReq_ReportSubj );
            }
            fprintf( file, "\"/>\n" );
        }
        
        fprintf( file, " </Application>\n" );
    }
    
    // *** system info

    DosQuerySysInfo( QSV_VERSION_MAJOR, QSV_VERSION_REVISION,
                     &aulBuf, sizeof(aulBuf) );
    // Warp 3 is reported as 20.30
    // Warp 4 is reported as 20.40
    // Aurora is reported as 20.45

    fprintf( file,
             " <System name=\"OS/2\" version=\"%u.%u.%u\"/>\n",
             aulBuf[0], aulBuf[1], aulBuf[2] );

    // *** process info

    if ( ppib )
    {
        fprintf( file,
                 " <Process ID=\"%04lX\" parentID=\"%04lX\">\n"
                 "  <Module ID=\"%04lX\" name=\"",
                ppib->pib_ulpid, ppib->pib_ulppid, ppib->pib_hmte );
        qt_excEscapeString( file, pszExeName );
        fprintf( file,
                 "\"/>\n"
                 " </Process>\n" );
    }
    else
        qt_excWriteErrorMsg( file, 1, "ppib is NULL" );

	fprintf( file, " <Threads>\n" );
	
    if ( ptib && ptib->tib_ptib2 )
        qt_excWriteThreadInfo( file, ptib, pReportRec, pContextRec );
    else if ( !ptib )
        qt_excWriteErrorMsg( file, 1, "ptib is NULL" );
    else
        qt_excWriteErrorMsg( file, 1, "ptib->tib_ptib2 is NULL" );

	fprintf( file, " </Threads>\n" );
}

/**
 *  Opens an unique log file using \a pcszBasePath as the base path.
 *  On input, \a pszFileName is the desired base file name w/o extension.
 *  If it doesn't fit into the file name length limit (ENAMETOOLONG) for the
 *  FS of the given disk, then a 8x3 file name is attemptet as a fallback.
 *  On output, \a pszFileName will contain the full file name of the opened
 *  log file. Note that \a pszFileName must be at least CCHMAXPATH bytes length.
 */
static FILE *qt_excOpenLogFile( const char *pcszBasePath, char *pszFileName )
{
    FILE *file = NULL;

    char szFileName[CCHMAXPATH];
    char szDir[] = "\\.qt3traps";
    char szFile[] = "\\12345678.123";
    enum { cbFileName = sizeof(szFileName) };
    enum { cbDir = sizeof(szDir) };
    enum { cbFile = sizeof(szFile) };

    ULONG ul = 0;
    ULONG cbDirLen = 0;
    ULONG cbLen = 0;
    ULONG ulStamp = 0;
    char *pszStamp = NULL; 
    
    if ( pcszBasePath == NULL || pszFileName == NULL )
        return NULL;
    
    if ( access( pcszBasePath, F_OK ) != 0 )
        return NULL;
    
    if ( strlen( pcszBasePath ) + cbDir + cbFile - 2 >= CCHMAXPATH )
        return NULL;

    // get the full path if it's not 'X:'
    if ( strlen(pcszBasePath) != 2 ||
         pcszBasePath[1] != ':' )
    {
        if( DosQueryPathInfo( pcszBasePath, FIL_QUERYFULLNAME,
                              szFileName, cbFileName ) != NO_ERROR )
            return NULL;
    }
    else
        strcpy( szFileName, pcszBasePath );
    
    strcat( szFileName, szDir );
    if ( access( szFileName, F_OK ) != 0 )
        if ( mkdir( szFileName, 0777 ) != 0 )
            return NULL;
    
    cbDirLen = strlen( szFileName );
        
    // first, try to use the desired base file name

    // we will append -NN to the file name to add some 'fraction of a
    // second' granularity to avoid name conflicts (0 <= NNN <= 99)
    cbLen = strlen( pszFileName );
    if ( cbDirLen + cbLen + 8 /* \-NN.xml */ < CCHMAXPATH )
    {
        strcat( szFileName, "\\" );
        strcat( szFileName, pszFileName );
        cbLen += cbDirLen + 1 /* \ */;
        for( ul = 0; ul < 100; ++ul )
        {
            sprintf( szFileName + cbLen, "-%02ld.xml", ul );
            if ( access( szFileName, F_OK ) == 0 )
                continue;
            file = fopen( szFileName, "wt" );
            if ( file )
                break;
            if ( errno == ENAMETOOLONG )
                break;
        }
    }

    // next, try a time stamp as a 8x3 file name
    if ( file == NULL )
    {
        pszStamp = szFileName + cbDirLen + 1;
        strcat( szFileName, szFile );
        
        ulStamp = time( NULL );
    
        // In order to add some 'fraction of a second' granularity to the
        // timestamp, we shift it by 5 bits. This gives us 0x07FFFFFF seconds
        // which is a period of approx. 4,25 years. 5 bits in turn give us 32
        // fractions of a second.
        ulStamp <<= 5;
        
        // try some few adajcent stamps if the first one fails
        for ( ul = 0; ul < 32; ++ul, ++ulStamp )
        {
            sprintf( pszStamp, "%08lX.xml", ulStamp );
            if ( access( szFileName, F_OK ) == 0 )
                continue;
            file = fopen( szFileName, "wt" );
            if ( file )
                break;
        }
    }
    
    if ( file )
    {
        strncpy( pszFileName, szFileName, CCHMAXPATH - 1 );
        pszFileName[CCHMAXPATH - 1] = '\0';
    }
    
    return file;
}

/*! \internal
    Qt Exception handler.
*/
/* static */
ULONG APIENTRY
QtOS2SysXcptMainHandler::handler( PEXCEPTIONREPORTRECORD pReportRec,
                                  PEXCEPTIONREGISTRATIONRECORD pRegRec,
                                  PCONTEXTRECORD pContextRec,
                                  PVOID pv )
{
    PTIB ptib = NULL;
    PPIB ppib = NULL;
    
    /* From the VAC++3 docs:
     *      "The first thing an exception handler should do is check the
     *      exception flags. If EH_EXIT_UNWIND is set, meaning
     *      the thread is ending, the handler tells the operating system
     *      to pass the exception to the next exception handler. It does the
     *      same if the EH_UNWINDING flag is set, the flag that indicates
     *      this exception handler is being removed.
     *      The EH_NESTED_CALL flag indicates whether the exception
     *      occurred within an exception handler. If the handler does
     *      not check this flag, recursive exceptions could occur until
     *      there is no stack remaining."
     *
     *      So for all these conditions, we exit immediately.
     */

    if ( pReportRec->fHandlerFlags &
         (EH_EXIT_UNWIND | EH_UNWINDING | EH_NESTED_CALL) )
        return XCPT_CONTINUE_SEARCH;

    // can it ever fail?...
    DosGetInfoBlocks( &ptib, &ppib );
        
    switch (pReportRec->ExceptionNum)
    {
        case XCPT_ACCESS_VIOLATION:
        case XCPT_INTEGER_DIVIDE_BY_ZERO:
        case XCPT_ILLEGAL_INSTRUCTION:
        case XCPT_PRIVILEGED_INSTRUCTION:
        case XCPT_INVALID_LOCK_SEQUENCE:
        case XCPT_INTEGER_OVERFLOW:
        {
            // "real" exceptions:

            DATETIME dt;
            char szFileName[CCHMAXPATH];
            FILE *file = NULL;
            char szExeName[CCHMAXPATH] = "unknown";
            char szExeBase[CCHMAXPATH];
            
            DosBeep( 880, 200 );

            if ( ppib )
            {
                // get the main module name
                DosQueryModuleName( ppib->pib_hmte, sizeof(szExeName),
                                    szExeName );
                
                // find the base name (w/o path and extension) 
                char *psz = strrchr( szExeName, '\\' );
                if ( psz )
                    ++ psz;
                if ( !psz )
                    psz = szExeName;
                strcpy( szExeBase, psz );
                psz = strrchr( szExeBase, '.' );
                if ( psz )
                    *psz = '\0';
            }

            // compose the desired base file name for the log file
            // (<datetime>-<exe>)
            DosGetDateTime( &dt );
            sprintf( szFileName, "%04d%02d%02d%02d%02d%02d-%s",
                     dt.year, dt.month, dt.day,
                     dt.hours, dt.minutes, dt.seconds,
                     szExeBase );
            
            // open the log file:
            
            // first try the home directory, then the root drive
            // (see QDir::homeDirPath())
            file = qt_excOpenLogFile( getenv( "HOME" ), szFileName );
            if ( file == NULL )
            {
                char *pszHomeDrive = getenv( "HOMEDRIVE" );
                char *pszHomePath = getenv( "HOMEPATH" );
                if ( pszHomeDrive && pszHomePath &&
                     strlen( pszHomeDrive ) + strlen( pszHomePath ) < CCHMAXPATH )
                {
                    char szHomePath[CCHMAXPATH];
                    strcpy( szHomePath, pszHomeDrive );
                    strcat( szHomePath, pszHomePath );
                    file = qt_excOpenLogFile( szHomePath, szFileName );
                }
            }
            if ( file == NULL )
            {
                static char szBootDrive[] = "\0:";
                if ( !szBootDrive[0] )
                {
                    ULONG ulBootDrive;
                    DosQuerySysInfo( QSV_BOOT_DRIVE, QSV_BOOT_DRIVE,
                                     &ulBootDrive, sizeof(ulBootDrive) );
                    szBootDrive[0] = (char) ulBootDrive + 'A' - 1;
                }
                file = qt_excOpenLogFile( szBootDrive, szFileName );
            }

            if ( file != NULL )
            {
                fprintf( file,
                        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
                        "<!--\n"
                        "  A fatal error has occured while running this application.\n"
                        "  Please contact the application vendor, describe what happened\n"
                        "  and send the contents of this file saved locally as\n"
                        "  '%s'.\n"
                        "-->\n",
                        szFileName );

                fprintf( file,
                         "<Trap timestamp=\"%04d-%02d-%02dT%02d:%02d:%02d%c%02d:%02d\" "
                               "version=\"1.0\"\n"
                         "      generator=\"Qt Library Version %s\"\n"
                         "      xmlns=\"http://db.hugaida.com/xml/os2/sys/Trap\">\n",
                         dt.year, dt.month, dt.day,
                         dt.hours, dt.minutes, dt.seconds,
                         (dt.timezone > 0 ? '-' : '+'),
                         abs( dt.timezone / 60 ),
                         abs( dt.timezone % 60 ),
                         QT_VERSION_STR );

                XcptPvt::file = file;
                
                // write trap information
                qt_excWriteException( file, szExeName, szExeBase,
                                      ppib, ptib, pReportRec, pContextRec );

                XcptPvt::file = NULL;
                
                fprintf( file, "</Trap>\n\n" );
                fclose( file );
                
                // attempt to display the created file
                {
                    STARTDATA SData       = {0};
                    PID       pid         = 0;
                    ULONG     ulSessID    = 0;
                    
                    SData.Length  = sizeof(STARTDATA);
                    SData.PgmName = "E.EXE";
                    SData.PgmInputs = szFileName;
                    
                    DosStartSession( &SData, &ulSessID, &pid );
                }
            }
            else
                DosBeep( 220, 200 );
        }
        break;
    }

    // we never handle the exception ourselves

    if ( ptib && ptib->tib_ptib2 && ptib->tib_ptib2->tib2_ultid == 1 &&
         QtOS2SysXcptMainHandler::libcHandler != NULL )
    {
        // we are on the main thread and were installed from qt_init() during
        // QApplication initialization using a hack; pass control back to the
        // LIBC exception handler
        return QtOS2SysXcptMainHandler::libcHandler( pReportRec, pRegRec,
                                                     pContextRec, pv );
    }
    
    return XCPT_CONTINUE_SEARCH; 
}

#endif // !defined(QT_PM_NO_SYSEXCEPTIONS)
