/* $Id: lxsuper.cpp,v 1.1 1999-09-06 02:20:06 bird Exp $ */

/*
 * PE2LX LX code
 *
 * Copyright 1998 Sander van Leeuwen (sandervl@xs4all.nl)
 * Copyright 1998 Knut St. Osmundsen
 * Copyright 1998 Peter Fitzsimmons
 *
 *
 * Project Odin Software License can be found in LICENSE.TXT
 *
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "pe2lx.h"  //KSO Tue 2nd June 1998

#include <versionos2.h>

char *hex(ULONG num);


/** Borrowed dos header from executables generated by Watcom */
unsigned char achDosHeader[DOSHEADER_SIZE] = {
    0x4D, 0x5A, 0x80, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00,
    0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x0E, 0x1F, 0xBA, 0x0E, 0x00, 0xB4, 0x09, 0xCD, 0x21, 0xB8, 0x01, 0x4C, 0xCD, 0x21, 0x74, 0x68,
    0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x61, 0x6E, 0x20, 0x4F, 0x53, 0x2F, 0x32, 0x20, 0x33, 0x32,
    0x2D, 0x62, 0x69, 0x74, 0x20, 0x65, 0x78, 0x65, 0x63, 0x75, 0x74, 0x61, 0x62, 0x6C, 0x65, 0x0D,
    0x0A, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};


/** Register module in kernel32 (resource managment) for EXEs:
// push  [esp+4]            ;instance handle
// push  internal pe2lx version
// push  version resource id
// push  name_lookup_tableid
// push  orgres_lookup_tableid
// mov   ecx, KERNEL32:RegisterResourceInfo (stdcall)
// call  ecx
//
// call  original_entrypoint
//  ret
*/
/** <br>Exe TIBFix code */
unsigned char achEXETIBFix[EXETIBFIX_SIZE] = {
//push [esp+4]
    0xFF, 0x74, 0x24, 0x04,
//SvL: 18-7-'98: push internal pe2lx version
    0x68, 0x00, 0x00, 0x00, 0x00,
//SvL: 18-7-'98: push version resource id (-1 = no version resource present)
    0x68, 0xFF, 0xFF, 0xFF, 0xFF,
//push [table id1] (kernel32 proc depends on 0x77777777 & 0x88888888!!)
//Svl: So does void LXHeaderSuper::SaveConvertedNames()!
    0x68, 0x77, 0x77, 0x77, 0x77,
//push [table id1]
    0x68, 0x88, 0x88, 0x88, 0x88,
//mov  ecx, KERNEL32:RegisterResourceInfo (stdcall)
    0xB9, 0x99, 0x99, 0x99, 0x99,
//call ecx
    0xFF, 0xD1,
//call entrypoint
    0xE8, 0x8C, 0xFF, 0xFF, 0xFF,
//ret
    0xC3
};



/** Register module in kernel32 (resource managment) for DLLs:
// cmp   [esp+8], 1
// je    detach
// push  [esp+4]            ;instance handle
// push  internal pe2lx version
// push  version resource id
// push  name_lookup_tableid
// push  orgres_lookup_tableid
// mov   ecx, KERNEL32:RegisterDll (stdcall)
// call  ecx
//
// DLL entrypoint:
//   ;BOOL WINAPI DllMain(HINSTANCE hDLLInst, DWORD fdwReason, LPVOID lpvReserved)
// push  0
// mov   ecx, [esp+12]
// xor   ecx, 1          //0 -> 1, 1 -> 0
// push  ecx             //push entry type (1 = attach process, 0 = detach process)
// push  [esp+12]          //push module handle
// call  testje
// ret
//
// detach:
// push  [esp+4]         //instance handle
// call  KERNEL32:DllExitList
// ret
*/
/** <br>Dll TIBFix code
 * WARNING: Kernel32.dll depends on this layout (offsets of dllexitlist &
 * entry point call are hardcoded! (both calculated in RegisterDll)
 */
unsigned char achDLLTIBFix[DLLTIBFIX_SIZE] = {
//cmp [esp+8], 1
//  0xCC, 0x80, 0x7C, 0x24, 0x08, 0x01,
    0x80, 0x7C, 0x24, 0x08, 0x01,
//je detach
    0x74, 0x33,
//push [esp+4]
    0xFF, 0x74, 0x24, 0x04,
//SvL: 18-7-'98: push internal pe2lx version
    0x68, 0x00, 0x00, 0x00, 0x00,
//SvL: 18-7-'98: push version resource id (-1 = no version resource present)
    0x68, 0xFF, 0xFF, 0xFF, 0xFF,
//push [table id1] (kernel32 proc depends on 0x77777777 & 0x88888888!!)
//Svl: So does void LXHeaderSuper::SaveConvertedNames()!
    0x68, 0x77, 0x77, 0x77, 0x77,
//push [table id1]
    0x68, 0x88, 0x88, 0x88, 0x88,
//mov  ecx, KERNEL32:RegisterDll (stdcall)
    0xB9, 0x99, 0x99, 0x99, 0x99,
//call ecx
    0xFF, 0xD1,
//push 0
    0x6A, 0x00,
    0x8B, 0x4C, 0x24, 0x0C,
    0x83, 0xF1, 0x01,
    0x51, 0xFF, 0x74, 0x24, 0x0C,
//call entrypoint
    0xE8, 0x80, 0xFF, 0xFF, 0xFF,
//ret
    0xC3,
//detach:
//push [esp+4]
    0xFF, 0x74, 0x24, 0x04,
//call KERNEL32:DllExitList (stdcall)
    0xE8, 0x00, 0x00, 0x00, 0x00,
//ret
    0xC3
};



/**
 * Constructor
 * @remark  Exetype is not fully initialized - always use SetExeType(...)!
 */
LXHeaderSuper::LXHeaderSuper() : fFlags(0), EntryAddress(0), IsEXE(TRUE),
                    impmodules(NULL), impnames(NULL), impfixuprec(NULL),
                    nrimpfixups(0), nrofimpfixups(0), StackSize(0), nrsections(0),
                    nrintfixups(0), nrofintfixups(0), intfixuprec(NULL),
                    os2resource(NULL), nrids(0),nrofids(0),
                    nrresources(0), curresource(0), exports(NULL),exportsize(0), resids(NULL),
                    nrexports(0), curexport(0), expbundle(NULL), curexpb(NULL),
                    cvtname(NULL), curcvtname(0), nrcvtnames(0), datapage(NULL),
                    cvtnametableid(0), orgrestableid(0), orgrestable(NULL), nrorgres(0),
                    kernel32Object(-1), TIBOffKerImport(0), fConsole(FALSE), impnamesize(0),
                    impmodulesize(0),VersionResourceId(-1)
{
    ZERO(LXHdr);

    LXHdr.e32_magic[0] = E32MAGIC1;
    LXHdr.e32_magic[1] = E32MAGIC2;
    LXHdr.e32_border   = E32LEBO;
    LXHdr.e32_worder   = E32LEWO;
    LXHdr.e32_level    = E32LEVEL;
    LXHdr.e32_cpu      = E32CPU386;
    LXHdr.e32_os       = 0x01;  //OS/2
    LXHdr.e32_ver      = 0;     //who cares which version...
    LXHdr.e32_pagesize = 4096;  //default
    LXHdr.e32_pageshift= 0;     //default (doet watcom zo)
    LXHdr.e32_fixupsum = 0;     //no checksum supported
    LXHdr.e32_ldrsum   = 0;     //no checksum supported
    LXHdr.e32_mflags   = 0;
    LXHdr.e32_startobj = 1;     //code
    LXHdr.e32_objcnt   = 0;
    LXHdr.e32_objtab   = sizeof(LXHdr); //begin hiermee vlak na exe hdr

    //KSO Aug 15 1998 5:07pm: PESection is now dynamic!
    PESection          = NULL;
    cPESections        = 0;
    ZERO(ResSection);
}



/**
 *  Destructor
 * @remark  Pointers that are freed anywhere else *must* be set to NULL else we'll go INT 3
 */
LXHeaderSuper::~LXHeaderSuper()
{
    if (impnames)           free(impnames);
    if (impfixuprec)        free(impfixuprec);
    if (impmodules)         free(impmodules);
    if (os2resource)        free(os2resource);
    if (resids)             free(resids);
    if (cvtname)            free(cvtname);
    if (orgrestable)        free(orgrestable);
    if (exports)            free(exports);
    if (expbundle)          free(expbundle);
    if (datapage)           free(datapage);
    if (ResSection.rawdata) free((void*)ResSection.rawdata);
    if (PESection)          free(PESection);
}



/**
 * Set flag for no fixups.
 * @result  LXHdr.e32_mflags have the E32NOINTFIX flag set.
 */
void LXHeaderSuper::SetNoFixups()
{
    LXHdr.e32_mflags |= E32NOINTFIX;// | 0x00100000; //testing MTEADDRALLOCED
}



/**
 * Sets executable type to either EXE or DLL.
 * @param IsEXE  TRUE: EXE; FALSE: DLL
 * @result  LXHdr.e32_mflags have flags set EXE: E32MODEXE; DLL:E32LIBINIT, E32LIBTERM and E32MODDLL
 *          All TIBFIX stuff is set.
 */
void LXHeaderSuper::SetExeType(BOOL IsEXE)
{
    this->IsEXE = IsEXE;
    if (IsEXE)
    {
        szTIBFix          = &achEXETIBFix[0];
        TIBSize           = EXE_TIBSIZE;
        TIBOffEntry       = EXE_OFF_ENTRYFIX;
        TIBOffCall        = EXE_OFF_TIBCALL;
        TIBOffName        = EXE_OFF_NAMETABLE;
        TIBOffKerImport   = EXE_OFF_KERIMP;
        //SvL: 18-7-'98: Internal pe2lx version and version resource id offsets
        TIBOffPe2lxVer    = EXE_OFF_PE2LXVER;
        TIBOffVerResId    = EXE_OFF_VERRESID;
        LXHdr.e32_mflags &= ~(E32LIBINIT | E32LIBTERM | E32MODDLL); //mask off some dll flags
        LXHdr.e32_mflags |= E32MODEXE;
    }
    else
    {//Assuming per process initialization/termination...
        szTIBFix          = &achDLLTIBFix[0];
        TIBSize           = DLL_TIBSIZE;
        TIBOffEntry       = DLL_OFF_ENTRYFIX;
        TIBOffCall        = DLL_OFF_TIBCALL;
        TIBOffName        = DLL_OFF_NAMETABLE;
        TIBOffKerImport   = DLL_OFF_KERIMP;
        //SvL: 18-7-'98: Internal pe2lx version and version resource id offsets
        TIBOffPe2lxVer    = DLL_OFF_PE2LXVER;
        TIBOffVerResId    = DLL_OFF_VERRESID;
        LXHdr.e32_mflags &= ~E32MODEXE; //mask off exe flag.
        LXHdr.e32_mflags |= E32LIBINIT | E32LIBTERM | E32MODDLL;
    }
}



/**
 * Set entry point.
 * @param    vaEntryPoint  Virtualaddress of the entrypoint.
 * @remark   KSO Aug 15 1998: Rewritten
 */
BOOL LXHeaderSuper::SetEntryAddress(int vaEntryPoint)
{
   int i;
    i                  = GetSectionIdxByVA(vaEntryPoint);
    ltassert(i != -1);
   LXHdr.e32_startobj = GetObjectNoByVA(vaEntryPoint);
    ltassert(LXHdr.e32_startobj != -1);
    LXHdr.e32_eip      = vaEntryPoint - PESection[i].address;

   return TRUE;
}



/**
 * Sets Stacksize for thread 1.
 * @param  size  (Suggested) stacksize in bytes.
 */
void LXHeaderSuper::SetStackSize(int size)
{
    if (IsEXE)
    {
        //Open32 requires a stack size of at least 64k
        if (size < 128*1024)
            StackSize = 128*1024;
        else
            StackSize = size;
        LXHdr.e32_stacksize = StackSize;
    }

}



/**
 * Sets Module Name
 * @remark  This functions now stores the modulename in its right format.
 */
void LXHeaderSuper::SetModuleName(char *filename)
{
    //KSO: modulename is now stored in the right format (chLen, sName, wOrd)
    char fname[CCHMAXPATH];
    memset(modulename, 0, sizeof(modulename));
    strcpy((char*)SSToDS(fname), filename);
    strcpy(&modulename[1], StripExtension(StripPath((char*)SSToDS(fname))));
    UpCase(&modulename[1]);
    modulename[0] = (char)strlen(&modulename[1]);
}



/**
 * Stores an PE Section in the PESection[] array sorted by (virtual)address.
 * @param   rawdata      If ring0: offset of rawdata from an PE sectionheader,
 *                           else: pointer to data in memory.
 *                       If this pointer is NULL, there is no rawdata for it.
 * @param   rawsize      SizeOfRawdata from an PE sectionheader.
 * @param   virtualsize  Misc.VirtualSize from an PE sectionheader.
 * @param   address      VirtualAddress from an PE sectionheader.
 * @param   type         Section type - defined in lx.h - SECTION_*
 * @remark  Be aware of that you can't add more sections then you have allocated space for. (SetNrSections)
 */
void LXHeaderSuper::StoreSection(RAWDATA rawdata, int rawsize, int virtualsize, int address, int type)
{
    int i;
    int j;

    //Several exes have sections with virtual size 0
    if (virtualsize == 0)  virtualsize = rawsize;

    /*find place in section table (sort on address)*/
    i = 0;
    while (i < nrsections/*idx?*/ && PESection[i].address < address )
        i++;

    if ( i != nrsections )
    {
        /* move sections from this one entry back */
        j = nrsections;
        while (j > i)
        {
            memcpy(&PESection[j], &PESection[j-1], sizeof(PESection[j-1]));
            j--;
        }
    }

    PESection[i].type        = type;
#ifndef RING0
    /** NB! for PE2LX no raw data is represented with 0 */
    PESection[i].rawdata  = rawdata;
#else
    /** and for RING0 represented with ~0 */
    PESection[i].rawdata  = rawdata ? (rawdata) : ~0;
#endif
    PESection[i].address     = address;
    PESection[i].rawsize     = min(rawsize,virtualsize);
    PESection[i].virtualsize = max(virtualsize,rawsize);
    PESection[i].nrpages     = ((virtualsize + 0xFFF) & ~0xFFF) >> PAGE_SHIFT;
    PESection[i].endaddress  = address + PESection[i].nrpages*PAGE_SIZE - 1;

    /* Inc number of sections */
    nrsections++;
}



/**
 * Aligns Section in executables with no fixups. If section alignment is not a multiple
 * of 64kb all sections are put into one big section.
 * @return  TRUE : success<br>
 *          FALSE: failure - fatal - dead object!
 * @precond <b>ASSUMPTIONS</b>:<br>
 *          -Code section comes first<br>
 *          -Sections that aren't adjacent are aligned at 64kb boundary<br>
 *          I could fix the second assumption, but I'm too lazy right now.
 * @sketch  SvL: Align code section at 64 kb boundary<br>
 *          SvL: Put adjacent data section in one big section and align it to a 64 kb boundary <br>
 */
BOOL LXHeaderSuper::AlignSections(PIMAGE_OPTIONAL_HEADER pOH)
{
    int      iFirst;
    int      i;
    int      j;
    int      rawsize;
    int      idx;
    Section  CombinedData;

    //KSO Aug 15 1998: TIBFixObject - code removed

    /* More to do? */
    if (!(LXHdr.e32_mflags & E32NOINTFIX))
        return TRUE;    //rest not needed if there are internal fixups

    //KSO Aug 16 1998: Ignore TIBFixSection.
    iFirst = GetTIBSectionIdx() == 0;

    PESection[iFirst].nrinvalidpages = (PESection[iFirst].address & 0xFFFF) >> PAGE_SHIFT;
    PESection[iFirst].address       &= 0xFFFF0000;

    //SvL: Concatenate code & data sections into one LX object
    memcpy(SSToDS(&CombinedData), &PESection[iFirst], sizeof(Section));

    //Insert invalid pages to align section at 64 kb boundary
    CombinedData.type        = SECTION_COMBINEDDATA;
    CombinedData.rawdata     = NULL;    //not used here (just use the originals)

    //too much, but just to be sure (in case we need to add extra invalid pages)
    //TODO: This should be done safer in the future
    //datapage = (DataPage*)malloc((GetNrPages()+256)*sizeof(DataPage));
    //KSO : done - use IMAGESIZE... let's try!
    datapage = (DataPage*)malloc(((pOH->SizeOfImage >> PAGE_SHIFT) + 10)*sizeof(DataPage));
    ltassert(datapage != NULL);

    //process pages of first section (code)
    ltassert(PESection[iFirst].type & SECTION_CODE);
    rawsize = CombinedData.rawsize; //?

    //KSO Aug 15 1998: TIBFixObject - code removed

    j = idx = 0;

    for (j; j<CombinedData.nrpages; j++)
    {
        if (rawsize < PAGE_SIZE)
        {
            datapage[idx].size = rawsize;
            rawsize            = 0;
        }
        else
        {
            datapage[idx].size = PAGE_SIZE;
            rawsize           -= PAGE_SIZE;
        }
        datapage[idx].flags= 0;
        idx++;
    }

    /* The rest of the section */
    for ( i = iFirst + 1; i < nrsections; i++)
    {
        //Ignore TIBFixSection
        if (i == GetTIBSectionIdx())
        {
            cout << "Debuginfo 3: " << i <<endl;;
            ltassert(i+1 == nrsections); //TIBFixSection should be the last one
            continue;
        }

        if (PESection[i].address != CombinedData.endaddress + 1)
        {
            if ((PESection[i].address & 0xFFFF) == 0)    //Watcom generated exe
                return TRUE;    //all sections are already properly aligned
            cout << "Section not aligned at 64 kb boundary!!!" << endl;
            //Damn, more work to do.
            int insertpages;

            insertpages = (PESection[i].address - CombinedData.endaddress-1) >> PAGE_SHIFT;
            for (j=0;j<insertpages;j++)
            {
                datapage[idx].size  = 0;
                datapage[idx].flags = ZEROED;
                idx++;
            }
            CombinedData.nrpages += insertpages;
        }

        CombinedData.nrpages     += PESection[i].nrpages;
        CombinedData.rawsize     += PESection[i].rawsize;
        CombinedData.virtualsize += PESection[i].virtualsize;
        CombinedData.endaddress   = PESection[i].endaddress;

        rawsize = PESection[i].rawsize;
        for (j=0;j<PESection[i].nrpages;j++)
        {
            if (PESection[i].type == SECTION_UNINITDATA || rawsize == 0)
            {
                datapage[idx].size  = 0;
                datapage[idx].flags = ZEROED;
            }
            else
            {
                if (rawsize < PAGE_SIZE)
                {
                    datapage[idx].size = rawsize;
                    rawsize        = 0;
                }
                else
                {
                    datapage[idx].size = PAGE_SIZE;
                    rawsize           -= PAGE_SIZE;
                }
                datapage[idx].flags= 0;
            }
            idx++;
        }
        PESection[i].fInvalid = TRUE;
    }

   //KSO Aug 16 1998: Dynamic PESection and TIBFixSection
    Section *pTmp;
    pTmp = (Section*)realloc(PESection,(nrsections+1)*sizeof(Section));
    ltassert(pTmp != NULL);
    PESection = pTmp;
    PESection[iFirst].fInvalid = TRUE;

    memcpy(&PESection[nrsections++], SSToDS(&CombinedData), sizeof(Section));

    return TRUE;
}



/**
 * Get number of objects in the virtual LX-file.
 * @return  Number of Ojbects in the virtual LX-file.
 * @remark  Does <b>NOT</b> count the resource section.
 */
int LXHeaderSuper::GetNrObjects()
{
    int i;
    int nrobjs = 0;

    for ( i = 0; i < nrsections; i++)
    {
        if (!PESection[i].fInvalid)
            nrobjs++;
    }

    return (nrobjs);
}



/**
 * Get number of objects in the virtual LX-file.
 * @return  Number of Ojbects in the virtual LX-file.
 * @remark  <b>Does</b> count the resource section.
 */
int LXHeaderSuper::GetNrObjects2()
{
    int i;
    int nrobjs = 0;

    for ( i = 0; i < nrsections; i++)
    {
        if (!PESection[i].fInvalid)
            nrobjs++;
    }

    nrobjs += nrresources != 0;
    return (nrobjs);
}



/**
 * Adds a name export
 * @param   address  Entrypoint address.
 * @param   name     Pointer to the namezerostring.
 * @param   ordinal  Ordinalvalue of export.
 * @return  TRUE : on success<br>
 *          FALSE: on failure
 */
BOOL LXHeaderSuper::AddNameExport(int address, char *name, int ordinal)
{
    int  len = strlen(name);

    ltassert(exportsize > (ULONG)(curexport-exports) + min(len,127)+1+2);

    if (len > (127-3))
    {
        name[126] = 0;   //OS/2 maximum
        len = 127;
        cout << "LXHeaderSuper::AddNameExport: WARNING - Had to truncate export name." << endl;
    }

    *curexport = (BYTE)len;
    curexport++;
    strcpy(curexport, name);
    curexport += len;
    *(USHORT *)curexport = (USHORT)(ordinal);
    curexport += sizeof(USHORT);

    return TRUE;
}


/**
 *  Adds ordinal export.
 * @param   address  Entrypoint address of export.
 * @param   ordinal  Ordinal value of export.
 * @return  TRUE : on success<br>
 *          FALSE: on failure
 */
BOOL LXHeaderSuper::AddOrdExport(int address, int ordinal)
{
    /* static int lastord = 0;  KOS: statics should be data members to work in RING0 */
    int i;

    if (ordinal > lastord+1)
    {//add empty record(s)
        int diff = ordinal - lastord - 1;
        while (diff)
        {
            curexpb->b32_cnt  = min(diff, 255);
            diff             -= curexpb->b32_cnt;
            curexpb->b32_type = EMPTY;
            curexpb = (exportbundle *)((int)curexpb + 2*sizeof(char));
        }
    }

    curexpb->b32_cnt    = 1;
    curexpb->b32_type   = ENTRY32;
    i                   = GetObjectNoByVA(address);
    ltassert(i != -1);
    curexpb->b32_obj    = (USHORT)i;

    curexpb->e32_flags  = E32EXPORT | E32SHARED;
    i                   = GetSectionIdxByVA(address);
    ltassert(i != -1);
    curexpb->e32_offset = address - PESection[i].address;
    lastord             = ordinal;
    curexpb++;

    return TRUE;
}



/**
 *  Sets number of exports. (?? External Fixups ??)
 * @param   nr  Number of exports.
 * @return  TRUE : on success<br>
 *          FALSE: on failure
 */
BOOL LXHeaderSuper::SetNrExtFixups(int nr)
{
    exportsize = 64+nr*(127+1+2);
    exports    = (char *)malloc(exportsize); //length byte, name (max 127) + ordinal (USHORT)
    ltassert(exports != NULL);

    memset(exports, 0, exportsize);
    if (nr > 0)
    {
        expbundle = (exportbundle *)malloc(nr*max(sizeof(exportbundle), sizeof(forwardbundle)));
        ltassert(expbundle != NULL); //KSO: fatal error - exports will be freed in the destructor
        memset(expbundle, 0, nr*max(sizeof(exportbundle), sizeof(forwardbundle)));
    }
    curexpb   = expbundle;
    curexport = exports;

    //First entry in resident name table is module name
    //KSO: module name now have the right format! -  Jun 16 1998 4:12pm
    memcpy(exports, modulename, modulename[0]+1);
    curexport += *curexport + 1;

    *(USHORT *)curexport = 0;
    curexport           += sizeof(USHORT);
#if 0
    //first entry in export table is unused
    curexpb->b32_cnt    = 1;
    curexpb->b32_type   = EMPTY;
    curexpb = (exportbundle *)&curexpb->b32_obj;
#endif

    return TRUE;
}



/**
 * Adds forwarder.
 * @param   name       Pointer to the exportname (zerostring).
 * @param   ordinal  Ordinal number of export.
 * @param   forward  Pointer to the forwardinfo (zerostring).
 * @return  TRUE : on success<br>
 *          FALSE: on failure
 * @sketch forward gives the DLL name and name of the export (i.e. MYDLL.expfunc or MYDLL.#27)
 * @remark <b>TODO:</b> finish this <br>
 */
BOOL LXHeaderSuper::AddForwarder(char *name, int ordinal, char *forward)
{
#if 0
    int len = strlen(name);

    ltassert(curexport + len - exports >= exportsize)

    *curexport = len;
    curexport++;
    strncpy(curexport, name, 127);
    curexport += len;
    *(USHORT *)curexport = (USHORT)ordinal;
    curexport += sizeof(USHORT);

    nrexports++;
    return TRUE;
#else
    return FALSE;
#endif

}



/**
 * Sets number of internal offset fixups.
 * @param   nr  number of internal offset fixups.
 * @return  TRUE : on success<br>
 *          FALSE: on failure
 * @remark  iets te veel, but who cares (KSO: Sander could you translate)<br>
 *          IMPORTANT: Do not Add more Off32Fixups than you have Set!
 */
BOOL LXHeaderSuper::SetNrOff32Fixups(int nr)
{
    nrofintfixups = nr;
    intfixuprec = (intfixup *)malloc(nr*sizeof(intfixup));

    ltassert(intfixuprec != NULL || nr == 0);
    memset(intfixuprec, 0, nr*sizeof(intfixup));

    return TRUE;
}



/**
 * Adds internal 32-bit offsetfixup
 * @param   address  Address of fixup. (Virtual)
 * @remark  This function is not used in RING0 (LXFile)
 */
void LXHeaderSuper::AddOff32Fixup(int address)
{
    static intfixup  crossedpage;
    static BOOL      fCrossed = FALSE;
    static int       crossedpageaddress = 0;
    intfixup        *frec;
    int              i;
    int              no;
    int              targetaddr;

    frec = &intfixuprec[nrintfixups];

    //Check if we need to insert a crossed page fixup
    if (fCrossed == TRUE && address > crossedpageaddress)
    {
        memcpy((char *)frec, (char *)&crossedpage, sizeof(crossedpage));
        fCrossed           = FALSE;
        crossedpageaddress = 0;
        nrintfixups++;
        frec++;
        cout << "CrossPageFixup: Finished." << endl;
    }
    frec->nr_stype   = 0x07;                    //32 bits offset fixup
    frec->nr_flags   = 0x00 | 0x10 | 0x40; //internal fixup + 32 bits offset + 16 bits object nr
    frec->r32_soff   = (SHORT)(address & 0xFFF);

    i  = GetOriginalSectionIdxByVA(address); //KSO: Combineddata sections don't have rawdata.
    no = GetObjectNoByVA(address);
    if (no == -1 || i == -1)
    {
        cout << "Oops, fixup error 1...." << endl;
        return;
    }
    frec->srcpage  = (USHORT)GetPageNoByVA(address);
    frec->srcobj   = (USHORT)no;

    //SvL: address could be aligned at 64 kb boundary, so correct address calculation
    //KSO Aug 16 1998: TIBFixSection - removed code -  don't worry about TIB anymore
    targetaddr = *(int *)(PESection[i].rawdata + (address - (PESection[i].address + PESection[i].nrinvalidpages*PAGE_SIZE)));
    cout << " targetaddr: " << hex(targetaddr);
    cout << endl;
    cout << " address: " << hex(address) << endl;

    //SvL: The target object might be the combined data section, so don't use
    //     GetSectionByAddress
    i  = GetSectionIdxByVA(targetaddr); //KSO: not original!
    no = GetObjectNoByVA(targetaddr);
    if (no == -1 || i == -1)
    {
        cout << "Oops, fixup error 2...." << endl;
        return;
    }
    frec->targetaddr = targetaddr - PESection[i].address;   //relative
    frec->targetobj = (USHORT)no;

    //  cout << "src address " << frec->r32_soff << " srcpage " << frec->srcpage << " target " << frec->targetaddr << " frec " << (int)frec << endl;

    //SvL: Check for fixups that cross page boundaries
    //     If I read the lx doc right, you need to add two fixups in
    //     this case, one for original page and one for original page + 1
    //TODO: Doesn't work if the last page contains such a fixup!
    if (frec->r32_soff + 4 > PAGE_SIZE)
    {
        memcpy((char *)&crossedpage, (char *)frec, sizeof(crossedpage));
        //crossed page boundary -> negative offset in next page
        crossedpage.r32_soff  = (SHORT)((frec->r32_soff & 0xFFF) - PAGE_SIZE);
        crossedpage.srcpage   = frec->srcpage+1;
        crossedpageaddress    = (address + PAGE_SIZE) & ~0xFFF;
        fCrossed              = TRUE;
    }

    nrintfixups++;
}


/**
 * Store and sort import fixups.
 * @param    newfrec  Pointer to namefixup to add.
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 * @precond  SvL: 30-10-'97, assumption about sorting of fixups is incorrect, so:<br>
 *           ASSUMPTION: namefixup & ordfixup of same size
 */
BOOL LXHeaderSuper::StoreAndSortImport(namefixup *newfrec)
{
    namefixup *frec;
    namefixup *tmp;
    int        i;


    /*KSO: Out of nrimpfixups? Try to alloc some more */
    if (nrimpfixups >= nrofimpfixups)
    {
        namefixup *pTmp;
        pTmp = (namefixup*)realloc(impfixuprec, nrofimpfixups+0x20);
        if (pTmp != NULL)
        {
            impfixuprec = pTmp;
            nrofimpfixups += 0x20;
        }
        else
        {
            cout << "LXHeaderSuper::StoreAndSortImport: Can't realloc more impfixuprec. Sorry" << endl;
            return (FALSE);
        }
    }



    if (nrimpfixups != 0)
    {
        frec = &impfixuprec[nrimpfixups - 1];

        if (newfrec->srcpage >= frec->srcpage)
        {
            memcpy(&impfixuprec[nrimpfixups], newfrec, sizeof(namefixup));
            nrimpfixups++;
            return TRUE;
        }
    }
    else
    {
        memcpy(&impfixuprec[0], newfrec, sizeof(namefixup));
        nrimpfixups++;
        return TRUE;
    }

    frec = &impfixuprec[0];
    for (i = 0; i < nrimpfixups; i++)
    {
        if (newfrec->srcpage < frec->srcpage)
        {
#ifndef _asmutils_h_
            tmp = (namefixup *)malloc(nrimpfixups*sizeof(namefixup));
            ltassert(tmp != NULL);
            memcpy(tmp, frec, (nrimpfixups-i)*sizeof(namefixup));
            memcpy(frec, newfrec, sizeof(namefixup));
            memcpy(frec+1, tmp, (nrimpfixups-i)*sizeof(namefixup));
            free(tmp);
#else
            memmov(frec, sizeof(frec), (nrimpfixups-i)*sizeof(namefixup));
            memcpy(frec, newfrec, sizeof(namefixup));
#endif
            break;

        }
        frec++;
    }
    ltassert(i != nrimpfixups);
    nrimpfixups++;
    return TRUE;
}



/**
 * Store name import.
 * @param    modname Pointer to a the name(sz) of the module(DLL) from which the function is imported.
 * @param    idxmod  Index of the module (DLL) from which the function is imported.
 * @param    name    Pointer to the name of the imported function (zerostring).
 * @param    offset  Offset(address) of fixup. (Virtual)
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 */
BOOL LXHeaderSuper::StoreImportByName(char *modname, int idxmod, char *name, int offset)
{
    namefixup frec;

    if (impfixuprec == NULL)
    {
        nrofimpfixups = MAX_IMPORT;
        impfixuprec = (namefixup *)malloc(nrofimpfixups*sizeof(namefixup));
        ltassert(impfixuprec != NULL);
        memset(impfixuprec, 0, MAX_IMPORT*sizeof(namefixup));
    }
    if (impnames == NULL)
    {
        impnames   = (char *)malloc(64*1024);   //moet genoeg zijn
        ltassert(impnames != NULL);
        memset(impnames, 0, 64*1024);
        impnameoff  = impnames + 1; //SvL: Bugfix: base 1
        impnamesize = 1;        //include ending 0
    }

    /*KSO: if there is to few impfixups we will try to alloc some more */
    if (nrimpfixups >= nrofimpfixups)
    {
        namefixup *pTmp;
        pTmp = (namefixup*)realloc(impfixuprec, nrofimpfixups+0x20);
        if (pTmp != NULL)
        {
            impfixuprec = pTmp;
            nrofimpfixups += 0x20;
        }
        else
        {
            cout << "LXHeaderSuper::StoreImportByName: Can't realloc more impfixuprec. Sorry" << endl;
            return (FALSE);
        }
    }

    frec.nr_stype   = 0x07;     //32 bits offset
    frec.nr_flags   = 0x02 | 0x40; //import by name + mod offset = 16 bits
    frec.r32_soff   = (SHORT)(offset & 0xFFF); //offset within specific page
    frec.r32_objmod = (USHORT)(idxmod+1);    //vanaf 1!!!!!

    impnameoff[0]    = (char)strlen(name);
    strcpy(&impnameoff[1], name);
    frec.proc       = (USHORT)((int)impnameoff-(int)impnames); //32 bits target flag not set!
    frec.srcpage    = (USHORT)GetPageNoByVA(offset);

    impnamesize    += impnameoff[0]+1;
    impnameoff      += impnameoff[0]+1;

    return StoreAndSortImport((namefixup *)SSToDS(&frec));
}



/**
 * Store (add) ordinal import.
 * @param    idxmod  Index of the module (DLL) from which the function is imported.
 * @param    ordinal Ordinal number of the function to be imported.
 * @param    offset  Offset(address) of fixup. (Virtual)
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 */
BOOL LXHeaderSuper::StoreImportByOrd(int idxmod, int ordinal, int offset)
{
    ordfixup frec;

    if (impfixuprec == NULL)
    {
        nrofimpfixups = MAX_IMPORT;
        impfixuprec = (namefixup *)malloc(nrofimpfixups*sizeof(namefixup)); //this'll have to be more accurate - later
        ltassert(impfixuprec != NULL);
        memset(impfixuprec, 0, MAX_IMPORT*sizeof(namefixup));
    }

    /*KSO: if there is to few impfixups we will try to alloc some more */
    if (nrimpfixups >= nrofimpfixups)
    {
        namefixup *pTmp;
        pTmp = (namefixup*)realloc(impfixuprec, nrofimpfixups+0x20);
        if (pTmp != NULL)
        {
            impfixuprec = pTmp;
            nrofimpfixups += 0x20;
        }
        else
        {
            cout << "LXHeaderSuper::StoreImportByName: Can't realloc more impfixuprec. Sorry" << endl;
            return (FALSE);
        }
    }

    frec.nr_stype   = 0x07;      //32 bits offset
    frec.nr_flags   = 0x01 | 0x40; //import by ordinal + mod offset = 16 bits
    frec.r32_soff   = (SHORT)(offset & 0xFFF); //offset within specific page
    frec.r32_objmod = (USHORT)(idxmod+1);       //vanaf 1!!!!!
    frec.ord        = (USHORT)ordinal;
    frec.srcpage    = (USHORT)GetPageNoByVA(offset);

    return StoreAndSortImport((namefixup *)SSToDS(&frec));
}



/**
 * Store importmodules.
 * @param    moduels  Pointer to array of modulenames (zerostrings).
 * @param    nrmod    Number of modules in array.
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 */
BOOL LXHeaderSuper::StoreImportModules(char *modules, int nrmod)
{
    int size, i;
    char *src, *dest;

    //KSO:determin size first
    int msize;

    size = 0;
    msize = _msize(modules);
    src = modules;
    i = nrmod;
    kernel32Object = 0;
    while ( i > 0 )
    {
        ltassert((src-modules) < msize);
        int len = strlen(src)+1;
        if (!kernel32Object)
            kernel32Object = stricmp(src, "KERNEL32.DLL") == 0 ? 1 : 0;
        size += len - 4 + 1;    //-".dll" + lenght-char
        src += len;
        i--;
    }
    size += !kernel32Object ? 10 : 1;
    kernel32Object = -1;
    //KSO - end

    impmodules = (char *)malloc(size);
    ltassert(impmodules != NULL);
    impmodulesize    = 0;

    src  = modules;
    dest = impmodules;
    for (i = 0; i < nrmod; i++)
    {
        size = min(strlen(src), 12);
        dest[0] = (char)(size-4);
        dest++;
        strncpy(dest, src, size-4);
        dest[size-4] = 0;     // !!!
        UpCase(dest);
        if (strcmp(dest, "NETAPI32") == 0)
        {//SvL: Name conflict with MUGLIB\DLL\NETAPI32.DLL
            strcpy(dest, "WNETAP32");
            dest += size-4;
        }
        else
        {
            if (strcmp(dest, "KERNEL32") == 0)
                kernel32Object = i;
            dest += size-4;
        }
        src += strlen(src) + 1; //+ null byte
    }

    if (kernel32Object == -1)
    {//add kernel32
        kernel32Object = nrmod;
        dest[0] = 8;
        strcpy(&dest[1], "KERNEL32");
        dest += 9;
        nrmod++;
    }

    LXHdr.e32_impmodcnt = nrmod;
    dest[0] = 0;     //end of array
    impmodulesize = (int)dest - (int)impmodules;

    return TRUE;
}


/**
 * Systemmodules.
 */
char *sysmodules[] = {"KERNEL32",
    "USER32",
    "GDI32",
    "WINMM",
    "DSOUND",
    "DDRAW",
    "WNETAP32",
    "MPR",
    "OLE32",
    "ADVAPI32",
    "COMMDLG",
    "NTDLL",
    "WINSPOOL",
    "SHELL32",
    "TAPI32",
    "CAPI32",
    "VERSION",
    "WSOCK32"};



/**
 * Determins where a module is a (win32os2-) systemmodule.
 * @param    mod  Pointer to modulename (string)
 * @param    size Size of modulename
 * @return   TRUE : on systemmodule<br>
 *           FALSE:
 */
BOOL LXHeaderSuper::IsSystemModule(char *mod, int size)
{
    int i;

    ltasserthp(mod);

    for (i=0;i<sizeof(sysmodules)/sizeof(char *);i++)
    {
        if (!strncmp(mod, sysmodules[i], size))
            return (TRUE);
    }
    return (FALSE);
}



/**
 * Find a name given by index in an array of Pascal strings.
 * @param    table   Pointer to a an array of Pascal strings.
 * @param    index   Index of
 * @return   Offset into the table of the name requested by index.
 * @remark   Not currently used.
 */
int LXHeaderSuper::FindName(char *table, int index)
{
    int i, offset = 0;

    for (i=0;i<index-1;i++)
    {
        offset += table[0] + 1;
        table = table + table[0] + 1;
    }
    return (offset);
}



/**
 * Get the number of pagemap entries in the virtual LX-file.
 * @return   Number of pagemap entries.
 * @remark   Requires fixing the resource size (as done in savenewexe)
 */
int LXHeaderSuper::GetNrPages()
{
    int i;
    int nrpages = 0;

    for (i = 0; i < nrsections; i++)
    {
        if (!PESection[i].fInvalid)
            nrpages += PESection[i].nrpages + PESection[i].nrinvalidpages;
    }
//SvL: 13 aug 1998
#if 0
    if (IsEXE)
    {
        nrpages += StackSize >> PAGE_SHIFT;
        if (StackSize & 0xFFF)
            nrpages++;
    }
#endif

    if (nrresources)
    {
        nrpages += ResSection.virtualsize >> PAGE_SHIFT;
        if (ResSection.rawsize & 0xFFF)
            nrpages++;
    }
    return (nrpages);
}



/**
 *  ASCII(us) uppercase.
 * @param   mixedcase  Pointer to the string to capitalize.
 * @result  The mixedcase string is in uppercase.
 * @remark  Mixedcase should of course not be in a readonly page/object (RING3).
 */
void LXHeaderSuper::UpCase(char *mixedcase)
{
    int i;

    for (i=0;i<strlen(mixedcase);i++)
    {
        if (mixedcase[i] >= 'a' && mixedcase[i] <= 'z')
        {
            mixedcase[i] += 'A' - 'a';
        }
    }
}



/**
 * Sets module type. GUI/text-mode.
 * @param   type  Moduletype - see SYSTEM_* defines in lx.h
 */
void LXHeaderSuper::SetModuleType(int type)
{
    LXHdr.e32_mflags   |= E32PMAPI;

    if (IsEXE) //don't set it for dlls (in case this flag is accidentally set in the header)
        fConsole = (type == SYSTEM_CHARACTER);
}



/**
 * Get the page number which address points to. (Of course in the Virtual LX-file.)
 * @param   vaAddress  A virtual address in the page.
 * @return  Page number.
 * @remark  WAS: GetAddressPage(int address)
 */
int LXHeaderSuper::GetPageNoByVA(int vaAddress)
{
    int i, page = 0;

    for (i = 0; i < nrsections; i++)
    {
        if (PESection[i].fInvalid)
            continue;

        if (vaAddress > PESection[i].endaddress)
        {
            page += PESection[i].nrpages + PESection[i].nrinvalidpages;
        }
        else
        {
            ltassert(vaAddress >= PESection[i].address) //KSO Aug 15 1998 3:36pm
            page += ((vaAddress - PESection[i].address) >> PAGE_SHIFT);
            break;
        }
    }

    return (page);
}



/**
 * Gets section number by type.
 * @param    type  Section type. See SECTION_* in lx.h
 * @return   Section number on success<br>
 *           -1 on failure
 */
int LXHeaderSuper::GetSection(int type)
{
    int i;

    for (i=0;i<nrsections;i++)
    {
        if (PESection[i].type & type)
        {
            return (i);
        }
    }
    return (-1);
}



/**
 * Get index, into PESection, of the "original" section which vaAddress points into.
 * "original" - all sections but COMBINEDDATA ignoring invalid sections.
 * @param    vaAddress  Virtual address.
 * @return   Section index (into PESection) on success.<br>
 *           -1 on failure.
 * @remark   WAS: GetSectionByAddress(int address) was used.<br>
 *           KSO: Now uses type instead of rawdata to find Combined data section.
 */
int LXHeaderSuper::GetOriginalSectionIdxByVA(int vaAddress)
{
    int i;

    for (i = 0; i < nrsections; i++)
    {
        if (!(PESection[i].type & SECTION_COMBINEDDATA))
        {
            if (PESection[i].address <= vaAddress && PESection[i].endaddress > vaAddress)
                return i;
        }
    }
    return -1;
}



/**
 * Get index, into PESection, of the section which vaAddress points into.
 * @param    vaAddress  Virtual address.
 * @return   Section index (into PESection) on success.<br>
 *           -1 on failure.
 * @remark   WAS: GetSectionByAddress(int address) was used<br>
 *           KSO: Now uses type instead of rawdata to find Combined data section.
 */
int LXHeaderSuper::GetSectionIdxByVA(int vaAddress)
{
    int i;

    for (i = 0; i < nrsections; i++)
    {
        if (!(PESection[i].fInvalid))
        {
            if (PESection[i].address <= vaAddress && PESection[i].endaddress > vaAddress)
                return i;
        }
    }
    return -1;
}



/**
 * Get the object number (1 based), in the virtual LX-File, of the section
 * which vaAddress points into.
 * @param    vaAddress  Virtual address.
 * @return   Object number (1 based) on success.<br>
 *           -1 on failure.
 * @remark   SvL: Ignore sections combined into one data section<br>
 *           KSO: Do not count Invalid section! and 1 based<br>
 * @remark   RENAMED : was GetRealSectionByAddress(int address)<br>
 *           QUESTION: why do you check rawdata != 0 ?
 */
int LXHeaderSuper::GetObjectNoByVA(int vaAddress)
{
    int i, ObjNo;

    for (i = ObjNo = 0; i < nrsections; i++)
    {
        if (!PESection[i].fInvalid)
        {
            ObjNo++;
            #if 0
            if (PESection[i].rawdata || PESection[i].type == SECTION_COMBINEDDATA)
            {
                if (PESection[i].address <= vaAddress && PESection[i].endaddress > vaAddress )
                    return (ObjNo);
            }
            #else

            if (PESection[i].address <= vaAddress && PESection[i].endaddress > vaAddress )
                return (ObjNo);

            #endif
        }
    }

    return -1;
}



/**
 * Gets section index by address (index into PESection).
 * @return   Section index on success.<br>
 *           -1 on failure.
 * @remark   Ignores invalid sections.<br>
 * @remark   Unused!
 */
int LXHeaderSuper::GetSectionLEAddress(int address)
{
    int i;

    for (i=0;i<nrsections;i++)
    {
        if (PESection[i].fInvalid)   continue;
        if (PESection[i].rawdata)
        {
            if (PESection[i].address > address)
            {
                return (i-1);
            }
        }
    }
    return (-1);
}



/**
 *  Strip off path.
 * @param    path  Pointer to a filename or directoryname with path. (sz)
 * @return   Pointer (into the path string) to the end of the path (beginning of filename).
 * @remark   Does not change anything in the string
 */
char *LXHeaderSuper::StripPath(char *path)      //niets aan de string veranderen!!
{
    char *pszRet = path;

    while (*path != '\0')
    {
        if (*path == '\\' || *path == '/' || *path == ':')
            pszRet = path + 1;

        path++;
    }

    return pszRet;
}



/**
 * Strip off extention (if any)
 * @param   fname  Pointer to a filename (without path) (sz)
 * @return  fname without extension
 * @remark  This function changes the fname string. (Must be writable - RING3)
 */
char *LXHeaderSuper::StripExtension(char *fname)
{
    int i = 0;
    int j = -1;

    while (fname[i] != '\0')
    {
        if (fname[i] == '.')
            j = i;
        i++;
    }
    if (j >= 0)
        fname[j] = '\0';
    return fname;
}



/**
 * Gets a unique resource id.
 * @return   Unique resource id.
 */
int LXHeaderSuper::GetUniqueId()
{
    int i, j, id;

    for (j=0;j<65536;j++)
    {
        for (i=0;i<nrids;i++)
        {
            if (uniqueId == resids[i])
            {
                break;
            }
        }
        if (i == nrids)
        {
            id = uniqueId;
            resids[nrids] = id;
            nrids++;
            if (nrids <= nrofids)
            {
                int *pTmp;
                pTmp = (int*)realloc(resids, nrofids + 0x20);
                if (pTmp != NULL)
                {
                    resids = pTmp;
                    nrofids += 0x20;
                }
                else
                    ltassert(FALSE && "NO ids left");
            }
            uniqueId = (uniqueId + 1) & 0xFFFF;
            return (id);
        }
        uniqueId = (uniqueId + 1) & 0xFFFF;
    }
    cout << "No ids left!!!!" << endl;
    ltassert(FALSE && "NO ids left");
    return (0);
}



/**
 * Set number of resources.
 * @param    cnt  Count of resources.
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 */
BOOL LXHeaderSuper::SetNrResources(int cnt)
{
    cout << "Nr of resources: " << cnt << endl;

    //+2 for resource name lookup table and original res lookup table
    nrresources = cnt+2;

    //reserve space for original win32 resources + strings
    os2resource = (rsrc32 *)malloc((nrresources*16)*sizeof(rsrc32));
    ltassert(os2resource != NULL);

    memset(os2resource, 0, nrresources*16*sizeof(rsrc32));
    cvtname    = (NameId *)malloc(cnt*(sizeof(NameId)+256));
    ltassert2(cvtname != NULL, free(os2resource);os2resource = NULL);
    curcvtname = (NameId *)((char *)cvtname+sizeof(USHORT));     //first USHORT = nr of converted names
    ltasserthp2(curcvtname, free(os2resource);os2resource = NULL);
    memset(cvtname, 0, cnt*(sizeof(NameId)+256));

    //temporary storage for used resource ids (*16 since menus use getuniqueid too)
    nrofids = nrresources*16;
    resids = (int *)malloc(nrofids * sizeof(int));
    ltassert2(resids != NULL, free(cvtname);cvtname = NULL;free(os2resource);os2resource = NULL);
    memset(resids, 0, nrresources*16*sizeof(int));

    //table used in LockResource (kernel32) to return the original win32 resource
    //layout:   1st ULONG   nr of entries
    //      nth ULONG (n>1) 0xAAAABBBB, AAAA=org id, BBBB=OS/2 id
    orgrestable = (ULONG *)malloc((nrresources+1)*sizeof(ULONG));
    ltassert2(orgrestable != NULL, free(cvtname);cvtname = NULL; free(os2resource);os2resource = NULL;free(resids);resids = NULL);
    memset(orgrestable, 0, (nrresources+1)*sizeof(ULONG));

    return TRUE;
}



/**
 * Store a resource id.
 * @param    id  Resource id.
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 * @remark   All ids should be stored before we ask for any unique resource id.
 */
BOOL LXHeaderSuper::StoreResourceId(int id)
{
    resids[nrids] = id;
    nrids++;
    if (nrids >= nrofids)
    {
        int *pTmp;
        pTmp = (int*)realloc(resids, sizeof(int)*(nrofids + 0x20));
        if (pTmp != NULL)
        {
            resids = resids;
            nrofids += 0x20;
        }
        else
            ltassert(FALSE && "NO ids left");
    }

    return TRUE;
}



/**
 * Stores a win32 resource.
 * @param    id            Resource id.
 * @param    type          (OS/2) Resource type. (ignored)
 * @param    size          Size of resourcedata (bytes).
 * @param    resourcedata  Pointer to resource data.
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 * @remark   The following resources are currently stored in Win32 format:<br>
 *           Accelerators<br>
 *           Dialogs<br>
 *           Menus<br>
 */
BOOL LXHeaderSuper::StoreWin32Resource(int id, int type, int size, char *resourcedata)
{
    int newid = GetUniqueId();
    ltassert(newid != 0);
    ltassert(nrorgres+1 <= nrresources); //(resourcedata is asserted in StoreResource)

    //First entry used for nr of entries, so + 1
    orgrestable[++nrorgres] = ((id & 0xFFFF) << 16) + (newid & 0xFFFF);

    //store them all as rcdata
    return StoreResource(newid, RT_RCDATA, size, resourcedata);
}



/**
 * Stores a resource.
 * @param    id            Resource id.
 * @param    type          OS/2 resource type.
 * @param    size          Size of resourcedata (bytes).
 * @param    resourcedata  Pointer to resource data.
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 */
BOOL LXHeaderSuper::StoreResource(int id, int type, int size, char *resourcedata)
{
    //static BOOL fFirstIcon = FALSE; //KSO: data member in class
    int i, newsize;

    //Calc aligned size (4 bytes align)
    newsize = (size + 3) & ~3;
    if (ResSection.curoff + newsize > ResSection.rawsize)
    {
        char *pTmp;
        pTmp = (char*)realloc((char*)ResSection.rawdata, ResSection.rawsize + max(size,0x2000));
        if (pTmp != NULL)
        {
#ifndef RING0
            ResSection.rawdata = pTmp;
#else
            ResSection.rawdata = (ULONG)pTmp;
#endif
            ResSection.rawsize = ResSection.virtualsize = ResSection.rawsize + max(size,0x2000);
        }
        else
        {
            kprintf(("LXHeaderSuper::StoreResource: failed to realloc - rawsize(=%d) + size(=%d) ",ResSection.rawsize,max(size,0x2000)));
            return FALSE;
        }
    }
    ltassert((ULONG)resourcedata > MINPTR && (ULONG)resourcedata < MAXPTR)

    if (type == RT_POINTER && fFirstIcon == FALSE && curresource > 0)
    {
        //put this one in front of all resources so OS/2 recognizes
        //it as the program object icon
#ifndef _asmutils_h_
        char *tmpbuf = (char *)malloc(curresource*sizeof(rsrc32)+ResSection.curoff);
        ltassert(tmpbuf != NULL);
        memcpy(tmpbuf, (char *)os2resource, curresource*sizeof(rsrc32));
        memcpy((char *)&os2resource[1], tmpbuf, curresource*sizeof(rsrc32));
        memcpy(tmpbuf, (char*)ResSection.rawdata, ResSection.curoff);
        memcpy((char*)ResSection.rawdata, resourcedata, size);
        memcpy((char*)ResSection.rawdata+newsize, tmpbuf, ResSection.curoff);
        free(tmpbuf);
#else
        //memmov - found in asmutils
        memmov((void*)os2resource, sizeof(rsrc32), curresource*sizeof(rsrc32));
        memmov((char*)ResSection.rawdata, newsize, ResSection.curoff);
        memcpy((char*)ResSection.rawdata, resourcedata, size);
#endif

        //Padd alignment
        for (i=0;i<newsize-size;i++)
        {
            ((char*) ResSection.rawdata)[size+i] = 0;
        }

        os2resource[0].type    = (USHORT)type;
        os2resource[0].name    = (USHORT)id;
        os2resource[0].cb      = size;
        os2resource[0].offset  = 0;

        //modify all offsets
        for (i=1;i<=curresource;i++)
            os2resource[i].offset += newsize;
    }
    else
    {
        memcpy((char*)ResSection.rawdata+ResSection.curoff, resourcedata, size);

        //Padd alignment
        for (i=0;i<newsize-size;i++)
            ((char*)ResSection.rawdata)[ResSection.curoff+size+i] = 0;

        os2resource[curresource].type   = (USHORT)type;
        os2resource[curresource].name   = (USHORT)id;
        os2resource[curresource].cb     = size;
        os2resource[curresource].offset = ResSection.curoff;
    }

    ResSection.curoff              += newsize;
    curresource++;

    return TRUE;
}



/**
 * Sets size and allocate heap for resource section.
 * @param    size  Size in bytes.
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 * @remark   KSO: TODO - size*3 is not good enough, it wastes to much heap!
 */
BOOL LXHeaderSuper::SetResourceSize(int size)
{
    ResSection.virtualsize = ResSection.rawsize = size*3;    //just to be sure (KSO:must not be too sure... *3)
#ifndef RING0
    ResSection.rawdata     = (char *)malloc(ResSection.rawsize);
#else
    ResSection.rawdata     = (ULONG)malloc(ResSection.rawsize);
#endif
    ltassert((char*)ResSection.rawdata != NULL);
    memset((char*)ResSection.rawdata, 0, ResSection.rawsize);

    return TRUE;
}



/**
 *  Convert Win32 resource name to a resource id.
 * @param    name  Pointer to name (sz).
 * @return   New resource id.
 */
int LXHeaderSuper::ConvertNametoId(char *name)
{
    NameId *resname = (NameId *)((USHORT *)cvtname + 1);
    int id;
    int i;

    for (i=0;i<nrcvtnames;i++)
    {//check if this name is already converted
        if (stricmp(name, resname->name) == 0)
            return (resname->id);
        resname = (NameId *)((int)resname + sizeof(NameId) + strlen(resname->name));
    }
    //Not found, so create one
    id = GetUniqueId();
    curcvtname->id = id;
    strcpy(curcvtname->name, name);
    curcvtname = (NameId *)((int)curcvtname + sizeof(NameId) + strlen(name));
    nrcvtnames++;
    return (id);
}



/**
 * Saves the resource which contains the converted names.
 * @return   TRUE : on success<br>
 *           FALSE: on failure
 * @remark   Should allways be called after resources are processed.<br>
 *           It also addjusts the TIB.
 */
BOOL LXHeaderSuper::SaveConvertedNames()
{
    BOOL rc = TRUE;
    if (nrcvtnames)
    {
        //first USHORT = nr of converted names
        *(USHORT *)cvtname = nrcvtnames;
        cvtnametableid = GetUniqueId();
        ltassert(cvtnametableid != 0);
        //fill in parameter for RegisterResourceInfo
        *(LONG *)&szTIBFix[TIBOffName] = cvtnametableid;

        rc = StoreResource(cvtnametableid, RT_RCDATA, (int)curcvtname - (int)cvtname, (char *)cvtname);
    }

    if (nrorgres && rc)
    {
        //first ULONG = nr of original win32 resources stored in OS/2 exe/dll
        orgrestable[0] = nrorgres;
        orgrestableid = GetUniqueId();
        ltassert(orgrestableid != 0);

        //fill in parameter for RegisterResourceInfo
        *(LONG *)&szTIBFix[TIBOffName+5] = (fConsole << 24) | orgrestableid;

        rc = StoreResource(orgrestableid, RT_RCDATA, (nrorgres+1)*sizeof(ULONG), (char *)orgrestable);
    }
    else
        *(LONG *)&szTIBFix[TIBOffName+5]    = (fConsole << 24) | 0x888888;

    //SvL: 18-7-'98: Store internal pe2lx version and version resource id
    *(LONG *)&szTIBFix[TIBOffPe2lxVer] = PE2LX_VERSION;
    *(LONG *)&szTIBFix[TIBOffVerResId] = VersionResourceId;

    return rc;
}


/**
 * Set number of section in PESection array. Alloc heap for them and zero it out.
 * @param    cSections  Count of section.
 * @return   TRUE : On success.
 *           FALSE: On failure - fatal - object unusable
 */
BOOL LXHeaderSuper::SetNrSections(int cSections)
{
    //cPESections = cSections + (options.TIBFixObject && !(options.noExeTIBFix && IsEXE));
    cPESections = cSections + 1; //KSO Aug 16 1998: TIBFixSection;
    PESection   = (Section*)malloc(cPESections*sizeof(Section));
    ltassert(PESection != NULL);
    memset(PESection, 0, cPESections*sizeof(Section));

    return TRUE;
}



/**
 * Creates the TIBFix Section
 * @param    pOH  Pointer to optionalheader from the PE file.
 * @return   TRUE : On success.
 *           FALSE: On failure - fatal - object unusable
 */
BOOL LXHeaderSuper::CreateTIBSection(PIMAGE_OPTIONAL_HEADER pOH)
{
    ULONG base;

    /** @sketch
     * Where to place the TIBSection
     * if EXE then <br>
    * begin<br>
     *   if imagebase is not 0x10000 then<br>
     *     base = 0x10000<br>
     *   else<br>
     *     base = imagebase + imagesize (pageboundrary)<br>
    * end<br>
    *
     * if DLL then
     *   base = imagebase + imagesize (pageboundrary)<br>
     * We could easily change this.
     */
    if (IsEXE && pOH->ImageBase >= 0x10000)
        base = 0x10000;
    else
        base = pOH->ImageBase + pOH->SizeOfImage;

    StoreSection(
        (RAWDATA)szTIBFix,
        TIBSize,
        TIBSize,
        base,
        SECTION_CODE | SECTION_TIBFIX
        );

    return TRUE;
}



/**
 * Get Sectionindex of the TIBSection in the PESection array.
 * @return   Index into PESection array on success<br>
 *           -1 on failure
 */
int LXHeaderSuper::GetTIBSectionIdx()
{
    int i;

    if (IsEXE && options.noExeTIBFix)
         return -1;
    for (i = 0; i < nrsections; i++)
    {
        if (!PESection[i].fInvalid && PESection[i].type & SECTION_TIBFIX)
            return i;
    }

    return -1;
}



/**
 * Get Object number of the TIBSection (in the Virtual LX-file).
 * @return   Object number on success<br>
 *           -1 on failure
 */
int LXHeaderSuper::GetTIBObjectNo()
{
    int i,j;

    if (IsEXE && options.noExeTIBFix)
         return -1;

    for (j = i = 0; i < nrsections; i++)
    {
        if (!PESection[i].fInvalid)
        {
            j++;
            if (PESection[i].type & SECTION_TIBFIX)
                return j;
        }
    }

    return -1;
}



/** KSO: Why is this uniqueId static ??? */
int LXHeaderSuper::uniqueId = RESID_CONVERTEDNAMES;









