#!/bin/sh
#
# This script will filter $input_file through a number of sed scripts,
# building an output suitable for GNU C command-line help facility
#
# Currently it works only for invoke.texi, g77.texi and gcj.texi
#
# This script needs following tools:
# (ba)sh,sed,uniq
#
# There are some assumptions about input file which this script makes:
# a) Some sections contains options, some contains other useful info,
#    sections are identified by their name. If new sections are added
#    which would be useful to see in command-line help it would be needed
#    to modify a bit the script.
# b) Options are defined using @item{} or @itemx{}.
# c) After each "@item -option" should follow either another "@item -option"
#    or a short sentence describing what the options above perform. The
#    sentence should be delimited by '.'. Example:
#
# @item -Wundef
# Warn if an undefined identifier is evaluated in an @samp{#if} directive.
#
# d) @pxref{} should not span across multiple lines.
#    Example:
#
# @pxref{abc,,abc: a test program}
#
#    is good, while
#
# @pxref{abc,,abc:
# a test program}
#
# is bad.

input_file=$1
mode=$2

if [ -z $mode ]; then
 echo "Usage: $0 input_file conversion_mode"
 exit 1;
fi

# Some common SED patterns
common_transform='
/^@table/d
/^@end/d
/^ *$/d
/^$/d
/^@c /d
/^@.index.*/d
/^@opindex.*/d
s/\(@.\) *{/\1{/g
s/\\/\\\\/g
s/"/\\"/g
s/@\././g
s/@://g
s/@\*//g
s/^@item[x]* @var\({file}.*\)/@x{\1}/
s/^@item[x]* *\(-.*\)/@t{\1}/
s/@math/@i/g
s/@gccoptlist/@b/g
s/@option/@b/g
s/@command/@b/g
s/@gol//g
s/@section *\(.*\)/@1{\1}/g
s/@subsection *\(.*\)/@2{\1}/g
s/@samp/@b/g
s/@cite/@b/g
s/@var/@i/g
s/@code/@b/g
s/@emph/@b/g
s/@file/@u/g
s/@strong/@b/g
s/@minus{}/-/g
s/@dots{}/.../g
s/@dfn{\([^}]*\)}/\1/g
s/@r{\([^}]*\)}/\1/g
s/@pxref{[^,]*,[^,]*,\([^,]*\)}/see also: \1/
s/@pxref{\(.*\)}/see also: \1/
s/  / /g
s/i\.e\./i.e._/
s/\. [^.]*$/./
s/i\.e\._/i.e. /
s/^\(.*\)$/"\1 "/
'

fnexthelp="@b{@z -h\\\"File name suffixes\\\"}) "
case "${mode}" in
    gcc)
        lang='C/C++'
        ;;
    g77)
        lang='Fortran'
        ;;
    gcj)
        lang='Java'
        fnexthelp="@b{@z -h\\\"Input and output files\\\"}) "
        ;;
    cpp)
        lang='C/C++ preprocessor'
        fnexthelp="@b{@z -h\\\"Input and output files\\\"}) "
        ;;
esac

echo "/* This file was automatically made by $0 from \"$input_file\" */"

#------------------------------------------------------------------------------
# Below goes the default help text for all drivers
#------------------------------------------------------------------------------
echo 'const char *clh_default_text = {\'
echo '"@0{GNU compiler suite '${lang}' driver}"'
echo '"@nUsage: @z [option]... [file]...@n@n"'
case "${mode}" in
    cpp)
	echo '"Preprocess input files (or read from @b{stdin} if no files are specified)@n@n"'
	;;
    *)
	echo '"Compile a set of @b{'${lang}'} source code files (for details run '$fnexthelp'"'
	echo '"then (depending on options) link resulting object files to produce a executable.@n@n"'
	;;
esac
echo '"The program has a large amount of options, to see the help index type "'
echo '"@b{@z --help=*}. Below is a list of most common options:@n@n"'
[ $mode != cpp ] && \
echo '"@b{-c}		Compile or assemble the source files, but do not link@n"'
echo '"@b{-D@i{macro}{=@i{defn}}}	Define @i{macro} as @b{1} (if @i{defn} is missing) or @i{defn}@n"'
[ $mode != cpp ] && \
echo '"@b{-E}		Stop after the preprocessing stage; do not run the compiler@n"'
[ $mode != cpp ] && \
echo '"@b{-g}		Produce debugging information@n"'
echo '"@b{-h@i{topic}}		Display command-line help on @i{topic} (* = index)@n"'
echo '"@b{-H}		Print the name of each header file used@n"'
echo '"@b{-I@i{dir}}		Search the directory @i{dir} for header files@n"'
[ $mode != cpp ] && \
echo '"@b{-l@i{library}}	Search the library named @i{library} when linking@n"'
[ $mode != cpp ] && \
echo '"@b{-L@i{dir}}		Search the directory @i{dir} for @b{-l} files@n"'
echo '"@b{-M}		Output a dependence rule for @b{make} for each input file@n"'
echo '"@b{-MM}		Same as @b{-M} but ignore system include files@n"'
echo '"@b{-M{M}D}		Like @b{-M{M}} but put dependency information in a .d file@n"'
echo '"@b{-o @i{file}}		Place output into @i{file}@n"'
[ $mode != cpp ] && \
echo '"@b{-O{1..3,s}}	Enable optimizations (1-few ... 3-most, s-optimize by size)@n"'
[ $mode != cpp ] && \
echo '"@b{-p{g}}		Generate extra code to write profile information@n"'
[ $mode != cpp ] && \
echo '"@b{-Q}		Compiler prints out each function name as it is compiled@n"'
[ $mode != cpp ] && \
echo '"@b{-S}		Stop after the stage of compilation proper; do not assemble@n"'
echo '"@b{-v}		Print the commands executed to run the stages of compilation@n"'
[ $mode != cpp ] && \
echo '"@b{-x @i{language}}	Specify explicitly the @i{language} for the following files@n"'
[ $mode != cpp ] && \
echo '"@b{-w}		Inhibit all warning messages@n"'
[ $mode != cpp ] && \
echo '"@b{-Wall}		Enable most warning messages@n"'
echo '};'
echo ''

echo 'const char *gcc_clh_text = {\'
echo '"@0@b{GNU '${lang}' Command-Line Help Database Index}@n@n"'
echo '"Please specify the help section by using --help=x.y.z where x is "'
echo '"the section number, y is the subsection number and so on. You also "'
echo '"can specify a few starting characters of the option or section name(-s) "'
echo '"you are interested in, e.g. @b{\"@z --help=@i{-W}\"} will display "'
echo '"a description of all the options which starts with @b{-W}, "'
echo '"@b{\"@z --help=@i{VAX}\"} will display all sections which begins "'
echo '"with @i{VAX} and so on. As a equivalent to @b{--help} you can also "'
echo '"use @b{-h} which is shorter to type (for example, \"@b{--help=1.12.6}\" "'
echo '"is equivalent to \"@b{-h1.12.6}\").@n"'
 
case "${mode}" in
#-------------------------------------------------------------------------------
    gcc)
        echo '"@nThe following sections are available:@n@n"'
      #
      # Pass one: Option summary
      #
        sed -n -e '
         /^@section Option Summary/,/^@menu/p
        ' $input_file | \
        sed -n -e '
         /^@[sub]*section \(.*\)/p
         /^@xref/d
         /^@smallexample/d
         /^@item/,/^@end/p
        ' | \
        sed -e '
         s/^@item[x]* *\(.*\)/@2{\1}/
         s/^@emph{\([^}]*\)}/@3{\1}/
        ' | sed -e "$common_transform" && \
      #
      # Pass two: Options Controlling the Kind of Output
      #
        sed -n -e '
         /^@section Options Controlling the Kind of Output/,/^@node/p
        ' $input_file | \
        sed -n -e '
         /^@[sub]*section/p
         /^@item[x]* *-/,/\./p
         s/^@cindex file name suffix/@t{File name suffixes}@n/p
         /^@item[x]* @var{file}/,/^ *$/p
         /@example/,/^@end/p
        ' | \
        sed -e '
         s/^@example.*/Possible values for @var{language} are:@n@b{/
         s/^@end.*/}/
        ' | sed -e "$common_transform" | sed -e "/^ *$/d" && \
      #
      # Pass three: All other options
      #
        sed -n -e '
         /^@section Options Controlling C Dialect/,/^@node/p
         /^@section Options Controlling C++ Dialect/,/^@node/p
         /^@section Options Controlling Objective-C Dialect/,/^@node/p
         /^@section Options to Control Diagnostic Messages Formatting/,/^@node/p
         /^@section Options to Request or Suppress Warnings/,/^@node/p
         /^@section Options for Debugging Your Program or GNU CC/,/^@node/p
         /^@section Options That Control Optimization/,/^@node/p
         /^@section Options Controlling the Preprocessor/,/^@node/p
         /^@section Passing Options to the Assembler/,/^@node/p
         /^@section Options for Linking/,/^@node/p
         /^@section Options for Directory Search/,/^@node/p
         /^@section Specifying Target Machine and Compiler Version/,/^@node/p
         /^@section Hardware Models and Configurations/,/^@section/p
         /^@section Options for Code Generation Conventions/,/^@node/p
        ' $input_file | \
        sed -n -e '
         /^@[sub]*section/p
         /^@item[x]* *-/{
          p
          : nextline
          n
          s/\. .*/\./
          s/\.$/\./
          p
          t
          b nextline
         }
        ' | uniq | sed -e "$common_transform" && \
      #
      # Pass four: Environment variables
      #
        sed -n -e '
         /^@section Environment Variables Affecting GNU CC/,/^@node/p
        ' $input_file | \
        sed -n -e '
         /^@[sub]*section/p
         /^@item.*/,/\./p
        ' | \
        sed -e '
         s/^@item[x]* *\(.*\)/@t{\1}/
        ' | sed -e "$common_transform"
        ;;
#-------------------------------------------------------------------------------
    g77)
        echo '"The @b{@z} driver supports all the options supported by @b{gcc}. "'
        echo '"For information on the non-Fortran-specific aspects of the "'
        echo '"@b{gcc} driver (and, therefore, the @b{@z} driver) please use @b{gcc} "'
        echo '"command-line help system.@n"'
        echo '"@nThe following sections are available:@n@n"'
      #
      # Pass one: Option summary
      #
        sed -n -e '
         /^@section Option Summary/,/^@menu/p
        ' $input_file | \
        sed -n -e '
         /^@[sub]*section \(.*\)/p
         /^@xref/d
         /^@smallexample/d
         /^@item/,/^@end/p
        ' | \
        sed -e '
         s/^@item[x]* *\(.*\)/@2{\1}/
         s/^@emph{\([^}]*\)}/@3{\1}/
        ' | sed -e "$common_transform" && \
      #
      # Pass two: Options Controlling the Kind of Output
      #
        sed -n -e '
         /^@section Options Controlling the Kind of Output/,/^@node/p
        ' $input_file | \
        sed -n -e '
         /^@[sub]*section/p
         /^@item[x]* *-/,/\./p
         /^@cindex *#/,/\./p
         s/^@cindex file name suffix/@t{File name suffixes}@n/p
         /^@item[x]* @var{file}.*/,/^ *$/p
        ' | \
        sed -e '
         s/@cindex *\(#.*\)/@t{\1}/g
         /@end table/d
        ' | sed -e "$common_transform" | sed -e "/^ *$/d" && \
      #
      # Pass three: All other options
      #
        sed -n -e '
         /^@section Shorthand Options/,/@node/p
         /^@section Options Controlling Fortran Dialect/,/@node/p
         /^@section Options to Request or Suppress Warnings/,/@node/p
         /^@section Options for Debugging Your Program or GNU Fortran/,/@node/p
         /^@section Options That Control Optimization/,/@node/p
         /^@section Options for Directory Search/,/@node/p
         /^@section Options for Code Generation Conventions/,/@node/p
        ' $input_file | \
        sed -n -e '
         /^@[sub]*section/p
         /^@item[x]* *-/{
          p
          : nextline
          n
          s/\. .*/\./
          s/\.$/\./
          p
          t
          b nextline
         }
        ' | sed -e "$common_transform"
        ;;
#-------------------------------------------------------------------------------
    gcj)
        echo '"@n@nThe following sections are available:@n@n"'

        java_common_transform_1='
            /^@chapter \(.*\)/p
            /^@[sub]*section \(.*\)/p
            /^@itemx* /,/\./p
            '
        java_common_transform_2='
            s/@chapter *\(.*\)/@1{\1}/g
            s/@section *\(.*\)/@2{\1}/g
            s/^@itemx* *\(.*\)/@t{\1}/
            '

      #
      # Pass one: Invoking gcj
      #
        sed -n -e '
         /^@chapter Invoking gcj$/,/^@node Compatibility/p
        ' $input_file | \
        sed -n -e "/@node Encodings/,/@node Warnings/d" \
               -e "$java_common_transform_1" | \
        sed -e "$java_common_transform_2" \
            -e "$common_transform"
      #
      # Pass two: Invoking gcjh, Invoking jv-scan, Invoking jcf-dump, Invoking gij
      #
        sed -n -e '
         /^@chapter Invoking gcjh/,/^@node Resources/p
        ' $input_file | \
        sed -n -e "$java_common_transform_1" | \
        sed -e "$java_common_transform_2" \
            -e "$common_transform"
        ;;
#-------------------------------------------------------------------------------
    cpp)
        echo '"@nThe following sections are available:@n@n"'
        sed -n -e '
         /^@section Options Controlling the Preprocessor/,/^@node/p
         /^@section Options for Directory Search/,/^@node/p
        ' $input_file | \
        sed -n -e '
         /^@[sub]*section/p
         /^@item[x]* *-/{
          p
          : nextline
          n
          s/\. .*/\./
          s/\.$/\./
          p
          t
          b nextline
         }
        ' | sed -e "$common_transform" && \
        sed -n -e '
         /^@section Environment Variables Affecting GNU CC/,/^@node/p
        ' $input_file | \
        sed -n -e '
         /^@[sub]*section/p
         /^@item.*/,/\./p
        ' | \
        sed -e '
         s/^@item[x]* *\(.*\)/@t{\1}/
        ' | sed -e "$common_transform"
        ;;
    *)
        echo "ERROR: $0 does not know how to filter $input_file"
        ;;
esac

echo '};'

exit 0;
