/* emxomfld.c --  Provide an ld-like interface to the IBM and M$ linkers
   Copyright (c) 1992-1998 Eberhard Mattes
   Copyright (c) 2003 InnoTek Systemberatung GmbH
   Copyright (c) 2003-2004 Knut St. Osmundsen

This file is part of emxomld.

emxomfld is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

emxomfld is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emxomfld; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#include <stdio.h>
#include <stdlib.h>
#include <alloca.h>
#include <errno.h>
#include <string.h>
#include <process.h>
#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/utime.h>
#include <sys/moddef.h>
#include <emx/getopt.h>
#include <alloca.h>
#include <sys/omflib.h>
#include "defs.h"
#include "weakld.h"

#define FALSE 0
#define TRUE  1

/* A member of a linked list of strings such as file names. */
typedef struct name_list
{
  struct name_list *next;
  char *name;
} name_list;


/* Whether or not linker tracing is enabled. */
static int opt_t;

/* The output file name, specified by the -o option.  */
static const char *output_fname = NULL;

/* The map file name (output), set by the -Zmap option. */
static const char *map_fname = NULL;
static int map_flag = FALSE;

/* The module definition file name (input), set if a file matching
   *.def is given on the command line. */
static const char *def_fname = NULL;

/* The binary resource file name (input), set if a file matching *.res
   is given on the command line. */
static const char *res_fname = NULL;

/* Base address of the excecutable file, specified by the -T
   option. */
static const char *base = NULL;

/* List of directories searched for libraries.  Each -L option adds a
   directory to this list.  add_libdirs is used to add another entry
   at the end of the list. */
static name_list *libdirs = NULL;
static name_list **add_libdirs = &libdirs;

/* List of object files.  Each file given on the command line which
   does not match *.def, *.lib and *.res is added to this list.
   add_obj_fnames is used to add another entry at the end of the
   list. */
static name_list *obj_fnames = NULL;
static name_list **add_obj_fnames = &obj_fnames;

/* List of library files.  Each file matching *.lib given on the
   command line is added to this list.  The -l option also adds an
   entry to this list. add_lib_fnames is used to add another entry at
   the end of the list. */
static name_list *lib_fnames = NULL;
static name_list **add_lib_fnames = &lib_fnames;

/* List of linker options.  Linker options can be specified with the
   -O option.  add_options is used to add another entry at the end of
   the list. */
static name_list *options = NULL;
static name_list **add_options = &options;

/* The command line passed to the linker. */
static char command_line[260];

/* The current length of the command line. */
static int line_len;

/* Non-zero if arguments go into the response file instead of
   command_line. */
static int response_flag;

/* The name of the response file. */
static char response_fname[L_tmpnam] = "";

/* The response file. */
static FILE *response_file = NULL;

/* Force the use of a response file from the next put_arg(). */
static int force_response_file = FALSE;

/* Weak alias object file. */
static char weakobj_fname[_MAX_PATH + 1];

/* Weak definition file (modified def_fname). */
static char weakdef_fname[_MAX_PATH + 1];

/* list of converted libraries and objects which must be removed upon exit. */
static name_list *conv_list = NULL;

/* Non-zero if debugging information is to be omitted.  Set by the -s
   and -S options. */
static int strip_symbols = FALSE;

/* Non-zero if emxomfld should create an .exe file and touch the
   output file.  Set by the -Zexe option. */
static int exe_flag = FALSE;

/* Non-zero when creating a dynamic link library.  Set by the -Zdll
   option. */
static int dll_flag = FALSE;

/* The stack size, specified by the -Zstack option, in Kbyte.  If the
   -Zstack option is not used, this variable is 0. */
static long stack_size = 0;

/* The name of the linker to use.  By default, ilink is used.  This
   can be overridden with the EMXOMFLD_LINKER environment variable. */
static const char *linker_name = "ilink.exe";

/* The type of linker to use. By default we assume it's VAC365 or later
   version of ilink. This can be overridden with the EMXOMFLD_TYPE env.
   var. using any of the value VAC365, VAC308 and LINK386. */
static const char *linker_type = "VAC365";

/* Non-zero if emxomfld should automatically convert a.out objects and
   archives to the OMF equivalents during linking. */
static int autoconvert_flag = 1;


/* Prototypes. */

static void usage (void) NORETURN2;
extern void *xmalloc (size_t n);
extern void *xrealloc (void *ptr, size_t n);
extern char *xstrdup (const char *s);
static void add_name_list (name_list ***add, const char *src);
static void conv_path (char *name);
static void put_arg (const char *src, int path, int quotable);
static void put_args (const name_list *list, int paths);
static void make_env (void);
static void cleanup (void);
static void arg_init (int rsp);
static void arg_end (void);
int main (int argc, char *argv[]);

/* To avoid including os2.h... */
#ifndef _System
#define _System
#endif
extern int _System DosCopy (char *, char *, int);

/* Tell the user how to run this program. */

static void usage (void)
{
  fprintf (stderr, "emxomfld " VERSION INNOTEK_VERSION "\n"
           "Copyright (c) 1992-1996 by Eberhard Mattes\n"
           "Copyright (c) 2003 by InnoTek Systemberatung GmbH\n\n");

  fprintf (stderr,
           "Usage: emxomfld -o <file> [-l <lib>] [-L <libdir>] [-T <base>] [-igtsS]\n"
           "                [-Zexe] [-Zdll] [-Zstack <size>] [-Zmap[=<map_file>]]\n"
           "                [-Z[no-]autoconv] [-O <option>] <file>...\n"
           "\n"
           "Options:\n"
           " -Zno-autoconv / -Zautoconv:\n"
           "    Turns off/on the automatic conversion of a.out libs and objs.\n"
           "    default: -Zautoconv\n"
           "\n"
           "Environment variables:\n"
           "  EMXOMFLD_TYPE:\n"
           "    The type of linker we're using. Values: VAC365, VAC308, LINK386.\n"
           "        VAC365   ilink.exe from IBM C and C++ Compilers for OS/2 v3.6 or later.\n"
           "        VAC308   ilink.exe from Visual Age for C++ v3.08.\n"
           "        LINK386  link386 form OS/2 install or DDK.\n"
           "  EMXOMFLD_LINKER:\n"
           "    Name of the linker to use and optionally extra parameters. Spaces in the\n"
           "    linker name or path is not supported. Quotes are not supported either.\n"
           "The default values for these two variables are VAC365 and ilink.exe.\n");
  exit (1);
}


/* Allocate N bytes of memory.  Quit on failure.  This function is
   used like malloc(), but we don't have to check the return value. */

void *xmalloc (size_t n)
{
  void *p;

  p = malloc (n);
  if (p == NULL && n)
    {
      fprintf (stderr, "emxomfld: out of memory\n");
      exit (2);
    }
  return p;
}


/* Change the allocation of PTR to N bytes.  Quit on failure.  This
   function is used like realloc(), but we don't have to check the
   return value. */

void *xrealloc (void *ptr, size_t n)
{
  void *p;

  p = realloc (ptr, n);
  if (p == NULL && n)
    {
      fprintf (stderr, "emxomfld: out of memory\n");
      exit (2);
    }
  return p;
}



/* Create a duplicate of the string S on the heap.  Quit on failure.
   This function is used like strdup(), but we don't have to check the
   return value. */

char *xstrdup (const char *s)
{
  char *p;
  int cch = strlen (s) + 1;

  p = xmalloc (cch);
  memcpy (p, s, cch);
  return p;
}


/* Add the name SRC to a list.  ADD is a pointer to the pointer of the
   end of the list.  We duplicate the string before adding it to the
   list. */

static void add_name_list (name_list ***add, const char *src)
{
  name_list *new;

  new = xmalloc (sizeof (name_list));
  new->next = NULL;
  new->name = xstrdup (src);
  *(*add) = new;
  (*add) = &new->next;
}


/* Replace forward slashes `/' in NAME with backslashes `\'.  The linkers
   requires backslashes in path names. */

static void conv_path (char *name)
{
  char *p;

  for (p = name; *p != 0; ++p)
    if (*p == '/')
      *p = '\\';
}


/* Add the argument SRC to the command line or to the response file.
   If PATH is non-zero, SRC is a path name and slashes are to be
   replaced by backslashes.  If the command line gets too long, a
   response file is created.
   If quotable is non-zero SRC will be quoted. This is required for
   supporting files names which includes '+' and spaces. */

static void put_arg (const char *src, int path, int quotable)
{
  int len, max_len;
  char *tmp;

  if (src != NULL)
    {

      /* Instead of a comma, we write a newline to the response
         file. */

      if (response_file != NULL && strcmp (src, ",") == 0)
        {
          fputc ('\n', response_file);
          line_len = 0;
          return;
        }

      /* Make a local copy of SRC to be able to modify it.  Then,
         translate forward slashes to backslashes if PATH is
         non-zero. */

      len = strlen (src);
      tmp = alloca (len + (quotable ? 3 : 1));
      if (path)
        {
          /* needs quoting? */
          if (quotable)
            {
              *tmp = '"';
              strcpy (tmp+1, src);
              tmp[++len] = '"';
              tmp[++len] = '\0';
            }
          else
            strcpy (tmp, src);
          conv_path (tmp);
        }
      else
        strcpy (tmp, src);


      /* Check if we've reached the maximum line length.  If the
         maximum command line length is exceeded, create a response
         file and write the remaining arguments to that file instead
         of putting them on the command line. */

      max_len = (response_file == NULL ? 110 : 52);
      if (   line_len + len + 1 > max_len
          || (force_response_file && !response_file))
        {

          /* If SRC is a single comma or a single semicolon, copy it
             to the output, ignoring the maximum line length.  This is
             to meet the IBM/M$ linker command syntax. The maximum line
             length allows for enough commas and semicolons added this
             way. */

          if ((*tmp == ',' || *tmp == ';') && tmp[1] == 0)
            {
              if (response_file == NULL)
                {
                  command_line[line_len+0] = *tmp;
                  command_line[line_len+1] = 0;
                }
              else
                fputc (*tmp, response_file);
              ++line_len;
              return;
            }

          /* If a response file has not yet been opened, open it. */

          if (response_file == NULL)
            {

              /* Complain if we are not allowed to use a response
                 file. */

              if (!response_flag)
                {
                  fprintf (stderr, "emxomfld: command line too long\n");
                  exit (2);
                }

              /* Choose a unique file name and create the response
                 file. */

              strcpy (response_fname, "ldXXXXXX");
              if (mktemp (response_fname) == NULL)
                {
                  perror ("emxomfld");
                  exit (2);
                }
              response_file = fopen (response_fname, "wt");
              if (response_file == NULL)
                {
                  perror ("emxomfld");
                  exit (2);
                }

              /* Add the name of the response file to the command
                 line. */

              command_line[line_len++] = ' ';
              command_line[line_len++] = '@';
              strcpy (command_line+line_len, response_fname);

              if (force_response_file)
                force_response_file = FALSE;
            }
          else if (line_len != 0)
            {

              /* Start a new line in the response file. */

              fputs (" +\n", response_file);
            }
          line_len = 0;
        }

      /* Separate command line arguments by spaces (unless the
         argument to be added starts with a delimiter. */

      if (line_len != 0 && *src != ',' && *src != ';')
        {
          if (response_file == NULL)
            command_line[line_len++] = ' ';
          else
            fputc (' ', response_file);
        }

      /* Finally write the argument to the command line or to the
         response file and adjust the current line length. */

      if (response_file == NULL)
        strcpy (command_line + line_len, tmp);
      else
        fputs (tmp, response_file);
      line_len += len;
    }
}


/* Put a list of arguments onto the command line or into the response
   file.  If PATHS is non-zero, the arguments are path names and
   slashes are to be replaced by backslashes. */

static void put_args (const name_list *list, int paths)
{
  while (list != NULL)
    {
      put_arg (list->name, paths, paths);
      list = list->next;
    }
}


/* Build the environment for the IBM/M$ Linkers: define the LIB
   environment variable. */

static void make_env (void)
{
  static char tmp[4096];
  char *p;
  int len;
  const name_list *list;

  /* Create a string for putenv(). */

  strcpy (tmp, "LIB=");
  len = strlen (tmp);

  /* Add the library directories to LIB, using `;' as separator. */

  for (list = libdirs; list != NULL; list = list->next)
    {
      if (list != libdirs && tmp[len-1] != ';')
        tmp[len++] = ';';
      strcpy (tmp+len, list->name);
      conv_path (tmp+len);
      len += strlen (list->name);
    }

  /* Append to the end the previous definition of LIB. */

  p = getenv ("LIB");
  if (p != NULL)
    {
      if (tmp[len-1] != ';')
        tmp[len++] = ';';
      strcpy (tmp+len, p);
    }


  /* Put the new value of LIB into the environment. */

  putenv (tmp);

  if (opt_t)
    fprintf(stderr, "*** %s\n", tmp);
}

/* Check if the given file is a valid OMF library. */

static int check_omf_library (FILE *f)
{
#pragma pack(1)
  struct
  {
    byte rec_type;
    word rec_len;
    dword dict_offset;
    word dict_blocks;
    byte flags;
  } libhdr;
#pragma pack()

  if (fread (&libhdr, 1, sizeof(libhdr), f) == sizeof (libhdr)
   && !fseek (f, 0, SEEK_SET)
   && libhdr.rec_type == LIBHDR
   && libhdr.flags <= 1 /* ASSUME only first bit is used... */
      )
    {
      int page_size = libhdr.rec_len + 3;
      if (page_size >= 16
       && page_size <= 32768
       && !(page_size & (page_size - 1)) != 0)
        return 1;
    }

  return 0;
}


/* Checks if the file handle is to an omf object or library file. */

static int check_omf (FILE *f)
{
#pragma pack(1)
  struct
  {
    byte rec_type;
    word rec_len;
  } omfhdr;
#pragma pack()
  if (fread (&omfhdr, 1, sizeof(omfhdr), f) == sizeof (omfhdr)
   && omfhdr.rec_type == THEADR
   && omfhdr.rec_len >= sizeof(omfhdr)
   && !fseek (f, 0, SEEK_SET)
      )
    {
      return 1;
    }

  return !fseek (f, 0, SEEK_SET) && check_omf_library (f);
}


/**
 * Generates an unique temporary file.
 *
 * @returns 0 on success.
 * @returns -1 on failure.
 * @param   pszFile     Where to put the filename.
 * @param   pszPrefix   Prefix.
 * @param   pszSuffix   Suffix.
 * @remark  The code is nicked from the weak linker.
 */
static int      make_tempfile(char *pszFile, const char *pszPrefix, const char *pszSuffix)
{
    struct stat     s;
    unsigned        c = 0;
    pid_t           pid = getpid();
    const char *    pszTmp = getenv("TMP");
    if (!pszTmp)    pszTmp = getenv("TMPDIR");
    if (!pszTmp)    pszTmp = getenv("TEMP");
    if (!pszTmp)    pszTmp = ".";

    do
    {
        struct timeval  tv = {0,0};
        if (c++ >= 200)
            return -1;
        gettimeofday(&tv, NULL);
        sprintf(pszFile, "%s\\%s%x%lx%d%lx%s", pszTmp, pszPrefix, pid, tv.tv_sec, c, tv.tv_usec, pszSuffix);
    } while (!stat(pszFile, &s));

    return 0;
}


/* Converts the file indicated by pf & pszFilename to omf closing f and
   updating filename with the new (temporary filename). A successful
   return from the function returns a pointe to the filestream for the
   converted file. On failure NULL is returned, f is closed adn filename
   is undefined. */

static FILE *aout_to_omf (FILE *pf, char *pszFilename, int fLibrary)
{
    int         rc;
    char *      pszNewFile;
    name_list  *pName;

    fclose(pf);                         /* don't need this! */

    if (opt_t)
      fprintf(stderr, "emxomfld: info: converting %s %s to OMF.\n",
              fLibrary ? "lib" : "obj", pszFilename);

    /*
     * Make temporary file.
     */
    pName = xmalloc(sizeof(name_list));
    pName->name = pszNewFile = xmalloc(_MAX_PATH);
    if (make_tempfile(pszNewFile, "ldconv", fLibrary ? ".lib" : ".obj"))
    {
        free(pszNewFile);
        return NULL;
    }

    /*
     * Do the conversion.
     */
    rc = spawnlp(P_WAIT, "emxomf.exe", "emxomf.exe",
                 "-o", pszNewFile, pszFilename, NULL);
    if (!rc)
    {
        /* open the file */
        pf = fopen(pszNewFile, "rb");
        if (pf)
        {
            /* add to auto delete list for removal on exit(). */
            pName->next = conv_list;
            conv_list = pName;

            strcpy(pszFilename, pszNewFile);

            if (opt_t)
              fprintf(stderr, "emxomfld: info: convert result '%s'.\n",
                      pszFilename);
            return pf;
        }
        remove(pszNewFile);
    }
    free(pszNewFile);
    free(pName);

    fprintf (stderr, "emxomfld: a.out to omf conversion failed for '%s'.\n",
             pszFilename);
    exit (2);
    return NULL;
}


/* Finds the full path of a OMF library or object file.

   This function may perform conversion from a.out to omf if that feature
   is enabled.

   Object files and libraries will be searched for using the LIB env.var
   (which is exported so that it is used by ILINK/LINK386 as well),
   if not found by their path directly. Note that finding object files in
   the LIB paths contradicts the standard Unix behaviour, so it is a kind
   of a hack, but this is the way ILINK and LINK386 work.

   If file has no extension, the .obj and .o extensions are tried in turn
   for object files, and .lib then .a are tried in turn for libraries.
   Additionaly, for libraries we check for a file with the 'lib' prefix,
   if one without the 'lib' prefix was not found. The '.o' and '.a'
   extensions are tried in the last case since otherwise we can find
   a 'something.a' when 'libsomething.lib' is available.

   Returns pointer to file stream and FULLNAME set to it's name on success.
   Returns NULL and FULLNAME a copy of NAME on failure.
   */

static FILE *find_objlib (char *fullname, const char *name, int is_library)
{
  static const char *exts [2][2] = { { ".obj", ".o" }, { ".lib", ".a" } };
  FILE *f;
  size_t pathlen, namelen = strlen (name);
  const char *libpath, *x;
  int i, has_ext = 0, has_path;

  /* Check if the file name has a path. If it has, we won't check the
     LIB directories, and won't check if it has a object file extension. */
  has_path = (strpbrk (name, ":/\\") != NULL);
  if (has_path)
    has_ext = 1;
  else
    {
      /* Check if file has a real extension. If it is a object file it should
         have either .o or .obj, and if it is a library it should have either
         the .a or .lib extension. Otherwise we don't consider it a extension. */
      x = strrchr (name, '.');
      if (x)
        for (i = 0; i < 2; i++)
          if (!stricmp (x, exts [is_library][i]))
            {
              has_ext = 1;
              break;
            }
    }

  /* First loop search file name as-is, then search the paths
     in the LIB environment variable. */
  libpath = getenv ("LIB");
  pathlen = 0;
  if (!has_path)
    {
      memcpy(fullname, ".\\", 2);
      pathlen = 2;
    }
  do
    {
      /* Step 0: check path+filename (+'.obj/.lib' if !has_ext)
       * Step 1: if is_library check path+'lib'+filename (+'.lib' if !has_ext)
       * Step 2: check path+filename (+'.o/.a' if !has_ext)
       * Step 3: if is_library check path+'lib'+filename (+'.a' if !has_ext)
       */
      for (i = 0; i < 4; i++)
        {
          char *cur = fullname + pathlen;

          if ((i & 1) && (!is_library || has_path))
            continue;

          if (i & 1)
            {
              memcpy (cur, "lib", 3);
              cur += 3;
            }

          /* Include final zero in the case we won't append the extension */
          memcpy (cur, name, namelen + 1);
          cur += namelen;

          if (!has_ext)
            {
              strcpy (cur, exts [is_library][i/2]);
              cur = strchr (cur, 0);
            }

          f = fopen (fullname, "rb");
          if (f)
            {
              if (autoconvert_flag && !check_omf (f))
                f = aout_to_omf (f, fullname, is_library);

              if (is_library && !check_omf_library (f) /* not sure if this makes sense */)
                fclose (f);
              else
                {
                  /* Replace forward slashes with backslashes (link386). */
                  while ((fullname = strchr (fullname, '/')) != NULL)
                    *fullname++ = '\\';
                  return f;
                }
            }
        }

      if (!libpath || has_path)
        break;

      /* Find the next LIB component.
         We're skipping empty components. */
      for (;;)
        {
          x = strchr (libpath, ';');
          if (!x) x = strchr (libpath, 0);
          pathlen = x - libpath;
          if (pathlen)
            {
              memcpy (fullname, libpath, pathlen);
              libpath = x + (*x == ';');
              /* Append last slash if it is not there */
              if (fullname [pathlen - 1] != '/'
               && fullname [pathlen - 1] != '\\')
                fullname [pathlen++] = '\\';
              break;
            }
          if (!*x)
            break;
          libpath = x + (*x == ';');
        }
    } while (pathlen);


  strcpy (fullname, name);
  return NULL;
}

/* Weak prelinking for Method 2 Weak support. */

static void weak_prelink ()
{
  int           rc = 0;
  name_list *   pOpt;
  PWLD          pwld;
  unsigned      fFlags = 0;

  /* look for ilinker options. */
  if (opt_t)
      fFlags |= WLDC_VERBOSE;
  if (!stricmp(linker_type, "LINK386"))
      fFlags |= WLDC_LINKER_LINK386;

  for (pOpt = options; pOpt; pOpt = pOpt->next)
    if (    !strnicmp(pOpt->name, "/NOE", 4)
        ||  !strnicmp(pOpt->name, "-NOE", 4))
        fFlags |= WLDC_NO_EXTENDED_DICTIONARY_SEARCH;
    else
    if (    !strnicmp(pOpt->name, "/INC", 4)
        ||  !strnicmp(pOpt->name, "-INC", 4))
        fFlags = fFlags;                /* Ignore for now. */
    else
    if (    !strnicmp(pOpt->name, "/IG", 3)
        ||  !strnicmp(pOpt->name, "-IG", 3))
        fFlags |= WLDC_CASE_INSENSITIVE;
    else
    if (    !strnicmp(pOpt->name, "/I", 2)
        ||  !strnicmp(pOpt->name, "-I", 2))
        fFlags = fFlags;                /* Ignore - require opt_t. */
    else
    if (    !strnicmp(pOpt->name, "/NOIN", 5)
        ||  !strnicmp(pOpt->name, "-NOIN", 5))
        fFlags &= ~WLDC_VERBOSE;
    else
    if (    !strnicmp(pOpt->name, "/NOI", 4)
        ||  !strnicmp(pOpt->name, "/NOI", 4))
        fFlags &= ~WLDC_CASE_INSENSITIVE;

  /* create the linker and to the linking. */
  if (opt_t)
    fprintf(stderr, "*** Invoking weak prelinker with flags %x.\n", fFlags);
  pwld = WLDCreate (fFlags);
  if (pwld)
    {
      name_list *   pcur;
      FILE         *phfile;
      char          szname[_MAX_PATH + 1];

      /* definition file if any */
      if (def_fname && def_fname[0])
        {
          phfile = fopen (def_fname, "r");
          rc = WLDAddDefFile (pwld, phfile, def_fname);
        }

      /* objects */
      for (pcur = obj_fnames; !rc && pcur; pcur = pcur->next)
        {
          phfile = find_objlib (szname, pcur->name, FALSE);
          rc = WLDAddObject (pwld, phfile, szname);
        }

      /* libraries */
      for (pcur = lib_fnames; !rc && pcur; pcur = pcur->next)
        {
          phfile = find_objlib (szname, pcur->name, TRUE);
          rc = WLDAddLibrary (pwld, phfile, szname);
          free(pcur->name);
          pcur->name = xstrdup(szname);
        }

      /* complete pass 1 */
      if (!rc)
        {
          rc = WLDPass1 (pwld);
          /* ignore unresolved externals for now. */
          if (rc == 42)
            {
              rc = 0;
              fprintf(stderr, "Ignoring unresolved externals reported from weak prelinker.\n");
            }
        }

      /* generate weak aliases. */
      if (!rc)
        rc = WLDGenerateWeakAliases (pwld, weakobj_fname, weakdef_fname);
      if (!rc && weakobj_fname[0])
        add_name_list (&add_obj_fnames, weakobj_fname);
      if (!rc && weakdef_fname[0])
        def_fname = weakdef_fname;

      /* cleanup the linker */
      WLDDestroy (pwld);

      /* last words */
      if (rc)
        fprintf (stderr, "emxomfld: weak prelinker failed. (rc=%d)\n", rc);
    }
  else
    {
      rc = 8;
      fprintf (stderr, "emxomfld: failed to create weak prelinker.\n");
    }

  /* die on error. */
  if (rc)
    exit(rc);

  /* verbose */
  if (opt_t)
    fprintf(stderr, "*** Weak prelinker done\n");
}


/* Start a new set of command line arguments.  If RSP is non-zero, we
   are allowed to use a response file. */

static void arg_init (int rsp)
{
  response_fname[0] = '\0';
  command_line[0] = '\0';
  line_len = 0;
  response_flag = rsp;
  force_response_file = FALSE;
}


/* Call this after adding all the command line arguments.  If a
   response file has been created, add a newline and close it. */

static void arg_end (void)
{
  if (response_file != NULL)
    {
      fputc ('\n', response_file);
      if (fflush (response_file) != 0 || fclose (response_file) != 0)
        {
          perror ("emxomfld");
          exit (2);
        }
      response_file = NULL;
    }
}


/* -t output. We dump the commandline and responsefile. */
static void  show_spawn(const char *pszwhat)
{
  if (!opt_t)
    return;
  fprintf(stderr, "*** Invoking %s\n %s\n", pszwhat, command_line);
  if (response_fname[0])
    { /* display the responsfile content. */
      char sz[4096];
      FILE *phfile = fopen(response_fname, "r");
      fprintf(stderr, "--- Response file %s:\n", response_fname);
      sz[0] = '\0';
      while (fgets(sz, sizeof(sz), phfile))
          fprintf(stderr, "%s", sz);
      fclose(phfile);
      if (sz[strlen(sz) - 1] != '\n')
          fprintf(stderr, "\n");
      fprintf(stderr, "--- End of Response File\n");
    }
}


/* Execute commandline and returns the result.
   pszwhat is used for opt_t trace information. */

static int emxomfld_spawn(char *pszcmd, const char *pszwhat)
{
  int       argi;
  char **   argv;
  char *    psz;
  int       rc;

  if (opt_t)
      show_spawn(pszwhat);

  /* construct spawnvp() argument array */
  argi = 0;
  argv = NULL;
  psz = pszcmd;
  while (psz && *psz)
    {
      char *psz2 = psz;

      /* skip blanks. */
      while (*psz2 == '\t' || *psz2 == ' ')
          psz2++;

      /* find end of argument taking in account in arg quoting. */
      while (*psz2 && *psz2 != '\t' && *psz2 != ' ')
        {
          if (*psz2 == '"' || *psz2 == '\'')
            {
              char chQuote = *psz2++;
              while (*psz2 && *psz2 != chQuote)
                psz2++;
            }
          psz2++;
        }

      /* terminate and set psz2 to point to next */
      if (*psz2)
        *psz2++ = '\0';

      /* add argument to argument vector. */
      if (!(argi % 32))
        argv = xrealloc(argv, sizeof(argv[0]) * (argi + 32 + 1));
      argv[argi++] = psz;

      /* next */
      psz = psz2;
    }
  argv[argi] = NULL;

  /* Spawn process. */
  rc = spawnvp(P_WAIT, argv[0], argv);
  if (opt_t)
    fprintf(stderr, "*** Return from %s is %d\n", pszwhat, rc);

  free(argv);
  return rc;
}


/* Cleanup by closing (if open) and deleting (if pressent) the
   response file.  This function is used with atexit(). */

static void cleanup (void)
{
  if (response_file != NULL)
    {
      fclose (response_file);
      response_file = NULL;
    }
  if (response_fname[0] != '\0')
    {
      remove (response_fname);
      response_fname[0] = '\0';
    }
  if (weakobj_fname[0] != '\0')
    {
      remove (weakobj_fname);
      weakobj_fname[0] = '\0';
    }
  if (weakdef_fname[0] != '\0')
    {
      remove (weakdef_fname);
      weakdef_fname[0] = '\0';
    }
  for (; conv_list; conv_list = conv_list->next)
      remove (conv_list->name);
}


/* Main function of emxomf.  Parse the command line and call the IBM/M$
   linker (and optionally RC). */

int main (int argc, char *argv[])
{
  int c, rc, files;
  const char *ext;
  char tmp[512], *t;
  char execname[512];
  name_list *pcur;

  /* Get options from response files (@filename) and wildcard (*.o) on the command. */

  _response (&argc, &argv);
  _wildcard (&argc, &argv);

  /* Close and delete the response file on exit. */

  atexit (cleanup);

  /* Prepare parsing of the command line. */

  files = 0;
  opterr = FALSE; optmode = GETOPT_KEEP;
  if (argc < 2)
    usage ();

  /* Parse the command line options and other arguments. */

  while ((c = getopt (argc, argv, "o:O:itl:vL:T:sSxXZ:")) != EOF)
    switch (c)
      {
      case 't':
      case 'i':                 /* Trace the linking process, sending /INFO to the IBM/M$ linker. */
        opt_t = TRUE;
        break;

      case 'l':                 /* Add library */
        add_name_list (&add_lib_fnames, optarg);
        break;

      case 'o':                 /* Set output file name */
        output_fname = optarg;
        break;

      case 'L':                 /* Add library directory */
        add_name_list (&add_libdirs, optarg);
        break;

      case 'T':                 /* Set base address */
        base = optarg;
        break;

      case 's':                 /* Strip all symbols */
      case 'S':                 /* Strip debugging symbols */
        strip_symbols = TRUE;
        break;

      case 'x':                 /* Discard all local symbols */
      case 'X':                 /* Discard local symbols starting with L */
        break;

      case 'v':                 /* For compatibility */
        break;

      case 'O':                 /* Specify Linker option */
        add_name_list (&add_options, optarg);
        break;

      case 'Z':                 /* -Zdll, -Zexe, -Zmap, and -Zstack */
        if (strcmp (optarg, "dll") == 0)
          dll_flag = TRUE;
        else if (strcmp (optarg, "exe") == 0)
          exe_flag = TRUE;
        else if (strcmp (optarg, "map") == 0)
          map_flag = TRUE;
        else if (strncmp (optarg, "map=", 4) == 0)
          {
            if (map_fname != NULL)
              {
                fprintf (stderr, "emxomfld: multiple map files files\n");
                usage ();
              }
            map_fname = optarg + 4;
            map_flag = TRUE;
          }
        else if (strcmp (optarg, "stack") == 0)
          {
            /* This makes assumptions on the internals of getopt(). */

            if (optind >= argc || argv[optind][0] == '-')
              usage ();
            errno = 0;
            stack_size = strtol (argv[optind], &t, 0);
            if (errno != 0 || *t != 0 || t == argv[optind])
              usage ();
            ++optind;
          }
        else if (strcmp (optarg, "autoconv") == 0)
          autoconvert_flag = 1;
        else if (strcmp (optarg, "no-autoconv") == 0)
          autoconvert_flag = 0;
        else
          {
            fprintf (stderr, "emxomfld: invalid option (%s)\n", optarg);
            usage ();
          }
        break;

      case 0:                   /* Non-option argument */

        /* Extract the extension to see what to do with this
           argument. */

        ext = _getext (optarg);

        if (ext == NULL)
          {
            /* GCC's temporary files don't have an extension.  Add a
               dot to the end of the name to prevent the linker from
               adding `.obj'. */

            sprintf (tmp, "%s.", optarg);
            add_name_list (&add_obj_fnames, tmp);
          }

        /* If it's a .def file, use it as module definition file
           (input). */

        else if (stricmp (ext, ".def") == 0)
          {
            if (def_fname != NULL)
              {
                fprintf (stderr,
                         "emxomfld: multiple module definition files\n");
                usage ();
              }
            def_fname = optarg;
          }

        /* If it's a .res file, use it as binary resource file
           (input). */

        else if (stricmp (ext, ".res") == 0)
          {
            if (res_fname != NULL)
              {
                fprintf (stderr,
                         "emxomfld: multiple binary resource files\n");
                usage ();
              }
            res_fname = optarg;
          }

        /* If it's a .lib file, use it as library file.  We also
           accept .a files for those who use OMF files disguised as
           a.out files (to simplify their make files). */

        else if (stricmp (ext, ".lib") == 0 || stricmp (ext, ".a") == 0)
          add_name_list (&add_lib_fnames, optarg);

        /* Otherwise, assume it's an object file. */

        else
          add_name_list (&add_obj_fnames, optarg);
        ++files;
        break;

      default:
        fprintf (stderr, "emxomfld: invalid option (%s)\n", argv[optind - 1]);
        usage ();
      }

  /* Set default value for output file. */

  if (output_fname == NULL)
    {
      fprintf (stderr,
               "emxomfld: no output file, creating $$$.exe or $$$.dll\n");
      output_fname = "$$$";
    }

  /* Check if there are any input files. */

  if (files == 0)
    {
      fprintf (stderr, "emxomfld: no input files\n");
      usage ();
    }

  /* Remove the output file if -Zexe is given. */

  if (exe_flag)
    remove (output_fname);

  /* If neither -Zmap nor -Zmap=file is used, pass "nul" to the linker in
     the map file field.  If -Zmap is used, construct the name of the
     .map file.  If -Zmap=file is used, use `file' as the name of the
     .map file. */

  if (!map_flag)
    map_fname = "nul";
  else if (map_fname == NULL)
    {
      t = xstrdup (output_fname);
      _remext (t);
      map_fname = t;
    }

  /* Build the environment for the linker. */

  make_env ();

  /* EMXOMFLD_TYPE contains VAC365, VAC308 or LINK386 if set. If non of these
     we assume VAC365.
     EMXOMFLD_LINKER contains the linker name and perhaps extra arguments. If
     not set we'll use the default linker, ilink.  */

  t = getenv ("EMXOMFLD_TYPE");
  if (    t
      &&  stricmp(t, "VAC365")
      &&  stricmp(t, "VAC308")
      &&  stricmp(t, "LINK386")
      )
    fprintf (stderr, "emxomfld: warning: '%s' is an invalid value for EMXOMFLD_TYPE.\n", t);
  else if (t)
    linker_type = t;

  t = getenv ("EMXOMFLD_LINKER");
  if (t)
    linker_name = t;
  if (opt_t)
    fprintf(stderr, "*** Linker     : %s\n"
                    "*** Linker type: %s\n", linker_name, linker_type);

  /* apply object & library hacks */
  for (pcur = obj_fnames, rc = 0; !rc && pcur; pcur = pcur->next)
    {
      char szname[_MAX_PATH + 1];
      FILE *phfile = find_objlib (szname, pcur->name, FALSE);
      if (!phfile)
        continue;
      free (pcur->name);
      pcur->name = xstrdup(szname);
      fclose(phfile);
    }

  for (pcur = lib_fnames, rc = 0; !rc && pcur; pcur = pcur->next)
    {
      char szname[_MAX_PATH + 1];
      FILE *phfile = find_objlib (szname, pcur->name, TRUE);
      if (!phfile)
        continue;
      free (pcur->name);
      pcur->name = xstrdup(szname);
      fclose(phfile);
    }


  /* Do the weak prelinking unless GCC_WEAKSYMS are set.
     Important that this is done after make_env(). */

  if (!getenv("GCC_WEAKSYMS"))
      weak_prelink ();

  /* Start building the linker command line.  We can use a response
     file if the command line gets too long. */

  arg_init (TRUE);

  /*
     For VAC365 and VAC308 the default options is:

     /NOFR[EEFORMAT]    Use /NOFREEFORMAT to allow a LINK386-compatible
                        command line syntax, in which different types of file
                        are grouped and separated by commas.

     /DBGPACK           If !strip_symbols then we'll add this option, which
                        will cause type tables to be merged into one global
                        table and so eliminating a lot of duplicate info.

     For VAC365 additional default option is:

     /STUB:<emxomfld-path>\os2stub.bin
                        Causes this MZ stub to be used when linking the
                        executables instead of the default on for the linker.

     For LINK386 the default options is:

     /BATCH             Run in batch mode (disable prompting, don't
                        echo response file)

     The default options for all linkers are:

     /NOLOGO            Don't display sign-on banner

     /NOEXTDICTIONARY   Don't use extended dictionary (redefining
                        library symbols is quite common)

     /NOIGNORECASE      Make symbols case-sensitive

     /PACKCODE          Group neighboring code segments (this is the
                        default unless the SEGMENTS module definition
                        statement is used for a segment of class
                        'CODE').  Not grouping neighboring code
                        segments would break sets

     For non DLLs targets:

     /BASE:0x10000      Base the executable an so removing extra fixups.

  */

  /* issue commandline */
  put_arg (linker_name, TRUE, FALSE);

  /* the next part depends on the linker type. */
  if (!stricmp (linker_type, "LINK386"))
      put_arg ("/bat", FALSE, FALSE);
  else /* vac3xx: */
    {
      put_arg ("/nofree", FALSE, FALSE);
      if (!strip_symbols)
        put_arg ("/db", FALSE, FALSE);
      if (map_flag)
        put_arg ("/map", FALSE, FALSE);
    }
  put_arg ("/nol", FALSE, FALSE);
  put_arg ("/noe", FALSE, FALSE);
  put_arg ("/noi", FALSE, FALSE);
  put_arg ("/packc", FALSE, FALSE);


  /* VAC365: check if we have os2stub.bin.
     We must to this after the above stuff else /nol might end up in the
     response file and we'll get the component output. */

  if (!stricmp (linker_type, "VAC365"))
    {
      struct stat s;
      /* gklayout show that the linker isn't capable of determining a
         decent value for this parameter. 32MB make gklayout link. */
      put_arg ("/ocache:0x02000000", FALSE, FALSE);

      _execname(&execname[0], sizeof(execname));
      strcpy(_getname(&execname[0]), "os2stub.bin");
      if (!stat (execname, &s))
        {
          sprintf (tmp, "/STUB:%s", &execname[0]);
          put_arg (tmp, FALSE, FALSE);
        }
    }

  /* Add the /INFORMATION option if the -i or -t option was given.  This is
     for debugging. */

  if (opt_t)
    put_arg ("/i", FALSE, FALSE);

  /* Add the /DEBUG option if the -s option was not given.  Without
     this, the linker throws away debugging information. */

  if (!strip_symbols)
    put_arg ("/de", FALSE, FALSE);

  /* Add the /BASE:n option to set the base address.  This specifies
     the preferred load address of object 1.  The base address being
     used is 0x10000 unless a DLL is generated or the -T option was
     given.  -Tno can be used to suppress the /BASE:n option. */

  if (base == NULL && !dll_flag)
    {
      struct _md *md;

      if (def_fname != NULL)
        {
          int token;
          md = _md_open (def_fname);
          if (md == NULL)
            {
              fprintf (stderr, "emxomfld: cannot open `%s'\n", def_fname);
              exit (2);
            }
          token = _md_next_token (md);
          if (token == _MD_LIBRARY || token == _MD_PHYSICAL || token == _MD_VIRTUAL)
            dll_flag = TRUE;
          _md_close (md);
        }
    }
  if (base == NULL && !dll_flag)
    base = "0x10000";
  if (base != NULL && stricmp (base, "no") != 0)
    {
      sprintf (tmp, "/bas:%s", base);
      put_arg (tmp, FALSE, FALSE);
    }

  /* Add the /STACK:n option if the -Zstack option was given. */

  if (stack_size != 0 && !dll_flag)
    {
      sprintf (tmp, "/st:0x%lx", stack_size * 1024);
      put_arg (tmp, FALSE, FALSE);
    }

  /* Add the linker options specified with -O. */

  put_args (options, FALSE);

  /* Put the object file names onto the command line. */

  force_response_file = TRUE;           /* link386 workaround. */
  put_args (obj_fnames, TRUE);
  put_arg (",", FALSE, FALSE);

  /* Put the output file name onto the command line. */

  put_arg (output_fname, TRUE, TRUE);
  put_arg (",", FALSE, FALSE);

  /* Put the map file name onto the command line. */

  put_arg (map_fname, TRUE, TRUE);
  put_arg (",", FALSE, FALSE);

  /* Put the library file names onto the command line. */

  put_args (lib_fnames, TRUE);
  put_arg (",", FALSE, FALSE);

  /* Put the name of the module definition file onto the command line. */

  put_arg (def_fname, TRUE, TRUE);
  put_arg (";", FALSE, FALSE);
  arg_end ();

  /* Call Linker and abort on failure. */

  rc = emxomfld_spawn (command_line, "Linker");
  if (rc == 4 && !strnicmp(linker_type, "VAC3", 4)) /* Ignore iLink warnings. */
      rc = 0;
  if (rc < 0)
    {
      perror (linker_name);
      exit (2);
    }

  /* Run RC if Linker completed successfully and a binary resource
     file was given on the command line. */

  if (rc == 0 && res_fname != NULL)
    {
      arg_init (TRUE);
      put_arg ("rc.exe", TRUE, FALSE);
      put_arg ("-n", FALSE, FALSE);
      put_arg (res_fname, TRUE, FALSE);
      put_arg (output_fname, TRUE, FALSE);
      arg_end ();
      rc = emxomfld_spawn (command_line, "Resource Linker");
      if (rc < 0)
        {
          perror ("emxomfld: rc");
          exit (2);
        }
    }

  /* If both Linker and RC completed successfully and the -Zexe option
     was given, touch the output file (without .exe) to keep `make'
     happy. */

  if (rc == 0 && exe_flag)
    {
      /* find target and source filenames. */
      t = xstrdup (output_fname);
      _remext (t);
      _execname(&execname[0], sizeof(execname));
      strcpy(_getname(&execname[0]), "ldstub.bin");

      /* Copy stub into file */
      if (opt_t)
          fprintf(stderr, "*** copy %s to %s (-Zexe)", execname, t);
      DosCopy(&execname[0], t, 4);

      /* Now touch it */
      if (utime(t, NULL))
        {
          perror ("emxomfld");
          exit (2);
        }
      free (t);
    }

  /* Return the return code of Linker or RC. */

  return rc;
}
