/*
   Compiler Driver Command-line Help facility
   Copyright (C) 1997 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   In other words, you are welcome to use, share and improve this program.
   You are forbidden to forbid anyone else to use, share and improve
   what you give them. Help stamp out software-hoarding!
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#ifndef HAVE_CONFIG_H
#include <io.h>
#endif

/* External variables */

/* Help text displayed when user specified no selector */
extern char *clh_default_text;
/* argv[0] */
extern char *programname;

/******************************************************************************

                          Terminal control routines

 Since terminal control is highly system dependent, clh library allows almost
 everything to be overriden from a host-dependent config file. The following
 macros can be redefined:

 CLH_GET_WIDTH
 CLH_GET_FILE_WIDTH
 CLH_SET_ATTR(flags, flagmask)
 CLH_FINISH

 A detailed description of each is below.

******************************************************************************/


#define CLH_ATTR_HEAD   0x0001	/* Header text */
#define CLH_ATTR_OPTN   0x0002	/* Option text */
#define CLH_ATTR_BOLD   0x0004	/* Bold text */
#define CLH_ATTR_ITAL   0x0008	/* Italic text */
#define CLH_ATTR_UNDL   0x0010	/* Underlined text */

/* Host configuration file can query the following definition */
/* to place additional code into the source code */
#define IN_CLH

#define CLH_DEFAULT_GET_WIDTH		term_getwidth ()
#define CLH_DEFAULT_GET_FILE_WIDTH	CLH_GET_WIDTH
#define CLH_DEFAULT_SET_ATTR(flags, flagmask) \
					term_setattr (flags, flagmask)
#define CLH_DEFAULT_FINISH		term_finish ()

/* Forward declaration of functions that use termcap */
/* so that host configuration file can use CLH_DEFAULT_XXX */
int term_getwidth (void);
int term_out (char *name);
int term_setattr (int flags, int flagmask);
void term_finish (void);

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/*
   This macro should return the width of the terminal (or window if windowed)
   and maybe perform other initialization. This macro is called only if
   isatty (1) returned >0.
 */
#ifndef CLH_GET_WIDTH
#define CLH_USE_TERMCAP
#define CLH_GET_WIDTH			CLH_DEFAULT_GET_WIDTH
#endif

/*
   This macro can be overriden to return the width of the text which will
   be output when stdout is redirected to something other than a terminal.
   Default value is CLH_GET_WIDTH
 */
#ifndef CLH_GET_FILE_WIDTH
#define CLH_USE_TERMCAP
#define CLH_GET_FILE_WIDTH		CLH_DEFAULT_GET_FILE_WIDTH
#endif

/*
   This should set the terminal into such mode that the following output will
   have appearance specified by CLH_ATTR_XXXX flags. flagmask contains a '1' in
   those positions that have been changed since last call to CLH_SET_ATTR.
   You can use it to reduce the volume of output to terminal. If a particular
   terminal cannot produce somewhat-looking characters (for example, most
   terminals cannot do italic characters) it can approximate the appearance
   mode with something other, for example with color. It can as well completely
   ignore some appearance modes. It should return the number of positions that
   cursor has been moved horizontally. Usually this is zero.
 */
#ifndef CLH_SET_ATTR
#define CLH_USE_TERMCAP
#define CLH_SET_ATTR(flags, flagmask)	CLH_DEFAULT_SET_ATTR(flags, flagmask)
#endif

/*
   This macro should reset terminal to its original state and free any
   previously-allocated memory.
 */
#ifndef CLH_FINISH
#define CLH_USE_TERMCAP
#define CLH_FINISH			CLH_DEFAULT_FINISH
#endif

#ifdef CLH_USE_TERMCAP

/*******************************************************************\
 *  Generic terminal control routines using TERMCAP library.       *
 *                                                                 *
 *  There should be no difference which TERMCAP library is used -- *
 *  there are no specific features used.                           *
\*******************************************************************/

static char termbuf[2048];

extern int tgetent (char *buffer, char *termtype);
extern int tgetnum (char *name);
extern int tgetflag (char *name);
extern char *tgetstr (char *name, char **area);

static int
term_getwidth ()
{
  int ret;
  char *term = getenv ("CLH_TERM");
  if (!term || (strcmp (term, "-") == 0))
    term = getenv ("TERM");

  if ((ret = tgetent (termbuf, term)) < 0)
    {
      if (term)
	  fprintf (stderr, "\n\nIncorrect CLH_TERM or TERM environment variable\n\n");
      else
	  fprintf (stderr, "\n\nThe CLH_TERM or TERM environment variable is not set\n\n");
      return 80;
    }
  if (ret == 0)
    termbuf[0] = 0;

  ret = tgetnum ("co");
  if (ret < 1)
    return 80;
  else
    return ret;
}

static int
term_out (char *name)
{
  char *buff = (char *) malloc (64);
  char *tmp = buff, *cap;
  int rc = 0;

  cap = tgetstr (name, &buff);
  if (cap)
    {
      write (1, cap, strlen (cap));
      rc = 1;
    }
  free (tmp);
  return rc;
}

static int
term_setattr (int flags, int flagmask)
{
  int fromscratch = 0;		/* Reset all attributes and restart from scratch */
  int width = 0;
  int cookiewidth = tgetnum ("sg");

  if (cookiewidth < 0)
    cookiewidth = 0;

  /* Attributes should come in reverse order by they importance, */
  /* since later attributes can overwrite previous */

  if (flagmask & CLH_ATTR_UNDL)
    if (flags & CLH_ATTR_UNDL)
      {
	if (term_out ("us"))
	  width += cookiewidth;
      }
    else
      {
	if (term_out ("ue"))
	  width += cookiewidth;
      }

  if (flagmask & CLH_ATTR_ITAL)
    if (flags & CLH_ATTR_ITAL)
      {
	if (term_out ("mb"))
	  width += cookiewidth;
      }
    else
      fromscratch = 1;

  if (flagmask & CLH_ATTR_BOLD)
    if (flags & CLH_ATTR_BOLD)
      {
	if (term_out ("md"))
	  width += cookiewidth;
      }
    else
      fromscratch = 1;

  if (flagmask & CLH_ATTR_OPTN)
    if (flags & CLH_ATTR_OPTN)
      {
	if (term_out ("mr"))
	  width += cookiewidth;
      }
    else
      fromscratch = 1;

  if (flagmask & CLH_ATTR_HEAD)
    if (flags & CLH_ATTR_HEAD)
      {
	if (term_out ("so"))
	  width += cookiewidth;
      }
    else
      {
	if (term_out ("se"))
	  width += cookiewidth;
      }

  if (fromscratch)
    {
      term_out ("me");
      if (flags)
	width += term_setattr (flags, flags);
    }
  return (width);
}

static void
term_finish ()
{
  term_out ("me");
}

#endif /* CLH_USE_TERMCAP */

/******************************************************************************

                         A simple parser for .txh files

 To produce command-line help you should provide a text in a simplified
 texi-like language. The following commands are accepted:

 @0, @1, @2 ... @9 {section name text}

  This identifies a section. The number after @ identifies its position within
  help tree. For example, a help tree can look like this:

  @0
    @1
    @1
      @2
    @1
      @2
        @3
        @3 <-
        @3
    @1
    @1
  @0

  Each one-level-deeper subsection within a given section is given an ordinal
  number which will be displayed at run time, and you can choose the path
  within command-line help tree by specifying a list of numbers separated by
  dots, for example --help=3,1,2 will show text from section number 3 at
  level 1, subsection 1 at level 2 and sub-subsection 2 at level 3. In the
  above example this section is marked with an arrow.
  For a more complex example, look at the end of this comment.

 @t{text}

  This defines a topic. The characters between curly brackets will
  (hopefully) be highlighted in some way, so the eye will easily find them.
  Help Topics can be selected directly with --help switch, for example
  program --help=--version will bring the help text on the "--version"
  topic. Text in brackets is prefixed and postfixed by newlines.

 @x{text}

  The text in brackets is a topic with same properties as @t
  except that it will not be selectable by --help=topic command.

 @y{text}

  Same as @x but the {text} will not be preceeded and followed by newlines.

 @b{text}

  The text in brackets will be (maybe) displayed in boldface.

 @i{text}

  The text in brackets will be (maybe) displayed italicised.

 @u{text}

  The text in brackets will be (maybe) displayed underlined.

 @n

  Produces a new line. By default, all words are re-formatted to terminal
  width, a new-line is automatically placed only between @0-@9, @o and @x
  commands.

 @z

  Inserts "programname" into text

 @@

  A literal `@'

 ----------------------------------------------------------------------------
  A simple example of a help tree:
 ----------------------------------------------------------------------------

  @0{Default help text}
    This text will be displayed when no parameters are given at all (for
    example, when user entered @b{program --help} or @b{program -h}). It
    should contain some general directions how to use the command-line help
    system, for example: Please look below and choose the number of the
    section you are interested in, then enter @b{program --help=?} where ?
    is the number of that section.

  @1{The section A of @b{command-line switches}}
    This section is so big, so big, that it has been split into subsections.
    Please look below and choose the ID of the subsection you like, then
    re-run the program with @b{--help=1?} switch, where ? is the ID of
    the subsection you like.

  @2{The section A-1}
    blah-blah-blah
  @2{The section A-2}
    blah-blah-blah
  @2{The section A-3}
    blah-blah-blah

  @1{The section B}
  @2{Section B-1}
    blah-blah-blah
    this section can be selected with --help=2,1
  @0
   This text will be displayed at the end of the level-zero text.
   For example, it can contain the copyright, distribution license and so on.

******************************************************************************/

#define TOKEN_EOF      -1
#define TOKEN_SPACE     0
#define TOKEN_WORD      1
#define TOKEN_COMMAND  32
#define TOKEN_END     128

/* The following structure encapsulates a command-line help class */
struct clhStat
{
/* The following is used for text input stream */
  const char *inp;		/* Input text pointer */
  int level;			/* Current level inside help tree */
  int path[10];			/* Help tree path selector */
  int pathlen;			/* Path length; if zero selector is used */
  const char *selector;		/* Option selector */
  int levcount[10];		/* Count sublevels here */
  int indent;			/* Left column to start at */
  int visible;			/* Display text? */
  int visiblesection;		/* Override 'visible' */
  char success;			/* Success indicator */

/* This stack is used for placing command arguments (i.e. @i{argument}) */
/* This provides support for up to TOKEN_STACK_SIZE nested commands */
#define TOKEN_STACK_SIZE 16
  const char *stack[TOKEN_STACK_SIZE];
  char *orgstack[TOKEN_STACK_SIZE];
  char stackcmd[TOKEN_STACK_SIZE];	/* Command that has this argument */
  int stackp;			/* Stack top pointer */

/* The following is used for output */
  int attrsup;			/* Has support for attributes? */
  int column;			/* Current cursor column */
  int width;			/* Terminal width */
  int curattr;			/* Text attributes terminal is set to */
};

/* Fetch a token from input stream and return:
 *  ?             -- if next token is @?
 *  TOKEN_END + ? -- if argument for command ? finished
 *  TOKEN_SPACE   -- if next word consists of spaces
 *  TOKEN_WORD    -- next word consists of non-spaces
 *  TOKEN_EOF     -- when input stream exhausted
 */
static int
gettoken (struct clhStat *stat, char *word, int wordlen)
{
  const char *in;		/* input pointer */
  char c;			/* next character read */
  int ret = TOKEN_EOF;		/* return value */
  int wordpos = 0;		/* next position in "word" */

  if (stat->stackp)
    in = stat->stack[stat->stackp - 1];
  else
    in = stat->inp;

  /* If next token is a command, process it */
  if (in [0] == '@')
    {
      in++;
      c = *in;
      in++;

      /* Command has arguments? */
      if (*in == '{')
	{
	  const char *tmp = in + sizeof (char);
	  int deep = 1;

	  /* Eat its argument */
	  while (deep && *tmp)
	    {
	      if (*tmp == '{')
		deep++;
	      if (*tmp == '}')
		deep--;
	      tmp++;
	    }
	  /* Adjust input pointer past end of argument */
	  if (stat->stackp)
	    stat->stack[stat->stackp - 1] = tmp;
	  else
	    stat->inp = tmp;
	  if (stat->stackp >= TOKEN_STACK_SIZE)
          {
            fprintf (stderr, "\nCLH FATAL ERROR: too many embedded tokens; token stack exceeded!\n");
	    abort ();		/* Avoid assert () */
          }

	  /* Now allocate a new slot in stack and put the argument there */
	  stat->orgstack[stat->stackp] = malloc (tmp - in - 1);
	  memcpy (stat->orgstack[stat->stackp], in + 1, tmp - in - 2);
	  stat->orgstack[stat->stackp][tmp - in - 2] = 0;
	  /* Adjust input pointer */
	  in = stat->orgstack[stat->stackp];
	  /* Remember the command which has the argument */
	  stat->stackcmd[stat->stackp] = c;
	  stat->stackp++;
	}
      ret = c;
      goto tokexit;
    }

  /* If this is the end of a particular input stream, */
  /* but this is not the last one */
  if ((*in == 0) && (stat->stackp))
    {
      /* Return TOKEN_END+command */
      ret = TOKEN_END + stat->stackcmd[--stat->stackp];
      /* Delete the argument */
      free (stat->orgstack[stat->stackp]);
      /* Adjust input pointer */
      if (stat->stackp)
	in = stat->stack[stat->stackp - 1];
      else
	in = stat->inp;
      goto tokexit;
    }

  /* If next word begins with a space, eat all */
  /* successive spaces and spit them out in one go */
  if ((*in == ' ') || (*in == '\n'))
    {
      do
	{
	  c = *in;
	  if (c == '\n')
	    c = ' ';
	  word[wordpos++] = c;
	  in++;
	}
      while ((*in == ' ') || (*in == '\n'));
      ret = TOKEN_SPACE;
      goto tokexit;
    }

  /* While we can fetch characters, put them into "word" */
  while ((*in) && (wordpos < wordlen))
    {
      ret = TOKEN_WORD;
      if (*in == '\n')
	c = ' ';
      else
	c = *in;
      /* If the word finished, get out */
      if ((c == ' ') || (c == '@'))
	goto tokexit;
      /* Go forth */
      in++;
      word[wordpos++] = c;
    }

tokexit:
  /* Keep the input pointer */
  if (stat->stackp)
    stat->stack[stat->stackp - 1] = in;
  else
    stat->inp = in;
  /* Finish the "word" with a zero */
  word[wordpos] = 0;
  return ret;
}

/* This text will be displayed in case of insuccess */
static const char *clh_notfound =
{
  "@0{--[ @i{help topic missing} ]--}"
  "Sorry, the help database does not contain more information on topic "
  "you selected"
};

/* Output a word taking care to output a newline */
/* if it doesnt fit in the current line */
/* If current path does is not along the path user wants, exit */
/* Return 1 if the word has been output */
static int
clh_output (struct clhStat *stat, const char *word)
{
  int i, spaces;
  int wordlen = strlen (word);
  int curattr = 0;

  if (!stat->visiblesection)	/* Section should be displayed? */
    if (!stat->visible)		/* Topic should be displayed? */
      return (0);

  /* Compute attributes for the text we are about to output */
  for (i = 0; i < stat->stackp; i++)
    switch (stat->stackcmd[i])
      {
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	curattr |= CLH_ATTR_HEAD;
	break;
      case 't':
      case 'x':
      case 'y':
	curattr |= CLH_ATTR_OPTN;
	break;
      case 'b':
	curattr |= CLH_ATTR_BOLD;
	break;
      case 'i':
	curattr |= CLH_ATTR_ITAL;
	break;
      case 'u':
	curattr |= CLH_ATTR_UNDL;
	break;
      }

  /* Compute indentation (spaces skipped from left margin)
     for the text we should output */
  if (stat->level == 0)
    stat->indent = 1;
  else
    stat->indent = stat->level + ((curattr & CLH_ATTR_HEAD) ? 0 :
				  ((curattr & CLH_ATTR_OPTN) ? 1 : 2));

  /* If user selected a path (i.e. 1.2.3.x), check it */
  if (stat->pathlen)
    {
      /* If we are more than one level deeper than user requested,
         or we are simply deeper and we're not displaying a heading,
         exit */
      if ((stat->pathlen <= stat->level)
      && (((curattr & CLH_ATTR_HEAD) == 0) || (stat->pathlen < stat->level)))
	return (0);

      /* If this is the section user wants, signal success */
      stat->success |= (stat->level + 1 >= stat->pathlen);
    }

  /* If we should output a newline or next word
     crosses the right margin, output a newline */
  if ((*word == '\n') || (stat->column + wordlen >= stat->width))
    {
      /* Reset attributes to nothing before newline */
      if (stat->attrsup && stat->curattr)
	{
	  CLH_SET_ATTR (0, stat->curattr);
	  stat->curattr = 0;
	}
      write (1, "\n", 1);
      stat->column = 1;
      /* If we're required to output a newline, we're done */
      if (*word == '\n')
	return (0);
    }

  /* If current column is less than needed indentation, fill with spaces */
  if (stat->column < stat->indent)
    {
      /* Check first if we`re requested to output a bunch of spaces */
      for (i = 0, spaces = 1; i < wordlen; i++)
	if (word[i] != ' ')
	  {
	    spaces--;
	    break;
	  }

      /* If so, exit (skip spaces at the left margin) */
      if (spaces)
	return (0);

      /* Reset attributes to nothing before newline */
      if (stat->attrsup && stat->curattr)
	{
	  stat->column += CLH_SET_ATTR (0, stat->curattr);
	  stat->curattr = 0;
	}
      /* If we're still unaligned, align */
      if (stat->indent - stat->column > 0)
	{
	  /* Since no more than 10 (0..9) nested levels are allowed, we'll
	     never have a indentation greater than 10 spaces (see
	     indentation computing formula above) */
	  write (1, "           ", stat->indent - stat->column);
	  stat->column = stat->indent;
	}
    }

  /* If the attributes for the text we should output differs from */
  /* current attributes, set new attributes */
  if ((i = curattr ^ stat->curattr))
    {
      if (stat->attrsup)
	stat->column += CLH_SET_ATTR (curattr, i);
      stat->curattr = curattr;
    }

  /* And finally, output the word */
  write (1, word, wordlen);
  stat->column += wordlen;
  return (1);
}

/* Compare regions of strings, ignoring case if str2 does not start */
/* with a '-', and return 0 if are equal, != 0 otherwise */
static int
clh_memcmp (const char *str1, const char *str2, int len)
{
  int i;

  if (*str2 == '-')
    return memcmp (str1, str2, len);
  else
    for (i = 0; i < len; i++)
      if (tolower (str1[i]) != tolower (str2[i]))
	return 1;
  return 0;
}

/* This routine is called when section changes
   to determine whenever the section should be displayed or not */
static int
clh_calcvisible (struct clhStat *stat, const char *section)
{
  /* User specified a help path? */
  if (stat->pathlen)
    {
      int i, pl;

      /* Do not display level zero when at least
         one-step step was specified */
      if ((stat->pathlen > 1) && (stat->level == 0))
	return 0;

      /* Check if we're on the right path */
      pl = stat->level + 1;
      if (pl > stat->pathlen)
	pl = stat->pathlen;
      for (i = 0; i < pl; i++)
	if (stat->path[i] != stat->levcount[i])
	  return 0;
    }
  else
    /* User specified a help topic */
    {
      int len = strlen (section);
      int sellen = strlen (stat->selector);

      /* Check if selector is a substring of section name */
      if ((stat->level == 0)
	  || (len < sellen)
	  || (clh_memcmp (section, stat->selector, sellen)))
	return 0;

      /* Signal success */
      stat->success = 1;
    }

  /* Display it */
  return 1;
}

/* This is the main (and the only) interface routine for CLH library
 * It accepts a pointer to help text database, and a pointer to
 * topic selector. Topic selector can be either of the form
 * "x.y.z{...}" where x,y,z etc are numbers, otherwise selector
 * is considered to be the starting portion of a help topic.
 * If topic does not begin with a '-', string are compared
 * case-insensitive, otherwise the comparison is case-sensitive.
 * Function returns 0 if topic has been found and -1 if topic
 * has not been found in database.
 */
int
clh_display (const char *text, const char *selector)
{
  char word[256];
  int token;
  struct clhStat stat;
  char *term;
  int emptytopic = 0;

  /* If no selector given, select default help text */
  if (!selector || !*selector)
    text = clh_default_text;

  /* If selector is equal to '*', user wants to see help index */
  if (selector && (selector[0] == '*') && (selector[1] == 0))
    selector = "";

  /* Construct a clhStat object */
  memset (&stat, 0, sizeof (stat));
  stat.attrsup = isatty (1);
  if (stat.attrsup < 0)		/* No attributes if error */
    stat.attrsup = 0;
  if (stat.attrsup)
    stat.width = CLH_GET_WIDTH;
  else
    stat.width = CLH_GET_FILE_WIDTH;
  stat.column = 1;
  stat.inp = text;
  stat.stackp = 0;
  stat.level = 0;
  stat.curattr = 0;
  stat.pathlen = 1;		/* Assume we have to display only 'default' section */
  stat.path[0] = 1;		/* Level zero always has only one section */
  stat.indent = 1;		/* Indent to 1st column */
  stat.visible = 0;		/* Topic is invisible */
  stat.visiblesection = 0;	/* Section is invisible */

  /* Check CLH_TERM variable, and if it is "-", disable attributes */
  term = getenv ("CLH_TERM");
  if (term)
    {
      if (strcmp (term, "-") == 0)
	stat.attrsup = 0;
    }

  if (stat.attrsup)		/* Set text attributes to nothing */
    stat.column += CLH_SET_ATTR (stat.curattr, 0);

  if (selector)
    {
      int i;
      const char *tmp = selector;

      /* See if selector is a series of numbers separated by dots */
      for (i = 1; (i <= 9) && (*tmp); i++)
	{
	  int val = 0;
	  while ((*tmp >= '0') && (*tmp <= '9'))
	    {
	      val = val * 10 + (*tmp - '0');
	      tmp++;
	    }
	  stat.pathlen = i + 1;
	  stat.path[i] = val;
	  while (*tmp == '.')
	    tmp++;
	}
      if (*tmp)			/* If there are non-numbers, consider selector a topic */
	stat.pathlen = 0;
    }

  if (stat.pathlen)
    stat.selector = NULL;
  else
    stat.selector = selector;

  /* Loop through all tokens in input text */
  while ((token = gettoken (&stat, word, sizeof (word))) != TOKEN_EOF)
    {
      if (token >= TOKEN_COMMAND)	/* Got a command? */
	{
	  char finished = token > TOKEN_END;
	  if (finished)
	    token -= TOKEN_END;

	  /* Which one? */
	  switch (token)
	    {
	    case '0':
	    case '1':
	    case '2':
	    case '3':
	    case '4':
	    case '5':
	    case '6':
	    case '7':
	    case '8':
	    case '9':
	      /* Section separator - start or end of argument */
	      {
		int i;

		stat.level = token - '0';
		stat.levcount[stat.level + 1] = 0;

		/* If token started, see if section is visible */
		if (!finished)
		  {
		    stat.levcount[stat.level]++;
		    stat.visible = 0;	/* Topic is invisible */
		    stat.visiblesection = clh_calcvisible (&stat,
							   (stat.stackp && (stat.stackcmd[stat.stackp - 1] == token))
					       ? stat.stack[stat.stackp - 1]
							   : "");
		  }
		/* If cursor is past indention column, newline */
		if (stat.column > stat.indent)
		  clh_output (&stat, "\n");

		/* If argument started and level is greater than zero,
		   print section number */
		if (!finished && stat.level)
		  {
		    for (i = 1; i <= stat.level; i++)
		      {
			snprintf (word, sizeof (word), "%d.", stat.levcount[i]);
			clh_output (&stat, word);
		      }
		    clh_output (&stat, " ");
		  }
		break;
	      }
	    case 't':		/* Help topic - argument started or finished */
	      if (!finished && (stat.pathlen == 0) && (!emptytopic || !stat.visible))
		{
		  /* Check if topic is visible */
		  stat.visible = clh_calcvisible (&stat,
		  (stat.stackp && (stat.stackcmd[stat.stackp - 1] == token))
					       ? stat.stack[stat.stackp - 1]
						  : "");
		}
	      emptytopic = 1;
	      /* continue on next case */
	    case 'x':		/* Unselectable help topic */
	      /* If cursor is past indention column, newline */
	      if (stat.column > stat.indent)
		clh_output (&stat, "\n");
	      break;
	    case 'y':		/* Unselectable help topic not followed by \n */
	      break;
	    case 'n':		/* Newline */
	      clh_output (&stat, "\n");
	      break;
	    case 'b':		/* Bold */
	      break;
	    case 'i':		/* Italic */
	      break;
	    case 'u':		/* Underline */
	      break;
	    case 'z':
	      clh_output (&stat, programname);
	      break;
            case '@':
	      clh_output (&stat, "@");
	      break;
	    default:
              {
		fprintf (stderr, "\nCLH FATAL ERROR: Unexpected token @%c\n", token);
		abort ();	/* Incorrect help text */
              }
	    }
	}
      else
	{
	  /* Display next word of text if it's worth it */
	  if (clh_output (&stat, word))
	    emptytopic = 0;
	}
    }

  /* Finish terminal if it supports attributes :-) */
  if (isatty (1))
    CLH_FINISH;

  /* Output a newline */
  if (stat.column > 1)
    write (1, "\n", 1);

  /* If we didn't succeeded to find the topic user asked for,
     display 'not found' message and return -1 */
  if ((!stat.success) && ((stat.pathlen > 1) || (stat.selector)))
    {
      clh_display (clh_notfound, "*");
      return -1;
    }
  return 0;
}

#ifdef MAIN

/* The following is used only for stand-alone testing */

#include "getopt.h"

static char *program_name;
static char *program_version = "0.0";

/* Command-line help text */
char *gcc_clh_text =
{
  "@0"
  "@u{Welcome to the GNU Compiler Suite Command-Line Help System!}@n"
  "The following sections are available:@n"
#include "clh.txh"
};

static struct option long_options[] =
{
  {"help", optional_argument, 0, 'h'},
  {"version", no_argument, 0, 'v'},
  {0, no_argument, 0, 0}
};

int
main (int argc, char *argv[])
{
  int c;

#if defined (__EMX__)
  program_name = _getname (argv[0]);
#else
  program_name = argv[0];
#endif /* __EMX__ */

  while ((c = getopt_long (argc, argv, "h::v", long_options, (int *) 0)) != EOF)
    switch (c)
      {
      case '?':
	break;
      case 'h':
	exit (clh_display (gcc_clh_text, optarg));
      case 'v':
	printf ("%s version %s\n", program_name, program_version);
	exit (0);
      }

  return (0);
}

#endif /* MAIN */
