/* $Id: bsdselect.c 886 2003-12-08 05:45:15Z bird $ */
/** @file
 *
 * bsdselect().
 *
 * Copyright (c) 2003 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of Innotek LIBC.
 *
 * Innotek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Innotek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Innotek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#include <errno.h>
#include <sys/socket.h>
#include <sys/fcntl.h>
#include <emx/io.h>
#include "socket.h"


/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
#ifdef TCPV40HDRS

#define MY_FD_SET(fd, set)      FD_SET(fd, set)
#define MY_FD_ISSET(fd, set)    FD_ISSET(fd, set)
#define MY_FD_ZERO(set, cb)     memset(set, 0, cb);
#define my_fd_set               fd_set

#else

#define MY_FD_SET(fd, set)      V5_FD_SET(fd, set)
#define	MY_FD_ISSET(fd, set)    V5_FD_ISSET(fd, set)
#define MY_FD_ZERO(set, cb)     memset(set, 0, cb);
#define my_fd_set               v5_fd_set

#endif


/**
 * Calculate the size required for the converted set structure.
 * @returns number of bytes.
 * @param   c       Number of file descriptors specified in the call.
 * @param   pFrom   The select input fd_set structure.
 */
static inline int calcsize(int c, const struct fd_set *pFrom)
{
    int cbRet;
#ifdef TCPV40HDRS
    cbRet = (c + 7) / 8;
#else
    int i;
    for (cbRet = sizeof(u_short), i = 0; i < c; i++)
        if (FD_ISSET(i, pFrom))
            cbRet++;
#endif
    if (cbRet < sizeof(struct my_fd_set))
        return sizeof(struct my_fd_set);
    return cbRet;
}

/** Converts the LIBC fd_set strcture pointed to by pFrom with it's LIBC socket handles,
 * to the fd_set strcuture used by the OS/2 tcpip and the OS/2 socket handles.
 * @returns 0 on success.
 * @returns -1 on failure, both errnos set.
 * @param   c       Number of file descriptors specified in the call.
 * @param   cb      Size of pTo. (used for zeroing it)
 * @param   pFrom   The select input fd_set structure.
 * @param   pTo     The structure we present to OS/2 TCPIP.
 *                  This will be initialized.
 */
static inline int convert(int c, int cb, const struct fd_set *pFrom, struct my_fd_set *pTo)
{
    int i;
    MY_FD_ZERO(pTo, cb)
    for (i = 0; i < c; i++)
    {
        if (FD_ISSET(i, pFrom))
        {
            PLIBCSOCKETFH   pFHSocket = __libsocket_FH(i);
            if (!pFHSocket)
                return -1;
            MY_FD_SET(pFHSocket->iSocket, pTo);
        }
    }
    return 0;
}

/**
 * Updates the pTo structure with the fds marked ready in pFrom.
 *
 * @param   c       Number of file descriptors specified in the call.
 * @param   pFrom   The structure returned from OS/2 TCPIP select.
 * @param   pTo     The structure passed in to select which have to
 *                  be updated for the return.
 */
static inline void update(int c, const struct my_fd_set *pFrom, struct fd_set *pTo)
{
    int i;
    for (i = 0; i < c; i++)
    {
        if (FD_ISSET(i, pTo))
        {
            PLIBCSOCKETFH   pFHSocket = __libsocket_FH(i);
            if (pFHSocket)
            {
                if (!MY_FD_ISSET(pFHSocket->iSocket, pFrom))
                    FD_CLR(i, pTo);
            }
        }
    }
}



int bsdselect(int nfds, struct fd_set *readfds, struct fd_set *writefds, struct fd_set *exceptfds, struct timeval *tv)
{
#ifdef TCPV40HDRS
    struct fd_set      *pRead;
    struct fd_set      *pWrite;
    struct fd_set      *pExcept;
#else
    struct v5_fd_set   *pRead;
    struct v5_fd_set   *pWrite;
    struct v5_fd_set   *pExcept;
#endif
    int                 cb;
    int                 rc;

    /*
     * Allocate new bitmaps.
     */
    pRead = NULL;
    if (readfds)
    {
        cb = calcsize(nfds, readfds);
        pRead = alloca(cb);
        if (!pRead)
        {
            __libsocket_setErrno(ENOMEM);
            return -1;
        }
        if (convert(nfds, cb, readfds, pRead))
            return -1;
    }

    pWrite = NULL;
    if (writefds)
    {
        cb = calcsize(nfds, writefds);
        pWrite = alloca(cb);
        if (!pWrite)
        {
            __libsocket_setErrno(ENOMEM);
            return -1;
        }
        if (convert(nfds, cb, writefds, pWrite))
            return -1;
    }

    pExcept = NULL;
    if (exceptfds)
    {
        cb = calcsize(nfds, exceptfds);
        pExcept = alloca(cb);
        if (!pExcept)
        {
            __libsocket_setErrno(ENOMEM);
            return -1;
        }
        if (convert(nfds, cb, exceptfds, pExcept))
            return -1;
    }

    /*
     * Do the call.
     */
    rc = __libsocket_bsdselect(nfds, pRead, pWrite, pExcept, tv);
    if (rc < 0)
    {
        __libsocket_setLibcErrno();
        return rc;
    }

    /*
     * Timeout?
     *  Just clear the bitmaps and return.
     */
    if (rc == 0)
    {
        cb = (nfds + 7) / 8;
        if (readfds)
            memset(readfds, 0, cb);
        if (writefds)
            memset(writefds, 0, cb);
        if (exceptfds)
            memset(exceptfds, 0, cb);
        return 0;
    }

    /*
     * Convert the bitmaps.
     */
    if (readfds)
        update(nfds, pRead, readfds);
    if (writefds)
        update(nfds, pWrite, writefds);
    if (exceptfds)
        update(nfds, pExcept, exceptfds);

    return rc;
}



