/* $Id: heaphigh.c 826 2003-10-10 23:25:48Z bird $ */
/** @file
 *
 * LIBC Base Functions for high memory heap.
 * Note. High memory heap does not mess around with sbrk().
 *
 * Copyright (c) 2003 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with This program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_DOSMEMMGR
#define INCL_ERRORS
#include <os2emx.h>
#include <string.h>
#include <emx/umalloc.h>
#include "syscalls.h"


/*******************************************************************************
*   Structures and Typedefs                                                    *
*******************************************************************************/
/** One chunk of memory
 * This structure is located at the very start of the chunk.
 */
typedef struct _HighChunk
{
    /** Magic / padding the struct to 16 bytes. */
    char                szMagic[8];
    /** Pointer to the next one. */
    struct _HighChunk  *pNext;
    /** Size of the chunk. */
    size_t              cb;
} HIGHCHUNK, *PHIGHCHUNK;


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/** List of chuncks. Protected by _sys_gmtxHimem. */
static PHIGHCHUNK   pHimemHead;



/**
 * Heap callback function for allocating high memory.
 *
 * @returns Pointer to the allocated memory on success.
 * @returns NULL on failure.
 * @param   Heap        Heap in question.
 * @param   pcb         Input the requested block size.
 *                      Output the amount of memory actually allocated.
 * @param   pfClean     On output this will tell the call if the memory
 *                      pointed to by the returned address is clean
 *                      (i.e zeroed) or not.
 */
void *__libc_HimemDefaultAlloc(Heap_t Heap, size_t *pcb, int *pfClean)
{
    size_t      cbAlloc;
    int         rc;
    PHIGHCHUNK  pChunk;

    /*
     * Allocate a (rather big) memory block, there is generally speaking
     * more than enough high memory to allocate from. So, we'll align the
     * chunks on a rather high limit, 4MB to save kernel calls and list
     * elements.
     */
    cbAlloc = *pcb + sizeof(HIGHCHUNK);
    cbAlloc = (cbAlloc + (4*1024*1024 - 1)) & ~(4*1024*1024 - 1);

    rc = DosAllocMem((PPVOID)&pChunk, cbAlloc, PAG_READ | PAG_WRITE | PAG_COMMIT | OBJ_ANY);
    if (rc == ERROR_NOT_ENOUGH_MEMORY)
    {   /*
         * Retry with page alignment instead of 4MB.
         */
        cbAlloc = *pcb + sizeof(HIGHCHUNK);
        cbAlloc = (cbAlloc + (0x1000 - 1)) & ~(0x1000 - 1);
        rc = DosAllocMem((PPVOID)&pChunk, cbAlloc, PAG_READ | PAG_WRITE | PAG_COMMIT | OBJ_ANY);
    }

    if (rc)
        return NULL;

    /*
     * We've got a block. Now it's gotta go into the list.
     */
    memcpy(pChunk->szMagic, "HiChunk", sizeof(pChunk->szMagic));
    pChunk->cb = cbAlloc;
    if (_fmutex_request(&_sys_gmtxHimem, _FMR_IGNINT) != 0)
    {
        DosFreeMem(pChunk);
        return NULL;
    }
    pChunk->pNext = pHimemHead;
    pHimemHead = pChunk;
    _fmutex_release(&_sys_gmtxHimem);


    /*
     * Set the output values and return.
     */
    *pfClean = _BLOCK_CLEAN;
    *pcb = cbAlloc - sizeof(HIGHCHUNK);
    return &pChunk[1];
}


/**
 * Heap callback function for releaseing high memory.
 *
 * @param   Heap    Heap in question.
 * @param   pv      Pointer to block.
 * @param   cb      Size of block.
 */
void __libc_HimemDefaultRelease(Heap_t Heap, void *pv, size_t cb)
{
    int         rc;
    PHIGHCHUNK  pChunk;
    PHIGHCHUNK  pPrevChunk;

    /*
     * Find and unlink the chunk.
     */
    if (_fmutex_request(&_sys_gmtxHimem, _FMR_IGNINT) != 0)
        return;
    for (pChunk = pHimemHead, pPrevChunk = NULL; pChunk; pPrevChunk = pChunk, pChunk = pChunk->pNext)
        if (&pChunk[1] == pv)
        {
            if (pPrevChunk)
                pPrevChunk->pNext = pChunk->pNext;
            else
                pHimemHead = pChunk->pNext;
            break;
        }
    _fmutex_release(&_sys_gmtxHimem);

    /*
     * Precautions and assertions.
     */
    if (!pChunk || cb + sizeof(HIGHCHUNK) != pChunk->cb)
    {
        #ifdef DEBUG
        __asm__("int $3");
        #endif
        return;
    }

    /*
     * Free it.
     */
    rc = DosFreeMem(pChunk);
    if (rc)
    {
        #ifdef DEBUG
        __asm__("int $3");
        #endif
    }
}


/**
 * Query if the system have support for high memory.
 *
 * @returns 1 if the system have more than 512MB of user address space.
 * @returns 0 if the system only have 512MB of user address space.
 */
int     __libc_HasHighMem(void)
{
    return _sys_gcbVirtualAddressLimit > 512*1024*1024;
}

