/* iconv wrapper based on OS/2 Unicode API. */

#define INCL_FSMACROS
#include <os2emx.h>
#include <uconv.h>

typedef struct _iconv_t
{
  UconvObject from;		/* "From" conversion handle */
  UconvObject to;		/* "To" conversion handle */
} *iconv_t;

/* Tell "iconv.h" to not define iconv_t by itself.  */
#define _ICONV_T
#include "iconv.h"

#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <alloca.h>

extern void
__convert_codepage (const char *cp, UniChar *ucp);

iconv_t
iconv_open (const char *cp_to, const char *cp_from)
{
  UniChar *ucp;
  iconv_t conv;
  uconv_attribute_t attr;
  FS_VAR();

  conv = (iconv_t) malloc (sizeof (struct _iconv_t));
  if (conv == NULL)
    {
      errno = ENOMEM;
      return (iconv_t)(-1);
    }

  FS_SAVE_LOAD();
  ucp = (UniChar *) alloca ((strlen (cp_from) + 2 + 1) * sizeof (UniChar));
  __convert_codepage (cp_from, ucp);
  if (UniCreateUconvObject (ucp, &conv->from))
    {
      free (conv);
      errno = EINVAL;
      FS_RESTORE();
      return (iconv_t)(-1);
    }

  ucp = (UniChar *) alloca ((strlen (cp_to) + 2 + 1) * sizeof (UniChar));
  __convert_codepage (cp_to, ucp);
  if (UniCreateUconvObject (ucp, &conv->to))
    {
      UniFreeUconvObject (conv->from);
      free (conv);
      errno = EINVAL;
      FS_RESTORE();
      return (iconv_t)(-1);
    }

  UniQueryUconvObject (conv->from, &attr, sizeof (attr), NULL, NULL, NULL);
  /* Do not treat 0x7f as a control character
     (don't understand what it exactly means but without it MBCS prefix
     character detection sometimes could fail (when 0x7f is a prefix)).
     And don't treat the string as a path (the docs also don't explain
     what it exactly means, but I'm pretty sure converted texts will
     mostly not be paths).  */
  attr.converttype &= ~(CVTTYPE_CTRL7F | CVTTYPE_PATH);
  UniSetUconvObject (conv->from, &attr);

  FS_RESTORE();
  return conv;
}

size_t
iconv (iconv_t conv,
       const char **in, size_t *in_left,
       char **out, size_t *out_left)
{
  int rc;
  size_t sl = *in_left, nonid;
  UniChar *ucs = (UniChar *) alloca (sl * sizeof (UniChar));
  UniChar *orig_ucs = ucs;
  size_t retval = 0;
  FS_VAR();

  FS_SAVE_LOAD();
  rc = UniUconvToUcs (conv->from, (void **)in, in_left, &ucs, &sl, &retval);
  if (rc)
    goto error;
  sl = ucs - orig_ucs;
  ucs = orig_ucs;
  /* UniUconvFromUcs will stop at first null byte (huh? indeed?)
     while we want ALL the bytes converted.  */
#if 1
  rc = UniUconvFromUcs (conv->to, &ucs, &sl, (void **)out, out_left, &nonid);
  if (rc)
    goto error;
  retval += nonid;
#else
  while (sl)
    {
      size_t usl = 0;
      while (sl && (ucs[usl] != 0))
        usl++, sl--;
      rc = UniUconvFromUcs (conv->to, &ucs, &usl, (void **)out, out_left, &nonid);
      if (rc)
        goto error;
      retval += nonid;
      if (sl && *out_left)
        {
          *(*out)++ = 0;
          (*out_left)--;
          ucs++; sl--;
        }
    }
#endif
  FS_RESTORE();
  return 0;

error:
  /* Convert OS/2 error code to errno.  */
  switch (rc)
  {
    case ULS_ILLEGALSEQUENCE:
      errno = EILSEQ;
      break;
    case ULS_INVALID:
      errno = EINVAL;
      break;
    case ULS_BUFFERFULL:
      errno = E2BIG;
      break;
    default:
      errno = EBADF;
      break;
  }
  FS_RESTORE();
  return (size_t)(-1);
}

int
iconv_close (iconv_t conv)
{
  if (conv != (iconv_t)(-1))
    {
      FS_VAR();
      FS_SAVE_LOAD();
      UniFreeUconvObject (conv->to);
      UniFreeUconvObject (conv->from);
      FS_RESTORE();
      free (conv);
    }
  return 0;
}
