/* sys/fcntl.c (emx+gcc) -- Copyright (c) 1992-1996 by Eberhard Mattes */

#include "libc-alias.h"
#include <fcntl.h>
#include <errno.h>
#include <limits.h>
#include <io.h>
#define INCL_FSMACROS
#include <os2emx.h>
#include <emx/syscalls.h>
#include "syscalls.h"

static int __fcntl_locking(int handle, int request, struct flock *pflock);

int __fcntl (int handle, int request, int arg)
{
  ULONG rc;
  ULONG state, new_state;
  FS_VAR();

  switch (request)
    {
    case F_SETFL:
      return 0;
    case F_GETFD:
      FS_SAVE_LOAD();
      rc = DosQueryFHState (handle, &state);
      FS_RESTORE();
      if (rc != 0)
        {
          _sys_set_errno (rc);
          return -1;
        }
      return ((state & OPEN_FLAGS_NOINHERIT) ? 1 : 0);
    case F_SETFD:
      FS_SAVE_LOAD();
      rc = DosQueryFHState (handle, &state);
      FS_RESTORE();
      if (rc != 0)
        {
          _sys_set_errno (rc);
          return -1;
        }
      if (arg & 1)
        new_state = state | OPEN_FLAGS_NOINHERIT;
      else
        new_state = state & ~OPEN_FLAGS_NOINHERIT;
      if (new_state != state)
        {
          new_state &= 0x7f88;
          FS_SAVE_LOAD();
          rc = DosSetFHState (handle, new_state);
          FS_RESTORE();
          if (rc != 0)
            {
              _sys_set_errno (rc);
              return -1;
            }
        }
      return 0;

    case F_GETLK:   /* get record locking information */
    case F_SETLK:   /* set record locking information */
    case F_SETLKW:  /* F_SETLK; wait if blocked */
      return __fcntl_locking(handle, request, (struct flock*)arg);

    default:
      errno = EINVAL;
      return -1;
    }
}


/* Handle locking requests.

   Please excuse the Hungarian and indenting used in this code,
   it's take from elsewhere. */

static int __fcntl_locking (int hFile, int request, struct flock *pflock)
{
    APIRET        rc;
    union
    {
        FILESTATUS3     fsts3;
        FILESTATUS3L    fsts3L;
    } info;
#if OFF_MAX > LONG_MAX
    int     fLarge = 0;
#endif
    FS_VAR();

    /* check input */
    /** @todo: Implement F_GETLK */
    if (!pflock || request == F_GETLK)
    {
        errno = EINVAL;
        return -1;
    }

    /* check hFile & get filesize. */
    FS_SAVE_LOAD();
#if OFF_MAX > LONG_MAX
    if (__pfnDosOpenL)
    {
        rc = DosQueryFileInfo(hFile, FIL_STANDARDL, &info, sizeof(info.fsts3L));
        fLarge = 1;
    }
    else
#endif
        rc = DosQueryFileInfo(hFile, FIL_STANDARD, &info, sizeof(info.fsts3));
    FS_RESTORE();
    if (!rc)
    {
        ULONG       fAccess;
        int         fLock;
        ULONG       ulTimeout;
        off_t       cbFile;
        off_t       offStart;
        off_t       cbRange;
#if OFF_MAX > LONG_MAX
        if (fLarge)
            cbFile = info.fsts3L.cbFile;
        else
#endif
            cbFile = info.fsts3.cbFile;

        /* range */
        cbRange = pflock->l_len ? pflock->l_len : OFF_MAX;

        /* offset */
        switch (pflock->l_whence)
        {
            case SEEK_SET:  offStart = pflock->l_start; break;
            case SEEK_CUR:  offStart = tell(hFile) + pflock->l_start; break;
            case SEEK_END:  offStart = cbFile - pflock->l_start; break;
            default:
                errno = EINVAL;
                return -1;
        }
        if (    offStart < 0
            ||  cbRange + offStart < 0 )
        {
            errno = EINVAL;
            return -1;
        }

        /* flags and order */
        fAccess = 0; /* exclusive */
        switch (pflock->l_type)
        {
            case F_UNLCK:
                fLock = 0;
                break;

            case F_RDLCK:
                fAccess = 1; /* shared */
            case F_WRLCK:
                fLock = 1;
                break;

            default:
                errno = EINVAL;
                return -1;
        }

        /* timeout */
        if (request == F_SETLKW)
            ulTimeout = SEM_INDEFINITE_WAIT;
        else
            ulTimeout = SEM_IMMEDIATE_RETURN;

        /* Do work. */
#if OFF_MAX > LONG_MAX
        if (__pfnDosSetFileLocksL)
        {
            FILELOCKL   aflock[2];
            memset(&aflock[(fLock + 1) & 1], 0, sizeof(aflock[0]));
            aflock[fLock].lOffset = offStart;
            aflock[fLock].lRange  = cbRange;
            FS_SAVE_LOAD();
            rc = __pfnDosSetFileLocksL(hFile, &aflock[0], &aflock[1], ulTimeout, fAccess);
            FS_RESTORE();
        }
        else
#endif
        {
            FILELOCK    aflock[2];
#if OFF_MAX > LONG_MAX
            if (    offStart > LONG_MAX
                ||  cbRange > LONG_MAX
                ||  offStart + cbRange > LONG_MAX
                )
              {
                errno = EOVERFLOW;
                return -1;
              }
#endif
            memset(&aflock[(fLock + 1) & 1], 0, sizeof(aflock[0]));
            aflock[fLock].lOffset = offStart;
            aflock[fLock].lRange  = cbRange;
            FS_SAVE_LOAD();
            rc = DosSetFileLocks(hFile, &aflock[0], &aflock[1], ulTimeout, fAccess);
            FS_RESTORE();
        }
    }

    /* done */
    if (!rc)
        return 0;
    _sys_set_errno (rc);
    return -1;
}

