/* emx.c: Functions for emx as target system.

Original version by Eberhard Mattes, based on i386.c.
Heavily modified by Andrew Zabolotny and Knut St. Osmundsen.

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "config.h"
#include "system.h"
#include "rtl.h"
#include "regs.h"
#include "hard-reg-set.h"
#include "output.h"
#include "tree.h"
#include "toplev.h"
#include "flags.h"
#include "i386-protos.h"

/* The size of the target's pointer type.  */
#ifndef PTR_SIZE
#define PTR_SIZE (POINTER_SIZE / BITS_PER_UNIT)
#endif

/** @todo remove debug code */
/*#define BIRD_DEBUG - DO NOT COMMIT WITH THIS DEFINED!!! */

#ifdef BIRD_DEBUG
static const char *code(tree node)
{
    if (node)
    {
        switch (TREE_CODE (node))
        {
            case FUNCTION_TYPE: return "FUNCTION_TYPE";
            case FUNCTION_DECL: return "FUNCTION_DECL";
            case METHOD_TYPE:   return "METHOD_TYPE";
            case FIELD_DECL:    return "FIELD_DECL";
            case TYPE_DECL:     return "TYPE_DECL";
            case VAR_DECL:      return "VAR_DECL";
            case PARM_DECL:     return "PARM_DECL";
            case RESULT_DECL:   return "RESULT_DECL";
            case CONST_DECL:    return "CONST_DECL";
            case POINTER_TYPE:  return "POINTER_TYPE";
            case VOID_TYPE:     return "VOID_TYPE";
            case INTEGER_TYPE:  return "INTEGER_TYPE";
            case CHAR_TYPE:     return "CHAR_TYPE";
            case SET_TYPE:      return "SET_TYPE";
            case ARRAY_TYPE:    return "ARRAY_TYPE";
            case RECORD_TYPE:   return "RECORD_TYPE";
            case QUAL_UNION_TYPE:  return "QUAL_UNION_TYPE";
            case UNION_TYPE:    return "UNION_TYPE";
            default:
                break;
        }
    }
    return "";
}

const char *birddump_callingconv(tree node);
const char *birddump_callingconv(tree node)
{
    static const char *apsz[] = {"system", "optlink", "stdcall"};
    static const char *psznone = "none";
    static const char *pszdefault = "default";
    tree        attr;
    unsigned    i;

    if (!node)
        return psznone;

    attr = (DECL_P (node) ? DECL_ATTRIBUTES (node) : TYPE_ATTRIBUTES (node));
    if (!attr)
        return pszdefault;

    for (i = 0; i < sizeof(apsz) / sizeof(apsz[0]); i++)
        if (node && attr && lookup_attribute (apsz[i], attr))
            return apsz[i];

    return pszdefault;
}

void birddump (tree node);
void birddump (tree node)
{
    tree type, type2, context, name;
    if (!node)
        return;

    type = TREE_TYPE (node);
    type2 = type ? TREE_TYPE(type) : NULL;
    context = DECL_P (node) ? DECL_CONTEXT (node) : NULL_TREE;
    name = DECL_P (node) ? DECL_NAME (node) : NULL_TREE;

    fprintf(stderr, "dbg: node=%d %s %p %s  type=%d %s %p %s  type_type=%d %s %p %s  context=%d %s %p  name=%s\n",
            TREE_CODE(node), code(node), (void*)node, birddump_callingconv(node),
            type ? (int)TREE_CODE(type) : -1, code(type), (void*)type, birddump_callingconv(type),
            type2 ? (int)TREE_CODE(type2) : -1, code(type2), (void*)type2, birddump_callingconv(type2),
            context ? (int)TREE_CODE(context) : -1, code(context), (void*)context,
            name ? IDENTIFIER_POINTER (name) : "<none>");
}

#define dfprintf(a) fprintf a
#define DUMP(node) birddump(node)
#else
#define dfprintf(a) do {} while (0)
#define DUMP(node) do {} while (0)
#endif

int emx_c_set_decl_assembler_name (tree decl, bool fclass)
{
  static int recurse;
  const char *oldsym;
  char *newsym;
  size_t sl;
  tree id, type;
  int rc = 1;
  
  /*
  dfprintf((stderr, "emx_c_set_decl_assembler_name\n"));
  DUMP(decl);
  */

  /* Sometimes we recursively call DECL_ASSEMBLER_NAME to apply the default
     mangling rules for current compiler. */
  if (recurse)
    return 0;

  /* Only functions declarations are subject to mangling. */
  if (TREE_CODE (decl) != FUNCTION_DECL)
    return 0;

  recurse++;
  type = TREE_TYPE (decl);

  if (lookup_attribute ("system", TYPE_ATTRIBUTES (type)))
    {
      /* Here we mangle _System functions as defined by IBM specs.
         The function always gets its name as-is (unless it is a method,
         which is a undefined case as VACPP always use _Optlink for methods,
         at least that's what I have understood from the docs). */

      oldsym = IDENTIFIER_POINTER (DECL_NAME (decl));
      if (fclass)
        return 0;

      /* Specifying '*' as first symbol character tells gcc (see varasm.c,
         function assemble_name()) to output the label as-is rather than
         invoking the ASM_OUTPUT_LABELREF macro (which prepends a underscore) */

      sl = strlen (oldsym);
      newsym = xmalloc (sl + 2);
      newsym [0] = '*';
      memcpy (newsym + 1, oldsym, sl + 1);

      SET_DECL_ASSEMBLER_NAME (decl, get_identifier (newsym));

      dfprintf ((stderr, "dbg: system %s -> %s\n", oldsym, newsym));
    }
  else if (lookup_attribute ("optlink", TYPE_ATTRIBUTES (type)))
    {
      /* At the moment we're only implementing OS/2 VAC linking
         compatibility for the C language. This means that no leading
         underscore.
         For C++ we are not compatible. It doesn't make that much sense
         either since we're not VFT compatible either. For simplicity
         and safety we are removing the leading underscore from the
         default mangled names to catch invalid declarations in the
         linking. */

      id = DECL_ASSEMBLER_NAME (decl);

      /* Remove the leading underscore. */
      oldsym = IDENTIFIER_POINTER (id);
      sl = strlen (oldsym);
      newsym = xmalloc (sl + 2);
      newsym [0] = '*';
      memcpy (newsym + 1, oldsym, sl + 1);

      SET_DECL_ASSEMBLER_NAME (decl, get_identifier (newsym));

      dfprintf ((stderr, "dbg: optlink %s -> %s\n", oldsym, newsym));
    }
  else if (lookup_attribute ("stdcall", TYPE_ATTRIBUTES (type)))
    {
      /* Mangle decl as the former assembler name modified with a
         suffix consisting of an atsign (@) followed by the number of bytes of
         arguments.
         For C++ the same applies as for optlink. See above. */

      int total = 0;

      /* If function does not have ellipsis as last argument, count total args size */
      if (TYPE_ARG_TYPES (type))
        if (TREE_VALUE (tree_last (TYPE_ARG_TYPES (type))) == void_type_node)
          {
            tree formal_type = TYPE_ARG_TYPES (type);

            while (TREE_VALUE (formal_type) != void_type_node)
              {
                int parm_size = TREE_INT_CST_LOW (TYPE_SIZE (TREE_VALUE (formal_type)));
                /* Must round up to include padding.  This is done the same
                   way as in store_one_arg.  */
                parm_size = ((parm_size + PARM_BOUNDARY - 1) / PARM_BOUNDARY * PARM_BOUNDARY);
                total += parm_size;
                formal_type = TREE_CHAIN (formal_type);
              }
          }

      id = DECL_ASSEMBLER_NAME (decl);
      oldsym = IDENTIFIER_POINTER (id);
      newsym = xmalloc (strlen (oldsym) + 10);
      sprintf (newsym, "%s@%d", oldsym, total / BITS_PER_UNIT);

      SET_DECL_ASSEMBLER_NAME (decl, get_identifier (newsym));

      dfprintf ((stderr, "dbg: stdcall %s -> %s\n", oldsym, newsym));
    }
  else
    rc = 0;

  recurse--;
  return rc;
}

#if 0 /* choose between the smaller non working version, and the
         large ugly one which is working. */

tree emx_handle_vacpp_attribute (tree *node, tree name, tree args,
  int flags, bool *no_add_attrs)
{
  tree *type;
  (void) args;
  (void) flags;

  dfprintf((stderr, "emx_handle_vacpp_attribute\n"));
  DUMP (*node);

  if (TREE_CODE (*node) != FUNCTION_TYPE
      && TREE_CODE (*node) != METHOD_TYPE
      && TREE_CODE (*node) != FIELD_DECL
      && TREE_CODE (*node) != TYPE_DECL)
    {
      warning ("`%s' attribute only applies to functions",
	       IDENTIFIER_POINTER (name));
      *no_add_attrs = true;
    }

  if (TARGET_64BIT)
    {
      warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
      *no_add_attrs = true;
    }

  /* Check if the attributes are compatible */
  if (ix86_check_append_attr (*node, name, no_add_attrs))
    return NULL_TREE;

  /* Now, a new type with our attribute please!
     If the FUNCTION_TYPE/METHOD_TYPE already have the attribute associated,
     there is no need to duplicate the type.
     @todo: I'm unsure wether or not we should do this from node and down... */
  for (type = node;
       TREE_CODE (*type) != FUNCTION_TYPE && TREE_CODE (*type) != METHOD_TYPE;
       type = &TREE_TYPE (*type))
      /* do nothing */;

  if (lookup_attribute (IDENTIFIER_POINTER (name), TYPE_ATTRIBUTES (*type)))
      return NULL_TREE;

  *type = build_type_copy (*type);
  *type = build_type_attribute_variant (*type, tree_cons (name, args, TYPE_ATTRIBUTES (*type)) );
  /**no_add_attrs = true; ?? */

  dfprintf((stderr, "emx_handle_vacpp_attribute: new type (%p)\n", (void*)type));
  DUMP (*node);
  return NULL_TREE;
}

#else

/* This is the working version which we belive duplicate some which could be
   done by the generic attribute stuff in GCC... */
tree emx_handle_vacpp_attribute (tree *node, tree name, tree args,
  int flags, bool *no_add_attrs)
{
  tree type;

  dfprintf((stderr, "emx_handle_vacpp_attribute: node=%p\n", node));
  DUMP (*node);

  switch (TREE_CODE (*node))
    {
      /* Declarations!
         We need to attach the attribute to the type of the declaration.
         (Name mangling is done by emx_c_set_decl_assembler_name().)  */
      case FUNCTION_DECL:/* Function declaration. */
      case TYPE_DECL:    /* Function or function pointer type. */
      case FIELD_DECL:   /* Function pointer (or function?) as a struct, union or class member. */
      case PARM_DECL:    /* Function pointer as a parameter. */
      case VAR_DECL:     /* Function pointer variable. */
        /* If this is a type declaration with our attribute, we allow it
           only if it is a pointer-to-a-function type or a function type. */
        type = TREE_TYPE (*node);
        if (TREE_CODE (type) == POINTER_TYPE)
          type = TREE_TYPE(type);
        if (   TREE_CODE (type) != FUNCTION_TYPE
            && TREE_CODE (type) != METHOD_TYPE)
          {
            warning ("`%s' attribute only applies to functions and function types, not to '%T'.",
                     IDENTIFIER_POINTER (name), type);
            *no_add_attrs = true;
            break;
          }

        if (ix86_check_append_attr (type, name, no_add_attrs))
          break;

        /* If required we'll make a variant of the type for this attribute. */
        if (!lookup_attribute (IDENTIFIER_POINTER (name),
                               TYPE_ATTRIBUTES (type)))
          {
            tree *ptype = node;
            while (*ptype != type)
                ptype = &TREE_TYPE (*ptype);
            *ptype = build_type_copy (*ptype);
            *ptype = build_type_attribute_variant (
                *ptype, tree_cons (name, args, TYPE_ATTRIBUTES (*ptype)) );
            dfprintf((stderr, "emx_handle_vacpp_attribute: new type\n"));
            DUMP(*node);
          }
        else
          dfprintf((stderr, "emx_handle_vacpp_attribute: use old type\n"));
        *no_add_attrs = true;
        break;

      /* Types!
         For types involving functions we need to convince decl_attributes()
         (and its callers) to supply a declaration so we safely can change
         the type. */
      case POINTER_TYPE:
        /* We allow:
           This being the return type of a function which is coming soon.
           This being a function pointer which declaration is coming next.
           Everything else is considered inappropriate use of the attribute. */
        if (   !(flags & ATTR_FLAG_FUNCTION_NEXT)
            && (    !(flags & ATTR_FLAG_DECL_NEXT)
                ||  !(type = TREE_TYPE (*node))
                || (   TREE_CODE (type) != FUNCTION_TYPE
                    && TREE_CODE (type) != METHOD_TYPE)))
        {
            warning ("`%s' attribute only applies to functions and function types",
                     IDENTIFIER_POINTER (name));
            *no_add_attrs = true;
            break;
        }
        /* fall thru */
      case FUNCTION_TYPE:
      case METHOD_TYPE:
#if 0 /* bird: This fixes optlink/tst4.c and doesn't seem to break anything.
               If set to true we are in some cases imposing _Optlink onto following
               declarations. This is weird stuff!
            !! This problem is actually what we see with the other implementation!! */
        *no_add_attrs = true;
#endif
        return tree_cons (name, args, NULL_TREE);

      default:
        warning ("`%s' attribute only applies to functions and function types (code=%d)",
                 IDENTIFIER_POINTER (name), TREE_CODE (*node));
        *no_add_attrs = true;
        break;
    }

  return NULL_TREE;
}
#endif

void
emx_eh_frame_section ()
{
/*
  tree label = get_file_function_name ('F');
  The __ehInit undefined external here is to drag __ehInit/__ehInitDLL into
  the linking so DLLs are initiated correctly. (#577)
*/
  data_section ();
  ASM_OUTPUT_ALIGN (asm_out_file, floor_log2 (PTR_SIZE));
  fputs ("\t.stabs\t\"___eh_frame__\",24,0,0,Lframe1\n" /* N_SETD */
         "\t.stabs\t\"___ehInit\",1,0,0,0\n",  /* N_UNDEF | N_EXT */
         asm_out_file);
}

/* Add a __POST$xxx label before epilogue if -mepilogue specified */
void emx_output_function_begin_epilogue (FILE *f)
{
  if (TARGET_EPILOGUE && TREE_PUBLIC (current_function_decl))
  {
    const char *func_label = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (current_function_decl));
    char *lbl = alloca (strlen (func_label) + 8);
    strcpy (lbl, "__POST$");
    if (func_label[0] == '*')
      func_label++;
    strcat (lbl, func_label);
    ASM_OUTPUT_LABEL (f, lbl);
  }
}


/* Checks if the function is using either optlink or system calling convention
   and returns 1 in those cases forcing the return value to be in memory.
   The problem (#631) was that structures less than 8 bytes were returned in 
   registers. _System and _Optlink requires them to be passed in as a hidden
   parameter.
   @type is the return type.
   @fntype is function type, call expression, function declaration or null.
   The return value is 1 to force the return value into memory. Return 0 if we
   don't know. */

int emx_return_in_memory_with_fntype (tree type, tree fntype)
{
  /* (from aggregate_value_p() CVS trunk) */
  if (fntype)
    switch (TREE_CODE (fntype))
      {
      case CALL_EXPR:
        fntype = get_callee_fndecl (fntype);
        fntype = fntype ? TREE_TYPE (fntype) : 0;
        break;
      case FUNCTION_DECL:
        fntype = TREE_TYPE (fntype);
        break;
      case FUNCTION_TYPE:
      case METHOD_TYPE:
        break;
      case IDENTIFIER_NODE:
        fntype = 0;
        break;
      default:
        /* We don't expect other rtl types here.  */
        abort();
      }

  /* (future targetm.calls.return_in_memory additions) */
  if (fntype
   && AGGREGATE_TYPE_P (type)
   && (   lookup_attribute ("optlink", TYPE_ATTRIBUTES (fntype))
       || lookup_attribute ("system", TYPE_ATTRIBUTES (fntype))))
    {
      dfprintf((stderr, "emx_return_in_memory_with_fntype: returns 1\n"));
      return 1;
    }

  /* return ix86_return_in_memory (exp); - future */
  dfprintf((stderr, "emx_return_in_memory_with_fntype: returns 0\n"));
  return 0;
}

