/* emx.c: Functions for emx as target system.

Original version by Eberhard Mattes, based on i386.c.
Heavily modified by Andrew Zabolotny and Knut St. Osmundsen.

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "config.h"
#include "system.h"
#include "rtl.h"
#include "regs.h"
#include "hard-reg-set.h"
#include "output.h"
#include "tree.h"
#include "toplev.h"
#include "flags.h"

/* The size of the target's pointer type.  */
#ifndef PTR_SIZE
#define PTR_SIZE (POINTER_SIZE / BITS_PER_UNIT)
#endif
#ifdef BIRD_DEBUG
static const char *code(tree node)
{
    if (node)
    {
        switch (TREE_CODE (node))
        {
            case FUNCTION_TYPE: return "FUNCTION_TYPE";
            case FUNCTION_DECL: return "FUNCTION_DECL";
            case METHOD_TYPE:   return "METHOD_TYPE";
            case FIELD_DECL:    return "FIELD_DECL";
            case TYPE_DECL:     return "TYPE_DECL";
            case POINTER_TYPE:  return "POINTER_TYPE";
            case VOID_TYPE:  return "VOID_TYPE";
            case INTEGER_TYPE:  return "INTEGER_TYPE";
            case CHAR_TYPE:  return "CHAR_TYPE";
            case SET_TYPE:  return "SET_TYPE";
            case ARRAY_TYPE:  return "ARRAY_TYPE";
            case RECORD_TYPE:  return "RECORD_TYPE";
            case QUAL_UNION_TYPE:  return "QUAL_UNION_TYPE";
            case UNION_TYPE:  return "UNION_TYPE";
            default:
                break;
        }
    }
    return "";
}

void dump (tree node);
void dump (tree node)
{
    tree        type, type2, context;
    if (!node)
        return;

    type = TREE_TYPE (node);
    type2 = type ? TREE_TYPE(type) : NULL;
    context = DECL_P (node) ? DECL_CONTEXT (node) : NULL_TREE;
    fprintf(stderr, "dbg: node=%d %s type=%d %s type_type=%d %s context=%d %s\n",
            TREE_CODE(node), code(node),
            type ? TREE_CODE(type) : -1, code(type),
            type2 ? TREE_CODE(type2) : -1, code(type2),
            context ? TREE_CODE(context) : -1, code(context));
}

#define dfprintf(a) fprintf a
#define DUMP(node) dump(node)
#else
#define dfprintf(a) do {} while (0)
#define DUMP(node) do {} while (0)
#endif


tree ix86_handle_system_attribute (tree *node, tree name, tree args,
  int flags, bool *no_add_attrs)
{
  tree          type;
  (void)args; (void)flags;

dfprintf ((stderr, "dbg: %s flags=%x\n", __FUNCTION__, flags));
DUMP (*node);

  type = TREE_TYPE (*node);
  if (DECL_P (*node))
    switch (TREE_CODE (type))
      {
        case FUNCTION_TYPE:
          {
            tree          context;
            size_t        sl;
            const char *  oldsym;
            char *        newsym;

            /* typedef and field may end up here. they should not get their name changed! */
            if (TREE_CODE (*node) == FIELD_DECL | TREE_CODE (*node) == TYPE_DECL)
                return NULL_TREE;

            /* class method (static) ends up here too, they should be get their
               name changed. Try check if context is RECORD_TYPE and pray that
               means it's a class...
               @todo: verify the class detection here! CLASSTYPE_DECLARED_CLASS?  */
            context = DECL_CONTEXT (*node);
            DUMP ((context));
            if (context && (    TREE_CODE (context) == RECORD_TYPE
                            ||  TREE_CODE (context) == UNION_TYPE))
              return NULL_TREE;
            /** @todo more decl code checks? */


            /* _System mangling!
               Specifying '*' as first symbol character tells gcc (see varasm.c,
               function assemble_name()) to output the label as-is rather than
               invoking the ASM_OUTPUT_LABELREF macro (which prepends a underscore) */
            oldsym = IDENTIFIER_POINTER (DECL_NAME (*node));
            sl = strlen (oldsym);
            newsym = xmalloc (sl + 2);
            newsym [0] = '*';
            memcpy (newsym + 1, oldsym, sl + 1);
            SET_DECL_ASSEMBLER_NAME (*node, get_identifier (newsym));
            dfprintf((stderr, "dbg: name change: %s -> %s\n", oldsym, newsym));
            return NULL_TREE;
          }
        case POINTER_TYPE:
          /* Pointer-to-a-function type is allowed */
	  if (TREE_CODE (TREE_TYPE (type)) != FUNCTION_TYPE
           && TREE_CODE (TREE_TYPE (type)) != METHOD_TYPE)
            break;
        case METHOD_TYPE:
        case FIELD_DECL:
        case TYPE_DECL:
          /* Not a error but no action either */
          return NULL_TREE;
        default:
          break;
      }
  else if (TREE_CODE (*node) == POINTER_TYPE && type)
    { /* Pointer-to-a-function type is allowed */
      switch (TREE_CODE (type))
        {
          case FUNCTION_TYPE:
          case METHOD_TYPE:
            return NULL_TREE;
          default:
            break;
        }
    }
  /* potential warnings get here */

  /* IF there is a function in the works, we don't complain and
     make sure _System is tried again on the function declaration.
     OR if this is a function type we need to pass it up to the
     declaration which is likely to be a some kind of type or field.
     */
  if (   (flags & (int)ATTR_FLAG_FUNCTION_NEXT)
      || TREE_CODE (*node) == FUNCTION_TYPE
      || TREE_CODE (*node) == METHOD_TYPE)
    return tree_cons (name, args, NULL_TREE);

  warning ("`%s' attribute only applies to functions and function types",
           IDENTIFIER_POINTER (name));
  *no_add_attrs = true;

  return NULL_TREE;
}


tree ix86_handle_optlink_attribute (tree *node, tree name, tree args,
  int flags, bool *no_add_attrs)
{
  tree id, type, typeset;
  size_t sl;
  const char *oldsym;
  char *newsym;

  (void)args; (void)flags;

  typeset = NULL;
  type = TREE_TYPE (*node);
  if (DECL_P (*node))
    switch (TREE_CODE (type))
      {
        case FUNCTION_TYPE:
        case METHOD_TYPE:
          typeset = type;
        case FIELD_DECL:                /* how come these needs mangling? */
        case TYPE_DECL:
          id = DECL_ASSEMBLER_NAME (*node);
          /* Remove the leading underscore */
          oldsym = IDENTIFIER_POINTER (id);
          sl = strlen (oldsym);
          newsym = xmalloc (sl + 2);
          newsym [0] = '*';
          memcpy (newsym + 1, oldsym, sl + 1);
          XEXP (DECL_RTL (*node), 0) = gen_rtx (SYMBOL_REF, Pmode,
            IDENTIFIER_POINTER (get_identifier (newsym)));
          break;
        case POINTER_TYPE:
          /* Pointer-to-a-function type is allowed */
	  if (TREE_CODE (TREE_TYPE (type)) != FUNCTION_TYPE
           && TREE_CODE (TREE_TYPE (type)) != METHOD_TYPE)
            goto warn;
/*@@@@@todo: not finished*/
          typeset = type;
          break;
        default:
          goto warn;
      }
  else if (TREE_CODE (*node) == POINTER_TYPE && type)
    { /* Pointer-to-a-function type is allowed */
      switch (TREE_CODE (type))
        {
          case FUNCTION_TYPE:
          case METHOD_TYPE:
            typeset = type;
            break;
          default:
            goto warn;
        }
    }
  else
    {
warn: warning ("`%s' attribute only applies to functions and function types",
               IDENTIFIER_POINTER (name));
      *no_add_attrs = true;
      DUMP (*node);
    }

  /* attach attribute to type. */
  if (typeset)
    {
      id = tree_cons (name, args, NULL_TREE);
      TYPE_ATTRIBUTES (typeset) = chainon (TYPE_ATTRIBUTES (typeset), id);
      return id;
    }

  return NULL_TREE;
}

void
emx_eh_frame_section ()
{
/*
  tree label = get_file_function_name ('F');
  The __ehInit undefined external here is to drag __ehInit/__ehInitDLL into
  the linking so DLLs are initiated correctly. (#577)
*/
  data_section ();
  ASM_OUTPUT_ALIGN (asm_out_file, floor_log2 (PTR_SIZE));
  fprintf (asm_out_file,
           "\t.stabs\t\"___eh_frame__\",24,0,0,Lframe1\n" /* N_SETD */
           "\t.stabs\t\"___ehInit\",1,0,0,0\n");          /* N_UNDEF | N_EXT */
}

/* Add a __POST$xxx label before epilogue if -mepilogue specified */
void emx_output_function_begin_epilogue (FILE *f)
{
  if (TARGET_EPILOGUE && TREE_PUBLIC (current_function_decl))
  {
    const char *func_label = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (current_function_decl));
    char *lbl = alloca (strlen (func_label) + 8);
    strcpy (lbl, "__POST$");
    if (func_label[0] == '*')
        func_label++;
    strcat (lbl, func_label);
    ASM_OUTPUT_LABEL (f, lbl);
  }
}

/* Return string which is the former assembler name modified with a
   suffix consisting of an atsign (@) followed by the number of bytes of
   arguments */

const char *
gen_stdcall_suffix (tree decl)
{
  int total = 0;
  /* ??? This probably should use XSTR (XEXP (DECL_RTL (decl), 0), 0) instead
     of DECL_ASSEMBLER_NAME.  */
  const char *asmname = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (decl));
  char *newsym;

  if (TYPE_ARG_TYPES (TREE_TYPE (decl)))
    if (TREE_VALUE (tree_last (TYPE_ARG_TYPES (TREE_TYPE (decl))))
        == void_type_node)
      {
	tree formal_type = TYPE_ARG_TYPES (TREE_TYPE (decl));

	while (TREE_VALUE (formal_type) != void_type_node)
	  {
	    int parm_size
	      = TREE_INT_CST_LOW (TYPE_SIZE (TREE_VALUE (formal_type)));
	    /* Must round up to include padding.  This is done the same
	       way as in store_one_arg.  */
	    parm_size = ((parm_size + PARM_BOUNDARY - 1)
			 / PARM_BOUNDARY * PARM_BOUNDARY);
	    total += parm_size;
	    formal_type = TREE_CHAIN (formal_type);
	  }
      }

  newsym = xmalloc (strlen (asmname) + 10);
  sprintf (newsym, "%s@%d", asmname, total/BITS_PER_UNIT);
  return IDENTIFIER_POINTER (get_identifier (newsym));
}

/* Cover function to implement ENCODE_SECTION_INFO.
   Note that this could be implemented much better by doing the
   stdcall mangling like optlink and system, but unfortunately */

void
emx_encode_section_info (tree decl)
{
  /* The default ENCODE_SECTION_INFO from i386.h */
  if (flag_pic)
    {
      rtx rtl = (TREE_CODE_CLASS (TREE_CODE (decl)) != 'd'
                 ? TREE_CST_RTL (decl) : DECL_RTL (decl));

      if (GET_CODE (rtl) == MEM)
        {
          if (TARGET_DEBUG_ADDR
              && TREE_CODE_CLASS (TREE_CODE (decl)) == 'd')
            {
              fprintf (stderr, "Encode %s, public = %d\n",
                       IDENTIFIER_POINTER (DECL_NAME (decl)),
                       TREE_PUBLIC (decl));
            }

          SYMBOL_REF_FLAG (XEXP (rtl, 0))
            = (TREE_CODE_CLASS (TREE_CODE (decl)) != 'd'
               || ! TREE_PUBLIC (decl));
        }
    }

  /* If declaring a function, mangle it if it's stdcall */
  if (TREE_CODE (decl) == FUNCTION_DECL)
    {
      if (lookup_attribute ("stdcall", TYPE_ATTRIBUTES (TREE_TYPE (decl))))
        XEXP (DECL_RTL (decl), 0) =
          gen_rtx (SYMBOL_REF, Pmode, gen_stdcall_suffix (decl));
    }
}
