/* emx.c: Functions for emx as target system.

Original version by Eberhard Mattes, based on i386.c.
Heavily modified by Andrew Zabolotny and Knut St. Osmundsen.

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "config.h"
#include "system.h"
#include "rtl.h"
#include "regs.h"
#include "hard-reg-set.h"
#include "output.h"
#include "tree.h"
#include "toplev.h"
#include "flags.h"

/* The size of the target's pointer type.  */
#ifndef PTR_SIZE
#define PTR_SIZE (POINTER_SIZE / BITS_PER_UNIT)
#endif

static void (*old_lang_set_decl_assembler_name) (tree decl);

static void emx_lang_set_decl_assembler_name (tree decl)
{
  tree id;

  if (lookup_attribute ("optlink", TYPE_ATTRIBUTES (TREE_TYPE (decl))))
    {
      size_t sl;
      const char *oldsym;
      char *newsym;

      old_lang_set_decl_assembler_name (decl);
      id = DECL_ASSEMBLER_NAME (decl);
      /* Remove the leading underscore */
remove_underscore:
      oldsym = IDENTIFIER_POINTER (id);
      sl = strlen (oldsym) + 2;
      newsym = xmalloc (sl);
      /* Specifying '*' as first symbol character tells gcc (see varasm.c,
         function assemble_name()) to output the label as-is rather than
         invoking the ASM_OUTPUT_LABELREF macro (which prepends a underscore) */
      newsym [0] = '*';
      memcpy (newsym + 1, oldsym, sl + 1);
      XEXP (DECL_RTL (decl), 0) = gen_rtx (SYMBOL_REF, Pmode,
        IDENTIFIER_POINTER (get_identifier (newsym)));
    }
  else if (lookup_attribute ("system", TYPE_ATTRIBUTES (TREE_TYPE (decl))))
    {
      id = get_identifier (IDENTIFIER_POINTER (DECL_NAME (decl)));
      SET_DECL_ASSEMBLER_NAME (decl, id);
      goto remove_underscore;
    }
  else
    old_lang_set_decl_assembler_name (decl);
}

/* The first time we encounter a _Optlink or _System function we hook the
 * lang_set_decl_assembler_name function so that instead of calling the
 * original mangler function our function gets called. Then, if the function
 * attribute is _System or _Optlink we mangle it according to respective
 * rules (_System functions always get the name as-is (e.g. no underscore,
 * no C++ mangling) and _Optlink don't get the leading underscore.
 */
static void hook_mangler (void)
{
  if (lang_set_decl_assembler_name != emx_lang_set_decl_assembler_name)
    {
      old_lang_set_decl_assembler_name = lang_set_decl_assembler_name;
      lang_set_decl_assembler_name = emx_lang_set_decl_assembler_name;
    }
}

tree ix86_handle_system_attribute (tree *node, tree name, tree args,
  int flags, bool *no_add_attrs)
{
  (void)args; (void)flags;
  if (TREE_CODE (*node) != FUNCTION_TYPE
      && TREE_CODE (*node) != METHOD_TYPE
      && TREE_CODE (*node) != FIELD_DECL
      && TREE_CODE (*node) != TYPE_DECL)
    {
      warning ("`%s' attribute only applies to functions",
	       IDENTIFIER_POINTER (name));
      *no_add_attrs = true;
    }

  hook_mangler ();

  return NULL_TREE;
}

tree ix86_handle_optlink_attribute (tree *node, tree name, tree args,
  int flags, bool *no_add_attrs)
{
  (void)args; (void)flags;
  if (TREE_CODE (*node) != FUNCTION_TYPE
      && TREE_CODE (*node) != METHOD_TYPE
      && TREE_CODE (*node) != FIELD_DECL
      && TREE_CODE (*node) != TYPE_DECL)
    {
      warning ("`%s' attribute only applies to functions",
	       IDENTIFIER_POINTER (name));
      *no_add_attrs = true;
    }

  hook_mangler ();

  return NULL_TREE;
}

void
emx_eh_frame_section ()
{
/*
  tree label = get_file_function_name ('F');
  The __ehInit undefined external here is to drag __ehInit/__ehInitDLL into
  the linking so DLLs are initiated correctly. (#577)
*/
  data_section ();
  ASM_OUTPUT_ALIGN (asm_out_file, floor_log2 (PTR_SIZE));
  fprintf (asm_out_file,
           "\t.stabs\t\"___eh_frame__\",24,0,0,Lframe1\n" /* N_SETD */
           "\t.stabs\t\"___ehInit\",1,0,0,0\n");          /* N_UNDEF | N_EXT */
}

/* Add a __POST$xxx label before epilogue if -mepilogue specified */
void emx_output_function_begin_epilogue (FILE *f)
{
  if (TARGET_EPILOGUE && TREE_PUBLIC (current_function_decl))
  {
    const char *func_label = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (current_function_decl));
    char *lbl = alloca (strlen (func_label) + 8);
    strcpy (lbl, "__POST$");
    strcat (lbl, func_label);
    ASM_OUTPUT_LABEL (f, lbl);
  }
}

/* Return string which is the former assembler name modified with a
   suffix consisting of an atsign (@) followed by the number of bytes of
   arguments */

const char *
gen_stdcall_suffix (tree decl)
{
  int total = 0;
  /* ??? This probably should use XSTR (XEXP (DECL_RTL (decl), 0), 0) instead
     of DECL_ASSEMBLER_NAME.  */
  const char *asmname = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (decl));
  char *newsym;

  if (TYPE_ARG_TYPES (TREE_TYPE (decl)))
    if (TREE_VALUE (tree_last (TYPE_ARG_TYPES (TREE_TYPE (decl))))
        == void_type_node)
      {
	tree formal_type = TYPE_ARG_TYPES (TREE_TYPE (decl));

	while (TREE_VALUE (formal_type) != void_type_node)
	  {
	    int parm_size
	      = TREE_INT_CST_LOW (TYPE_SIZE (TREE_VALUE (formal_type)));
	    /* Must round up to include padding.  This is done the same
	       way as in store_one_arg.  */
	    parm_size = ((parm_size + PARM_BOUNDARY - 1)
			 / PARM_BOUNDARY * PARM_BOUNDARY);
	    total += parm_size;
	    formal_type = TREE_CHAIN (formal_type);
	  }
      }

  newsym = xmalloc (strlen (asmname) + 10);
  sprintf (newsym, "%s@%d", asmname, total/BITS_PER_UNIT);
  return IDENTIFIER_POINTER (get_identifier (newsym));
}

/* Cover function to implement ENCODE_SECTION_INFO.
   Note that this could be implemented much better by doing the
   stdcall mangling like optlink and system, but unfortunately */

void
emx_encode_section_info (tree decl)
{
  /* The default ENCODE_SECTION_INFO from i386.h */
  if (flag_pic)
    {
      rtx rtl = (TREE_CODE_CLASS (TREE_CODE (decl)) != 'd'
                 ? TREE_CST_RTL (decl) : DECL_RTL (decl));

      if (GET_CODE (rtl) == MEM)
        {
          if (TARGET_DEBUG_ADDR
              && TREE_CODE_CLASS (TREE_CODE (decl)) == 'd')
            {
              fprintf (stderr, "Encode %s, public = %d\n",
                       IDENTIFIER_POINTER (DECL_NAME (decl)),
                       TREE_PUBLIC (decl));
            }

          SYMBOL_REF_FLAG (XEXP (rtl, 0))
            = (TREE_CODE_CLASS (TREE_CODE (decl)) != 'd'
               || ! TREE_PUBLIC (decl));
        }
    }

  /* If declaring a function, mangle it if it's stdcall */
  if (TREE_CODE (decl) == FUNCTION_DECL)
    {
      if (lookup_attribute ("stdcall", TYPE_ATTRIBUTES (TREE_TYPE (decl))))
        XEXP (DECL_RTL (decl), 0) =
          gen_rtx (SYMBOL_REF, Pmode, gen_stdcall_suffix (decl));
    }
}
