/*
    Locale support implementation through OS/2 Unicode API.
    Copyright (c) 1992-1996 by Eberhard Mattes
    Copyright (c) 2003 InnoTek Systemberatung GmbH

    For conditions of distribution and use, see the file COPYING.

    Format time. The output string is written to the array of size characters
    pointed to by string, including the terminating null character. Like
    sprintf(), strftime() copies the string pointed to by format to the
    array pointed to by string, replacing format specifications with formatted
    data from t. Ordinary characters are copied unmodified.

    On success, strftime() returns the number of characters copied to the
    array pointed to by string, excluding the terminating null character.
    On failure (size exceeded), strftime() returns 0.
*/

#define __INTERNAL_DEFS
#include "libc-alias.h"
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <limits.h>
#include <sys/locale.h>
#include <emx/time.h>
#include <sys/builtin.h>

#define INS(STR) \
  if (!(i = ins (string, size, STR))) \
    return 0; \
  else \
    string += i, size -= i
#define NUM2(X) \
  if (!(i = num2 (string, size, X, minlen))) \
    return 0; \
  else \
    string += i, size -= i
#define NUM3(X) \
  if (!(i = num3 (string, size, X, minlen))) \
    return 0; \
  else \
    string += i, size -= i
#define FMT(F) \
  if (!(i = strftime (string, size, F, t))) \
    return 0; \
  else \
    string += i, size -= i
#define CHR(C) \
  if (!size) \
    return 0; \
  else \
    *string++ = C, size--

static size_t ins (char *string, size_t size, const char *s)
{
  size_t len = strlen (s);

  if (len >= size)
    return 0;

  memcpy (string, s, len);
  return len;
}

static size_t num2 (char *string, size_t size, int x, unsigned char minlen)
{
  char *orgstring = string;
  long r;

  if (size < 2)
    return 0;

  if (x < 0)
    x = 0;

  x = __ldivmod (x, 10, &r);
  if (x > 9)
    x = x % 10;

  if (minlen > 1 || x > 0)
    *string++ = x + '0';
  *string++ = r + '0';
  return string - orgstring;
}

static size_t num3 (char *string, size_t size, int x, unsigned char minlen)
{
  char *orgstring = string;
  long r1, r2;

  if (size < 3)
    return 0;

  if (x < 0)
    x = 0;

  x = __ldivmod (x, 10, &r1);
  x = __ldivmod (x, 10, &r2);
  if (x > 9)
    x = x % 10;

  if (minlen >= 3 || x > 0)
    *string++ = x + '0', minlen = 2;
  if (minlen >= 2 || r2 > 0)
    *string++ = r2 + '0';
  *string++ = r1 + '0';
  return string - orgstring;
}

static int week (const struct tm *t, int first)
{
  int wd, tmp;

  wd = t->tm_wday - first;    /* wd = relative day in week (w.r.t. first) */
  if (wd < 0) wd += 7;
  tmp = t->tm_yday - wd;      /* Start of current week */
  if (tmp <= 0) return 0;     /* In partial first week */
  return (tmp + 6) / 7;       /* Week number */
}

size_t _STD(strftime) (char *string, size_t size, const char *format,
  const struct tm *t)
{
  size_t i;
  unsigned char c, minlen;
  char *orgstring = string;

  if (!_tzset_flag) tzset ();

  while ((c = *format) != 0)
  {
    if (c == '%')
    {
      minlen = 9;
nextformat:
      ++format;
      switch (*format)
      {
        case 0:
        case '%':
          CHR ('%');
          if (*format == 0)
            --format;
          break;

        /* Handle prefixes first */
        case 'E':
          /* Alternative date/time formats not supported for now */
        case 'O':
          /* Alternative numeric formats not supported for now */
          goto nextformat;

        case '1':
          /* Undocumented in Unicode API reference, encountered in Russian
             locale. I think it should mean that the next number should occupy
             such much characters how it should (e.g. %1H:%M = 1:20 or 23:00). */
          minlen = 1;
          goto nextformat;

        case 'a':
          INS (__locale_time.swdays [t->tm_wday]);
          break;
        case 'A':
          INS (__locale_time.lwdays [t->tm_wday]);
          break;
        case 'b':
        case 'h':
          INS (__locale_time.smonths [t->tm_mon]);
          break;
        case 'B':
          INS (__locale_time.lmonths [t->tm_mon]);
          break;
        case 'c':
          FMT (__locale_time.date_time_fmt);
          break;
        case 'C':
          /* 2000 A.D. is still 20th century (not 21st) */
          NUM2 (1 + (t->tm_year - 1) / 100);
          break;
        case 'd':
          NUM2 (t->tm_mday);
          break;
        case 'D':
          FMT ("%m/%d/%y");
          break;
        case 'e':
          NUM2 (t->tm_mday);
          if (string [-2] == '0')
            string [-2] = ' ';
          break;
        case 'H':
          NUM2 (t->tm_hour);
          break;
        case 'I':
          NUM2 ((t->tm_hour + 11) % 12 + 1);
          break;
        case 'j':
          NUM3 (t->tm_yday + 1);
          break;
        case 'm':
          NUM2 (t->tm_mon + 1);
          break;
        case 'M':
          NUM2 (t->tm_min);
          break;
        case 'n':
          CHR ('\n');
          break;
        case 'p':
          INS (t->tm_hour >= 12 ? __locale_time.pm : __locale_time.am);
          break;
        case 'r':
          FMT ("%I:%M:%S %p");
          break;
        case 'S':
          NUM2 (t->tm_sec);
          break;
        case 't':
          CHR ('\t');
          break;
        case 'T':
          FMT ("%H:%M:%S");
          break;
        case 'U':
          NUM2 (week (t, 0));
          break;
        case 'w':
          NUM2 (t->tm_wday);
          break;
        case 'W':
          NUM2 (week (t, 1));
          break;
        case 'x':
          FMT (__locale_time.date_fmt);
          break;
        case 'X':
          FMT (__locale_time.time_fmt);
          break;
        case 'y':
          NUM2 (t->tm_year % 100);
          break;
        case 'Y':
          CHR ('0' + (1900 + t->tm_year) / 1000);
          NUM3 ((t->tm_year + 1900) % 1000);
          break;
        case 'Z':
          if (t->tm_isdst >= 0)
          {
            INS (_tzname [(t->tm_isdst ? 1 : 0)]);
          }
          break;
      }
    }
    else if (!CHK_MBCS_PREFIX (__locale_ctype, c, i))
      CHR (c);
    else
    {
      if (i > size)
        return 0;
      memcpy (string, format, i);
      string += i;
      format += i - 1;
    }
    ++format;
  }

  if (size)
  {
    *string = 0;
    return string - orgstring;
  }

  return 0;
}
