/* $Id: weakld.c 489 2003-07-31 15:10:56Z bird $
 *
 * Weak Symbol Pre-Linker.
 *
 * Copyright (c) 2003 InnoTek Systemberatung GmbH
 * Author: knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with This program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/** @page weakld        Weak Pre-Linker
 * 
 * In order to get the weak symbols somewhat right it looks like we have to do
 * the pass1 of the linking process in order to resolve the weak symbols.
 *
 *
 *
 * @subsection          Symbols
 *
 * There is a couple of symbol types, but we can skip most of them for this 
 * pre-linking operation. We use one symbol type which is public or global
 * symbols if you like. Perhaps it would be wise to devide them into separat
 * type groups, but the choice was made to differenciate this using flags.
 * So, symbol enumeration is done using flag masks.
 *
 *
 */

/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
#define WLDSYM_HASH_SIZE    211
#define OMF_MAX_REC         1024


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/omflib.h>
#include "defs.h"
#include "grow.h"
#include "weakld.h"

 
/*******************************************************************************
*   Structures and Typedefs                                                    *
*******************************************************************************/
/**
 * Library structure 
 */
typedef struct wldlib
{
    /** Library name. */
    const char *pszLibName;
    /** Filehandle if open */
    FILE *      phFile;
    /** Linked list next pointer. */
    struct wldlib * pNext;
} WLDLIB, *PWLDLIB;
 

/**
 * Module structure 
 */    
typedef struct wldmod
{
    /** Module name. */
    const char *pszModName;
    /** Filehandle if open. */
    FILE *      phFile;
    /** Module offset into the file. */
    off_t       off;
    /** Library relation - not used for libraries added thru wld_add_object(). */
    PWLDLIB     pLib;
    /* Linked list next pointer */
    struct wldmod *pNext;
} WLDMOD, *PWLDMOD;
 

/**
 * Symbol structure.
 */
typedef struct wldsym
{
    /** Symbol name. */
    const char *        pszName;
    /** Weak name - for weak symbols only. */
    const char *        pszWeakName;

    /** Symbol type. */
    enum {
        WLDST_PUBLIC
        }               eType;

    /** Symbol flags. */
    enum {
        /** @group Symbol state 
         * @{ */
        /* Mask of the symbol state. */
        WLDSF_STATEMASK = 0x000f,
        /** Strong symbol.
         * A strong definition exists for this symbol (PUBDEF). */
        WLDSF_STRONG    = 0x0001,
        /** Communal data/code. 
         * Communal definition exists for this symbol.
         * If a PUBDEF is found for this symbol it will become strong. */
        WLDSF_COMM      = 0x0002,
        /** Undefined symbol.
         * This symbol is not yet defined anywhere. */
        WLDSF_UNDEF     = 0x0003,
        /** Weak external.
         * This symbol doesn't need to be resolved as it have a default
         * resolution. pWeakDefault is pointing to the default resolution.
         * If an non weak EXTDEF is found in another module, it will become
         * WLDSF_UNDEF.
         */
        WLDSF_WKEXT     = 0x0004,
        /** @} */

        /** Weak symbol. 
         * This symbol doesn't need to be resolved (if WLDSF_UNDEF), or 
         * it may be overridden by a EXTDEF with no WKEXT.
         * If this is an weak undefined symbol (extern weak, local default)
         * pWeakDefault Will point at it.
         */
        WLDSF_WEAK      = 0x0100,

        /** Uncertain undefined symbol.
         * We're still processing the module containing this and the uncertainty
         * we're facing is that a WLDSF_UNDEF may changed to an WLDSF_WKEXT 
         * upon encountering a WKEXT COMENT record. */
        WLDSF_UNCERTAIN = 0x0200,


        /** Import symbol. 
         * This symbol is imported. */
        WLDSF_IMPORT    = 0x0400,
        /** Exported symbol. 
         * This symbol is to be exported. */
        WLDSF_EXPORT    = 0x0800,
        /** Alias symbol.
         * This symbol is an alias for another symbol. pAliasFor will be set. */
        WLDSF_ALIAS     = 0x1000,
    }                   eFlags;

    /** The module this symbol is defined in. */
    PWLDMOD             pMod;


    /** Import module - only if WLDSF_IMPORT is set. */
    const char*         pszImpMod;

    /** Import Ordinal (WLDSF_IMPORT). 
     * 0 means no ordinal. */
    unsigned            iImpOrd;


    /** Export flags */
    struct 
    {
        unsigned    fResidentName : 1;  /* Put name in resident or non resident table. */
        unsigned    fNoName : 1;        /* No name at all. */
        unsigned    fNoData : 1;        /* ?? */
        unsigned    cParams;            /* callgate stuff, # of words to push */
    }                   fExport;

    /** Import Ordinal (WLDSF_EXPORT). 
     * 0 means no ordinal. */
    unsigned            iExpOrd;


    /** Symbol this is an alias for. */
    struct wldsym *     pAliasFor;

    /** Weak default resolution. 
     * If this symbol in state WLDSF_WKEXT or have to WLDSF_WEAK flag set this
     * member indicates a default resolution for the symbol. For WLDSF_WEAK 
     * this only make sense if the state is WLDSF_UNDEF.
     */
    struct wldsym *     pWeakDefault;

    /** Next node in the hash bucket. */
    struct wldsym *     pHashNext;
} WLDSYM, *PWLDSYM;


/**
 * Symbol table.
 */
typedef struct wldsymtab
{
    PWLDSYM     ap[WLDSYM_HASH_SIZE];
} WLDSYMTAB, *PWLDSYMTAB;

 
/** 
 * Weak Pre-Linker Instance.
 */
struct wld
{
    /** Linker flags. */
    unsigned            fFlags;

    /** Global symbols. */
    WLDSYMTAB           Global;

    /** Linked list (FIFO) of objects included in the link. */
    PWLDMOD             pObjs;
    PWLDMOD *           ppObjsAdd;

    /** Linked list (FIFO) of libraries to be searched in the link. */
    PWLDMOD             pLibs;
    PWLDMOD *           ppLibsAdd;

    /** string pool for miscellaneous string. */
    struct strpool *    pStrMisc;
};
typedef struct wld WLD;
 

/** @group OMF stuff 
 * @{ */

/** OMF record header. */
#pragma pack(1)
typedef struct OMFREC
{
  unsigned char     chType;
  unsigned short    cb;
} OMFREC, *POMFREC;
#pragma pack()


/** OMF library header. */
#pragma pack(1)
typedef struct OMFLIBHDR
{
  unsigned char     chType;
  unsigned short    cb;
  unsigned long     offDict;
  unsigned short    cDictBlocks;
  unsigned char     fchFlags;
} OMFLIBHDR, *POMFLIBHDR;
#pragma pack()

/** @} */


extern void *xrealloc (void *ptr, size_t n);
extern void *xmalloc (size_t n);

static unsigned     hash(const char* pszSym, unsigned cch);
static FILE *       modOpen(PWLDMOD pMod);
static void         modClose(PWLDMOD pMod);



/*******************************************************************************
*  
*  H e l p e r s
*  H e l p e r s
*  H e l p e r s
*  
*******************************************************************************/

/** 
 * Calculate the hash value of a symbol.
 * @returns hash value.
 * @param   pszSym  Symbol to calculate it for.
 * @param   cch     Symbol length.
 */
static unsigned     hash(const char* pszSym, unsigned cch)
{
    unsigned uHash = 0;
    while (cch && *pszSym && *pszSym != '$')
    {
        uHash = uHash * 65599 + *pszSym;
        pszSym++;
        cch--;
    }
    uHash %= WLDSYM_HASH_SIZE;
    return uHash;  
}

/**
 * Report error in module.
 * @param   pMod        Pointer to module to report error on.
 * @param   pszFormat   Format string.
 * @param   ...         Format arguments.
 */
static void     modError(PWLDMOD pMod, const char *pszFormat, ...)
{
    va_list     args;
    if (pMod->pLib)
        fprintf(stderr, "weakld: %s(%s) - error: ", pMod->pLib->pszLibName, pMod->pszModName);
    else
        fprintf(stderr, "weakld: %s - error: ", pMod->pszModName);

    va_start(args, pszFormat);
    vfprintf(stderr, pszFormat, args);
    va_end(args);
    if (pszFormat[strlen(pszFormat) - 1] != '\n')
        fputc('\n', stderr);
}


/**
 * Report warning in module.
 * @param   pMod        Pointer to module to report warning on.
 * @param   pszFormat   Format string.
 * @param   ...         Format arguments.
 */
static void     modWarn(PWLDMOD pMod, const char *pszFormat, ...)
{
    va_list     args;
    if (pMod->pLib)
        fprintf(stderr, "weakld: %s(%s) - warning: ", pMod->pLib->pszLibName, pMod->pszModName);
    else
        fprintf(stderr, "weakld: %s - warning: ", pMod->pszModName);

    va_start(args, pszFormat);
    vfprintf(stderr, pszFormat, args);
    va_end(args);
    if (pszFormat[strlen(pszFormat) - 1] != '\n')
        fputc('\n', stderr);
}



/**
 * Opens the module (if required) file for reading.
 *
 * @returns Pointer to file stream.
 * @param   pMod    Module to open.
 */
static FILE *   modOpen(PWLDMOD pMod)
{
    const char *pszFilename;

    /* open the file */
    if (!pMod->phFile)
    {
        if (pMod->pLib && pMod->pLib->phFile)
            pMod->phFile = pMod->pLib->phFile;
        else
        {   /* fopen it */
            if (pMod->pLib)
            {
                pszFilename = pMod->pLib->pszLibName;
                pMod->phFile = pMod->pLib->phFile = fopen(pszFilename, "rb");
            }
            else
            {
                pszFilename = pMod->pszModName;
                pMod->phFile = fopen(pszFilename, "rb");
            }
        }
    }

    /* Position the stream at the start of the module. */
    if (!pMod->phFile)
        modErrro(pMod, "failed to reopen.");
    else
    {
        if (fseek(pMod->phFile, pMod->off, SEEK_SET))
        {
            modErrro(pMod, "failed to seek to module start (%#x).", pMod->off);
            modClose(pMod);
            return NULL;
        }
    }
        
    return pMod->phFile;
}


/**
 * Closes the module.
 *
 * @param   pMod    Module to close.
 */
static void     modClose(PWLDMOD pMod)
{
    if (!pMod->phFile)
        return;
    if (!pMod->pLib || pMod->pLib->phFile != pMod->phFile)
        fclose(pMod->phFile);
    pMod->phFile = NULL;
}


/**
 * Reads an OMF module from a file.
 *
 * This may be part of a library file so, we'll only read from THEADR to 
 * past the first MODEND or the Pass 1 comment record.
 * The function will return the module stream positioned after the last 
 * record it read.
 *
 * @returns 0 on success.
 * @returns non zero on failure.
 * @param   pWld    Pointer to linker instance.
 * @param   pMod    Pointer to module
 */
static unsigned     pass1ReadOMFMod(PWLD pWld, PWLDMOD pMod)
{
    FILE *          phFile;             /* Input file. */
    PWLDSYM         papExts[] = NULL;   /* Pointer to an array of EXTDEFs (as they appear) */
                                        /* We need them for the WKEXT processing. */
    int             cExts = 0;          /* Number of Entries in papExts. */
    int             fFirst = 1;         /* First record indicator. */
    /* generic stuff we'll use alot with not status associated. */
    int             cch;                
    PWLDSYM         pSym;
    PWLDSYM         pSym2;



    /* open and position the file. */
    phFile = modOpen(pMod);

    /* loop till we get a MODEND */
    for (;;)
    {
        OMFREC          OmfRec;             
        unsigned char   achBuffer[OMF_MAX_REC + 8];
        union 
        {
            unsigned char *     puch;
            signed char *       pch;
            unsigned short *    pus;
            signed short *      ps;
            unsigned long *     pul;
            signed long *       pl;
        } u, u1, u2, u3;
        /** macro for getting a OMF index out of the buffer */
        #define OMF_GETINDEX()  (*u.puch & 0x80 ? ((*u.pch++ & 0x7f) << 8) + *u.pch++ : *u.pch++)
        #define OMF_BYTE()      (*u.puch++)
        #define OMF_WORD()      (*u.pus++)
        #define OMF_MORE()      (u.puch - &achBuffer[0] < (int)OmfRec.cb - 1)

        u.pv = &achBuffer[0];

        /* read omf record header */
        if (fread(&OmfRec, sizeof(OmfRec), 1, phFile) != 1)
        {
            modError(pMod, "read error. (offset ~= %#x).", ftell(phFile));
            goto failure;
        }
        if (fFirst)
        {   /* some extra check for the first record. */
            fFirst = 0;
            if (OmfRec.chType != THEADR)
            {
                modError(pMod, "invalid object module (offset %#x).", pMod->off);
                goto failure;
            }
        }
    
        /* Read or skip the record. */
        switch (OmfRec.chType)
        {
            /* done */
            case MODEND: case MODEND | REC32:
            case LIBEND: 
                fseek(phFile, OmfRec.cb, SEEK_CUR);
                goto done_skip;
            /* read */
            case EXTDEF: case EXTDEF | REC32: 
            case PUBDEF: case PUBDEF | REC32: 
            case ALIAS:  case ALIAS  | REC32:
            case COMDEF: case COMDEF | REC32: 
            case COMDAT: case COMDAT | REC32: 
            case COMENT: case COMENT | REC32: 
            case LIBHDR: 
                break;
            /* skip */
            default:
                fseek(phFile, OmfRec.cb, SEEK_CUR);
                continue;
        }
        if (fread(achBuffer, OmfRec.cb, 1, phFile) != 1)
        {
            modError(pMod, "read error. (offset ~= %#x)", ftell(phFile));
            goto failure;             
        }

        /* Switch on type. */
        switch (OmfRec.chType)
        {
            case COMENT: case COMENT | REC32: 
                switch (*++u.pch)
                {
                    case CLASS_PASS:
                         goto done_noskip;
                    case CLASS_WKEXT:
                    {   /* This is a bit tricky, we need to have an indexable array 
                         * of the extdefs for this module. In addition we'll need to 
                         * make sure we don't mark an EXTDEF from another module as
                         * weak. 
                         */
                        while (OMF_MORE())
                        {
                            int iWeak = OMF_GETINDEX();
                            int iDefault = OMF_GETINDEX();
                            if (   iWeak     >= cExts 
                                || iDefault  >= cExts
                                || !papExts[iWeak]
                                || !papExts[iDefault])
                                modError(pMod, "Invalid WKEXT record.");
                            if (papExts[iWeak]->eFlags & WLDSF_UNCERTAIN)
                            {
                                papExts[iWeak]->eFlags = (papExts[iWeak]->eFlags & ~WLDSF_STATEMASK) | WLDSF_WKEXT;
                                papExts[iWeak]->pWeakDefault = papExts[iDefault];
                            }
                            else if (   (papExts[iWeak]->eFlags & WLDSF_STATEMASK) == WLDSF_WKEXT
                                     &&  papExts[iWeak]->pWeakDefault != papExts[iDefault])
                                modWarn(pMod, "WKEXT '%s' allready declared with '%s' and not '%s' as default.", 
                                        papExts[iWeak]->pszName, papExts[iWeak]->pWeakDefault->pszName, papExts[iDefault]->pszName);
                        }
                        break;
                    }
                    
                    case CLASS_OMFEXT:
                    {
                        switch (OMF_BYTE())
                        {   /*
                             * Import definition.
                             */
                            case OMFEXT_IMPDEF:
                            {
                                int fOrd = OMF_BYTE();
                                cch = OMF_BYTE();
                                pSym = symNew(pWld, WLDSF_STRONG | WLDSF_IMPORT, u.pch, cch);
                                if (!pSym) goto failure;
                                u.pch += cch;
                                cch = OMF_BYTE();
                                pSym->pszImpMod = strpool_addn(pWld->pStrMisc, u.pch, cch);
                                u.pch += cch;
                                if (fOrd)
                                    pSym->iImpOrd = OMF_WORD();
                                break;
                            }

                            /* 
                             * Export definition.
                             * If it have an internal name the exported name will become
                             * an alias record.
                             */
                            case OMFEXT_EXPDEF:
                            {
                                u1 = u; u.pch++;
                                u2 = u; u.pch+ = 1 + *u2.pch;
                                u3 = u; u.pch+ = 1 + *u3.pch;
                                
                                pSym2 = NULL;
                                if (*u3.pch)
                                {
                                    pSym2 = symNew(pWld, WLDSF_UNDEF, u3.pch + 1, *u3.pch);
                                    if (!pSym2) goto failure;
                                    pSym = symNew(pWld, WLDSF_STRONG | WLDSF_ALIAS | WLDSF_EXPORT, u2.pch + 1, *u2.pch);
                                    if (!pSym) goto failure;
                                    pSym->pAliasFor = pSym2;
                                }
                                else
                                {
                                    pSym = symNew(pWld, WLDSF_UNDEF | WLDSF_EXPORT, u2.pch + 1, *u2.pch);
                                    if (!pSym) goto failure;
                                }
                                if (*u1.pch & 0x80)
                                    pSym->iExpOrd = GET_BYTE();
                                if (*u1.pch & 0x40)
                                    pSym->fExport.fResidentName = 1;
                                

                                break;
                            }
                        }

                    }

                }
                break;

            case EXTDEF: case EXTDEF | REC32:
            {

                break;
            }

            case PUBDEF: case PUBDEF | REC32: 
            {
                break;
            }

            case ALIAS: case ALIAS | REC32:
            {
                break;
            }

            case COMDEF: case COMDEF | REC32: 
            {
                break;
            }

            case COMDAT: case COMDAT | REC32: 
            {
                break;
            }

            case LIBHDR:
                modError(pMod, "invalid object module (LIBHDR).");
                goto failure;
            default:
                fprintf(stderr, "we shall not be here!!\n");
                asm ("int $3");
                break;
        }

        #undef OMF_GETINDEX
    }

done_skip:
    /* Skip to end of record */
    fseek(phFile, OmfRec.cb, SEEK_CUR);

done_noskip:
    /* Make all the EXTDEFs uncertain. */
    for (i = 0; i < cExts; i++)
        papExts[i]->eFlags &= ~WLDSF_UNCERTAIN;

    return 0;

failure:
    if (papExts)
        free(papExts);
    return -1;
}




/*******************************************************************************
*  
*  P u b l i c   I n t e r f a c e
*  P u b l i c   I n t e r f a c e
*  P u b l i c   I n t e r f a c e
*  
*******************************************************************************/




/**
 * Creates an instance of the linker.
 *
 * @returns Pointer to linker instance.
 * @param   fFlags  Linker flags as defined by enum wld_create_flags.
 */
PWLD    wld_create(int fFlags)
{
    PWLD    pWld = xmalloc(sizeof(*pWld));
    
    /* initiate it */
    memset(pWld, 0, sizeof(*pWld));
    pWld->fFlags = fFlags;
    pWld->pStrMisc = strpool_init();
    pWld->ppLibsAdd = &pWld->pObjs;
    pWld->ppObjsAdd = &pWld->pLibs;

    /* done */
    if (fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: linker created\n");
    return pWld;
}

/**
 * Destroys a linker cleaning up open files and memory.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 */
int     wld_destroy(PWLD pWld)
{
    if (!pWld)
        return 0;
    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: destroying linker instance\n");

    /* members and finally the instance structure. */
    strpool_free(pWld->pStrMisc);

    free(pWld);
    return 0;
}


/**
 * Adds a object module to the linking process.
 * The object module will be analysed and the file handle closed.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 * @param   phFile  File handle to pszName. 
 * @param   pszName Object file to add. This may be an OMF library too,
 *                  but that only means that all it's content is to be
 *                  linked in as if they have been all specified on the 
 *                  commandline.
 * @remark  Don't call wld_add_object after wld_add_library() 
 *          or wld_generate_weaklib()!
 */
int     wld_add_object(PWLD pWld, FILE *phFile, const char *pszName)
{
    OMFREC  OmfRec = {0,0};
    int     rc = 0;
    if (!phFile)
        phFile = fopen(pszName, "rb");
    if (!phFile)
    {
        fprintf(stderr, "weakld: cannot open object file '%s'.\n", pszName);
        return 8;
    }
    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: adding object %s.\n", pszName);

    /*
     * An object module is either a object or a library.
     * In anycase all the modules it contains is to be added to the link.
     */
    fread(&OmfRec, sizeof(OmfRec), 1, phFile);
    if (OmfRec.chType == THEADR)
    {
        /* Single Object */
        PWLDMOD pMod = xmalloc(sizeof(*pMod));
        memset(pMod, 0, sizeof(*pMod));
        pMod->pszModName = strpool_add(pWld->pStrMisc, pszName);
        pMod->phFile = phFile;
        *pWld->ppObjsAdd = pMod;
        pWld->ppObjsAdd = &pMod->pNext;
        rc = pass1ReadOMFMod(pWld, pMod);
    }
    else if (OmfRec.chType == LIBHDR)
    {
        /* Library of object modules */
        while (OmfRec.chType != LIBEND && OmfRec.chType != (LIBEND | REC32))
        {
            if (OmfRec.chType == THEADR || OmfRec.chType == (THEADR | REC32))
            {
                PWLDMOD     pMod = xmalloc(sizeof(*pMod));
                memset(pMod, 0, sizeof(*pMod));
                pMod->pszModName = strpool_add(pWld->pStrMisc, pszName);
                pMod->phFile = phFile;
                pMod->off = ftell(phFile) - sizeof(OmfRec);
                *pWld->ppObjsAdd = pMod;
                pWld->ppObjsAdd = &pMod->pNext;
                if (pWld->fFlags & WLDC_VERBOSE)
                {
                    char            achModName[256];
                    unsigned char   cchModName;
                    cchModName = fgetc(phFile);
                    achModName[fread(achModName, 1, cchModName, phFile)] = '\0';
                    fprintf(stderr, "weakld: info: adding library member '%s'.\n", achModName);
                }
                rc = pass1ReadOMFMod(pWld, pMod);
                if (rc)
                    break;
            }
            else
            {
                /* skip to the net record */
                fseek(phFile, OmfRec.cb, SEEK_CUR);
            }

            /* read next record */
            fread(&OmfRec, sizeof(OmfRec), 1, phFile);
        }
    }
    else
    {
        fprintf(stderr, "weakld: invalid object file '%s'.\n", pszName);
        rc = -1;
    }

    return rc;
}


/**
 * Adds an definition file to the linking process.
 * The definition file will be analysed and the file handle closed.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 * @param   phFile  File handle to pszName. If NULL we'll try open it.
 *                  The handle may be closed on exit, or latest when
 *                  we destroy the linker.
 * @param   pszName Definition file to add. 
 * @remark  Don't call wld_add_deffile after wld_add_library() 
 *          or wld_generate_weaklib()!
 */
int     wld_add_deffile(PWLD pWld, FILE *phFile, const char *pszName)
{
    if (!phFile)
        phFile = fopen(pszName, "r");
    if (!phFile)
    {
        fprintf(stderr, "weakld: cannot open deffile file '%s'.\n", pszName);
        return 8;
    }
    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: adding deffile %s\n", pszName);


    return 0;
}


/**
 * Adds a library module to the linking process.
 * The library module will be processed and the file handle eventually closed.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 * @param   phFile  File handle to pszName. If NULL we'll try open it.
 *                  The handle may be closed on exit, or latest when
 *                  we destroy the linker.
 * @param   pszName Library file to add. This may be an OMF object too.
 * @author  Don't call wld_add_library after wld_generate_weaklib()!
 */
int     wld_add_library(PWLD pWld, FILE *phFile, const char *pszName)
{
    if (!phFile)
        phFile = fopen(pszName, "r");
    if (!phFile)
    {
        fprintf(stderr, "weakld: cannot open library file '%s'.\n", pszName);
        return 8;
    }
    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: adding library %s\n", pszName);

    return 0;
}


/**
 * Generates a object file containing alias for resolving the weak 
 * symbol references in the linked executable.
 * The filename of the object will be generated, but of course it
 * ends with '.obj'.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 * @param   pszName Where to put the name of the generated object file.
 *                  This is an empty string if no weak symbols were found!
 */
int     wld_generate_weakobj(PWLD pWld, char *pszName)
{
    char *      psz;

    *pszName = '\0';
    psz = _tempnam(NULL, "weako");
    if (psz)
    {
        strcpy(pszName, psz);
        free(psz);
    }
    else
        tmpnam(pszName);
    strcat(pszName, "wk.obj");

    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: generating weakobj object file '%s'.\n", pszName);

    *pszName = '\0';
    return 0;

}

