/* $Id: weakld.c 485 2003-07-30 18:10:01Z bird $
 *
 * Weak Symbol Pre-Linker.
 *
 * Copyright (c) 2003 InnoTek Systemberatung GmbH
 * Author: knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with This program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/** @page weakld        Weak Pre-Linker
 * 
 * In order to get the weak symbols somewhat right it looks like we have to do
 * the pass1 of the linking process in order to resolve the weak symbols.
 *
 *
 *
 * @subsection          Symbols
 *
 * There is a couple of symbol types, but we can skip most of them for this 
 * pre-linking operation. We use one symbol type which is public or global
 * symbols if you like. Perhaps it would be wise to devide them into separat
 * type groups, but the choice was made to differenciate this using flags.
 * So, symbol enumeration is done using flag masks.
 *
 *
 */

/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
#define WLDSYM_HASH_SIZE    211


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include "defs.h"
#include "grow.h"
#include "weakld.h"
 
/*******************************************************************************
*   Structures and Typedefs                                                    *
*******************************************************************************/
/**
 * Library structure 
 */
typedef struct wldlib
{
    /** Library name. */
    const char *pszLibName;
    /** Filehandle if open */
    FILE *      phFile;
} WLDLIB, *PWLDLIB;
 

/**
 * Module structure 
 */    
typedef struct wldmod
{
    /** Module name. */
    const char *pszModName;
    /** Filehandle if open. */
    FILE *      phFile;

    /** Library relation. */
    PWLDLIB     pLib;
    /** Library offset. */
    off_t       offLib;
} WLDMOD, *PWLDMOD;
 

/**
 * Symbol structure.
 */
typedef struct wldsym
{
    /** Symbol name. */
    const char *        pszName;
    /** Weak name - for weak symbols only. */
    const char *        pszWeakName;

    /** Symbol type. */
    enum {
        WLDST_PUBLIC
        }               eType;

    /** Symbol flags. */
    enum {
        /* Strong symbol. */
        WLDSF_STRONG    = 0,
        /** Weak symbol, can become strong. */
        WLDSF_WEAK      = 1,
        /** Undefined symbol, can become strong. */
        WLDSF_UNDEF     = 2,
        /** Communal data/code, become strong. */
        WLDSF_COMM      = 4,
        /** Import symbol. */
        WLDSF_IMPORT    = 8,
        /** Exported symbol. */
        WLDSF_EXPORT    = 16,

    }                   eFlags;

    /** The module this symbol is defined in. */
    PWLDMOD             pMod;

    /** Import module - only if WLDSF_IMPORT is set. */
    const char*         pszImpMod;

    /** Symbol this is an alias for. */
    struct wldsym *     pAliasFor;

    /** Next node in the hash bucket. */
    struct wldsym *     pHashNext;
} WLDSYM, *PWLDSYM;


/**
 * Symbol table.
 */
typedef struct wldsymtab
{
    PWLDSYM     ap[WLDSYM_HASH_SIZE];
} WLDSYMTAB, *PWLDSYMTAB;

 
/** 
 * Weak Pre-Linker Instance.
 */
struct wld
{
    /** Linker flags. */
    unsigned            fFlags;

    /** Global symbols. */
    WLDSYMTAB           Global;

    /** string pool for miscellaneous string. */
    struct strpool *    pStrMisc;
};
typedef struct wld WLD;
 


extern void *xrealloc (void *ptr, size_t n);
extern void *xmalloc (size_t n);




/*******************************************************************************
*  
*  H e l p e r s
*  H e l p e r s
*  H e l p e r s
*  
*******************************************************************************/

/** 
 * Calculate the hash value of a symbol.
 * @returns hash value.
 * @param   pszSym  Symbol to calculate it for.
 * @param   cch     Symbol length.
 */
static unsigned     hash(const char* pszSym, unsigned cch)
{
    unsigned uHash = 0;
    while (cch && *pszSym && *pszSym != '$')
    {
        uHash = uHash * 65599 + *pszSym;
        pszSym++;
        cch--;
    }
    uHash %= WLDSYM_HASH_SIZE;
    return uHash;  
}




/*******************************************************************************
*  
*  P u b l i c   I n t e r f a c e
*  P u b l i c   I n t e r f a c e
*  P u b l i c   I n t e r f a c e
*  
*******************************************************************************/




/**
 * Creates an instance of the linker.
 *
 * @returns Pointer to linker instance.
 * @param   fFlags  Linker flags as defined by enum wld_create_flags.
 */
PWLD    wld_create(int fFlags)
{
    PWLD    pWld = xmalloc(sizeof(*pWld));
    
    /* initiate it */
    memset(pWld, 0, sizeof(*pWld));
    pWld->fFlags = fFlags;
    pWld->pStrMisc = strpool_init();

    /* done */
    if (fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: linker created\n");
    return pWld;
}

/**
 * Destroys a linker cleaning up open files and memory.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 */
int     wld_destroy(PWLD pWld)
{
    if (!pWld)
        return 0;
    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: destroying linker instance\n");

    /* members and finally the instance structure. */
    strpool_free(pWld->pStrMisc);

    free(pWld);
    return 0;
}


/**
 * Adds a object module to the linking process.
 * The object module will be analysed and the file handle closed.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 * @param   phFile  File handle to pszName. 
 * @param   pszName Object file to add. This may be an OMF library too,
 *                  but that only means that all it's content is to be
 *                  linked in as if they have been all specified on the 
 *                  commandline.
 * @remark  Don't call wld_add_object after wld_add_library() 
 *          or wld_generate_weaklib()!
 */
int     wld_add_object(PWLD pWld, FILE *phFile, const char *pszName)
{
    if (!phFile)
        phFile = fopen(pszName, "rb");
    if (!phFile)
    {
        fprintf(stderr, "weakld: cannot open object file '%s'.\n", pszName);
        return 8;
    }
    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: adding object %s\n", pszName);

    /* @todo process it! */

    return 0;
}


/**
 * Adds an definition file to the linking process.
 * The definition file will be analysed and the file handle closed.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 * @param   phFile  File handle to pszName. If NULL we'll try open it.
 *                  The handle may be closed on exit, or latest when
 *                  we destroy the linker.
 * @param   pszName Definition file to add. 
 * @remark  Don't call wld_add_deffile after wld_add_library() 
 *          or wld_generate_weaklib()!
 */
int     wld_add_deffile(PWLD pWld, FILE *phFile, const char *pszName)
{
    if (!phFile)
        phFile = fopen(pszName, "r");
    if (!phFile)
    {
        fprintf(stderr, "weakld: cannot open deffile file '%s'.\n", pszName);
        return 8;
    }
    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: adding deffile %s\n", pszName);


    return 0;
}


/**
 * Adds a library module to the linking process.
 * The library module will be processed and the file handle eventually closed.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 * @param   phFile  File handle to pszName. If NULL we'll try open it.
 *                  The handle may be closed on exit, or latest when
 *                  we destroy the linker.
 * @param   pszName Library file to add. This may be an OMF object too.
 * @author  Don't call wld_add_library after wld_generate_weaklib()!
 */
int     wld_add_library(PWLD pWld, FILE *phFile, const char *pszName)
{
    if (!phFile)
        phFile = fopen(pszName, "r");
    if (!phFile)
    {
        fprintf(stderr, "weakld: cannot open library file '%s'.\n", pszName);
        return 8;
    }
    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: adding library %s\n", pszName);

    return 0;
}


/**
 * Generates a object file containing alias for resolving the weak 
 * symbol references in the linked executable.
 * The filename of the object will be generated, but of course it
 * ends with '.obj'.
 *
 * @returns 0 on success.
 * @returns some unexplainable randomly picked number on failure.
 * @param   pWld    Linker instance to destroy.
 * @param   pszName Where to put the name of the generated object file.
 *                  This is an empty string if no weak symbols were found!
 */
int     wld_generate_weakobj(PWLD pWld, char *pszName)
{
    char *      psz;

    *pszName = '\0';
    psz = _tempnam(NULL, "weako");
    if (psz)
    {
        strcpy(pszName, psz);
        free(psz);
    }
    else
        tmpnam(pszName);
    strcat(pszName, "wk.obj");

    if (pWld->fFlags & WLDC_VERBOSE)
        fprintf(stderr, "weakld: info: generating weakobj object file '%s'.\n", pszName);

    *pszName = '\0';
    return 0;

}

