/*
 * Legalesy-free Unicode API interface for OS/2
 * Interface definitions for basic Unicode API functions
 *
 * Written by Andrew Zabolotny <bit@eltech.ru>
 *
 * This file is put into public domain. You are free to do
 * literally anything you wish with it: modify, print, sell,
 * rent, eat, throw out of window: in all (esp. in later)
 * cases I am not responsible for any damage it causes.
 */

#ifndef __UNIDEF_H__
#define __UNIDEF_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stddef.h>
#include <time.h>

#include <ulserrno.h>
#include <ulsitem.h>

typedef wchar_t UniChar;
typedef void *LocaleObject;
typedef unsigned int LocaleToken;
typedef void *AttrObject;
typedef void *XformObject;
typedef int ulsBool;

#ifndef TRUE
# define TRUE	1
#endif
#ifndef FALSE
# define FALSE	0
#endif
#ifndef APIENTRY
#define APIENTRY _System
#endif

#define UNI_TOKEN_POINTER	1
#define UNI_MBS_STRING_POINTER	2
#define UNI_UCS_STRING_POINTER	3

/* Locale categories */
#undef LANG
#undef LC_ALL
#undef LC_COLLATE
#undef LC_CTYPE
#undef LC_NUMERIC
#undef LC_MONETARY
#undef LC_TIME
#undef LC_MESSAGES

#define LANG			(-2)
#define LC_ALL			(-1)
#define LC_COLLATE		0
#define LC_CTYPE		1
#define LC_NUMERIC		2
#define LC_MONETARY		3
#define LC_TIME			4
#define LC_MESSAGES		5

/* Locale conventions structure */
typedef struct UniLconv
{
  UniChar *decimal_point;	/* non-monetary decimal point */
  UniChar *thousands_sep;	/* non-monetary thousands separator */
  short   *grouping;		/* non-monetary size of grouping */
  UniChar *int_curr_symbol;	/* int'l currency symbol and separator */
  UniChar *currency_symbol;	/* local currency symbol */
  UniChar *mon_decimal_point;	/* monetary decimal point */
  UniChar *mon_thousands_sep;	/* monetary thousands separator */
  short   *mon_grouping;	/* monetary size of grouping */
  UniChar *positive_sign;	/* non-negative values sign */
  UniChar *negative_sign;	/* negative values sign */
  short   int_frac_digits;	/* no of fractional digits int currency */
  short   frac_digits;		/* no of fractional digits loc currency */
  short   p_cs_precedes;	/* nonneg curr sym 1-precedes,0-succeeds */
  short   p_sep_by_space;	/* nonneg curr sym 1-space,0-no space */
  short   n_cs_precedes;	/* neg curr sym 1-precedes,0-succeeds */
  short   n_sep_by_space;	/* neg curr sym 1-space 0-no space */
  short   p_sign_posn;		/* positioning of nonneg monetary sign */
  short   n_sign_posn;		/* positioning of negative monetary sign */
  short   os2_mondecpt;		/* os2 curr sym positioning */
  UniChar *debit_sign;		/* non-neg-valued monetary sym - "DB" */
  UniChar *credit_sign;		/* negative-valued monetary sym - "CR" */
  UniChar *left_parenthesis;	/* negative-valued monetary sym - "(" */
  UniChar *right_parenthesis;	/* negative-valued monetary sym - ")" */
} UNILCONV;

/* Char/String Type (QueryCharType and UniQueryStringType). */
typedef struct UniCType {
    unsigned short  itype;
    char            bidi;
    char            charset;
    unsigned short  extend;
    unsigned short  codepage;
} UNICTYPE;


/* Locale Management Function Prototypes */

int APIENTRY UniCreateLocaleObject (int locale_spec_type, const void *locale_spec,
  LocaleObject *locale_object_ptr);
int APIENTRY UniQueryLocaleObject (const LocaleObject locale_object, int category,
  int locale_spec_type, void **locale_spec_ptr);
int APIENTRY UniFreeLocaleObject (LocaleObject locale_object);
int APIENTRY UniFreeMem (void *memory_ptr);
int APIENTRY UniLocaleStrToToken (int locale_string_type, const void *locale_string,
  LocaleToken *locale_token_ptr);
int APIENTRY UniLocaleTokenToStr (const LocaleToken locale_token,
  int locale_string_type, void **locale_string_ptr);

/* Locale Information Function Prototypes */
int APIENTRY UniQueryLocaleInfo (const LocaleObject locale_object,
  struct UniLconv **unilconv_addr_ptr);
int APIENTRY UniFreeLocaleInfo (struct UniLconv *unilconv_addr);
int APIENTRY UniQueryLocaleItem (const LocaleObject locale_object, LocaleItem item,
  UniChar **info_item_addr_ptr);

/* Date and Time Function Prototypes */
size_t APIENTRY UniStrftime (const LocaleObject locale_object, UniChar *ucs,
  size_t maxsize, const UniChar *format, const struct tm *time_ptr);
UniChar *APIENTRY UniStrptime (const LocaleObject locale_object, const UniChar *buf,
  const UniChar *format, struct tm *time_ptr);

/* Monetary Formatting Function Prototype */
int APIENTRY UniStrfmon (const LocaleObject locale_object, UniChar *ucs, size_t maxsize,
  const UniChar *format, ...);

/* String/Character Function Prototypes */
UniChar *APIENTRY UniStrcat (UniChar *ucs1, const UniChar *ucs2);
UniChar *APIENTRY UniStrchr (const UniChar *ucs, UniChar uc);
int APIENTRY UniStrcmp (const UniChar *ucs1, const UniChar *ucs2);
UniChar *APIENTRY UniStrcpy (UniChar *ucs1, const UniChar *ucs2);
size_t APIENTRY UniStrcspn (const UniChar *ucs1, const UniChar *ucs2);
size_t APIENTRY UniStrlen (const UniChar *ucs1);
UniChar *APIENTRY UniStrncat (UniChar *ucs1, const UniChar *ucs2, size_t n);
int APIENTRY UniStrncmp (const UniChar *ucs1, const UniChar *ucs2, size_t n);
UniChar *APIENTRY UniStrncpy (UniChar *ucs1, const UniChar *ucs2, size_t n);
UniChar *APIENTRY UniStrpbrk (const UniChar *ucs1, const UniChar *ucs2);
UniChar *APIENTRY UniStrrchr (const UniChar *ucs, UniChar uc);
size_t APIENTRY UniStrspn (const UniChar *ucs1, const UniChar *ucs2);
UniChar *APIENTRY UniStrstr (const UniChar *ucs1, const UniChar *ucs2);
UniChar *APIENTRY UniStrtok (UniChar *ucs1, const UniChar *ucs2);

/* Character Attribute Function Prototypes */
int APIENTRY UniCreateAttrObject (const LocaleObject locale_object,
  const UniChar *attr_name, AttrObject *attr_object_ptr);
int APIENTRY UniQueryCharAttr (AttrObject attr_object, UniChar uc);
int APIENTRY UniScanForAttr (AttrObject attr_object, const UniChar *ucs,
  size_t num_elems, ulsBool inverse_op, size_t *offset_ptr);
int APIENTRY UniFreeAttrObject (AttrObject attr_object);
int APIENTRY UniQueryAlnum (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryAlpha (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryBlank (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryCntrl (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryDigit (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryGraph (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryLower (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryPrint (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryPunct (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQuerySpace (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryUpper (const LocaleObject locale_object, UniChar uc);
int APIENTRY UniQueryXdigit (const LocaleObject locale_object, UniChar uc);

/* String Transformation Function Prototypes */
int APIENTRY UniCreateTransformObject (const LocaleObject locale_object,
  const UniChar *xtype, XformObject *xform_object_ptr);
int APIENTRY UniTransformStr (XformObject xform_object, const UniChar *inp_buf,
  int *inp_size, UniChar *out_buf, int *out_size);
int APIENTRY UniFreeTransformObject (XformObject xform_object);
UniChar APIENTRY UniTransLower (const LocaleObject locale_object, UniChar uc);
UniChar APIENTRY UniTransUpper (const LocaleObject locale_object, UniChar uc);

/* String Conversion Function Prototypes */
int APIENTRY UniStrtod (const LocaleObject locale_object, const UniChar *ucs,
  UniChar **end_ptr, double *result_ptr);
int APIENTRY UniStrtol (const LocaleObject locale_object, const UniChar *ucs,
  UniChar **end_ptr, int base, long int *result_ptr);
int APIENTRY UniStrtoul (const LocaleObject locale_object, const UniChar *ucs,
  UniChar **end_ptr, int base, unsigned long int *result_ptr);

/* String Comparison Function Prototypes */
int APIENTRY UniStrcoll (const LocaleObject locale_object,
  const UniChar *ucs1, const UniChar *ucs2);
size_t APIENTRY UniStrxfrm (const LocaleObject locale_object, UniChar *ucs1,
  const UniChar *ucs2, size_t n);
int APIENTRY UniStrcmpi (const LocaleObject locale_object,
  const UniChar *ucs1, const UniChar *ucs2);
int APIENTRY UniStrncmpi (const LocaleObject locale_object,
  const UniChar *ucs1, const UniChar *ucs2, const size_t n);

/* Unicode Case Mapping Function Prototypes */
UniChar APIENTRY UniToupper (UniChar uc);
UniChar APIENTRY UniTolower (UniChar uc);
UniChar *APIENTRY UniStrupr (UniChar *ucs);
UniChar *APIENTRY UniStrlwr (UniChar *ucs);

int APIENTRY UniMapCtryToLocale (unsigned long Country, UniChar *LocaleName, size_t n);

/* Locale independent character classification. */
int             APIENTRY UniQueryChar (UniChar uc, unsigned long attr);
unsigned long   APIENTRY UniQueryAttr (UniChar * name);
unsigned long   APIENTRY UniQueryStringType (UniChar * ustr, int size, unsigned short *outstr, int kind);
struct UniCType*APIENTRY UniQueryCharType (UniChar uchr);
unsigned long   APIENTRY UniQueryCharTypeTable (unsigned long * count, struct UniCType **unictype);
int             APIENTRY UniQueryNumericValue (UniChar uc);

/* Functions for user locals designed to be used by WPShell (local object). */
int APIENTRY UniSetUserLocaleItem (UniChar * locale, int item, int type, void * value);
int APIENTRY UniMakeUserLocale (UniChar * name, UniChar * basename);
int APIENTRY UniDeleteUserLocale (UniChar * locale);
int APIENTRY UniCompleteUserLocale (void);
int APIENTRY UniQueryLocaleList (int, UniChar *, int);
int APIENTRY UniQueryLanguageName (UniChar *lang, UniChar *isolang, UniChar **infoitem);
int APIENTRY UniQueryCountryName (UniChar *country, UniChar *isolang, UniChar **infoitem);


#ifdef __cplusplus
}
#endif

#endif /* __UNIDEF_H__ */
