/* $Id: b_dir.c 2316 2005-08-29 03:35:39Z bird $ */
/** @file
 *
 * LIBC SYS Backend - Directory Access.
 *
 * Copyright (c) 2005 knut st. osmundsen <bird@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_BASE
#define INCL_FSMACROS
#include <os2emx.h>
#include "b_fs.h"
#include "b_dir.h"
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <sys/fcntl.h>
#include <emx/umalloc.h>
#include <emx/io.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_BACK_IO
#include <InnoTekLIBC/logstrict.h>


/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
static int dirClose(__LIBC_PFH pFH, int fh);
static int dirRead(__LIBC_PFH pFH, int fh, void *pvBuf, size_t cbRead, size_t *pcbRead);
static int dirWrite(__LIBC_PFH pFH, int fh, const void *pvBuf, size_t cbWrite, size_t *pcbWritten);
static int dirDuplicate(__LIBC_PFH pFH, int fh, int *pfhNew);
static int dirFileControl(__LIBC_PFH pFH, int fh, int iRequest, int iArg, int *prc);
static int dirIOControl(__LIBC_PFH pFH, int fh, int iIOControl, int iArg, int *prc);
static int dirSelect(int cFHs, struct fd_set *pRead, struct fd_set *pWrite, struct fd_set *pExcept, struct timeval *tv, int *prc);
static int dirForkChild(__LIBC_PFH pFH, int fh, __LIBC_PFORKHANDLE pForkHandle, __LIBC_FORKOP enmOperation);


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/**
 * Directory file handle operations.
 */
static const __LIBC_FHOPS gDirOps =
{
    enmFH_Directory,
    dirClose,
    dirRead,
    dirWrite,
    dirDuplicate,
    dirFileControl,
    dirIOControl,
    dirSelect,
    NULL,
    dirForkChild
};


/**
 * Close operation.
 *
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 */
static int dirClose(__LIBC_PFH pFH, int fh)
{
    __LIBC_PFHDIR pFHDir = (__LIBC_PFHDIR)pFH;
    LIBCLOG_ENTER("pFH=%p:{.hDir=%lx} fh=%d\n", (void *)pFH, pFHDir->hDir, fh);

    if (pFHDir->hDir != HDIR_CREATE)
    {
        FS_VAR_SAVE_LOAD();
        int rc = DosFindClose(pFHDir->hDir);
        LIBC_ASSERTM(rc == NO_ERROR, "DosFindClose(%lx) -> %d\n", pFHDir->hDir, rc); (void)rc;
        pFHDir->hDir = HDIR_CREATE;
        FS_RESTORE();
    }
    if (pFHDir->uBuf.pv)
    {
        free(pFHDir->uBuf.pv);
        pFHDir->uBuf.pv = NULL;
    }
    if (pFHDir->Core.pszNativePath)
    {
        free(pFHDir->Core.pszNativePath);
        pFHDir->Core.pszNativePath = NULL;
    }

    LIBCLOG_RETURN_INT(0);
}


/**
 * Read operation.
 *
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   pvBuf       Pointer to the buffer to read into.
 * @param   cbRead      Number of bytes to read.
 * @param   pcbRead     Where to store the count of bytes actually read.
 */
static int dirRead(__LIBC_PFH pFH, int fh, void *pvBuf, size_t cbRead, size_t *pcbRead)
{
    /** @todo */
    return -ENOSYS;
}


/** Write operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   pvBuf       Pointer to the buffer which contains the data to write.
 * @param   cbWrite     Number of bytes to write.
 * @param   pcbWritten  Where to store the count of bytes actually written.
 */
static int dirWrite(__LIBC_PFH pFH, int fh, const void *pvBuf, size_t cbWrite, size_t *pcbWritten)
{
    LIBCLOG_ENTER("pFH=%p fh=%d pvBuf=%p cbWrite=%d pcbWritten=%p\n", (void *)pFH, fh, pvBuf, cbWrite, (void *)pcbWritten);
    LIBCLOG_ERROR_RETURN_INT(-EOPNOTSUPP);
}


/** Duplicate handle operation.
 * @returns 0 on success, OS/2 error code on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   pfhNew      Where to store the duplicate filehandle.
 *                      The input value describe how the handle is to be
 *                      duplicated. If it's -1 a new handle is allocated.
 *                      Any other value will result in that value to be
 *                      used as handle. Any existing handle with that
 *                      value will be closed.
 */
static int dirDuplicate(__LIBC_PFH pFH, int fh, int *pfhNew)
{
    /** @todo */
    return -EOPNOTSUPP;
}


/** File Control operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   iRequest    Which file file descriptior request to perform.
 * @param   iArg        Argument which content is specific to each
 *                      iRequest operation.
 * @param   prc         Where to store the value which upon success is
 *                      returned to the caller.
 */
static int dirFileControl(__LIBC_PFH pFH, int fh, int iRequest, int iArg, int *prc)
{
    LIBCLOG_ENTER("pFH=%p fh=%d iRequest=%d iArg=%d prc=%p\n", (void *)pFH, fh, iRequest, iArg, (void *)prc);
    LIBCLOG_ERROR_RETURN_INT(-EOPNOTSUPP);
}


/** I/O Control operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   iIOControl  Which I/O control operation to perform.
 * @param   iArg        Argument which content is specific to each
 *                      iIOControl operation.
 * @param   prc         Where to store the value which upon success is
 *                      returned to the caller.
 */
static int dirIOControl(__LIBC_PFH pFH, int fh, int iIOControl, int iArg, int *prc)
{
    LIBCLOG_ENTER("pFH=%p fh=%d iIOControl=%d iArg=%d prc=%p\n", (void *)pFH, fh, iIOControl, iArg, (void *)prc);
    LIBCLOG_ERROR_RETURN_INT(-EOPNOTSUPP);
}


/** Select operation.
 * The select operation is only performed if all handles have the same
 * select routine (the main worker checks this).
 *
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   cFHs        Range of handles to be tested.
 * @param   pRead       Bitmap for file handles to wait upon to become ready for reading.
 * @param   pWrite      Bitmap for file handles to wait upon to become ready for writing.
 * @param   pExcept     Bitmap of file handles to wait on (error) exceptions from.
 * @param   tv          Timeout value.
 * @param   prc         Where to store the value which upon success is
 *                      returned to the caller.
 */
static int dirSelect(int cFHs, struct fd_set *pRead, struct fd_set *pWrite, struct fd_set *pExcept, struct timeval *tv, int *prc)
{
    LIBCLOG_ENTER("cFHs=%d pRead=%p pWrite=%p pExcept=%p tv=%p prc=%p\n",
                  cFHs, (void *)pRead, (void *)pWrite, (void *)pExcept, (void *)tv, (void *)prc);
    LIBCLOG_ERROR_RETURN_INT(-EOPNOTSUPP);
}


/** Fork notification - child context.
 * Only the __LIBC_FORK_OP_FORK_CHILD operation is forwarded atm.
 * If NULL it's assumed that no notifiction is needed.
 *
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH             Pointer to the handle structure to operate on.
 * @param   fh              It's associated filehandle.
 * @param   pForkHandle     The fork handle.
 * @param   enmOperation    The fork operation.
 */
static int dirForkChild(__LIBC_PFH pFH, int fh, __LIBC_PFORKHANDLE pForkHandle, __LIBC_FORKOP enmOperation)
{
    /** @todo reopen the handle, same as for duplicate! */
    return 0;
}



int __libc_Back_dirOpen(const char *pszPath);


/**
 * Opens a directory specified by a path already made native.
 *
 * @returns File handle (>=0) on success.
 * @returns negative error code (errno.h) on failure.
 * @param   pszNativePath       Pointer to the native path. The buffer must be on stack
 *                              and must have space for 4 extra bytes!
 * @param   fInUnixTree         Set if the native path is in the unix tree.
 * @param   fLibc               The LIBC open() flags.
 * @param   pStat               Pointer to the stat structure for the directory.
 */
int __libc_Back_dirOpenNative(char *pszNativePath, unsigned fInUnixTree, unsigned fLibc, struct stat *pStat)
{
    LIBCLOG_ENTER("pszNativePath=%p:{%s} fInUnixTree=%d fLibc=%#x pStat=%p\n", (void *)pszNativePath, pszNativePath, fInUnixTree, fLibc, (void *)pStat);

    /*
     * Validate input.
     */
    if (!S_ISDIR(pStat->st_mode))
        LIBCLOG_ERROR_RETURN_INT(-ENOTDIR);
    if ((fLibc & O_ACCMODE) != O_RDONLY)
        LIBCLOG_ERROR_RETURN_INT(-EISDIR);
    if ((fLibc & (O_EXCL | O_CREAT)) == (O_EXCL | O_CREAT))
        LIBCLOG_ERROR_RETURN_INT(-EEXIST);
    if (fLibc & (O_TRUNC | O_APPEND))
        LIBCLOG_ERROR_RETURN_INT(-EPERM);
    if (fLibc & (O_TRUNC | O_APPEND | O_SIZE))
        LIBCLOG_ERROR_RETURN_INT(-EINVAL);

    /*
     * Directory handles are binary, period.
     */
    fLibc &= ~O_TEXT;
    fLibc |= O_BINARY;

    /*
     * Setting up a temporary handle; allocate buffer and suchlike.
     */
    int             rc = -ENOMEM;
    __LIBC_FHDIR    Tmp;
    Tmp.hDir        =  HDIR_CREATE;
    Tmp.cbBuf       = 0xf800;
    Tmp.uBuf.pv     = _lmalloc(Tmp.cbBuf);
    if (!Tmp.uBuf.pv)
    {
        Tmp.cbBuf   = 0x1000;
        Tmp.uBuf.pv = _lmalloc(Tmp.cbBuf);
    }
    if (Tmp.uBuf.pv)
    {
        Tmp.uCur.pv = Tmp.uBuf.pv;
        Tmp.cFiles  = Tmp.cbBuf / 40;
#if OFF_MAX > LONG_MAX
        Tmp.fType   = __libc_gpfnDosOpenL ? FIL_QUERYEASIZEL : FIL_QUERYEASIZE;
        if (Tmp.fType == FIL_QUERYEASIZEL)  /* the L version is buggy!! Make sure there is enough space. */
            Tmp.cFiles = Tmp.cbBuf / sizeof(FILEFINDBUF4L);
#else
        pFD->fType  = FIL_QUERYEASIZE;
#endif
        Tmp.Core.pszNativePath = _hstrdup(pszNativePath);
        if (Tmp.Core.pszNativePath)
        {
            /*
             * Try perform the search (find everything!).
             */
            char   *psz = strchr(pszNativePath, '\0');
            psz[0] = '/';
            psz[1] = '*';
            psz[2] = '\0';
            bzero(Tmp.uBuf.pv, Tmp.cbBuf);
            FS_VAR_SAVE_LOAD();
            rc = DosFindFirst((PCSZ)pszNativePath,
                              &Tmp.hDir,
                              FILE_NORMAL | FILE_READONLY | FILE_HIDDEN | FILE_SYSTEM | FILE_DIRECTORY | FILE_ARCHIVED,
                              Tmp.uBuf.pv,
                              Tmp.cbBuf,
                              &Tmp.cFiles,
                              Tmp.fType);
            if (!rc)
            {
                int fh;
                __LIBC_PFH pFH;
                rc = __libc_FHAllocate(-1, fLibc, sizeof(__LIBC_FHDIR), &gDirOps, &pFH, &fh);
                if (!rc)
                {
                    pFH->Dev            = __libc_back_fsPathCalcInodeAndDev(pszNativePath, &pFH->Inode);
                    pFH->pFsInfo        = __libc_back_fsInfoObjByDev(pFH->Dev);
                    pFH->pszNativePath  = Tmp.Core.pszNativePath;

                    __LIBC_PFHDIR pFHDir = (__LIBC_PFHDIR)pFH;
                    pFHDir->hDir        = Tmp.hDir;
                    pFHDir->fType       = Tmp.fType;
                    pFHDir->uBuf.pv     = Tmp.uBuf.pv;
                    pFHDir->uCur.pv     = Tmp.uCur.pv;
                    pFHDir->cFiles      = Tmp.cFiles;
                    pFHDir->cbBuf       = Tmp.cbBuf;

                    FS_RESTORE();
                    LIBCLOG_MSG("pFHDir=%p:{.hDir=%#lx, .fType=%d, .cFiles=%ld, .cbBuf=%#x} fh=%d\n",
                                (void *)pFHDir, pFHDir->hDir, pFHDir->fType, pFHDir->cFiles, pFHDir->cbBuf, fh);
                    LIBCLOG_RETURN_INT(fh);
                }

                /* bailout */
                DosFindClose(Tmp.hDir);
            }
            FS_RESTORE();
            free(Tmp.Core.pszNativePath);
        }
        free(Tmp.uBuf.pv);
    }

    if (rc > 0)
        rc = -__libc_native2errno(rc);
    LIBCLOG_ERROR_RETURN_INT(rc);
}

