/* $Id: b_ioFileOpen.c 2313 2005-08-28 06:19:49Z bird $ */
/** @file
 *
 * LIBC SYS Backend - open.
 *
 * Copyright (c) 2003-2005 knut st. osmundsen <bird@innotek.de>
 * Copyright (c) 1992-1996 by Eberhard Mattes
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_DOSERRORS
#define INCL_FSMACROS
#include <os2emx.h>
#include "b_fs.h"
#include "b_dir.h"
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <share.h>
#include <sys/stat.h>
#include <emx/umalloc.h>
#include <emx/syscalls.h>
#include <emx/io.h>
#include "syscalls.h"
#include <InnoTekLIBC/backend.h>
#include <InnoTekLIBC/libc.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_BACK_IO
#include <InnoTekLIBC/logstrict.h>

#define SH_MASK 0x70

/**
 * Opens a file.
 *
 * @returns Filehandle to the opened file on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszFile     Path to the file.
 * @param   fFlags      Open flags.
 * @param   cbInitial   Initial filesize.
 * @param   Mode        The specified permission mask.
 * @param   fLibc       LIBC filehandle flags.
 * @param   ppFH        Where to store the LIBC filehandle structure which was created
 *                      for the opened file.
 */
int __libc_Back_ioFileOpen(const char *pszFile, int fFlags, off_t cbInitial, mode_t Mode, unsigned fLibc, PLIBCFH *ppFH)
{
    LIBCLOG_ENTER("pszFile=%s fFlags=%#x cbInitial=%lld Mode=0%o fLibc=%#x ppFH=%p\n",
                  pszFile, fFlags, cbInitial, Mode, fLibc, (void*)ppFH);
    FS_VAR();

    /*
     * The SH_COMPAT mode is weird and unless supported by the host we map it to SH_DENYNO.
     */
    if ((fFlags & SH_MASK) == SH_COMPAT)
        fFlags = (fFlags & ~SH_MASK) | SH_DENYNO;

    /*
     * Extract the access mode and sharing mode bits.
     */
    ULONG flOpenMode = fFlags & 0x77;

    /*
     * File O_NOINHERIT and O_SYNC.
     */
    if (fFlags & _SO_NOINHERIT)
        flOpenMode |= OPEN_FLAGS_NOINHERIT;
    if (fFlags & _SO_SYNC)
        flOpenMode |= OPEN_FLAGS_WRITE_THROUGH;

    /*
     * Extract the file flAttribute bits.
     */
    ULONG flAttr = (fFlags >> 8) & 0xff;
    if (__libc_gfsUMask & S_IWUSR)
        flAttr |= FILE_READONLY;

    /*
     * Translate ERROR_OPEN_FAILED to ENOENT unless O_EXCL is set (see below).
     */
    int rcOpenFailed = -ENOENT;

    /*
     * Compute `flOpenFlags' depending on `fFlags'.  Note that _SO_CREAT is
     * set for O_CREAT.
     */
    ULONG flOpenFlags;
    if (fFlags & _SO_CREAT)
    {
        if (fFlags & _SO_EXCL)
        {
            flOpenFlags = OPEN_ACTION_FAIL_IF_EXISTS | OPEN_ACTION_CREATE_IF_NEW;
            rcOpenFailed = -EEXIST;
        }
        else if (fFlags & _SO_TRUNC)
            flOpenFlags = OPEN_ACTION_REPLACE_IF_EXISTS | OPEN_ACTION_CREATE_IF_NEW;
        else
            flOpenFlags = OPEN_ACTION_OPEN_IF_EXISTS | OPEN_ACTION_CREATE_IF_NEW;
    }
    else if (fFlags & _SO_TRUNC)
        flOpenFlags = OPEN_ACTION_REPLACE_IF_EXISTS | OPEN_ACTION_FAIL_IF_NEW;
    else
        flOpenFlags = OPEN_ACTION_OPEN_IF_EXISTS | OPEN_ACTION_FAIL_IF_NEW;

    if (!(fFlags & _SO_SIZE))
        cbInitial = 0;

    /*
     * Resolve the specified file path.
     */
    char szNativePath[PATH_MAX + 5];
    int rc = __libc_back_fsResolve(pszFile,
                                   (flOpenFlags & OPEN_ACTION_FAIL_IF_NEW
                                        ? BACKFS_FLAGS_RESOLVE_FULL       | BACKFS_FLAGS_RESOLVE_DIR_MAYBE
                                        : BACKFS_FLAGS_RESOLVE_FULL_MAYBE | BACKFS_FLAGS_RESOLVE_DIR_MAYBE),
                                   &szNativePath[0],
                                   NULL);
    if (rc)
        LIBCLOG_ERROR_RETURN_INT(rc);

    /*
     * Create Unix attributes for a new file.
     */
    dev_t   Dev = 0;
    ino_t   Inode = 0;
    PEAOP2 pEaOp2 = NULL;
    if (    (flOpenFlags & (OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS))
        &&  !__libc_gfNoUnix)
    {
        Mode &= ~__libc_gfsUMask;
        Mode &= S_IRWXG | S_IRWXO | S_IRWXU | S_ISUID | S_ISGID | S_ISTXT | S_ISVTX;
        Mode |= S_IFREG;

        pEaOp2 = alloca(sizeof(EAOP2) + sizeof(__libc_gFsUnixAttribsCreateFEA2List));
        struct __LIBC_FSUNIXATTRIBSCREATEFEA2LIST *pFEas = (struct __LIBC_FSUNIXATTRIBSCREATEFEA2LIST *)(pEaOp2 + 1);
        *pFEas = __libc_gFsUnixAttribsCreateFEA2List;
        Dev = __libc_back_fsUnixAttribsInit(pFEas, szNativePath, Mode);
        Inode = pFEas->u64INO;
        pEaOp2->fpGEA2List = NULL;
        pEaOp2->fpFEA2List = (PFEA2LIST)pFEas;
        pEaOp2->oError     = 0;
    }

    /*
     * Try to open the file.
     */
    FS_SAVE_LOAD();
    ULONG   cExpandRetries;
    ULONG   ulAction;
    HFILE   hFile;
    for (cExpandRetries = 0;;)
    {
#if OFF_MAX > LONG_MAX
        if (__libc_gpfnDosOpenL)
        {
            LONGLONG cbInitialTmp = cbInitial;
            rc = __libc_gpfnDosOpenL((PCSZ)&szNativePath[0], &hFile, &ulAction, cbInitialTmp, flAttr, flOpenFlags, flOpenMode, pEaOp2);
        }
        else
        {
            ULONG cbInitialTmp = (ULONG)cbInitial;
            if (cbInitial > LONG_MAX)
            {
                FS_RESTORE();
                LIBCLOG_ERROR_RETURN_INT(-EOVERFLOW);
            }
            rc = DosOpen((PCSZ)&szNativePath[0], &hFile, &ulAction, cbInitialTmp, flAttr, flOpenFlags, flOpenMode, pEaOp2);
        }
#else
        {
            ULONG cbInitialTmp = cbInitial;
            rc = DosOpen((PCSZ)&szNativePath[0], &hFile, &ulAction, cbInitialTmp, flAttr, flOpenFlags, flOpenMode, pEaOp2);
        }
#endif
        /* Check for EA errors. */
        if (pEaOp2 && rc == ERROR_EAS_NOT_SUPPORTED)
        {
            pEaOp2 = NULL;
            continue;
        }

        /* Check if we're out of handles. */
        if (rc != ERROR_TOO_MANY_OPEN_FILES)
            break;
        if (cExpandRetries++ >= 3)
            break;
        __libc_FHMoreHandles();
    }   /* ... retry 3 times ... */

    if (!rc)
    {
        ULONG   fulType;
        ULONG   fulDevFlags;

        /*
         * Figure the handle type.
         */
        rc = DosQueryHType(hFile, &fulType, &fulDevFlags);
        if (!rc)
        {
            switch (fulType & 0xff)
            {
                default: /* paranoia */
                case HANDTYPE_FILE:
                    fLibc |= F_FILE;
                    /* If a new file the unix EAs needs to be established. */
                    if (    !pEaOp2
                        ||  (ulAction != FILE_CREATED && ulAction != FILE_TRUNCATED)) /** @todo validate that FILE_TRUNCATED will have replaced EAs. */
                        Dev = __libc_back_fsPathCalcInodeAndDev(szNativePath, &Inode);
                    break;
                case HANDTYPE_DEVICE:
                    fLibc |= F_DEV;
                    if (!(fulDevFlags & 0xf))
                        Dev = makedev('c', 0);
                    else if (fulDevFlags & 1 /*KBD*/)
                        Dev = makedev('c', 1);
                    else if (fulDevFlags & 2 /*SCR*/)
                        Dev = makedev('c', 2);
                    else if (fulDevFlags & 4 /*NUL*/)
                        Dev = makedev('c', 4);
                    else /*if (fulDevFlags & 8 / *CLK* /)*/
                        Dev = makedev('c', 8);
                    __libc_back_fsPathCalcInodeAndDev(szNativePath, &Inode);
                    break;
                case HANDTYPE_PIPE:
                    fLibc |= F_PIPE;
                    Dev = makedev('p', 0);
                    __libc_back_fsPathCalcInodeAndDev(szNativePath, &Inode);
                    break;
            }

            /*
             * Register the handle and calc Dev and Inode.
             */
            __LIBC_PFH pFH;
            rc = __libc_FHAllocate(hFile, fLibc, sizeof(LIBCFH), NULL, &pFH, NULL);
            if (!rc)
            {
                pFH->Inode = Inode;
                pFH->Dev = Dev;
                pFH->pFsInfo = __libc_back_fsInfoObjByDev(Dev);
                pFH->pszNativePath = _hstrdup(szNativePath);

                if (ppFH)
                    *ppFH = pFH;
                LIBCLOG_MSG("pFH=%p hFile=%#lx fLibc=%#x fulType=%#lx ulAction=%lu Dev=%#x Inode=%#llx\n",
                            (void *)pFH, hFile, fLibc, fulType, ulAction, Dev, Inode);
            }
        }

        if (__predict_false(rc != NO_ERROR))
            DosClose(hFile);
    }
#if 0  /** @todo Try open directories. */
    else if (   rc == ERROR_???
             && (fFlags & ...) == ...
             )
    {
        /*
         * Try open directory.
         */
        rc = __libc_Back_dirOpenNative(szNativePath, fLibc);
        FS_RESTORE();
        LIBCLOG_RETURN_INT(rc);
    }
#endif
    FS_RESTORE();

    /*
     * Handle any errors.
     */
    if (__predict_false(rc != NO_ERROR))
    {
        if (rc == ERROR_OPEN_FAILED)
            rc = rcOpenFailed;
        else
            rc = -__libc_native2errno(rc);
        LIBCLOG_ERROR_RETURN_INT(rc);
    }
    LIBCLOG_RETURN_INT((int)hFile);
}

