/* $Id: safesems.c 2294 2005-08-21 09:03:21Z bird $ */
/** @file
 *
 * LIBC SYS Backend - Internal Signal-Safe Semaphores.
 *
 * Copyright (c) 2005 knut st. osmundsen <bird@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#define INCL_DOSSEMAPHORES
#define INCL_DOSEXCEPTIONS
#define INCL_DOSERRORS
#define INCL_DOSPROCESS
#define INCL_FSMACROS
#define INCL_EXAPIS
#include <os2emx.h>
#include <sys/errno.h>
#include "syscalls.h"
#include <emx/umalloc.h>
#include <InnoTekLIBC/thread.h>
#include <InnoTekLIBC/backend.h>
#include <InnoTekLIBC/sharedpm.h>
#define __LIBC_LOG_GROUP    __LIBC_LOG_GRP_BACK_IPC
#include <InnoTekLIBC/logstrict.h>


/*******************************************************************************
*   Structures and Typedefs                                                    *
*******************************************************************************/
/**
 * Event semaphore tracking structure.
 *
 * The event semaphore business is difficult because the lack of 
 * atomic mutex release + event wait apis in OS/2. We have to 
 * jump around the place to get this working nearly safly...
 */      
typedef struct __LIBC_SAFESEMEV
{
    /** The event semaphore. */
    HEV                 hev;
    /** Number of threads which are supposed to be blocking on the above event semaphore. */
    uint32_t volatile   cWaiters;
    /** Number of processes using the semaphore. */
    uint32_t volatile   cUsers;
    /** Set if the semaphore is shared. 
     * If shared this structure is allocated from SPM, else it's from the heap. */
    unsigned            fShared;
#ifdef __LIBC_STRICT
    /** The mutex semaphore used to protect the event semaphore. 
     * Strict builds only. */
    uintptr_t           hmtx;
#endif
} __LIBC_SAFESEMEV, *__LIBC_PSAFESEMEV;


/**
 * Arguments to semEvSleepSignalCallback().
 */
struct SignalArgs
{
    /** Set if pfnComplete have already been executed. */
    int volatile            fDone;
    /** Number of attempts at doing it. (in case of crashes in pfnComplete) */
    int volatile            cTries;
    /** Callback to execute. */
    void                  (*pfnComplete)(void *pvUser);
    /** User arg. */
    void                   *pvUser;
    /** The old priority - to be restored when we're unblocked. */
    ULONG                   ulOldPri;
    /** Pointer to the semaphore structure. */
    __LIBC_PSAFESEMEV       pEv;
};


/**
 * Creates a safe mutex sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   phmtx       Where to store the semaphore handle.
 * @param   fShared     Set if the semaphore should be sharable between processes.
 */
int __libc_Back_safesemMtxCreate(uintptr_t *phmtx, int fShared)
{
    FS_VAR_SAVE_LOAD();
    HMTX hmtx = NULLHANDLE;
    int rc = DosCreateMutexSemEx(NULL, &hmtx, fShared ? DC_SEM_SHARED : 0, FALSE);
    FS_RESTORE();
    if (rc)
        return -__libc_native2errno(rc);
    *phmtx = hmtx;
    return 0;
}


/**
 * Opens a shared safe mutex sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hmtx        The semaphore handle.
 */
int __libc_Back_safesemMtxOpen(uintptr_t hmtx)
{
    FS_VAR_SAVE_LOAD();
    HMTX hmtxOS2 = hmtx;
    int rc = DosOpenMutexSemEx(NULL, &hmtxOS2);
    FS_RESTORE();
    if (rc)
        return -__libc_native2errno(rc);
    return 0;
}


/**
 * Closes a shared safe mutex sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hmtx        The semaphore handle.
 */
int __libc_Back_safesemMtxClose(uintptr_t hmtx)
{
    FS_VAR_SAVE_LOAD();
    int rc = DosCloseMutexSemEx(hmtx);
    FS_RESTORE();
    if (rc)
        return -__libc_native2errno(rc);
    return 0;
}


/**
 * This function checks that there is at least 2k of writable
 * stack available. If there isn't, a crash is usually the
 * result.
 * @internal
 */
static int stackChecker(void)
{
    char volatile *pch = alloca(2048);
    if (!pch)
        return -1;
    /* With any luck the compiler doesn't optimize this away. */
    pch[0] = pch[1024] = pch[2044] = 0x7f;
    return 0;
}


/**
 * Locks a mutex semaphore.
 *
 * @returns 0 on success.
 * @returns Negative errno on failure.
 * @param   hmtx    Handle to the mutex.
 */
int __libc_Back_safesemMtxLock(uintptr_t hmtx)
{
    LIBCLOG_ENTER("hmtx=%#x\n", hmtx);
    ULONG       ul;
    int         rc;
    FS_VAR();

    /*
     * Check stack.
     */
    if (stackChecker())
    {
        LIBC_ASSERTM_FAILED("Too little stack left!\n");
        LIBCLOG_RETURN_INT(-EFAULT);
    }

    /*
     * Request semaphore and enter "must complete section" to avoid signal trouble.
     */
    FS_SAVE_LOAD();
    rc = DosRequestMutexSem(hmtx, 30*1000);
    DosEnterMustComplete(&ul);
    if (!rc)
    {
        FS_RESTORE();
        LIBCLOG_RETURN_INT(0);
    }

    /* failure out */
    DosExitMustComplete(&ul);
    LIBC_ASSERTM_FAILED("DosRequestMutexSem(%x) failed with rc=%d!\n", hmtx, rc);
    rc = -__libc_native2errno(rc);
    FS_RESTORE();
    LIBCLOG_RETURN_INT(rc);
}


/**
 * Unlocks a mutex semaphore.
 *
 * @returns 0 on success.
 * @returns Negative errno on failure.
 * @param   hmtx    Handle to the mutex.
 */
int __libc_Back_safesemMtxUnlock(uintptr_t hmtx)
{
    LIBCLOG_ENTER("hmtx=%#x\n", hmtx);
    ULONG   ul = 0;
    int     rc;
    FS_VAR();

    /*
     * Release the semaphore.
     */
    FS_SAVE_LOAD();
    rc = DosReleaseMutexSem(hmtx);
    if (rc)
    {
        FS_RESTORE();
        LIBC_ASSERTM_FAILED("DosReleaseMutexSem(%x) -> %d\n", hmtx, rc);
        rc = -__libc_native2errno(rc);
        LIBCLOG_RETURN_INT(rc);
    }

    DosExitMustComplete(&ul);
    FS_RESTORE();
    LIBCLOG_RETURN_INT(0);
}



/**
 * Creates a safe event sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   phev        Where to store the semaphore handle.
 * @param   fShared     Set if the semaphore should be sharable between processes.
 */
int __libc_Back_safesemEvCreate(uintptr_t *phev, int fShared)
{
    LIBCLOG_ENTER("phev=%p fShared=%d\n", (void *)phev, fShared);
    int rc;
    __LIBC_PSAFESEMEV pEv = NULL;
    FS_VAR_SAVE_LOAD();
    if (fShared)
    {
        /*
         * Allocate from SPM, be very careful.
         */
        __LIBC_SPMXCPTREGREC RegRec;
        rc = __libc_spmLock(&RegRec, NULL);
        if (!rc)
        {
            pEv = __libc_spmAllocLocked(sizeof(*pEv));
            if (pEv)
            {
                pEv->hev = NULLHANDLE;
                pEv->cWaiters = 0;
                pEv->cUsers = 1;
                pEv->fShared = fShared;
#ifdef __LIBC_STRICT
                pEv->hmtx = 0;
#endif 
            
                rc = DosCreateEventSemEx(NULL, &pEv->hev, fShared ? DC_SEM_SHARED : 0, FALSE);
                if (rc)
                {
                    __libc_spmFree(pEv);
                    rc = -__libc_native2errno(rc);
                }
            }
            else
                rc = -ENOMEM;
            __libc_spmUnlock(&RegRec);
        }
    }
    else
    {
        /*
         * Allocate from high mem heap.
         */
        pEv = _hmalloc(sizeof(*pEv));
        if (pEv)
        {
            pEv->hev = NULLHANDLE;
            pEv->cWaiters = 0;
            pEv->cUsers = 1;
            pEv->fShared = fShared;
#ifdef __LIBC_STRICT
            pEv->hmtx = 0;
#endif 

            rc = DosCreateEventSemEx(NULL, &pEv->hev, fShared ? DC_SEM_SHARED : 0, FALSE);
            if (rc)
            {
                free(pEv);
                rc = -__libc_native2errno(rc);
            }
        }
        else
            rc = -ENOMEM;
    }
    FS_RESTORE();

    if (!rc)
        *phev = (uintptr_t)pEv;
    LIBCLOG_RETURN_MSG(rc, "ret %d *phev=%#x\n", rc, *phev);
}


/**
 * Opens a shared safe event sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hev         The semaphore handle.
 */
int __libc_Back_safesemEvOpen(uintptr_t hev)
{
    LIBCLOG_ENTER("hev=%#x\n", hev);
    __LIBC_PSAFESEMEV pEv = (__LIBC_PSAFESEMEV)hev;
    FS_VAR_SAVE_LOAD();
    int rc = DosOpenEventSemEx(NULL, &pEv->hev);
    FS_RESTORE();
    if (!rc)
        __atomic_increment_u32(&pEv->cUsers);
    else
        rc = -__libc_native2errno(rc);
    LIBCLOG_RETURN_INT(rc);
}


/**
 * Closes a shared safe mutex sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hev         The semaphore handle.
 */
int __libc_Back_safesemEvClose(uintptr_t hev)
{
    LIBCLOG_ENTER("hev=%#x\n", hev);
    __LIBC_PSAFESEMEV pEv = (__LIBC_PSAFESEMEV)hev;
    FS_VAR_SAVE_LOAD();
    int rc = DosCloseEventSemEx(pEv->hev);
    FS_RESTORE();
    if (!rc)
    {
        if (    pEv->cUsers
            &&  !__atomic_decrement_u32(&pEv->cUsers))
        {
            pEv->hev = NULLHANDLE;
            if (pEv->fShared)
                __libc_spmFree(pEv);
            else
                free(pEv);
        }
    }
    else
        rc = -__libc_native2errno(rc);
    LIBCLOG_RETURN_INT(rc);
}


/**
 * Signal notification callback.
 *
 * This will call the completion handler and restore the thread priority before
 * any signal is executed. We're protect by the signal mutex/must-complete here.
 */
static void semEvSleepSignalCallback(int iSignal, void *pvUser)
{
    LIBCLOG_ENTER("iSignal=%d pvUser=%p\n", iSignal, pvUser);
    struct SignalArgs *pArgs = (struct SignalArgs *)pvUser;

    if (!pArgs->fDone && pArgs->cTries++ < 5)
    {
        pArgs->pfnComplete(pArgs->pvUser);
        int rc = DosSetPriority(PRTYS_THREAD, pArgs->ulOldPri >> 8, pArgs->ulOldPri & 0xff, 0);
        if (rc)
            __libc_Back_panic(0, NULL, "DosSetPriority(PRTYS_THREAD, %x, %x, 0) -> %d\n", 
                              (unsigned)pArgs->ulOldPri >> 8, (unsigned)pArgs->ulOldPri & 0xff, rc);
        pArgs->fDone = 1;
    }
    LIBCLOG_RETURN_VOID();
}


/**
 * Sleep on a semaphore.
 *
 * This is the most difficult thing we're doing in this file.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 *
 * @param   hev         The semaphore to sleep on.
 * @param   hmtx        The semaphore protecting hev and which is to be unlocked while
 *                      sleeping and reaquired upon waking up.
 * @param   pfnComplete Function to execute on signal and on wait completion.
 * @param   pvUser      User argument to pfnComplete.
 */
int __libc_Back_safesemEvSleep(uintptr_t hev, uintptr_t hmtx, void (*pfnComplete)(void *pvUser), void *pvUser)
{
    LIBCLOG_ENTER("hev=%#x hmtx=%#x pfnComplete=%p pvUser=%p\n", hev, hmtx, (void *)pfnComplete, pvUser);
    __LIBC_PSAFESEMEV pEv = (__LIBC_PSAFESEMEV)hev;

#ifdef __LIBC_STRICT
    /*
     * Keep track of hmtx.
     */
    if (pEv->hmtx)
        LIBC_ASSERTM(pEv->hmtx == hmtx, "pEv->hmtx=%#x hmtx=%#x\n", pEv->hmtx, hmtx);
    else
        pEv->hmtx = hmtx;
#endif 

    /*
     * Setup signal notification callback.
     */
    int rc;
    __LIBC_PTHREAD pThrd = __libc_threadCurrentNoAuto();
    if (pThrd)
    {
        FS_VAR_SAVE_LOAD();
        PTIB    pTib;
        PPIB    pPib;
        DosGetInfoBlocks(&pTib, &pPib);
    
        struct SignalArgs Args;
        Args.ulOldPri       = pTib->tib_ptib2->tib2_ulpri;
        Args.fDone          = 0;
        Args.cTries         = 0;
        Args.pfnComplete    = pfnComplete;
        Args.pvUser         = pvUser;
    
        pThrd->pvSigCallbackUser = &Args;
        pThrd->pfnSigCallback    = semEvSleepSignalCallback;
    
        /*
         * Raise priority to time critical to increase our chances of actually getting 
         * blocked before something bad like rescheduling or signaling strikes us.
         */
        rc = DosSetPriority(PRTYS_THREAD, PRTYC_TIMECRITICAL, 31, 0);
        LIBC_ASSERTM(!rc, "DosSetPriority(PRTYS_THREAD, PRTYC_TIMECRITICAL, 31, 0) -> %d\n", rc);
        if (rc && (Args.ulOldPri >> 8) != PRTYC_TIMECRITICAL)
        {
            rc = DosSetPriority(PRTYS_THREAD, PRTYC_TIMECRITICAL, 0, 0);
            LIBC_ASSERTM(!rc, "DosSetPriority(PRTYS_THREAD, PRTYC_TIMECRITICAL, 0, 0) -> %d\n", rc);
        }
    
        /*
         * Release the sempahore and exit the must complete section.
         */
        __atomic_increment_u32(&pEv->cWaiters);
        rc = __libc_Back_safesemMtxUnlock(hmtx);
        if (!rc)
        {
            /*
             * Now, lets block until something happens.
             */
            int rc2 = 0;
            if (!Args.fDone)
                rc2 = DosWaitEventSem(pEv->hev, SEM_INDEFINITE_WAIT);
    
            /*
             * Reclaim the ownership of the sempahore (w/must-complete) and deregister 
             * the signal notification callback before we check if we was interrupted or not during the wait.
             */
            __atomic_decrement_u32(&pEv->cWaiters);
            LIBCLOG_MSG("woke up - rc2=%d\n", rc2);
            rc = __libc_Back_safesemMtxLock(hmtx);
            pThrd->pfnSigCallback = NULL;
            pThrd->pvSigCallbackUser = NULL;
            if (!Args.fDone)
            {
                /* 
                 * Not interrupted.
                 * Check for errors, do completion and restore priority. 
                 */
                if (rc2)
                    rc = -__libc_native2errno(rc2);
                pfnComplete(pvUser);
                rc2 = DosSetPriority(PRTYS_THREAD, Args.ulOldPri >> 8, Args.ulOldPri & 0xff, 0);
                if (rc2)
                    __libc_Back_panic(0, NULL, "DosSetPriority(PRTYS_THREAD, %x, %x, 0) -> %d\n", 
                                      (unsigned)Args.ulOldPri >> 8, (unsigned)Args.ulOldPri & 0xff, rc2);
            }
            else if (!rc)
                rc = -EINTR;
    
            /*
             * Reset the event semaphore.
             */
            ULONG ulIgnore;
            rc2 = DosResetEventSem(pEv->hev, &ulIgnore);
            LIBC_ASSERTM(!rc2 && ERROR_ALREADY_RESET, "DosResetEventSem(%#lx,)->%d\n", pEv->hev, rc2);
        }
        FS_RESTORE();
    }
    else
        rc = -ENOSYS;

    LIBCLOG_RETURN_INT(rc);
}


/**
 * Wakes up all threads sleeping on a given event semaphore.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hev         Event semaphore to post.
 * @param   hmtx        The semaphore protecting hev (see __libc_Back_safesemEvSleep).
 *                      The caller should own this semaphore, it's purpose is debug strictness only!
 */
int __libc_Back_safesemEvWakeup(uintptr_t hev, uintptr_t hmtx)
{
    LIBCLOG_ENTER("hev=%#x hmtx=%#x\n", hev, hmtx);
    __LIBC_PSAFESEMEV pEv = (__LIBC_PSAFESEMEV)hev;
    FS_VAR_SAVE_LOAD();
    int rc;

#ifdef __LIBC_STRICT
    /*
     * Check hmtx.
     */
    LIBC_ASSERTM(!pEv->hmtx || pEv->hmtx == hmtx, "pEv->hmtx=%#x hmtx=%#x\n", pEv->hmtx, hmtx);
    ULONG   cNesting;
    PID     pid;
    TID     tid;
    rc = DosQueryMutexSem(hmtx, &pid, &tid, &cNesting);
    LIBC_ASSERTM(!rc, "DosQueryMutexSem(%#x,,,) -> %d\n", hmtx, rc);
    if (!rc)
        LIBC_ASSERTM(pid == fibGetPid() && tid == fibGetTid(),
                     "pid=%d fibGetPid()->%d  tid=%d fibGetTid()->%d\n", (int)pid, fibGetPid(), (int)tid, fibGetTid());
#endif 

    /*
     * Post it.
     */
    rc = DosPostEventSem(hev);
    if (!rc)
    {
        if (!rc && pEv->cWaiters)
        {
            /* give the waiting threads a fair chance to get past the DosWaitEventSem() call. */
            unsigned cYields = 3;
            do
            {
                DosSleep(0);
                LIBCLOG_MSG("cWaiters=%d cYields=%d\n", pEv->cWaiters, cYields);
            } while (pEv->cWaiters && --cYields > 0);
        }
    }
    else
    {
        if (rc == ERROR_ALREADY_POSTED || rc == ERROR_TOO_MANY_POSTS)
            rc = 0;
        else
            rc = -__libc_native2errno(rc);
    }
    FS_RESTORE();
    LIBCLOG_RETURN_INT(rc);
}

