/* sys/spawnve.c (emx+gcc) -- Copyright (c) 1992-1996 by Eberhard Mattes */

#include "libc-alias.h"
#include <stdlib.h>
#include <string.h>
#include <process.h>
#include <errno.h>
#include <alloca.h>
#include <386/builtin.h>
#include <sys/fmutex.h>
#define INCL_DOSPROCESS
#define INCL_FSMACROS
#define INCL_DOSERRORS
#include <os2emx.h>
#include "b_fs.h"
#include "b_signal.h"
#include "b_process.h"
#include <emx/syscalls.h>
#include <InnoTekLIBC/sharedpm.h>
#define __LIBC_LOG_GROUP    __LIBC_LOG_GRP_PROCESS
#include <InnoTekLIBC/logstrict.h>
#include "syscalls.h"


/**
 * Collects and creates the inherit stuff we send down to a LIBC child.
 * @returns Pointer to inherit structure on success.
 * @returns NULL and errno on failure.
 */
static __LIBC_PSPMINHERIT doInherit(void)
{
    LIBCLOG_ENTER("\n");
    __LIBC_PSPMINHERIT      pRet = NULL;

    /*
     * Get FH stuff.
     */
    size_t                  cbFH;
    __LIBC_PSPMINHFHBHDR    pFH;
    pFH = __libc_fhInheritPack(&cbFH);
    if (pFH)
    {
        /*
         * Get FS stuff.
         */
        size_t              cbFS;
        __LIBC_PSPMINHFS    pFS;
        if (!__libc_back_fsInheritPack(&pFS, &cbFS))
        {
            /*
             * Get Signal stuff
             */
            size_t              cbSig;
            __LIBC_PSPMINHSIG   pSig;
            if (!__libc_back_signalInheritPack(&pSig, &cbSig))
            {
                /*
                 * Allocate shared memory.
                 */
                size_t  cb = sizeof(__LIBC_SPMINHERIT) + ((cbFH + 3) & ~3) + ((cbFS + 3) & ~3) + ((cbSig + 3) & ~3);
                pRet = __libc_spmAlloc(cb);
                if (pRet)
                {
                    /* fh */
                    pRet->cb = sizeof(*pRet);
                    pRet->pFHBundles = (__LIBC_PSPMINHFHBHDR)(pRet + 1);
                    memcpy(pRet->pFHBundles, pFH, cbFH);
                    free(pFH);

                    /* fs */
                    if (pFS)
                    {
                        pRet->pFS = (__LIBC_PSPMINHFS)((char *)pRet->pFHBundles + ((cbFH + 3) & ~3));
                        memcpy(pRet->pFS, pFS, cbFS);
                        free(pFS);
                    }
                    else
                        pRet->pFS = NULL;

                    /* sig */
                    if (pSig)
                    {
                        pRet->pSig = (__LIBC_PSPMINHSIG)((char *)pRet->pFHBundles + ((cbFH + 3) & ~3) + ((cbFS + 3) & ~3));
                        memcpy(pRet->pSig, pSig, cbSig);
                        free(pSig);
                    }

                    /* done! */
                    LIBCLOG_RETURN_P(pRet);
                }
                free(pSig);
            }
            free(pFS);
        }

        /* cleanup on failure */
        __libc_fhInheritDone();
        free(pFH);
    }

    LIBCLOG_RETURN_P(pRet);
}


/**
 * Cleans up any globale inherit stuff.
 */
static void doInheritDone(void)
{
    __libc_fhInheritDone();
}

/* Note: We are allowed to call _trealloc() as this module is not used
   in an .a library. */

#define ADD(n) do { \
  while (cbArgs + n > cbArgsBuf) \
    { \
      cbArgsBuf += 512; \
      pszArgsBuf = _trealloc (pszArgsBuf, cbArgsBuf); \
      if (pszArgsBuf == NULL) \
        { \
          errno = ENOMEM; \
          LIBCLOG_RETURN_INT(-1); \
        } \
      pszArg = pszArgsBuf + cbArgs; \
    } \
  cbArgs += n; } while (0)


int __spawnve(struct _new_proc *np)
{
    LIBCLOG_ENTER("np=%p:{mode=%#x}\n", (void *)np, np->mode);
    FS_VAR();

    /*
     * Validate mode.
     */
    ULONG ulMode = np->mode;
    switch (ulMode & 0xff)
    {
        case P_WAIT:
        case P_NOWAIT:
        case P_OVERLAY:
            break;
        default:
            errno = EINVAL;
            LIBCLOG_ERROR_RETURN(-1, "ret -1 - invalid mode 0x%08lx\n", ulMode);
    }

    /*
     * Allocate space for the program name and resolve the filename.
     */
    size_t cch = strlen((const char *)np->fname_off);
    char *pszPgmName = alloca(cch + 5);
    memcpy(pszPgmName, (const char *)np->fname_off, cch + 1);
    _defext(pszPgmName, "exe");
    char szNativePath[PATH_MAX];
    int rc = __libc_back_fsResolve(pszPgmName, BACKFS_FLAGS_RESOLVE_FULL, &szNativePath[0], NULL);
    if (rc)
    {
        if (pszPgmName[cch])
        {
            pszPgmName[cch] = '\0';
            rc = __libc_back_fsResolve(pszPgmName, BACKFS_FLAGS_RESOLVE_FULL, &szNativePath[0], NULL);
        }
        if (rc)
        {
            errno = -rc;
            LIBCLOG_ERROR_RETURN(-1, "ret -1 - Failed to resolve program name: '%s' rc=%d.\n", pszPgmName, rc);
        }
    }
    pszPgmName = &szNativePath[0];

    /*
     * cmd.exe and 4os2.exe needs different argument handling.
     * (1 == cmd or 4os2 shell, 0 == anything else)
     */
    char *psz = _getname(pszPgmName);
    int method = stricmp(psz, "cmd.exe") == 0
              || stricmp(psz, "4os2.exe") == 0;

    /*
     * Construct the commandline.
     */
    const char *pszSrc = (const char *)np->arg_off;
    char       *pszArgsBuf = NULL;
    size_t      cbArgsBuf = 0;
    char       *pszArg = NULL;
    size_t      cbArgs = 0;
    if (np->arg_count > 0)
    {
        ++pszSrc;                    /* skip flags byte */
        cch = strlen(pszSrc) + 1;
        ADD(cch);
        memcpy(pszArg, pszSrc, cch);
        pszArg += cch; pszSrc += cch;
    }
    int i;
    for (i = 1; i < np->arg_count; ++i)
    {
        if (i > 1)
        {
            ADD(1);
            *pszArg++ = ' ';
        }
        ++pszSrc;                    /* skip flags byte */
        BOOL fQuote = FALSE;
        if (*pszSrc == 0)
            fQuote = TRUE;
        else if (ulMode & P_QUOTE)
        {
            if (pszSrc[0] == '@' && pszSrc[1] != 0)
                fQuote = TRUE;
            else
                for (psz = (char *)pszSrc; *psz != 0; ++psz)
                    if (*psz == '?' || *psz == '*')
                    {
                        fQuote = TRUE;
                        break;
                    }
        }
        if (!fQuote)
        {
            for (psz = (char *)pszSrc; *psz != 0; ++psz)
                if (*psz == ' ' || *psz == '\t' || (*psz == '"' && method == 1))
                {
                    fQuote = TRUE;
                    break;
                }
        }
        if (fQuote)
        {
            ADD(1);
            *pszArg++ = '"';
        }
        size_t bs = 0;
        while (*pszSrc != 0)
        {
            if (*pszSrc == '"' && method == 0)
            {
                ++bs;
                ADD(bs);
                memset(pszArg, '\\', bs); pszArg += bs;
                bs = 0;
            }
            else if (*pszSrc == '\\' && method == 0)
                ++bs;
            else
                bs = 0;
            ADD(1);
            *pszArg++ = *pszSrc;
            ++pszSrc;
        }
        if (fQuote)
        {
            ADD(1+bs);
            memset(pszArg, '\\', bs); pszArg += bs;
            *pszArg++ = '"';
        }
        ++pszSrc;
    }
    /* The arguments are an array of zero terminated strings, ending with an empty string. */
    ADD(2);
    *pszArg++ = '\0';
    *pszArg++ = '\0';


    /*
     * Now create an embryo process.
     */
    _fmutex_request(&__libc_gmtxExec, 0);
    __LIBC_PSPMPROCESS pEmbryo = __libc_spmCreateEmbryo(getpid());
    if (pEmbryo)
    {
        /*
         * Do inheritance stuff.
         */
        pEmbryo->pInherit = doInherit();
        if (pEmbryo->pInherit)
        {
            RESULTCODES resc;
            char        szObj[40];

            /*
             * Create the process.
             */
            FS_SAVE_LOAD();
            LIBCLOG_MSG("Calling DosExecPgm pgm: %s args: %s\\0%s\\0\\0\n", pszPgmName, pszArgsBuf, pszArgsBuf + strlen(pszArgsBuf) + 1);
            rc = DosExecPgm(szObj, sizeof(szObj), EXEC_ASYNCRESULT, (PCSZ)pszArgsBuf, (PCSZ)np->env_off, &resc, (PCSZ)pszPgmName);
            int cTries = 3;
            while (     (   rc == ERROR_INVALID_EXE_SIGNATURE
                         || rc == ERROR_BAD_EXE_FORMAT)
                   &&   --cTries > 0)
            {
                /*
                 * This could be a batch, rexx or hash bang script.
                 * The first two is recognized by the filename extension, the latter
                 * requires inspection of the first line of the file.
                 */
                char szLineBuf[256];
                const char *pszInterpreter = NULL;
                const char *pszInterpreterArgs = NULL;
                psz = _getext(pszPgmName);
                if (psz && (!stricmp(psz, ".cmd") || !stricmp(psz, ".bat") || !stricmp(psz, ".btm")))
                {
                    pszInterpreterArgs = "/C";
                    pszInterpreter = getenv("COMSPEC");
                    if (!pszInterpreter)
                    {
                        pszInterpreter = getenv("OS2_SHELL");
                        if (!pszInterpreter)
                            pszInterpreter = stricmp(psz, ".btm") ? "cmd.exe" : "4os2.exe";
                    }

                    /* make sure the slashes in the script name goes the DOS way. */
                    psz = szNativePath;
                    while ((psz = strchr(szNativePath, '/')) != NULL)
                        *psz++ = '\\';
                }
                else
                {
                    /*
                     * Read the first line of the file into szLineBuf and terminate
                     * it stripping trailing blanks.
                     */
                    HFILE hFile = NULLHANDLE;
                    ULONG ulAction = 0;
                    int rc2 = DosOpen((PCSZ)pszPgmName, &hFile, &ulAction, 0, FILE_NORMAL,
                                      OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
                                      OPEN_FLAGS_SEQUENTIAL | OPEN_FLAGS_NOINHERIT | OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY,
                                      NULL);
                    if (!rc2)
                    {
                        ULONG cbRead = 0;
                        rc2 = DosRead(hFile, szLineBuf, sizeof(szLineBuf) - 1, &cbRead);
                        DosClose(hFile);
                        if (!rc2)
                        {
                            szLineBuf[cbRead < sizeof(szLineBuf) ? cbRead : sizeof(szLineBuf) - 1] = '\0';
                            psz = strpbrk(szLineBuf, "\r\n");
                            if (psz)
                            {
                                register char ch;
                                while ((ch = *--psz) == ' ' || ch == '\t')
                                    /* nothing */;
                                psz[1] = '\0';

                                /*
                                 * Check for '#[ \t]*!'
                                 */
                                psz = &szLineBuf[0];
                                if (*psz++ == '#')
                                {
                                    while ((ch = *psz) == ' ' && ch == '\t')
                                        psz++;
                                    if (*psz++ == '!')
                                    {
                                        while ((ch = *psz) == ' ' && ch == '\t')
                                            psz++;
                                        pszInterpreter = psz;

                                        /*
                                         * Find end of interpreter and start of potential arguments.
                                         * I've never seen quoted interpreter names, so we won't bother with that yet.
                                         */
                                        while ((ch = *psz) != ' ' && ch != '\t' && ch != '\0')
                                            psz++;
                                        if (ch)
                                        {
                                            *psz++ = '\0';
                                            while ((ch = *psz) == ' ' && ch == '\t')
                                                psz++;
                                            if (ch)
                                                pszInterpreterArgs = psz;
                                        }
                                    } /* if bang */
                                } /* if hash */
                            } /* if full line */
                        } /* if read */
                    } /* if open */
                }
                if (!pszInterpreter)
                    break;

                /*
                 * Squeeze the interpreter arguments + the program name into
                 * the argument buffer after argv[0].
                 * ASSUME that the arguments and program name require no escaping.
                 */
                size_t cchPgmName = strlen(pszPgmName);
                int cchInterpreterArgs = pszInterpreterArgs ? strlen(pszInterpreterArgs) : -1;
                BOOL fQuote = strpbrk(pszPgmName, " \t") != NULL;
                cch = cchPgmName + cchInterpreterArgs + 2 + 2 * fQuote;

                /* grow and shift the argument buffer. */
                ADD(cch);
                psz = pszArgsBuf + strlen(pszArgsBuf) + 1;
                memmove(psz + cch, psz, cbArgs);

                /* add arguments */
                if (pszInterpreterArgs)
                {
                    memcpy(psz, pszInterpreterArgs, cchInterpreterArgs);
                    psz += cchInterpreterArgs;
                    *psz++ = ' ';
                }

                /* script name */
                if (fQuote)
                    *psz++ = '"';
                memcpy(psz, pszPgmName, cchPgmName);
                psz += cchPgmName;
                if (fQuote)
                    *psz++ = '"';
                *psz++ = ' ';

                /*
                 * Resolve the interpreter name.
                 * If the specified name fails, we'll try search the path for it and
                 * also adding an .exe extension before we give up.
                 */
                rc = __libc_back_fsResolve(pszInterpreter, BACKFS_FLAGS_RESOLVE_FULL, &szNativePath[0], NULL);
                if (rc)
                {
                    char szPath[512];
                    _searchenv(pszInterpreter, "PATH", szPath); /** @todo _searchenv is not safe, it can easily overflow szPath! */
                    if (!szPath[0])
                    {
                        cch = strlen(pszInterpreter);
                        memcpy(szNativePath, pszInterpreter, cch + 1);
                        _defext(szNativePath, "exe");
                        if (szNativePath[cch])
                            _searchenv(szNativePath, "PATH", szPath);
                    }
                    if (szPath[0])
                        rc = __libc_back_fsResolve(szPath, BACKFS_FLAGS_RESOLVE_FULL, &szNativePath[0], NULL);
                    if (rc)
                        break;
                }

                /*
                 * Try execute it.
                 */
                LIBCLOG_MSG("Calling DosExecPgm pgm: %s args: %s\\0%s\\0\\0\n", pszPgmName, pszArgsBuf, pszArgsBuf + strlen(pszArgsBuf) + 1);
                rc = DosExecPgm(szObj, sizeof(szObj), EXEC_ASYNCRESULT, (PCSZ)pszArgsBuf, (PCSZ)np->env_off, &resc, (PCSZ)pszPgmName);
            } /* while */
            FS_RESTORE();
            if (!rc)
            {
                __atomic_cmpxchg32((volatile uint32_t *)(void *)&pEmbryo->pid, (uint32_t)resc.codeTerminate, ~0);
                LIBCLOG_MSG("Spawned pid=%04lx (%ld)\n", resc.codeTerminate, resc.codeTerminate);
                __libc_back_processWaitNotifyExec(resc.codeTerminate);

                /* cleanup embryo and other stuff. */
                __libc_spmRelease(pEmbryo);
                doInheritDone();
                if (pszArgsBuf != NULL)
                    _tfree(pszArgsBuf);
                _fmutex_release(&__libc_gmtxExec);

                /*
                 * Exit depends on the mode.
                 */
                switch (ulMode & 0xff)
                {
                    /*
                     * Return the pid.
                     */
                    case P_NOWAIT:
                        LIBCLOG_RETURN_INT((int)resc.codeTerminate);

                    /*
                     * Wait for the child and return the result.
                     */
                    case P_WAIT:
                    {
                        pid_t pid = resc.codeTerminate;
                        LIBCLOG_MSG("Calling wait4(%d,,0,0)\n", pid);
                        int iStatus = 0;
                        pid_t pidEnded = wait4(pid, &iStatus, 0, NULL);
                        while (pidEnded < 0 && errno == EINTR)
                            pidEnded = wait4(pid, &iStatus, 0, NULL);
                        if (pidEnded > 0)
                        {
                            LIBCLOG_MSG("wait4(%d,,0,0) returned pidEnded=%d iStatus=%#x (%d)\n", pid, pidEnded, iStatus, iStatus);
                            LIBC_ASSERTM(pidEnded == pid, "Expected pid %d and got %d!\n", pid, pidEnded);
                            LIBCLOG_RETURN_INT(iStatus >> 8);
                        }

                        LIBC_ASSERTM_FAILED("Calling wait4(%d,,0,0) -> errno=%d\n", pid, errno);
                        LIBCLOG_RETURN_INT(-1);
                        break;
                    }

                    /*
                     * Wait for the child and exit this process with the same result.
                     */
                    case P_OVERLAY:
                    {
                        /*
                         * Ignore SIGCHLD signal.
                         */
                        /** @todo proxy job control */
                        bsd_signal(SIGCHLD, SIG_DFL);
                        bsd_signal(SIGSTOP, SIG_DFL);
                        bsd_signal(SIGTSTP, SIG_DFL);
                        bsd_signal(SIGCONT, SIG_DFL);

                        /*
                         * Wait
                         */
                        pid_t pid = resc.codeTerminate;
                        LIBCLOG_MSG("Calling __libc_Back_processWait(P_PID,%d,,WEXITED,NULL)\n", pid);
                        for (;;)
                        {
                            siginfo_t SigInfo = {0};
                            do
                                /** @todo proxy job control */
                                rc = __libc_Back_processWait(P_PID, pid, &SigInfo, WEXITED, NULL);
                            while (rc == -EINTR);
                            if (rc < 0)
                                break;
                            LIBCLOG_MSG("__libc_Back_processWait(P_PID,%d,,WEXITED,NULL) returned %d si_code=%d si_status=%#x (%d)\n",
                                        pid, rc, SigInfo.si_code, SigInfo.si_status, SigInfo.si_status);
                            LIBC_ASSERTM(SigInfo.si_pid == pid, "Expected pid %d and got %d!\n", pid, SigInfo.si_pid);
                            if (    SigInfo.si_code == CLD_STOPPED
                                ||  SigInfo.si_code == CLD_CONTINUED)
                            {
                                /* notify parent. */
                                /** @todo proxy job control */
                            }
                            else
                            {
                                /*
                                 * Terminate the process.
                                 */
                                int iStatus = SigInfo.si_status;
                                switch (SigInfo.si_code)
                                {
                                    default:
                                        LIBC_ASSERTM_FAILED("Invalid si_code=%#x si_status=%#x\n", SigInfo.si_code, SigInfo.si_status);
                                    case CLD_EXITED:
                                        __libc_spmTerm(__LIBC_EXIT_REASON_EXIT, iStatus);
                                        break;
                                    case CLD_KILLED:
                                        __libc_spmTerm(__LIBC_EXIT_REASON_SIGNAL_BASE + iStatus, 0);
                                        iStatus = 127;
                                        break;
                                    case CLD_DUMPED:
                                        if (iStatus == SIGSEGV || iStatus > SIGRTMAX || iStatus <= 0)
                                            __libc_spmTerm(__LIBC_EXIT_REASON_XCPT, 0);
                                        else
                                            __libc_spmTerm(__LIBC_EXIT_REASON_SIGNAL_BASE + iStatus, 0);
                                        iStatus = 127;
                                        break;
                                    case CLD_TRAPPED:
                                        if (iStatus <= SIGRTMAX && iStatus > 0)
                                            __libc_spmTerm(__LIBC_EXIT_REASON_SIGNAL_BASE + iStatus, 0);
                                        else
                                            __libc_spmTerm(__LIBC_EXIT_REASON_TRAP, 0);
                                        iStatus = 127;
                                        break;
                                }

                                LIBCLOG_MSG("Calling DosExit(,0)\n");
                                for (;;)
                                    DosExit(EXIT_PROCESS, iStatus);
                                break; /* won't get here */
                            }
                        }

                        LIBC_ASSERTM_FAILED("__libc_Back_processWait(P_PID,%d,,WEXITED,NULL) returned %d\n", pid, rc);
                        __libc_spmTerm(__LIBC_EXIT_REASON_KILL + SIGABRT, 123);
                        for (;;)
                            DosExit(EXIT_PROCESS, 123);
                        break; /* won't get here */
                    }

                        /* this can _NEVER_ happen! */
                    default:
                        errno = EINVAL;
                        LIBCLOG_ERROR_RETURN_INT(-1);
                }
                /* won't ever get here! */
            }
            else if (rc > 0)
                _sys_set_errno(rc);
            else
                errno = -rc;
            doInheritDone();
        }
        /* cleanup embryo */
        __libc_spmRelease(pEmbryo);
    }

    if (pszArgsBuf != NULL)
        _tfree(pszArgsBuf);
    _fmutex_release(&__libc_gmtxExec);
    LIBCLOG_ERROR_RETURN_INT(-1);
}
