/* $Id: b_fsUnlink.c 2254 2005-07-17 12:25:44Z bird $ */
/** @file
 *
 * LIBC SYS Backend - unlink.
 *
 * Copyright (c) 2005 knut st. osmundsen <bird@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_FSMACROS
#define INCL_ERRORS
#define INCL_DOSFILEMGR
#define INCL_DOSMISC
#include <os2emx.h>
#include "b_fs.h"
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <InnoTekLIBC/backend.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_BACK_FS
#include <InnoTekLIBC/logstrict.h>


/**
 * Unlinks a file, directory, symlink, dev, pipe or socket.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath         Path to the filesystem file/dir/symlink/whatever to remove.
 */
int __libc_Back_fsUnlink(const char *pszPath)
{
    LIBCLOG_ENTER("pszPath=%p:{%s}\n", (void *)pszPath, pszPath);

    /*
     * Validate input.
     */
    if (!*pszPath)
        LIBCLOG_ERROR_RETURN_INT(-ENOENT);

    /*
     * Resolve the path.
     */
    char szNativePath[PATH_MAX];
    int rc = __libc_back_fsResolve(pszPath, BACKFS_FLAGS_RESOLVE_FULL_SYMLINK | BACKFS_FLAGS_RESOLVE_DIR_MAYBE, &szNativePath[0], NULL);
    if (rc)
        LIBCLOG_ERROR_RETURN_INT(rc);

    FS_VAR_SAVE_LOAD();

    /*
     * DosDelete is kind of slow as it always scans the environment (in a R0
     * accessing R3 data safe manner to make matters worse even) for DELDIR.
     * Since only a limited number of programs actually changes this
     * environment block, we assume that it's ok to check the first time if
     * DELDIR processing is enabled or not. If it isn't there, we can use
     * DosForceDelete and save quite a bit of time if we're called some
     * times.
     */
    static int fUseForce = 0; /* state: 0 - uninit, 1 - DosForceDelete, -1 - DosDelete */
    if (fUseForce == 0)
    {
        PSZ psz = NULL;
        if (DosScanEnv((PCSZ)"DELDIR", &psz) || !psz)
            fUseForce = 1;
        else
            fUseForce = -1;
    }

    /*
     * We'll attempt delete it as a file first.
     */
    if (fUseForce == 1)
        rc = DosForceDelete((PCSZ)&szNativePath[0]);
    else
        rc = DosDelete((PCSZ)&szNativePath[0]);
    if (rc == ERROR_ACCESS_DENIED)
    {
        /*
         * Ok, try delete it as a directory then.
         */
        rc = DosDeleteDir((PCSZ)&szNativePath[0]);
        if (rc)
        {
            /*
             * OS/2 returns access denied when the directory
             * contains files or it is not a directory. Check for
             * directory/other and return failure accordingly.
             */
            if (rc == ERROR_ACCESS_DENIED)
            {
                struct stat s;
                rc = __libc_back_fsNativeFileStat(&szNativePath[0], &s);
                if (!rc && S_ISDIR(s.st_mode))
                    rc = -ENOTEMPTY;
                else
                    rc = -EACCES;
            }
            else
                rc = -__libc_native2errno(rc);
        }
    }
    else if (rc)
        rc = -__libc_native2errno(rc);

    FS_RESTORE();
    if (!rc)
        LIBCLOG_RETURN_INT(rc);
    LIBCLOG_ERROR_RETURN_INT(rc);
}

