/* $Id: b_fsNativeFileStat.c 2254 2005-07-17 12:25:44Z bird $ */
/** @file
 *
 * LIBC SYS Backend - internal stat.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird@innotek.de>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_BASE
#define INCL_FSMACROS
#include <os2emx.h>
#include "b_fs.h"
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <limits.h>
#include "syscalls.h"
#include <InnoTekLIBC/libc.h>
#include <InnoTekLIBC/pathrewrite.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_BACK_FS
#include <InnoTekLIBC/logstrict.h>


/**
 * Stats a native file.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszNativePath   Path to the file to stat. This path is resolved, no
 *                          processing required.
 * @param   pStat           Where to store the file stats.
 */
int __libc_back_fsNativeFileStat(const char *pszNativePath, struct stat *pStat)
{
    LIBCLOG_ENTER("pszNativePath=%p:{%s} pStat=%p\n", (void *)pszNativePath, pszNativePath, (void *)pStat);
    union
    {
        FILESTATUS4     fsts4;
        FILESTATUS4L    fsts4L;
    } info;
#if OFF_MAX > LONG_MAX
    int     fLarge = 0;
#endif
    FS_VAR();

    bzero(pStat, sizeof(*pStat));

    /*
     * Validate input, refusing named pipes.
     */
    if (    (pszNativePath[0] == '/' || pszNativePath[0] == '\\')
        &&  (pszNativePath[1] == 'p' || pszNativePath[1] == 'P')
        &&  (pszNativePath[2] == 'i' || pszNativePath[2] == 'I')
        &&  (pszNativePath[3] == 'p' || pszNativePath[3] == 'P')
        &&  (pszNativePath[4] == 'e' || pszNativePath[4] == 'E')
        &&  (pszNativePath[5] == '/' || pszNativePath[5] == '\\'))
        LIBCLOG_ERROR_RETURN_INT(-ENOENT);

    /*
     * If potential device, then perform real check.
     */
    /** @todo copy device check from the path resolver. */
    /*
     * Get path info.
     */
    FS_SAVE_LOAD();
    int rc;
#if OFF_MAX > LONG_MAX
    if (__libc_gpfnDosOpenL)
    {
        rc = DosQueryPathInfo((PCSZ)pszNativePath, FIL_QUERYEASIZEL, &info, sizeof(info.fsts4L));
        fLarge = 1;
    }
    else
#endif
        rc = DosQueryPathInfo((PCSZ)pszNativePath, FIL_QUERYEASIZE, &info, sizeof(info.fsts4));
    /* Now, if the file is open in write mode, we cannot even get the EA size. stupid. */
    if (rc == ERROR_SHARING_VIOLATION)
    {
#if OFF_MAX > LONG_MAX
        if (__libc_gpfnDosOpenL)
        {
            rc = DosQueryPathInfo((PCSZ)pszNativePath, FIL_STANDARDL, &info, sizeof(info.fsts4L) - sizeof(info.fsts4L.cbList));
            fLarge = 1;
        }
        else
#endif
            rc = DosQueryPathInfo((PCSZ)pszNativePath, FIL_STANDARD, &info, sizeof(info.fsts4) - sizeof(info.fsts4.cbList));
        info.fsts4L.cbList = 0;
    }
    FS_RESTORE();
    if (rc)
    {
        rc = -__libc_native2errno(rc);
        LIBCLOG_ERROR_RETURN_INT(rc);
    }

    /*
     * Format stats struct.
     *      We know the info struct layouts!
     *      Only cbFile, cbFileAlloc and attrFile need be accessed
     *      using the specific structure.
     */
    /* Times: FAT might not return create and access time. */
    pStat->st_mtime = _sys_p2t(info.fsts4.ftimeLastWrite, info.fsts4.fdateLastWrite);
    if (   FTIMEZEROP(info.fsts4.ftimeCreation)
        && FDATEZEROP(info.fsts4.fdateCreation))
        pStat->st_ctime = pStat->st_mtime;
    else
        pStat->st_ctime = _sys_p2t(info.fsts4.ftimeCreation, info.fsts4.fdateCreation);
    if (   FTIMEZEROP(info.fsts4.ftimeLastAccess)
        && FDATEZEROP(info.fsts4.fdateLastAccess))
        pStat->st_atime = pStat->st_mtime;
    else
        pStat->st_atime = _sys_p2t(info.fsts4.ftimeLastAccess, info.fsts4.fdateLastAccess);

#if OFF_MAX > LONG_MAX
    ULONG fAttributtes = fLarge ? info.fsts4L.attrFile : info.fsts4.attrFile;
#else
    ULONG fAttributtes = info.fsts4.attrFile;
#endif
    pStat->st_attr = fAttributtes;
    if (fAttributtes & FILE_DIRECTORY)
    {
        /* directory */
        pStat->st_mode = S_IFDIR;
        pStat->st_mode |= ((S_IREAD|S_IWRITE|S_IEXEC) >> 6) * 0111;
        pStat->st_size = 0;
    }
    else
    {
#if OFF_MAX > LONG_MAX
        if (fLarge)
        {
            pStat->st_size = info.fsts4L.cbFile;
            pStat->st_blocks = info.fsts4L.cbFileAlloc / S_BLKSIZE;
        }
        else
#endif
        {
            pStat->st_size = info.fsts4.cbFile;
            pStat->st_blocks = info.fsts4.cbFileAlloc / S_BLKSIZE;
        }
        pStat->st_mode = S_IFREG;
        if (rc & FILE_READONLY)
            pStat->st_mode |= (S_IREAD >> 6) * 0111;
        else
            pStat->st_mode |= ((S_IREAD|S_IWRITE) >> 6) * 0111;

        /* Mark .exe, .com, .cmd and .bat as executables. */
        if ((pStat->st_mode & (S_IFMT | ((S_IEXEC >> 6) * 0111))) == S_IFREG)
        {
            const char *pszExt = _getext(pszNativePath);
            if (   pszExt++
                && pszExt[0] && pszExt[1] && pszExt[2] && !pszExt[3]
                && strstr("!exe!Exe!EXe!EXE!ExE!eXe!eXE!exE"
                          "!com!Com!COm!COM!CoM!cOm!cOM!coM"
                          "!bat!Bat!BAt!BAT!BaT!bAt!bAT!baT"
                          "!btm!Btm!BTm!BTM!BtM!bTm!bTM!btM"
                          "!cmd!Cmd!CMd!CMD!CmD!cMd!cMD!cmD",
                          pszExt)
                   )
                pStat->st_mode |= (S_IEXEC >> 6) * 0111;
        }
    }

    /* fake unix stuff */
    pStat->st_uid = 0;
    pStat->st_gid = 0;
    pStat->st_dev = __libc_back_fsPathCalcInodeAndDev(pszNativePath, &pStat->st_ino);
    pStat->st_rdev = 0;
    pStat->st_nlink = 1;
    pStat->st_blksize = 4096 * 12; /* 48kb */
    /* If in unix mode we'll check the EAs (if any). */
    if (    !__libc_gfNoUnix
        && (fLarge ? info.fsts4L.cbList : info.fsts4.cbList) >= LIBC_UNIX_EA_MIN)
        __libc_back_fsUnixAttribsGet(-1, pszNativePath, pStat);

    LIBCLOG_MSG("st_dev:     %#x\n",  pStat->st_dev);
    LIBCLOG_MSG("st_ino:     %#llx\n",  pStat->st_ino);
    LIBCLOG_MSG("st_mode:    %#x\n",  pStat->st_mode);
    LIBCLOG_MSG("st_nlink:   %u\n",   pStat->st_nlink);
    LIBCLOG_MSG("st_uid:     %u\n",   pStat->st_uid);
    LIBCLOG_MSG("st_gid:     %u\n",   pStat->st_gid);
    LIBCLOG_MSG("st_rdev:    %#x\n",  pStat->st_rdev);
    LIBCLOG_MSG("st_atime:   %d\n",   pStat->st_atime);
    LIBCLOG_MSG("st_mtime:   %d\n",   pStat->st_mtime);
    LIBCLOG_MSG("st_ctime:   %d\n",   pStat->st_ctime);
    LIBCLOG_MSG("st_size:    %lld\n", pStat->st_size);
    LIBCLOG_MSG("st_blocks:  %lld\n", pStat->st_blocks);
    LIBCLOG_MSG("st_blksize: %u\n",   pStat->st_blksize);
    LIBCLOG_MSG("st_attr:    %ld\n",  pStat->st_attr);
    LIBCLOG_RETURN_INT(0);
}
