/* emx.c: Functions for emx as target system.

Original version by Eberhard Mattes, based on i386.c.
Heavily modified by Andrew Zabolotny and Knut St. Osmundsen.

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "config.h"
#include "system.h"
#include "rtl.h"
#include "regs.h"
#include "hard-reg-set.h"
#include "output.h"
#include "tree.h"
#include "toplev.h"
#include "flags.h"
#include "i386-protos.h"
#include "ggc.h"

/* The size of the target's pointer type.  */
#ifndef PTR_SIZE
#define PTR_SIZE (POINTER_SIZE / BITS_PER_UNIT)
#endif

/*#define BIRD_DEBUG - DO NOT COMMIT WITH THIS DEFINED!!! */

#ifdef BIRD_DEBUG
static const char *code(tree node)
{
    if (node)
    {
        switch (TREE_CODE (node))
        {
            case IDENTIFIER_NODE:   return "IDENTIFIER_NODE";
            case TREE_LIST:         return "TREE_LIST      ";
            case TREE_VEC:          return "TREE_VEC       ";
            case BLOCK:             return "BLOCK          ";
            case FUNCTION_TYPE:     return "FUNCTION_TYPE  ";
            case FUNCTION_DECL:     return "FUNCTION_DECL  ";
            case METHOD_TYPE:       return "METHOD_TYPE    ";
            case FIELD_DECL:        return "FIELD_DECL     ";
            case TYPE_DECL:         return "TYPE_DECL      ";
            case VAR_DECL:          return "VAR_DECL       ";
            case PARM_DECL:         return "PARM_DECL      ";
            case RESULT_DECL:       return "RESULT_DECL    ";
            case CONST_DECL:        return "CONST_DECL     ";
            case POINTER_TYPE:      return "POINTER_TYPE   ";
            case VOID_TYPE:         return "VOID_TYPE      ";
            case INTEGER_TYPE:      return "INTEGER_TYPE   ";
            case CHAR_TYPE:         return "CHAR_TYPE      ";
            case BOOLEAN_TYPE:      return "BOOLEAN_TYPE   ";
            case SET_TYPE:          return "SET_TYPE       ";
            case ARRAY_TYPE:        return "ARRAY_TYPE     ";
            case RECORD_TYPE:       return "RECORD_TYPE    ";
            case QUAL_UNION_TYPE:   return "QUAL_UNION_TYPE";
            case UNION_TYPE:        return "UNION_TYPE     ";
            case INTEGER_CST:       return "INTEGER_CST    ";
            default:
                break;
        }
    }
    return "";
}

const char *birddump_callingconv(tree node);
const char *birddump_callingconv(node)
    tree node;
{
    static const char *apsz[] = {"system", "optlink", "stdcall", "cdecl", "dllimport", "dllexport"};
    static const char *psznone = "none";
    static const char *pszdefault = "default";
    tree        attr;
    unsigned    i;

    if (!node)
        return psznone;

    attr = DECL_P (node) ? DECL_ATTRIBUTES (node) : TYPE_P (node) ? TYPE_ATTRIBUTES (node) : NULL_TREE;
    if (!attr)
        return pszdefault;

    for (i = 0; i < sizeof(apsz) / sizeof(apsz[0]); i++)
        if (node && attr && lookup_attribute (apsz[i], attr))
            return apsz[i];

    return pszdefault;
}

void birddump (tree node, const char *pszFunction)
{
    tree type, type2, context, name, tname;
    if (!node)
        return;

    name = DECL_P (node) ? DECL_NAME (node) : TYPE_P (node) ? TYPE_NAME (node) : NULL_TREE;
    type = TREE_TYPE (node);
    tname = type ? TYPE_NAME (type) : NULL_TREE;
    type2 = type ? TREE_TYPE(type) : NULL_TREE;
    context = DECL_P (node) ? DECL_CONTEXT (node) : NULL_TREE;

    fprintf(stderr, "dbg: node=%d %s %p %s '%s'  type=%d %s %p %s '%s'  type_type=%d %s %p %s  context=%d %s %p %s (%s)\n",
            TREE_CODE(node), code(node), (void*)node, birddump_callingconv(node), name ? IDENTIFIER_POINTER (name) : "<none>",
            type ? (int)TREE_CODE(type) : -1, code(type), (void*)type, birddump_callingconv(type), tname ? IDENTIFIER_POINTER (tname) : "<none>",
            type2 ? (int)TREE_CODE(type2) : -1, code(type2), (void*)type2, birddump_callingconv(type2),
            context ? (int)TREE_CODE(context) : -1, code(context), (void*)context, birddump_callingconv(context),
            pszFunction);
}

#define dfprintf(a) fprintf a
#define DUMP(node) birddump(node, __FUNCTION__)
#else
#define dfprintf(a) do {} while (0)
#define DUMP(node) do {} while (0)
#endif

int emx_c_set_decl_assembler_name (decl, fclass)
    tree decl;
    int fclass;
{
  static int recurse;
  const char *oldsym;
  char *newsym;
  size_t sl;
  tree id, type;
  int rc = 1;

  dfprintf((stderr, "emx_c_set_decl_assembler_name\n"));
  DUMP(decl);

  /* Sometimes we recursively call DECL_ASSEMBLER_NAME to apply the default
     mangling rules for current compiler. */
  if (recurse)
    return 0;

  /* Only functions declarations are subject to mangling. */
  if (TREE_CODE (decl) != FUNCTION_DECL)
    return 0;

  recurse++;
  type = TREE_TYPE (decl);

  if (lookup_attribute ("system", TYPE_ATTRIBUTES (type)))
    {
      /* Here we mangle _System functions as defined by IBM specs.
         The function always gets its name as-is (unless it is a method,
         which is a undefined case as VACPP always use _Optlink for methods,
         at least that's what I have understood from the docs). */

      oldsym = IDENTIFIER_POINTER (DECL_NAME (decl));
      if (fclass)
        return 0;

      /* Specifying '*' as first symbol character tells gcc (see varasm.c,
         function assemble_name()) to output the label as-is rather than
         invoking the ASM_OUTPUT_LABELREF macro (which prepends a underscore) */

      sl = strlen (oldsym);
      newsym = xmalloc (sl + 2);
      newsym [0] = '*';
      memcpy (newsym + 1, oldsym, sl + 1);

      SET_DECL_ASSEMBLER_NAME (decl, get_identifier (newsym));

      dfprintf ((stderr, "dbg: system %s -> %s\n", oldsym, newsym));
    }
  else if (lookup_attribute ("optlink", TYPE_ATTRIBUTES (type)))
    {
      /* At the moment we're only implementing OS/2 VAC linking
         compatibility for the C language. This means that no leading
         underscore.
         For C++ we are not compatible. It doesn't make that much sense
         either since we're not VFT compatible either. For simplicity
         and safety we are removing the leading underscore from the
         default mangled names to catch invalid declarations in the
         linking. */

      id = DECL_ASSEMBLER_NAME (decl);

      /* Remove the leading underscore. */
      oldsym = IDENTIFIER_POINTER (id);
      sl = strlen (oldsym);
      newsym = xmalloc (sl + 2);
      newsym [0] = '*';
      memcpy (newsym + 1, oldsym, sl + 1);

      SET_DECL_ASSEMBLER_NAME (decl, get_identifier (newsym));

      dfprintf ((stderr, "dbg: optlink %s -> %s\n", oldsym, newsym));
    }
  else if (lookup_attribute ("stdcall", TYPE_ATTRIBUTES (type)))
    {
      /* Mangle decl as the former assembler name modified with a
         suffix consisting of an atsign (@) followed by the number of bytes of
         arguments.
         For C++ the same applies as for optlink. See above. */

      int total = 0;

      /* If function does not have ellipsis as last argument, count total args size */
      if (TYPE_ARG_TYPES (type))
        if (TREE_VALUE (tree_last (TYPE_ARG_TYPES (type))) == void_type_node)
          {
            tree formal_type = TYPE_ARG_TYPES (type);

            while (TREE_VALUE (formal_type) != void_type_node)
              {
                int parm_size = TREE_INT_CST_LOW (TYPE_SIZE (TREE_VALUE (formal_type)));
                /* Must round up to include padding.  This is done the same
                   way as in store_one_arg.  */
                parm_size = ((parm_size + PARM_BOUNDARY - 1) / PARM_BOUNDARY * PARM_BOUNDARY);
                total += parm_size;
                formal_type = TREE_CHAIN (formal_type);
              }
          }

      id = DECL_ASSEMBLER_NAME (decl);
      oldsym = IDENTIFIER_POINTER (id);
      newsym = xmalloc (strlen (oldsym) + 10);
      sprintf (newsym, "%s@%d", oldsym, total / BITS_PER_UNIT);

      SET_DECL_ASSEMBLER_NAME (decl, get_identifier (newsym));

      dfprintf ((stderr, "dbg: stdcall %s -> %s\n", oldsym, newsym));
    }
  else
    rc = 0;

  recurse--;
  return rc;
}

#if 0 /* choose between the smaller non working version, and the
         large ugly one which is working. */

tree emx_handle_vacpp_attribute (tree *node, tree name, tree args,
  int flags, bool *no_add_attrs)
{
  tree *type;
  (void) args;
  (void) flags;

  dfprintf((stderr, "emx_handle_vacpp_attribute\n"));
  DUMP (*node);

  if (TREE_CODE (*node) != FUNCTION_TYPE
      && TREE_CODE (*node) != METHOD_TYPE
      && TREE_CODE (*node) != FIELD_DECL
      && TREE_CODE (*node) != TYPE_DECL)
    {
      warning ("`%s' attribute only applies to functions",
	       IDENTIFIER_POINTER (name));
      *no_add_attrs = true;
    }

  if (TARGET_64BIT)
    {
      warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
      *no_add_attrs = true;
    }

  /* Check if the attributes are compatible */
  if (ix86_check_append_attr (*node, name, no_add_attrs))
    return NULL_TREE;

  /* Now, a new type with our attribute please!
     If the FUNCTION_TYPE/METHOD_TYPE already have the attribute associated,
     there is no need to duplicate the type.
     @todo: I'm unsure wether or not we should do this from node and down... */
  for (type = node;
       TREE_CODE (*type) != FUNCTION_TYPE && TREE_CODE (*type) != METHOD_TYPE;
       type = &TREE_TYPE (*type))
      /* do nothing */;

  if (lookup_attribute (IDENTIFIER_POINTER (name), TYPE_ATTRIBUTES (*type)))
      return NULL_TREE;

  *type = build_type_copy (*type);
  *type = build_type_attribute_variant (*type, tree_cons (name, args, TYPE_ATTRIBUTES (*type)) );
  /**no_add_attrs = true; ?? */

  dfprintf((stderr, "emx_handle_vacpp_attribute: new type (%p)\n", (void*)type));
  DUMP (*node);
  return NULL_TREE;
}

#else

/* This is the working version which we belive duplicate some which could be
   done by the generic attribute stuff in GCC... */
tree emx_handle_vacpp_attribute (node, name, args, flags, no_add_attrs)
    tree *node, name, args; int flags; bool *no_add_attrs;
{
  tree type;

  dfprintf((stderr, "emx_handle_vacpp_attribute: node=%p\n", (void *)node));
  DUMP (*node);

  switch (TREE_CODE (*node))
    {
      /* Declarations!
         We need to attach the attribute to the type of the declaration.
         (Name mangling is done by emx_c_set_decl_assembler_name().)  */
      case FUNCTION_DECL:/* Function declaration. */
      case TYPE_DECL:    /* Function or function pointer type. */
      case FIELD_DECL:   /* Function pointer (or function?) as a struct, union or class member. */
      case PARM_DECL:    /* Function pointer as a parameter. */
      case VAR_DECL:     /* Function pointer variable. */
        /* If this is a type declaration with our attribute, we allow it
           only if it is a pointer-to-a-function type or a function type. */
        type = TREE_TYPE (*node);
        if (TREE_CODE (type) == POINTER_TYPE)
          type = TREE_TYPE(type);
        if (   TREE_CODE (type) != FUNCTION_TYPE
            && TREE_CODE (type) != METHOD_TYPE)
          {
            warning ("`%s' attribute only applies to functions and function types, not to '%T'.",
                     IDENTIFIER_POINTER (name), type);
            *no_add_attrs = true;
            break;
          }

        if (ix86_check_append_attr (type, name, no_add_attrs))
          break;

        /* If required we'll make a variant of the type for this attribute. */
        if (!lookup_attribute (IDENTIFIER_POINTER (name),
                               TYPE_ATTRIBUTES (type)))
          {
            tree *ptype = node;
            while (*ptype != type)
                ptype = &TREE_TYPE (*ptype);
            *ptype = build_type_copy (*ptype);
            *ptype = build_type_attribute_variant (
                *ptype, tree_cons (name, args, TYPE_ATTRIBUTES (*ptype)) );
            dfprintf((stderr, "emx_handle_vacpp_attribute: new type\n"));
            DUMP(*node);
          }
        else
          dfprintf((stderr, "emx_handle_vacpp_attribute: use old type\n"));
        *no_add_attrs = true;
        break;

      /* Types!
         For types involving functions we need to convince decl_attributes()
         (and its callers) to supply a declaration so we safely can change
         the type. */
      case POINTER_TYPE:
        /* We allow:
           This being the return type of a function which is coming soon.
           This being a function pointer which declaration is coming next.
           Everything else is considered inappropriate use of the attribute. */
        if (   !(flags & ATTR_FLAG_FUNCTION_NEXT)
            && (    !(flags & ATTR_FLAG_DECL_NEXT)
                ||  !(type = TREE_TYPE (*node))
                || (   TREE_CODE (type) != FUNCTION_TYPE
                    && TREE_CODE (type) != METHOD_TYPE)))
        {
            warning ("`%s' attribute only applies to functions and function types",
                     IDENTIFIER_POINTER (name));
            *no_add_attrs = true;
            break;
        }
        /* fall thru */
      case FUNCTION_TYPE:
      case METHOD_TYPE:
#if 0 /* bird: This fixes optlink/tst4.c and doesn't seem to break anything.
               If set to true we are in some cases imposing _Optlink onto following
               declarations. This is weird stuff!
            !! This problem is actually what we see with the other implementation!! */
        *no_add_attrs = true;
#endif
        return tree_cons (name, args, NULL_TREE);

      default:
        warning ("`%s' attribute only applies to functions and function types (code=%d)",
                 IDENTIFIER_POINTER (name), TREE_CODE (*node));
        *no_add_attrs = true;
        break;
    }

  return NULL_TREE;
}
#endif

void
emx_eh_frame_section ()
{
/*
  tree label = get_file_function_name ('F');
  The __ehInit undefined external here is to drag __ehInit/__ehInitDLL into
  the linking so DLLs are initiated correctly. (#577)
*/
  data_section ();
  ASM_OUTPUT_ALIGN (asm_out_file, floor_log2 (PTR_SIZE));
  fputs ("\t.stabs\t\"___eh_frame__\",24,0,0,Lframe1\n", asm_out_file); /* N_SETD */
  fputs ("\t.stabs\t\"___ehInit\",1,0,0,0\n", asm_out_file);  /* N_UNDEF | N_EXT */
}

/* Add a __POST$xxx label before epilogue if -mepilogue specified */
void emx_output_function_begin_epilogue (f)
    FILE *f;
{
  if (TARGET_EPILOGUE && TREE_PUBLIC (current_function_decl))
  {
    const char *func_label = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (current_function_decl));
    char *lbl = alloca (strlen (func_label) + 8);
    strcpy (lbl, "__POST$");
    if (func_label[0] == '*')
      func_label++;
    strcat (lbl, func_label);
    ASM_OUTPUT_LABEL (f, lbl);
  }
}


/* Checks if the function is using either optlink or system calling convention
   and returns 1 in those cases forcing the return value to be in memory.
   The problem (#631) was that structures less than 8 bytes were returned in
   registers. _System and _Optlink requires them to be passed in as a hidden
   parameter.
   @type is the return type.
   @fntype is function type, call expression, function declaration or null.
   The return value is 1 to force the return value into memory. Return 0 if we
   don't know. */

int emx_return_in_memory_with_fntype (type, fntype)
    tree type, fntype;
{
  dfprintf ((stderr, "trace: emx_return_in_memory_with_fntype\n"));
  /* (from aggregate_value_p() CVS trunk) */
  if (fntype)
    switch (TREE_CODE (fntype))
      {
      case CALL_EXPR:
        fntype = get_callee_fndecl (fntype);
        fntype = fntype ? TREE_TYPE (fntype) : 0;
        break;
      case FUNCTION_DECL:
        fntype = TREE_TYPE (fntype);
        break;
      case FUNCTION_TYPE:
      case METHOD_TYPE:
        break;
      case IDENTIFIER_NODE:
        fntype = 0;
        break;
      default:
        /* We don't expect other rtl types here.  */
        abort();
      }

  /* (future targetm.calls.return_in_memory additions) */
  if (fntype
   && AGGREGATE_TYPE_P (type)
   && (   lookup_attribute ("optlink", TYPE_ATTRIBUTES (fntype))
       || lookup_attribute ("system", TYPE_ATTRIBUTES (fntype))))
    {
      dfprintf((stderr, "emx_return_in_memory_with_fntype: returns 1\n"));
      return 1;
    }

  /* return ix86_return_in_memory (exp); - future */
  dfprintf((stderr, "emx_return_in_memory_with_fntype: returns 0\n"));
  return 0;
}

/* Strip only the leading encoding, leaving the stdcall suffix.  */

const char *
i386_emx_strip_name_encoding (str)
    const char *str;
{
  dfprintf ((stderr, "trace: i386_emx_strip_name_encoding(%s)\n", str));
  if (*str == DLL_IMPORT_EXPORT_PREFIX)
    str += 3;
  if (*str == '%')                      /* don't actually know what this is.. */
    str += 2;
  if (*str == '*')
    str += 1;
  return str;
}

/* Also strip the stdcall suffix.  */

const char * emx_strip_name_encoding_full (str)
    const char *str;
{
  const char *p;
  const char *ret;
  const char *name = i386_emx_strip_name_encoding (str);

  p = strchr (name, '@');
  if (p)
    ret = ggc_alloc_string (name, p - name);
  else 
    ret = name;
  dfprintf ((stderr, "trace: emx_strip_name_encoding_full: '%s' -> '%s'\n", str, ret));
  return ret;
}


/* copy&hack from winnt.c */

static tree associated_type PARAMS ((tree));
/*static const char * gen_stdcall_suffix PARAMS ((tree));*/
static int i386_emx_dllexport_p PARAMS ((tree));
static int i386_emx_dllimport_p PARAMS ((tree));
static void i386_emx_mark_dllexport PARAMS ((tree));
static void i386_emx_mark_dllimport PARAMS ((tree));

/* Handle a "dllimport" or "dllexport" attribute;
   arguments as in struct attribute_spec.handler.  */
tree
ix86_handle_dll_attribute (pnode, name, args, flags, no_add_attrs)
     tree * pnode;
     tree name;
     tree args;
     int flags;
     bool *no_add_attrs;
{
  tree node = *pnode;
  dfprintf((stderr, "trace: ix86_handle_dll_attribute\n"));
  DUMP (node);

  /* These attributes may apply to structure and union types being created,
     but otherwise should pass to the declaration involved.  */
  if (!DECL_P (node))
    {
      if (flags & ((int) ATTR_FLAG_DECL_NEXT | (int) ATTR_FLAG_FUNCTION_NEXT
		   | (int) ATTR_FLAG_ARRAY_NEXT))
	{
	  *no_add_attrs = true;
	  return tree_cons (name, args, NULL_TREE);
	}
      if (TREE_CODE (node) != RECORD_TYPE && TREE_CODE (node) != UNION_TYPE)
	{
	  warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
	  *no_add_attrs = true;
	}

      return NULL_TREE;
    }

  /* Report error on dllimport ambiguities seen now before they cause
     any damage.  */
  else if (is_attribute_p ("dllimport", name))
    {
      /* Like MS, treat definition of dllimported variables and
	 non-inlined functions on declaration as syntax errors.
	 We allow the attribute for function definitions if declared
	 inline, but just ignore it in i386_emx_dllimport_p.  */
      if (TREE_CODE (node) == FUNCTION_DECL  && DECL_INITIAL (node)
          && !DECL_INLINE (node))
	{
	  error_with_decl (node, "function `%s' definition is marked dllimport.");
	  *no_add_attrs = true;
	}

      else if (TREE_CODE (node) == VAR_DECL)
	{
	  if (DECL_INITIAL (node))
	    {
	      error_with_decl (node,"variable `%s' definition is marked dllimport.");
	      *no_add_attrs = true;
	    }

	  /* `extern' needn't be specified with dllimport.
	     Specify `extern' now and hope for the best.  Sigh.  */
	  DECL_EXTERNAL (node) = 1;
	  /* Also, implicitly give dllimport'd variables declared within
	     a function global scope, unless declared static.  */
	  if (current_function_decl != NULL_TREE && !TREE_STATIC (node))
  	    TREE_PUBLIC (node) = 1;
	}
    }

  /*  Report error if symbol is not accessible at global scope. */
  if (!TREE_PUBLIC (node)
      && (TREE_CODE (node) == VAR_DECL
	  || TREE_CODE (node) == FUNCTION_DECL))
    {
      error_with_decl (node, "external linkage required for symbol '%s' because of '%s' attribute.",
		       IDENTIFIER_POINTER (name));
      *no_add_attrs = true;
    }

  return NULL_TREE;
}

/* Handle a "shared" attribute;
   arguments as in struct attribute_spec.handler.  */
tree
ix86_handle_shared_attribute (node, name, args, flags, no_add_attrs)
     tree *node;
     tree name;
     tree args ATTRIBUTE_UNUSED;
     int flags ATTRIBUTE_UNUSED;
     bool *no_add_attrs;
{
  dfprintf ((stderr, "trace: ix86_handle_shared_attribute\n"));
  DUMP (*node);
  if (TREE_CODE (*node) != VAR_DECL)
    {
      warning ("`%s' attribute only applies to variables",
	       IDENTIFIER_POINTER (name));
      *no_add_attrs = true;
    }

  return NULL_TREE;
}

/* Return the type that we should use to determine if DECL is
   imported or exported.  */

static tree
associated_type (decl)
     tree decl;
{
  tree t = NULL_TREE;

  /* In the C++ frontend, DECL_CONTEXT for a method doesn't actually refer
     to the containing class.  So we look at the 'this' arg.  */
  if (TREE_CODE (TREE_TYPE (decl)) == METHOD_TYPE)
    {
#if 0
      /* Artificial methods are not affected by the import/export status of
	 their class unless they are virtual.  */
      if (! DECL_ARTIFICIAL (decl) || DECL_VINDEX (decl))
	t = TREE_TYPE (TREE_VALUE (TYPE_ARG_TYPES (TREE_TYPE (decl))));
#else /* from the CVS 3.4 branch. */
      /* Artificial methods are not affected by the import/export status
	 of their class unless they are COMDAT.  Implicit copy ctor's and
	 dtor's are not affected by class status but virtual and
	 non-virtual thunks are.  */
      if (!DECL_ARTIFICIAL (decl) || DECL_COMDAT (decl))
	t = TYPE_MAIN_VARIANT
	  (TREE_TYPE (TREE_VALUE (TYPE_ARG_TYPES (TREE_TYPE (decl)))));
      dfprintf((stderr, "trace: associated_type: decl=%p t=%p\n", decl, t));
      DUMP(decl);
      DUMP(t);
#endif 
    }
  else if (DECL_CONTEXT (decl)
	   && TREE_CODE_CLASS (TREE_CODE (DECL_CONTEXT (decl))) == 't')
    t = DECL_CONTEXT (decl);

  return t;
}

/* Return nonzero if DECL is a dllexport'd object.  */

static int
i386_emx_dllexport_p (decl)
     tree decl;
{
  tree exp;

  if (TREE_CODE (decl) != VAR_DECL
      && TREE_CODE (decl) != FUNCTION_DECL)
    return 0;
  exp = lookup_attribute ("dllexport", DECL_ATTRIBUTES (decl));
  if (exp)
    return 1;

  /* Class members get the dllexport status of their class.  */
  if (associated_type (decl))
    {
      DUMP (associated_type (decl));
      DUMP (decl);
      exp = lookup_attribute ("dllexport",
			      TYPE_ATTRIBUTES (associated_type (decl)));
      if (exp)
	return 1;
    }

  return 0;
}

/* Return nonzero if DECL is a dllimport'd object.  */

static int
i386_emx_dllimport_p (decl)
     tree decl;
{
  tree imp;
  int context_imp = 0;

  if (TREE_CODE (decl) == FUNCTION_DECL
      && TARGET_NOP_FUN_DLLIMPORT)
    return 0;

  if (TREE_CODE (decl) != VAR_DECL
      && TREE_CODE (decl) != FUNCTION_DECL)
    return 0;

  imp = lookup_attribute ("dllimport", DECL_ATTRIBUTES (decl));

  /* Class members get the dllimport status of their class.  */
  if (!imp && associated_type (decl))
    {
      imp = lookup_attribute ("dllimport",
			      TYPE_ATTRIBUTES (associated_type (decl)));
      if (imp)
	context_imp = 1;
    }

  if (imp)
    {
      /* Don't mark defined functions as dllimport.  If the definition
	 itself was marked with dllimport, than ix86_handle_dll_attribute
	 reports an error. This handles the case when the definition
	 overrides an earlier declaration.  */
      if (TREE_CODE (decl) ==  FUNCTION_DECL && DECL_INITIAL (decl)
	  && !DECL_INLINE (decl))
	{
	   /* Don't warn about artificial methods.  */
	  if (!DECL_ARTIFICIAL (decl))
	    warning_with_decl (decl,"function '%s' is defined after prior declaration as dllimport: attribute ignored.");
	  return 0;
	}

      /* We ignore the dllimport attribute for inline member functions.
	 This differs from MSVC behaviour which treats it like GNUC
     	 'extern inline' extension.   */
      else if (TREE_CODE (decl) == FUNCTION_DECL && DECL_INLINE (decl))
        {
	  if (extra_warnings)
	    warning_with_decl (decl, "inline function '%s' is declared as dllimport: attribute ignored.");
	  return 0;
	}

      /*  Don't allow definitions of static data members in dllimport class,
	  Just ignore attribute for vtable data.  */
      else if (TREE_CODE (decl) == VAR_DECL
	       && TREE_STATIC (decl) && TREE_PUBLIC (decl)
	       && !DECL_EXTERNAL (decl) && context_imp)
	{
	  if (!DECL_VIRTUAL_P (decl))
	      error_with_decl (decl, "definition of static data member '%s' of dllimport'd class.");
           return 0;
	}

#if 0
      /* Since we can't treat a pointer to a dllimport'd symbol as a
	 constant address, we turn off the attribute on C++ virtual
	 methods to allow creation of vtables using thunks. */
      else if (TREE_CODE (TREE_TYPE (decl)) == METHOD_TYPE
	       && (DECL_VIRTUAL_P (decl)))
           return 0;
#else /* From the CVS 3.4 branch */
      /* Since we can't treat a pointer to a dllimport'd symbol as a
	 constant address, we turn off the attribute on C++ virtual
	 methods to allow creation of vtables using thunks.  Don't mark
	 artificial methods either (in associated_type, only COMDAT
	 artificial method get import status from class context).  */
      else if (TREE_CODE (TREE_TYPE (decl)) == METHOD_TYPE
	       && (DECL_VIRTUAL_P (decl) || DECL_ARTIFICIAL (decl)))
	return 0;
#endif 

      return 1;
    }

  return 0;
}

/* Return nonzero if SYMBOL is marked as being dllexport'd.  */

int
i386_emx_dllexport_name_p (const char *symbol EMX_DBG_LOC_DECL)
{
  dfprintf ((stderr,
             "trace: i386_emx_dllexport_name_p: %s %s(%d)\n"
             "           symbol=%s\n", EMX_DBG_LOC_RARGS, symbol));
  return symbol[0] == DLL_IMPORT_EXPORT_PREFIX
         && symbol[1] == 'e' && symbol[2] == '.';
}

/* Return nonzero if SYMBOL is marked as being dllimport'd.  */

int
i386_emx_dllimport_name_p (const char *symbol EMX_DBG_LOC_DECL)
{
  dfprintf ((stderr,
             "trace: i386_emx_dllimport_name_p: %s %s(%d)\n"
             "           symbol='%s'\n", EMX_DBG_LOC_RARGS, symbol));
  return symbol[0] == DLL_IMPORT_EXPORT_PREFIX
         && symbol[1] == 'i' && symbol[2] == '.';
}

/* Mark a DECL as being dllexport'd.
   Note that we override the previous setting (eg: dllimport).  */

static void
i386_emx_mark_dllexport (decl)
     tree decl;
{
  const char *oldname;
  char  *newname;
  rtx rtlname;
  tree idp;
  dfprintf ((stderr, "trace: i386_emx_mark_dllexport\n"));
  DUMP (decl);

  rtlname = XEXP (DECL_RTL (decl), 0);
  if (GET_CODE (rtlname) == SYMBOL_REF)
    oldname = XSTR (rtlname, 0);
  else if (GET_CODE (rtlname) == MEM
	   && GET_CODE (XEXP (rtlname, 0)) == SYMBOL_REF)
    oldname = XSTR (XEXP (rtlname, 0), 0);
  else
    abort ();
  if (i386_emx_dllimport_name_p (oldname EMX_DBG_LOC_INARGS))
    {
      warning_with_decl (decl,"inconsistent dll linkage for '%s': dllexport assumed.");
     /* Remove DLL_IMPORT_PREFIX.  */
      oldname += 3; /* not 9 since we're not doing the __imp_ bit. */
      DECL_NON_ADDR_CONST_P (decl) = 0;
    }
  else if (i386_emx_dllexport_name_p (oldname EMX_DBG_LOC_INARGS))
    return; /* already done */

  newname = alloca (strlen (oldname) + 4);
  sprintf (newname, "%ce.%s", DLL_IMPORT_EXPORT_PREFIX, oldname);

  /* We pass newname through get_identifier to ensure it has a unique
     address.  RTL processing can sometimes peek inside the symbol ref
     and compare the string's addresses to see if two symbols are
     identical.  */
  idp = get_identifier (newname);

  XEXP (DECL_RTL (decl), 0) =
    gen_rtx (SYMBOL_REF, Pmode, IDENTIFIER_POINTER (idp));
}

/* Mark a DECL as being dllimport'd.  */

static void
i386_emx_mark_dllimport (decl)
     tree decl;
{
  const char *oldname;
  /*char  *newname;*/
  /*tree idp;*/
  rtx rtlname/*, newrtl*/;
  dfprintf ((stderr, "trace: i386_emx_mark_dllimport\n"));

  rtlname = XEXP (DECL_RTL (decl), 0);
  if (GET_CODE (rtlname) == SYMBOL_REF)
    oldname = XSTR (rtlname, 0);
  else if (GET_CODE (rtlname) == MEM
	   && GET_CODE (XEXP (rtlname, 0)) == SYMBOL_REF)
    oldname = XSTR (XEXP (rtlname, 0), 0);
  else
    abort ();
  if (i386_emx_dllexport_name_p (oldname EMX_DBG_LOC_INARGS))
    {
      error ("`%s' declared as both exported to and imported from a DLL",
             IDENTIFIER_POINTER (DECL_NAME (decl)));
      return;
    }
  else if (i386_emx_dllimport_name_p (oldname EMX_DBG_LOC_INARGS))
    {
      /* Already done, but do a sanity check to prevent assembler errors. */
      if (!DECL_EXTERNAL (decl) || !TREE_PUBLIC (decl))
	{
	  error_with_decl (decl, "failure in redeclaration of '%s': dllimport'd symbol lacks external linkage.");
	  abort();
	}
    return;
    }

#if 0 /* we're only tagging along for the error handling but need no special handling. */
  newname = alloca (strlen (oldname) + 11);
  sprintf (newname, "%ci.%s", DLL_IMPORT_EXPORT_PREFIX, oldname);

  /* We pass newname through get_identifier to ensure it has a unique
     address.  RTL processing can sometimes peek inside the symbol ref
     and compare the string's addresses to see if two symbols are
     identical.  */
  idp = get_identifier (newname);

  newrtl = gen_rtx (MEM, Pmode,
		    gen_rtx (SYMBOL_REF, Pmode,
			     IDENTIFIER_POINTER (idp)));
  XEXP (DECL_RTL (decl), 0) = newrtl;

  /* Can't treat a pointer to this as a constant address */
  DECL_NON_ADDR_CONST_P (decl) = 1;
#endif 
}

#if 0
/* Return string which is the former assembler name modified with a
   suffix consisting of an atsign (@) followed by the number of bytes of
   arguments */

const char *
gen_stdcall_suffix (decl)
  tree decl;
{
  int total = 0;
  /* ??? This probably should use XSTR (XEXP (DECL_RTL (decl), 0), 0) instead
     of DECL_ASSEMBLER_NAME.  */
  const char *asmname = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (decl));
  char *newsym;

  if (TYPE_ARG_TYPES (TREE_TYPE (decl)))
    if (TREE_VALUE (tree_last (TYPE_ARG_TYPES (TREE_TYPE (decl))))
        == void_type_node)
      {
	tree formal_type = TYPE_ARG_TYPES (TREE_TYPE (decl));

	while (TREE_VALUE (formal_type) != void_type_node)
	  {
	    int parm_size
	      = TREE_INT_CST_LOW (TYPE_SIZE (TREE_VALUE (formal_type)));
	    /* Must round up to include padding.  This is done the same
	       way as in store_one_arg.  */
	    parm_size = ((parm_size + PARM_BOUNDARY - 1)
			 / PARM_BOUNDARY * PARM_BOUNDARY);
	    total += parm_size;
	    formal_type = TREE_CHAIN (formal_type);
	  }
      }

  newsym = xmalloc (strlen (asmname) + 10);
  sprintf (newsym, "%s@%d", asmname, total/BITS_PER_UNIT);
  return IDENTIFIER_POINTER (get_identifier (newsym));
}
#endif

void
i386_emx_encode_section_info (decl, first)
     tree decl;
     int first ATTRIBUTE_UNUSED;
{
  dfprintf ((stderr, "trace: i386_emx_encode_section_info\n"));
  DUMP (decl);

  /* This bit is copied from i386.h.  */
  if (optimize > 0 && TREE_CONSTANT (decl)
      && (!flag_writable_strings || TREE_CODE (decl) != STRING_CST))
    {
      rtx rtl = (TREE_CODE_CLASS (TREE_CODE (decl)) != 'd'
                 ? TREE_CST_RTL (decl) : DECL_RTL (decl));
      SYMBOL_REF_FLAG (XEXP (rtl, 0)) = 1;
    }

#if 0
  if (TREE_CODE (decl) == FUNCTION_DECL)
    if (lookup_attribute ("stdcall",
			  TYPE_ATTRIBUTES (TREE_TYPE (decl))))
      XEXP (DECL_RTL (decl), 0) =
	gen_rtx (SYMBOL_REF, Pmode, gen_stdcall_suffix (decl));
#endif

  /* Mark the decl so we can tell from the rtl whether the object is
     dllexport'd or dllimport'd.  This also handles dllexport/dllimport
     override semantics.  */

  if (i386_emx_dllexport_p (decl))
    i386_emx_mark_dllexport (decl);
  else if (i386_emx_dllimport_p (decl))
    i386_emx_mark_dllimport (decl);
  /* It might be that DECL has already been marked as dllimport, but a
     subsequent definition nullified that.  The attribute is gone but
     DECL_RTL still has (DLL_IMPORT_EXPORT_PREFIX)i._imp__foo.  We need
     to remove that. Ditto for the DECL_NON_ADDR_CONST_P flag.  */
  else if ((TREE_CODE (decl) == FUNCTION_DECL
	    || TREE_CODE (decl) == VAR_DECL)
	   && DECL_RTL (decl) != NULL_RTX
	   && GET_CODE (DECL_RTL (decl)) == MEM
	   && GET_CODE (XEXP (DECL_RTL (decl), 0)) == MEM
	   && GET_CODE (XEXP (XEXP (DECL_RTL (decl), 0), 0)) == SYMBOL_REF
	   && i386_emx_dllimport_name_p (XSTR (XEXP (XEXP (DECL_RTL (decl), 0), 0), 0)  EMX_DBG_LOC_INARGS))
    {
      const char *oldname = XSTR (XEXP (XEXP (DECL_RTL (decl), 0), 0), 0);
      tree idp = get_identifier (oldname + 9);
      rtx newrtl = gen_rtx (SYMBOL_REF, Pmode, IDENTIFIER_POINTER (idp));

      warning_with_decl (decl, "'%s' %s after being referenced with dllimport linkage.",
	         	 (DECL_INITIAL (decl) || !DECL_EXTERNAL (decl))
			 ? "defined locally" : "redeclared without dllimport attribute");

      XEXP (DECL_RTL (decl), 0) = newrtl;

      DECL_NON_ADDR_CONST_P (decl) = 0;

      /* We previously set TREE_PUBLIC and DECL_EXTERNAL.
	 We leave these alone for now.  */
    }
}

/* [snip] */

#if 0 /* don't think we need this. */
/* Keep a list of external functions.  */

struct extern_list
{
  struct extern_list *next;
  const char *name;
};

static struct extern_list *extern_head;

/* Assemble an external function reference.  We need to keep a list of
   these, so that we can output the function types at the end of the
   assembly.  We can't output the types now, because we might see a
   definition of the function later on and emit debugging information
   for it then.  */

void
i386_emx_record_external_function (name)
     const char *name;
{
  struct extern_list *p;

  p = (struct extern_list *) xmalloc (sizeof *p);
  p->next = extern_head;
  p->name = name;
  extern_head = p;
}
#endif

/* Keep a list of exported symbols.  */

struct export_list
{
  struct export_list *next;
  tree decl;
  const char *name;
  const char *asm_name;
  const char *exported_name;
  unsigned    ordinal;
  int is_data;		/* used to type tag exported symbols.  */
};

static struct export_list *export_head;

/* Assemble an export symbol entry.  We need to keep a list of
   these, so that we can output the export list at the end of the
   assembly.  We used to output these export symbols in each function,
   but that causes problems with GNU ld when the sections are
   linkonce.  */

void
i386_emx_record_exported_symbol (tree decl, const char *name, int is_data EMX_DBG_LOC_DECL)
{
  struct export_list *p;
  dfprintf ((stderr,
             "trace: i386_emx_record_exported_symbol: %s %s(%d)\n"
             "           name='%s' is_data=%d\n", EMX_DBG_LOC_RARGS, name, is_data));
  DUMP (decl);

  p = (struct export_list *) xmalloc (sizeof *p);
  p->next = export_head;
  p->decl = decl;
  p->name = name;
  p->asm_name = NULL;
  p->exported_name = NULL;
  p->ordinal = 0;
  p->is_data = is_data;
  export_head = p;
}

/* This is called at the end of assembly.  For each external function
   which has not been defined, we output a declaration now.  We also
   output the .drectve section.  */

void
i386_emx_asm_file_end (file)
     FILE *file;
{
/*  struct extern_list *p; */

  ix86_asm_file_end (file);

#if 0 /* don't think we need this */
  for (p = extern_head; p != NULL; p = p->next)
    {
      tree decl;

      decl = get_identifier (p->name);

      /* Positively ensure only one declaration for any given symbol.  */
      if (! TREE_ASM_WRITTEN (decl) && TREE_SYMBOL_REFERENCED (decl))
	{
	  TREE_ASM_WRITTEN (decl) = 1;
	  i386_emx_declare_function_type (file, p->name, TREE_PUBLIC (decl));
	}
    }
#endif
  if (export_head)
    {
      struct export_list *q;
      fprintf (file, "\t%s Exports: \"<exportname>,<ordinal>=<asmname>,<code|data>\", N_EXP,0,0,-42\n", ASM_COMMENT_START);
      for (q = export_head; q != NULL; q = q->next)
	{
          static const char * const types[2] = {",code", ",data"};
          tree decl = q->decl;
          if (decl)
            {
              DUMP(decl);
              /* breaks thunks. Windows ain't doing this either.
                 If it turns out to break stuff, try add "|| DECL_COMDAT (decl)".
              if (!TREE_STATIC (decl))
                {
                  dfprintf ((stderr, "dbg: skipping %p. !TREE_STATIC\n", decl));
                  continue;
                }
               */
              if (!q->asm_name)
                q->asm_name = q->exported_name ? q->exported_name
                  : IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (decl));
            }
          if (!q->asm_name)
            q->asm_name = q->exported_name ? q->exported_name : q->name;
          if (!q->exported_name && !q->ordinal)
            q->exported_name = q->asm_name ? q->asm_name : q->name;
          fprintf (file, "\t.stabs\t\"");
          ASM_OUTPUT_LABELREF(file, q->exported_name);
          fprintf (file, ",%u=", q->ordinal);
          ASM_OUTPUT_LABELREF(file, q->asm_name);
          fprintf (file, "%s\",0x6c,0,0,-42\t%s %s\n", /* 0x6c == N_EXP */
                   types[q->is_data], ASM_COMMENT_START, q->name);
	}
    }
}


