/* $Id: safesems.c 1986 2005-05-09 05:02:45Z bird $ */
/** @file
 *
 * LIBC SYS Backend - Internal Signal-Safe Semaphores.
 *
 * Copyright (c) 2005 knut st. osmundsen <bird@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#define INCL_DOSSEMAPHORES
#define INCL_DOSEXCEPTIONS
#define INCL_DOSERRORS
#define INCL_FSMACROS
#define INCL_EXAPIS
#include <os2emx.h>
#include <sys/errno.h>
#include "syscalls.h"
#include <InnoTekLIBC/thread.h>
#include <InnoTekLIBC/backend.h>
#define __LIBC_LOG_GROUP    __LIBC_LOG_GRP_BACK_IPC
#include <InnoTekLIBC/logstrict.h>


/**
 * Creates a safe mutex sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   phmtx       Where to store the semaphore handle.
 * @param   fShared     Set if the semaphore should be sharable between processes.
 */
int __libc_Back_safesemMtxCreate(uintptr_t *phmtx, int fShared)
{
    FS_VAR_SAVE_LOAD();
    HMTX hmtx = NULLHANDLE;
    int rc = DosCreateMutexSemEx(NULL, &hmtx, fShared ? DC_SEM_SHARED : 0, FALSE);
    FS_RESTORE();
    if (rc)
        return -__libc_native2errno(rc);
    *phmtx = hmtx;
    return 0;
}


/**
 * Opens a shared safe mutex sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hmtx        The semaphore handle.
 */
int __libc_Back_safesemMtxOpen(uintptr_t hmtx)
{
    FS_VAR_SAVE_LOAD();
    HMTX hmtxOS2 = hmtx;
    int rc = DosOpenMutexSemEx(NULL, &hmtxOS2);
    FS_RESTORE();
    if (rc)
        return -__libc_native2errno(rc);
    return 0;
}


/**
 * Closes a shared safe mutex sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hmtx        The semaphore handle.
 */
int __libc_Back_safesemMtxClose(uintptr_t hmtx)
{
    FS_VAR_SAVE_LOAD();
    int rc = DosCloseMutexSemEx(hmtx);
    FS_RESTORE();
    if (rc)
        return -__libc_native2errno(rc);
    return 0;
}


/**
 * This function checks that there is at least 2k of writable
 * stack available. If there isn't, a crash is usually the
 * result.
 * @internal
 */
static int stackChecker(void)
{
    char volatile *pch = alloca(2048);
    if (!pch)
        return -1;
    /* With any luck the compiler doesn't optimize this away. */
    pch[0] = pch[1024] = pch[2044] = 0x7f;
    return 0;
}


/**
 * Locks a mutex semaphore.
 *
 * @returns 0 on success.
 * @returns Negative errno on failure.
 * @param   hmtx    Handle to the mutex.
 */
int __libc_Back_safesemMtxLock(uintptr_t hmtx)
{
    LIBCLOG_ENTER("hmtx=%#x\n", hmtx);
    ULONG       ul;
    int         rc;
    FS_VAR();

    /*
     * Check stack.
     */
    if (stackChecker())
    {
        LIBC_ASSERTM_FAILED("Too little stack left!\n");
        LIBCLOG_RETURN_INT(-EFAULT);
    }

    /*
     * Request semaphore and enter "must complete section" to avoid signal trouble.
     */
    FS_SAVE_LOAD();
    rc = DosRequestMutexSem(hmtx, 30*1000);
    DosEnterMustComplete(&ul);
    if (!rc)
    {
        FS_RESTORE();
        LIBCLOG_RETURN_INT(0);
    }

    /* failure out */
    DosExitMustComplete(&ul);
    LIBC_ASSERTM_FAILED("DosRequestMutexSem(%x) failed with rc=%d!\n", hmtx, rc);
    rc = -__libc_native2errno(rc);
    FS_RESTORE();
    LIBCLOG_RETURN_INT(rc);
}


/**
 * Unlocks a mutex semaphore.
 *
 * @returns 0 on success.
 * @returns Negative errno on failure.
 * @param   hmtx    Handle to the mutex.
 */
int __libc_Back_safesemMtxUnlock(uintptr_t hmtx)
{
    LIBCLOG_ENTER("hmtx=%#x\n", hmtx);
    ULONG   ul = 0;
    int     rc;
    FS_VAR();

    /*
     * Release the semaphore.
     */
    FS_SAVE_LOAD();
    rc = DosReleaseMutexSem(hmtx);
    if (rc)
    {
        FS_RESTORE();
        LIBC_ASSERTM_FAILED("DosReleaseMutexSem(%x) -> %d\n", hmtx, rc);
        rc = -__libc_native2errno(rc);
        LIBCLOG_RETURN_INT(rc);
    }

    DosExitMustComplete(&ul);
    FS_RESTORE();
    LIBCLOG_RETURN_INT(0);
}



/**
 * Creates a safe event sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   phev        Where to store the semaphore handle.
 * @param   fShared     Set if the semaphore should be sharable between processes.
 */
int __libc_Back_safesemEvCreate(uintptr_t *phev, int fShared)
{
    FS_VAR_SAVE_LOAD();
    HEV hev = NULLHANDLE;
    int rc = DosCreateEventSemEx(NULL, &hev, fShared ? DC_SEM_SHARED : 0, FALSE);
    FS_RESTORE();
    if (rc)
        return -__libc_native2errno(rc);
    *phev = hev;
    return 0;
}


/**
 * Opens a shared safe event sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hev         The semaphore handle.
 */
int __libc_Back_safesemEvOpen(uintptr_t hev)
{
    FS_VAR_SAVE_LOAD();
    HMTX hevOS2 = hev;
    int rc = DosOpenEventSemEx(NULL, &hevOS2);
    FS_RESTORE();
    if (rc)
        return -__libc_native2errno(rc);
    return 0;
}


/**
 * Closes a shared safe mutex sem.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hev         The semaphore handle.
 */
int __libc_Back_safesemEvClose(uintptr_t hev)
{
    FS_VAR_SAVE_LOAD();
    int rc = DosCloseEventSemEx(hev);
    FS_RESTORE();
    if (rc)
        return -__libc_native2errno(rc);
    return 0;
}



/**
 * Arguments to semEvSleepSignalCallback().
 */
struct SignalArgs
{
    /** Set if pfnComplete have already been executed. */
    volatile int            fDone;
    /** Number of attempts at doing it. (in case of crashes in pfnComplete) */
    volatile int            cTries;
    /** Callback to execute. */
    void                  (*pfnComplete)(void *pvUser);
    /** User arg. */
    void                   *pvUser;
};

/**
 * Signal notification callback.
 */
static void semEvSleepSignalCallback(int iSignal, void *pvUser)
{
    struct SignalArgs *pArgs = (struct SignalArgs *)pvUser;
    if (!pArgs->fDone && pArgs->cTries++ < 5)
    {
        pArgs->pfnComplete(pArgs->pvUser);
        pArgs->fDone = 1;
    }
}

/**
 * Sleep on a semaphore.
 *
 * This is the most difficult thing we're doing in this file.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 *
 * @param   hev         The semaphore to sleep on.
 * @param   hmtx        The semaphore protecting hev and which is to be unlocked while
 *                      sleeping and reaquired upon waking up.
 * @param   pfnComplete Function to execute on signal and on wait completion.
 * @param   pvUser      User argument to pfnComplete.
 */
int __libc_Back_safesemEvSleep(uintptr_t hev, uintptr_t hmtx, void (*pfnComplete)(void *pvUser), void *pvUser)
{
    LIBCLOG_ENTER("hev=%#x hmtx=%#x pfnComplete=%p pvUser=%p\n", hev, hmtx, (void *)pfnComplete, pvUser);

    /*
     * Setup signal notification.
     */
    __LIBC_PTHREAD pThrd = __libc_threadCurrentNoAuto();
    int rc;
    if (pThrd)
    {
        struct SignalArgs Args;
        Args.fDone          = 0;
        Args.cTries         = 0;
        Args.pfnComplete    = pfnComplete;
        Args.pvUser         = pvUser;
        pThrd->pvSigCallbackUser = &Args;
        pThrd->pfnSigCallback    = semEvSleepSignalCallback;

        /*
         * Reset the event semaphore.
         */
        ULONG ulIgnore;
        FS_VAR_SAVE_LOAD();
        rc = DosResetEventSem(hev, &ulIgnore);
        if (!rc || rc == ERROR_ALREADY_RESET)
        {
            /*
             * Release the sempahore and exit the must complete section.
             */
            rc = __libc_Back_safesemMtxUnlock(hmtx);
            if (!rc)
            {
                /*
                 * Now, lets wait.
                 */
                DosWaitEventSem(hev, SEM_INDEFINITE_WAIT);

                /*
                 * Regain access to the semaphore and must complete section
                 * before checking if we got an interrupt or not.
                 */
                rc = __libc_Back_safesemMtxLock(hmtx);
                if (!Args.fDone)
		    pfnComplete(pvUser);
                else if (!rc)
                    rc = -EINTR;
            }
        }
        else
            rc = -__libc_native2errno(rc);
        FS_RESTORE();
    }
    else
        rc = -ENOSYS;

    LIBCLOG_RETURN_INT(rc);
}


/**
 * Wakes up all threads sleeping on a given event semaphore.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   hev         Event semaphore to post.
 */
int __libc_Back_safesemEvWakeup(uintptr_t hev)
{
    LIBCLOG_ENTER("hev=%#x\n", hev);
    FS_VAR_SAVE_LOAD();
    int rc = DosPostEventSem(hev);
    FS_RESTORE();
    if (rc)
    {
        if (rc == ERROR_ALREADY_POSTED || rc == ERROR_TOO_MANY_POSTS)
            rc = 0;
        else
            rc = -__libc_native2errno(rc);
    }
    LIBCLOG_RETURN_INT(rc);
}

