/* $Id: b_fsFileModeSetFH.c 1942 2005-05-01 07:37:14Z bird $ */
/** @file
 *
 * LIBC SYS Backend - fchmod.
 *
 * Copyright (c) 2005 knut st. osmundsen <bird@innotek.de>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_FSMACROS
#include <os2emx.h>
#include "b_fs.h"
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <emx/io.h>
#include <emx/syscalls.h>
#include <limits.h>
#include "syscalls.h"
#include <InnoTekLIBC/libc.h>
#include <InnoTekLIBC/backend.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_BACK_FS
#include <InnoTekLIBC/logstrict.h>



/**
 * Sets the file access mode of a file by filehandle.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   fh      Handle to file.
 * @param   Mode    The filemode.
 */
int __libc_Back_fsFileModeSetFH(int fh, mode_t Mode)
{
    LIBCLOG_ENTER("fh=%d Mode=%#x\n", fh, Mode);

    /*
     * Get filehandle.
     */
    PLIBCFH pFH;

    int rc = __libc_FHEx(fh, &pFH);
    if (!pFH)
        LIBCLOG_RETURN_INT(rc);

    /*
     * Check the type.
     */
    switch (pFH->fFlags & __LIBC_FH_TYPEMASK)
    {
        /* fail */
        case F_SOCKET:
        case F_PIPE: /* treat as socket for now */
            LIBCLOG_RETURN_INT(-EINVAL);
        /* ignore */
        case F_DEV:
            LIBCLOG_RETURN_INT(0);

        /* treat */
        default:
        case F_FILE:
            break;
    }

    if (!pFH->pOps)
    {
        /*
         * Standard OS/2 file handle.
         */
        FS_VAR();
        FS_SAVE_LOAD();
        union
        {
            FILESTATUS4     fsts4;
            FILESTATUS4L    fsts4L;
        } info;
#if OFF_MAX > LONG_MAX
        int     fLarge = 0;
#endif

        /*
         * Get file info.
         * We have a little hack here, temporarily, write only files
         * cannot read EAs.
         */
#if OFF_MAX > LONG_MAX
        if (__libc_gpfnDosOpenL)
        {
            rc = DosQueryFileInfo(fh, FIL_QUERYEASIZEL, &info, sizeof(info.fsts4L));
            if (rc)
            {
                rc = DosQueryFileInfo(fh, FIL_STANDARDL, &info, sizeof(FILESTATUS3L));
                info.fsts4L.cbList = ~0;
            }
            fLarge = 1;
        }
        else
#endif
        {
            rc = DosQueryFileInfo(fh, FIL_QUERYEASIZE, &info, sizeof(info.fsts4));
            if (rc)
            {
                rc = DosQueryFileInfo(fh, FIL_STANDARD, &info, sizeof(FILESTATUS3));
                info.fsts4.cbList = ~0;
            }
        }
        if (rc)
        {
            FS_RESTORE();
            rc = -__libc_native2errno(rc);
            LIBCLOG_RETURN_INT(rc);
        }

        /* If in unix mode we'll check the EAs (if any). */
        if (    !__libc_gfNoUnix
            && (fLarge ? info.fsts4L.cbList : info.fsts4.cbList) >= LIBC_UNIX_EA_MIN)
        {
            //__libc_back_fsUnixAttribsGet(fh, 0, pStat);
            /* Update unix attributes */
        }

        /*
         * Update OS/2 bits.
         */
#if OFF_MAX > LONG_MAX
        if (fLarge)
        {
            if (Mode & S_IWRITE)
                info.fsts4L.attrFile &= ~FILE_READONLY;
            else
                info.fsts4L.attrFile = FILE_READONLY;
            rc = DosSetFileInfo(fh, FIL_STANDARDL, &info, sizeof(info.fsts4L) - sizeof(info.fsts4L.cbList));
        }
        else
#endif
        {
            if (Mode & S_IWRITE)
                info.fsts4.attrFile &= ~FILE_READONLY;
            else
                info.fsts4.attrFile |= FILE_READONLY;
            rc = DosSetFileInfo(fh, FIL_STANDARD, &info, sizeof(info.fsts4) - sizeof(info.fsts4.cbList));
        }
        FS_RESTORE();
        if (rc)
        {
            rc = -__libc_native2errno(rc);
            LIBCLOG_RETURN_INT(rc);
        }
    }
    else
    {
        /*
         * Non-standard handle - fail.
         */
        LIBCLOG_RETURN_INT(-EOPNOTSUPP);
    }

    LIBCLOG_RETURN_INT(0);
}

