/* $Id: b_ioFileOpen.c 1750 2004-12-21 01:38:04Z bird $ */
/** @file
 *
 * LIBC SYS Backend - open.
 *
 * Copyright (c) 2003-2004 knut st. osmundsen <bird@innotek.de>
 * Copyright (c) 1992-1996 by Eberhard Mattes
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_DOSERRORS
#define INCL_FSMACROS
#include <os2emx.h>
#include "b_fs.h"
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <sys/stat.h>
#include <emx/syscalls.h>
#include <emx/io.h>
#include "syscalls.h"
#include <InnoTekLIBC/backend.h>
#include <InnoTekLIBC/libc.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_BACK_IO
#include <InnoTekLIBC/logstrict.h>


/**
 * Opens a file.
 *
 * @returns Filehandle to the opened file on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszFile     Path to the file.
 * @param   fFlags      Open flags.
 * @param   cbInitial   Initial filesize.
 * @param   Mode        The specified permission mask.
 * @param   fLibc       LIBC filehandle flags.
 * @param   ppFH        Where to store the LIBC filehandle structure which was created
 *                      for the opened file.
 */
int __libc_Back_ioFileOpen(const char *pszFile, int fFlags, off_t cbInitial, mode_t Mode, unsigned fLibc, PLIBCFH *ppFH)
{
    LIBCLOG_ENTER("pszFile=%s fFlags=%#x cbInitial=%lld Mode=0%o fLibc=%#x ppFH=%p\n",
                  pszFile, fFlags, cbInitial, Mode, fLibc, (void*)ppFH);
    FS_VAR();

    /*
     * Extract the access mode and sharing mode bits.
     */
    ULONG flOpenMode = fFlags & 0x77;

    /*
     * File O_NOINHERIT and O_SYNC.
     */
    if (fFlags & _SO_NOINHERIT)
        flOpenMode |= OPEN_FLAGS_NOINHERIT;
    if (fFlags & _SO_SYNC)
        flOpenMode |= OPEN_FLAGS_WRITE_THROUGH;

    /*
     * Extract the file flAttribute bits.
     */
    ULONG flAttr = (fFlags >> 8) & 0xff;
    if (__libc_gfsUMask & S_IWUSR)
        flAttr |= FILE_READONLY;

    /*
     * Translate ERROR_OPEN_FAILED to ENOENT unless O_EXCL is set (see below).
     */
    int rcOpenFailed = -ENOENT;

    /*
     * Compute `flOpenFlags' depending on `fFlags'.  Note that _SO_CREAT is
     * set for O_CREAT.
     */
    ULONG flOpenFlags;
    if (fFlags & _SO_CREAT)
    {
        if (fFlags & _SO_EXCL)
        {
            flOpenFlags = OPEN_ACTION_FAIL_IF_EXISTS | OPEN_ACTION_CREATE_IF_NEW;
            rcOpenFailed = -EEXIST;
        }
        else if (fFlags & _SO_TRUNC)
            flOpenFlags = OPEN_ACTION_REPLACE_IF_EXISTS | OPEN_ACTION_CREATE_IF_NEW;
        else
            flOpenFlags = OPEN_ACTION_OPEN_IF_EXISTS | OPEN_ACTION_CREATE_IF_NEW;
    }
    else if (fFlags & _SO_TRUNC)
        flOpenFlags = OPEN_ACTION_REPLACE_IF_EXISTS | OPEN_ACTION_FAIL_IF_NEW;
    else
        flOpenFlags = OPEN_ACTION_OPEN_IF_EXISTS | OPEN_ACTION_FAIL_IF_NEW;

    if (!(fFlags & _SO_SIZE))
        cbInitial = 0;

    /*
     * Resolve the specified file path.
     */
    char szNativePath[PATH_MAX];
    int rc = __libc_back_fsResolve(pszFile,
                                   (flOpenFlags & OPEN_ACTION_FAIL_IF_NEW
                                        ? BACKFS_FLAGS_RESOLVE_FULL
                                        : BACKFS_FLAGS_RESOLVE_FULL_MAYBE),
                                   &szNativePath[0],
                                   NULL);
    if (rc)
        LIBCLOG_RETURN_INT(rc);


    /*
     * Try to open the file.
     */
    FS_SAVE_LOAD();
    ULONG   cExpandRetries;
    ULONG   ulAction;
    HFILE   hFile;
    for (cExpandRetries = 0;;)
    {
#if OFF_MAX > LONG_MAX
        if (__libc_gpfnDosOpenL)
        {
            LONGLONG cbInitialTmp = cbInitial;
            rc = __libc_gpfnDosOpenL((PCSZ)&szNativePath[0], &hFile, &ulAction, cbInitialTmp, flAttr, flOpenFlags, flOpenMode, NULL);
        }
        else
        {
            ULONG cbInitialTmp = (ULONG)cbInitial;
            if (cbInitial > LONG_MAX)
            {
                FS_RESTORE();
                LIBCLOG_RETURN_INT(-EOVERFLOW);
            }
            rc = DosOpen((PCSZ)&szNativePath[0], &hFile, &ulAction, cbInitialTmp, flAttr, flOpenFlags, flOpenMode, NULL);
        }
#else
        {
            ULONG cbInitialTmp = cbInitial;
            rc = DosOpen((PCSZ)&szNativePath[0], &hFile, &ulAction, cbInitialTmp, flAttr, flOpenFlags, flOpenMode, NULL);
        }
#endif
        /* Check if we're out of handles. */
        if (rc != ERROR_TOO_MANY_OPEN_FILES)
            break;
        if (cExpandRetries++ >= 3)
            break;
        __libc_FHMoreHandles();
    }   /* ... retry 3 times ... */

    if (!rc)
    {
        ULONG   fulType;
        ULONG   fulDevFlags;
        dev_t   Dev;
        ino_t   Inode;

        /*
         * If a new file the unix EAs needs to be established.
         */
        if (   (ulAction == FILE_CREATED)
            && !__libc_gfNoUnix)
            __libc_back_fsUnixAttribsSet(hFile, pszFile, Mode, &Dev, &Inode);
        else
            Dev = __libc_back_fsPathCalcInodeAndDev(szNativePath, &(*ppFH)->Inode);


        /*
         * Figure the handle type.
         */
        rc = DosQueryHType(hFile, &fulType, &fulDevFlags);
        if (!rc)
        {
            switch (fulType & 0xff)
            {
                default: /* paranoia */
                case HANDTYPE_FILE:
                    fLibc |= F_FILE;
                    break;
                case HANDTYPE_DEVICE:
                    fLibc |= F_DEV;
                    break;
                case HANDTYPE_PIPE:
                    fLibc |= F_PIPE;
                    break;
            }

            /*
             * Register the handle and calc Dev and Inode.
             * We calc the Dev/Inode regardless of whether it's stored in an EA or not.
             */
            rc = __libc_FHAllocate(hFile, fLibc, sizeof(LIBCFH), NULL, ppFH, NULL);
            if (!rc)
            {
                if (ppFH)
                {
                    (*ppFH)->Dev = Dev;
                    (*ppFH)->Inode = Inode;
                }
                LIBCLOG_MSG("hFile=%#lx fLibc=%#x fulType=%#lx ulAction=%lu Dev=%#x Inode=%#llx\n",
                            hFile, fLibc, fulType, ulAction, Dev, Inode);
            }
        }

        if (rc)
            DosClose(hFile);
    }
    FS_RESTORE();

    /*
     * Handle any errors.
     */
    if (rc)
    {
        if (rc == ERROR_OPEN_FAILED)
            rc = rcOpenFailed;
        else
            rc = -__libc_native2errno(rc);
        LIBCLOG_RETURN_INT(rc);
    }
    return hFile;
}

