/* $Id: b_fsFileStatFH.c 1750 2004-12-21 01:38:04Z bird $ */
/** @file
 *
 * LIBC SYS Backend - fstat.
 *
 * Copyright (c) 2003-2004 knut st. osmundsen <bird@innotek.de>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_FSMACROS
#include <os2emx.h>
#include "b_fs.h"
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <emx/io.h>
#include <emx/syscalls.h>
#include <limits.h>
#include "syscalls.h"
#include <InnoTekLIBC/libc.h>
#include <InnoTekLIBC/backend.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_BACK_FS
#include <InnoTekLIBC/logstrict.h>



/**
 * Gets the file stats for a file by filehandle.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure. The content
 *          of *pStat is undefined.
 * @param   fh      Handle to file.
 * @param   pStat   Where to store the stats.
 */
int __libc_Back_fsFileStatFH(int fh, struct stat *pStat)
{
    LIBCLOG_ENTER("fh=%d pStat=%p\n", fh, (void *)pStat);

    /*
     * Get filehandle.
     */
    PLIBCFH pFH;

    int rc = __libc_FHEx(fh, &pFH);
    if (!pFH)
        LIBCLOG_RETURN_INT(rc);

    if (!pFH->pOps)
    {
        /*
         * Standard OS/2 file handle.
         *
         * Use query handle type to figure out the file type.
         */
        bzero(pStat, sizeof(*pStat));
        switch (pFH->fFlags & __LIBC_FH_TYPEMASK)
        {
            case F_DEV:     pStat->st_mode = S_IFCHR; break;
            case F_SOCKET:  pStat->st_mode = S_IFSOCK; break;
            /** @todo needs to check if it's a named pipe or not. */
            /* Anonymous pipes are AF_UNIX sockets on (some?) unix systems. */
            case F_PIPE:    pStat->st_mode = S_IFSOCK; break;
            default:
            case F_FILE:    pStat->st_mode = S_IFREG; break;
        }

        /* fake unix stuff */
        pStat->st_uid = 0;
        pStat->st_gid = 0;
        pStat->st_ino = pFH->Inode;
        pStat->st_dev = pFH->Dev;
        pStat->st_rdev = 0;
        pStat->st_nlink = 1;
        pStat->st_blksize = 4096*12; /* 48KB */

        FS_VAR();
        FS_SAVE_LOAD();
        if (pStat->st_mode == S_IFREG)
        {
            union
            {
                FILESTATUS4     fsts4;
                FILESTATUS4L    fsts4L;
            } info;
#if OFF_MAX > LONG_MAX
            int     fLarge = 0;
#endif

            /*
             * Get file info.
             * We have a little hack here, temporarily, write only files
             * cannot read EAs.
             */
#if OFF_MAX > LONG_MAX
            if (__libc_gpfnDosOpenL)
            {
                rc = DosQueryFileInfo(fh, FIL_QUERYEASIZEL, &info, sizeof(info.fsts4L));
                if (rc)
                {
                    rc = DosQueryFileInfo(fh, FIL_STANDARDL, &info, sizeof(FILESTATUS3L));
                    info.fsts4L.cbList = ~0;
                }
                fLarge = 1;
            }
            else
#endif
            {
                rc = DosQueryFileInfo(fh, FIL_QUERYEASIZE, &info, sizeof(info.fsts4));
                if (rc)
                {
                    rc = DosQueryFileInfo(fh, FIL_STANDARD, &info, sizeof(FILESTATUS3));
                    info.fsts4.cbList = ~0;
                }
            }
            if (rc)
            {
                FS_RESTORE();
                rc = -__libc_native2errno(rc);
                LIBCLOG_RETURN_INT(rc);
            }

            /*
             * Format stats struct.
             *      We know the info struct layouts!
             *      Only cbFile, cbFileAlloc and attrFile need be accessed
             *      using the specific structure.
             */
            /* Times: FAT might not return create and access time. */
            pStat->st_mtime = _sys_p2t(info.fsts4.ftimeLastWrite, info.fsts4.fdateLastWrite);
            if (    FTIMEZEROP(info.fsts4.ftimeCreation)
                &&  FDATEZEROP(info.fsts4.fdateCreation))
                pStat->st_ctime = pStat->st_mtime;
            else
                pStat->st_ctime = _sys_p2t(info.fsts4.ftimeCreation, info.fsts4.fdateCreation);
            if (    FTIMEZEROP(info.fsts4.ftimeLastAccess)
                &&  FDATEZEROP(info.fsts4.fdateLastAccess))
                pStat->st_atime = pStat->st_mtime;
            else
                pStat->st_atime = _sys_p2t(info.fsts4.ftimeLastAccess, info.fsts4.fdateLastAccess);

#if OFF_MAX > LONG_MAX
            if (fLarge)
            {
                pStat->st_size = info.fsts4L.cbFile;
                pStat->st_blocks = info.fsts4L.cbFileAlloc / S_BLKSIZE;
                rc = info.fsts4L.attrFile;
            }
            else
#endif
            {
                pStat->st_size = info.fsts4.cbFile;
                pStat->st_blocks = info.fsts4.cbFileAlloc / S_BLKSIZE;
                rc = info.fsts4.attrFile;
            }
            pStat->st_attr = rc;
            if (rc & FILE_READONLY)
                pStat->st_mode |= (S_IREAD >> 6) * 0111;
            else
                pStat->st_mode |= ((S_IREAD|S_IWRITE) >> 6) * 0111;

            /* If in unix mode we'll check the EAs (if any). */
            if (    !__libc_gfNoUnix
                && (fLarge ? info.fsts4L.cbList : info.fsts4.cbList) >= LIBC_UNIX_EA_MIN)
                __libc_back_fsUnixAttribsGet(fh, 0, pStat);
        }
        else
        {
            if ((pFH->fFlags & O_ACCMODE) == O_RDONLY)
                pStat->st_mode |= (S_IREAD >> 6) * 0111;
            else
                pStat->st_mode |= ((S_IREAD|S_IWRITE) >> 6) * 0111;
        }

        FS_RESTORE();
    }
    else
    {
        /*
         * Non-standard handle - fail.
         */
        LIBCLOG_RETURN_INT(-EOPNOTSUPP);
    }

    LIBCLOG_MSG("st_dev:     %#x\n",  pStat->st_dev);
    LIBCLOG_MSG("st_ino:     %#llx\n",  pStat->st_ino);
    LIBCLOG_MSG("st_mode:    %#x\n",  pStat->st_mode);
    LIBCLOG_MSG("st_nlink:   %u\n",   pStat->st_nlink);
    LIBCLOG_MSG("st_uid:     %u\n",   pStat->st_uid);
    LIBCLOG_MSG("st_gid:     %u\n",   pStat->st_gid);
    LIBCLOG_MSG("st_rdev:    %#x\n",  pStat->st_rdev);
    LIBCLOG_MSG("st_atime:   %d\n",   pStat->st_atime);
    LIBCLOG_MSG("st_mtime:   %d\n",   pStat->st_mtime);
    LIBCLOG_MSG("st_ctime:   %d\n",   pStat->st_ctime);
    LIBCLOG_MSG("st_size:    %lld\n", pStat->st_size);
    LIBCLOG_MSG("st_blocks:  %lld\n", pStat->st_blocks);
    LIBCLOG_MSG("st_blksize: %u\n",   pStat->st_blksize);
    LIBCLOG_MSG("st_attr:    %ld\n",  pStat->st_attr);
    LIBCLOG_RETURN_INT(0);
}

