/* sys/spawnve.c (emx+gcc) -- Copyright (c) 1992-1996 by Eberhard Mattes */

#include "libc-alias.h"
#include <stdlib.h>
#include <string.h>
#include <process.h>
#include <errno.h>
#include <alloca.h>
#include <386/builtin.h>
#include <sys/fmutex.h>
#define INCL_DOSPROCESS
#define INCL_FSMACROS
#include <os2emx.h>
#include "b_fs.h"
#include "b_signal.h"
#include "b_process.h"
#include <emx/syscalls.h>
#include <InnoTekLIBC/sharedpm.h>
#define __LIBC_LOG_GROUP    __LIBC_LOG_GRP_PROCESS
#include <InnoTekLIBC/logstrict.h>
#include "syscalls.h"


/**
 * Collects and creates the inherit stuff we send down to a LIBC child.
 * @returns Pointer to inherit structure on success.
 * @returns NULL and errno on failure.
 */
static __LIBC_PSPMINHERIT doInherit(void)
{
    LIBCLOG_ENTER("\n");
    __LIBC_PSPMINHERIT      pRet = NULL;

    /*
     * Get FH stuff.
     */
    size_t                  cbFH;
    __LIBC_PSPMINHFHBHDR    pFH;
    pFH = __libc_fhInheritPack(&cbFH);
    if (pFH)
    {
        /*
         * Get FS stuff.
         */
        size_t              cbFS;
        __LIBC_PSPMINHFS    pFS;
        if (!__libc_back_fsInheritPack(&pFS, &cbFS))
        {
            /*
             * Get Signal stuff
             */
            size_t              cbSig;
            __LIBC_PSPMINHSIG   pSig;
            if (!__libc_back_signalInheritPack(&pSig, &cbSig))
            {
                /*
                 * Allocate shared memory.
                 */
                size_t  cb = sizeof(__LIBC_SPMINHERIT) + ((cbFH + 3) & ~3) + ((cbFS + 3) & ~3) + ((cbSig + 3) & ~3);
                pRet = __libc_spmAlloc(cb);
                if (pRet)
                {
                    /* fh */
                    pRet->cb = sizeof(*pRet);
                    pRet->pFHBundles = (__LIBC_PSPMINHFHBHDR)(pRet + 1);
                    memcpy(pRet->pFHBundles, pFH, cbFH);
                    free(pFH);

                    /* fs */
                    if (pFS)
                    {
                        pRet->pFS = (__LIBC_PSPMINHFS)((char *)pRet->pFHBundles + ((cbFH + 3) & ~3));
                        memcpy(pRet->pFS, pFS, cbFS);
                        free(pFS);
                    }
                    else
                        pRet->pFS = NULL;

                    /* sig */
                    if (pSig)
                    {
                        pRet->pSig = (__LIBC_PSPMINHSIG)((char *)pRet->pFHBundles + ((cbFH + 3) & ~3) + ((cbFS + 3) & ~3));
                        memcpy(pRet->pSig, pSig, cbSig);
                        free(pFS);
                    }

                    /* done! */
                    LIBCLOG_RETURN_P(pRet);
                }
                free(pSig);
            }
            free(pFS);
        }

        /* cleanup on failure */
        __libc_fhInheritDone();
        free(pFH);
    }

    LIBCLOG_RETURN_P(pRet);
}


/**
 * Cleans up any globale inherit stuff.
 */
static void doInheritDone(void)
{
    __libc_fhInheritDone();
}

/* Note: We are allowed to call _trealloc() as this module is not used
   in an .a library. */

#define ADD(n) do { \
  while (arg_size + n > arg_alloc) \
    { \
      arg_alloc += 512; \
      arg_buf = _trealloc (arg_buf, arg_alloc); \
      if (arg_buf == NULL) \
        { \
          errno = ENOMEM; \
          LIBCLOG_RETURN_INT(-1); \
        } \
      arg_ptr = arg_buf + arg_size; \
    } \
  arg_size += n; } while (0)


int __spawnve (struct _new_proc *np)
{
    LIBCLOG_ENTER("np=%p:{mode=%#x}\n", (void *)np, np->mode);
    __LIBC_PSPMPROCESS    pEmbryo;
    ULONG                 mode;
    char *arg_ptr, *arg_buf;
    char *pgm_name, *base;
    const char *src, *s;
    size_t arg_size, arg_alloc, len;
    int i, quote, bs, method;
    FS_VAR();

    arg_buf = NULL; arg_alloc = 0; arg_size = 0; arg_ptr = NULL;

    /*
     * Validate mode.
     */
    mode = np->mode;
    switch (mode & 0xff)
    {
        case P_WAIT:
        case P_NOWAIT:
        case P_OVERLAY:
            break;
        default:
            errno = EINVAL;
            LIBC_ASSERTM_FAILED("invalid mode 0x%08lx\n", mode);
            LIBCLOG_RETURN_INT(-1);
    }

    /*
     * Allocate space for the program name and check if it's an OS/2 shell.
     */
    pgm_name = alloca (strlen ((const char *)np->fname_off) + 5);
    strcpy (pgm_name, (const char *)np->fname_off);
    _defext (pgm_name, "exe");
    base = _getname (pgm_name);
    method = 0;
    if (stricmp (base, "cmd.exe") == 0 || stricmp (base, "4os2.exe") == 0)
        method = 1;
    char szNativePath[PATH_MAX];
    int rc = __libc_back_fsResolve(pgm_name, BACKFS_FLAGS_RESOLVE_FULL, &szNativePath[0], NULL);
    if (rc)
    {
        LIBC_ASSERTM_FAILED("Failed to resolve program name: '%s' rc=%d.\n", pgm_name, rc);
        errno = -rc;
        LIBCLOG_RETURN_INT(-1);
    }
    pgm_name = &szNativePath[0];

    /*
     * Process arguments.
     */
    src = (const char *)np->arg_off;
    if (np->arg_count > 0)
    {
        ++src;                    /* skip flags byte */
        len = strlen (src) + 1;
        ADD (len);
        memcpy (arg_ptr, src, len);
        arg_ptr += len; src += len;
    }
    for (i = 1; i < np->arg_count; ++i)
    {
        if (i > 1)
        {
            ADD (1);
            *arg_ptr++ = ' ';
        }
        ++src;                    /* skip flags byte */
        quote = FALSE;
        if (*src == 0)
            quote = TRUE;
        else if (mode & P_QUOTE)
        {
            if (src[0] == '@' && src[1] != 0)
                quote = TRUE;
            else
                for (s = src; *s != 0; ++s)
                    if (*s == '?' || *s == '*')
                    {
                        quote = TRUE;
                        break;
                    }
        }
        if (!quote)
            for (s = src; *s != 0; ++s)
                if (*s == ' ' || *s == '\t' || (*s == '"' && method == 1))
                {
                    quote = TRUE;
                    break;
                }
        if (quote)
        {
            ADD (1);
            *arg_ptr++ = '"';
        }
        bs = 0;
        while (*src != 0)
        {
            if (*src == '"' && method == 0)
            {
                ++bs;
                ADD (bs);
                memset (arg_ptr, '\\', bs); arg_ptr += bs;
                bs = 0;
            }
            else if (*src == '\\' && method == 0)
                ++bs;
            else
                bs = 0;
            ADD (1);
            *arg_ptr++ = *src;
            ++src;
        }
        if (quote)
        {
            ADD (1+bs);
            memset (arg_ptr, '\\', bs); arg_ptr += bs;
            *arg_ptr++ = '"';
        }
        ++src;
    }
    /* The arguments are an array of zero terminated strings, ending with an empty string. */
    ADD (2);
    *arg_ptr++ = '\0';
    *arg_ptr++ = '\0';


    /*
     * Now create an embryo process.
     */
    _fmutex_request(&__libc_gmtxExec, 0);
    pEmbryo = __libc_spmCreateEmbryo(getpid());
    if (pEmbryo)
    {
        /*
         * Do inheritance stuff.
         */
        pEmbryo->pInherit = doInherit();
        if (pEmbryo->pInherit)
        {
            RESULTCODES resc;
            char        szObj[40];
            int         rc;

            /*
             * Notify the process waiter system.
             */
            __libc_back_processWaitNotifyExec();

            /*
             * Create the process.
             */
            FS_SAVE_LOAD();
            LIBCLOG_MSG("Calling DosExecPgm pgm: %s args: %s\\0%s\\0\\0\n", pgm_name, arg_buf, arg_buf + strlen(arg_buf) + 1);
            rc = DosExecPgm(szObj, sizeof(szObj), EXEC_ASYNCRESULT, (PCSZ)arg_buf, (PCSZ)np->env_off, &resc, (PCSZ)pgm_name);
            FS_RESTORE();
            if (!rc)
            {
                __atomic_cmpxchg32((volatile uint32_t *)(void *)&pEmbryo->pid, (uint32_t)resc.codeTerminate, 0);
                LIBCLOG_MSG("Spawned pid=%04lx (%ld)\n", resc.codeTerminate, resc.codeTerminate);

                /* cleanup embryo and other stuff. */
                __libc_spmRelease(pEmbryo);
                doInheritDone();
                if (arg_buf != NULL)
                    _tfree (arg_buf);
                _fmutex_release(&__libc_gmtxExec);

                /*
                 * Exit depends on the mode.
                 */
                switch (mode & 0xff)
                {
                    /*
                     * Return the pid.
                     */
                    case P_NOWAIT:
                        LIBCLOG_RETURN_INT((int)resc.codeTerminate);

                        /*
                         * Wait for the child and return the result.
                         */
                    case P_WAIT:
                        /*
                         * Wait for the child and exit this process with the same result.
                         */
                    case P_OVERLAY:
                    {
                        PID pid = resc.codeTerminate;
                        PID pidEnded = 0;
                        FS_SAVE_LOAD();
                        LIBCLOG_MSG("Calling DosWaitChild(,,,,0x%04lx)\n", pid);
                        rc = DosWaitChild(DCWA_PROCESS, DCWW_WAIT, &resc, &pidEnded, pid);
                        if (!rc)
                        {
                            LIBCLOG_MSG("DosWaitChild(,,,,0x%04lx) returned pidEnded=0x%04lx resc.codeTerminate=%ld resc.codeResult=%ld\n",
                                        pid, pidEnded, resc.codeTerminate, resc.codeResult);
                            LIBC_ASSERTM(pidEnded == pid, "Expected pid 0x%04lx and got 0x%04lx!\n", pid, pidEnded);

                            if (mode == P_OVERLAY)
                            {
                                LIBCLOG_MSG("Calling DosExit(,0)\n");
                                while (pid == pidEnded)
                                    DosExit(EXIT_PROCESS, resc.codeResult);
                            }

                            LIBCLOG_RETURN_INT((int)resc.codeResult);
                        }

                        LIBC_ASSERTM_FAILED("Calling DosWaitChild(,,,,0x%04lx) -> rc=%d\n", pid, rc);
                        _sys_set_errno(rc);
                        LIBCLOG_RETURN_INT(-1);
                    }

                        /* this can _NEVER_ happen! */
                    default:
                        errno = EINVAL;
                        LIBCLOG_RETURN_INT(-1);
                }
                /* won't ever get here! */
            }
            else
                _sys_set_errno (rc);
            doInheritDone();
        }
        /* cleanup embryo */
        __libc_spmRelease(pEmbryo);
    }

    if (arg_buf != NULL)
        _tfree (arg_buf);
    _fmutex_release(&__libc_gmtxExec);
    LIBCLOG_RETURN_INT(-1);
}
