/* $Id: signals.c 1615 2004-11-06 08:35:32Z bird $ */
/** @file
 *
 * LIBC SYS Backend - Signals.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/** @page   LIBCSignals     Signals
 *
 * LIBC fully implements the posix signal handling.
 *
 * Interprocess signaling is only available when target is linked to LIBCxyz.DLL.
 * When the target is a LIBC process the signal is queued in SPM before the
 * target process is actually poked. When the target is a non-LIBC process the
 * signal is attempted converted to EMX and signaled in the EMX fashion. This
 * obviously means that certain restrictions applies when signaling EMX processes.
 *
 *
 * Signals are pending in on of three places depending on how fare they are
 * scheduled. Scheduling levels and pending location:
 *          - Pending on the SPM process, not scheduled.
 *          - Pending in process wide queues, not scheduled.
 *          - Pending on thread, scheduled.
 *
 * There are generally speaking fource sources signals:
 *          - Internal software signals in current thread, raise()/kill(getpid())/sigqueue(getpid()).
 *          - Other thread, pthread_kill(). This needs no process scheduling, only
 *            priority ordering on delivery.
 *          - OS/2 hardware exceptions. Generally no scheduling required, these must be
 *            handled promptly on the current thread.
 *          - External signals, queued in SPM, EMX signals, or OS/2 signal exceptions.
 *
 * When talking scheduling this means we'll reschedule the entire process when:
 *          - an external signal arrives.
 *          - an internal software signal occurs (raise/kill/sigqueue).
 *          - the signal mask of a thread changes.
 *            Two cases, 1) unblocking a signal no, 2) blocking signal no with
 *            pending signal. The 2nd case is very uncommon and could perhaps be
 *            ignored or handled differently. -bird: we do that now!
 *
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_BASE
#define INCL_DOSSIGNALS
#define INCL_FSMACROS
#define INCL_EXAPIS
#include <os2emx.h>

#include <signal.h>
#include <errno.h>
#include <string.h>
#include <malloc.h>
#include "syscalls.h"
#include <386/builtin.h>
#include <InnoTekLIBC/signals.h>
#include <InnoTekLIBC/thread.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_SIGNAL
#include <InnoTekLIBC/logstrict.h>


/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
/** OS/2 default standard error handle. */
#define HFILE_STDERR    ((HFILE)2)

/** @defgroup libc_back_signals_properties  Signal Properties
 * The return actions are ordered by precendece.
 * @{ */
/** Return Action: Terminate. */
#define SPR_KILL        0x0000
/** Return Action: Restart instruction. */
#define SPR_CONTINUE    0x0001
/** Return Action Mask */
#define SPR_MASK        0x0001

/** Action: Ignore the signal. */
#define SPA_IGNORE      0x0000
/** Action: Terminate the process. */
#define SPA_KILL        0x0010
/** Action: Perform coredump killing the process. */
#define SPA_CORE        0x0020
/** Action: Suspend the process. */
#define SPA_STOP        0x0030
/** Action: Suspend the process from the tty. */
#define SPA_STOPTTY     0x0040
/** Action: Resume(/continue) suspended process. */
#define SPA_RESUME      0x0050
/** Action: Next exception handler. */
#define SPA_NEXT        0x0060
/** Action: Terminate if primary exception handler, next exception handler if not. */
#define SPA_NEXT_KILL   0x0070
/** Action: Coredump+term if primary exception handler, next exception handler if not. */
#define SPA_NEXT_CORE   0x0080
/** Action Mask */
#define SPA_MASK        0x00f0

/** Property: Catchable but not blockable. */
#define SPP_NOBLOCK     0x0100
/** Property: Not catchable. */
#define SPP_NOCATCH     0x0200
/** Property: Anything can service this signal. */
#define SPP_ANYTHRD     0x0400
/** Property: This signal can be queued. */
#define SPP_QUEUED      0x0800
/** Property: Signal action cannot be automatically reset in SysV fashion. */
#define SPP_NORESET     0x1000
/** @} */


/** EMX signal numbers.
 * @{ */
#define EMX_SIGHUP    1
#define EMX_SIGINT    2
#define EMX_SIGQUIT   3
#define EMX_SIGILL    4
#define EMX_SIGTRAP   5
#define EMX_SIGABRT   6
#define EMX_SIGEMT    7
#define EMX_SIGFPE    8
#define EMX_SIGKILL   9
#define EMX_SIGBUS   10
#define EMX_SIGSEGV  11
#define EMX_SIGSYS   12
#define EMX_SIGPIPE  13
#define EMX_SIGALRM  14
#define EMX_SIGTERM  15
#define EMX_SIGUSR1  16
#define EMX_SIGUSR2  17
#define EMX_SIGCHLD  18
#define EMX_SIGBREAK 21
#define EMX_SIGWINCH 28
/** @} */


/*******************************************************************************
*   Structures and Typedefs                                                    *
*******************************************************************************/

/** A queued signal.
 * (Per process variant.)
 */
typedef struct SignalQueued
{
    /** Pointer to the next signal in the queue. */
    struct SignalQueued    *pNext;
    /** Pointer to the previous signal in the queue. */
    struct SignalQueued    *pPrev;
    /** How to free this signal. */
    enum enmSignalQueuedHowToFree
    {
        /** Free from pre allocated chunk. */
        enmHowFree_PreAllocFree,
        /** Free from shared heap. */
        enmHowFree_HeapFree,
        /** Don't free, keep within thread, it's stack. */
        enmHowFree_Temporary
    }                       enmHowFree;
    /** Signal info. */
    siginfo_t               SigInfo;
} SIGQUEUED, *PSIGQUEUED;


/** Thread enumeration state data.
 * Used by signalScheduleThread() when enumerating threads.
 */
typedef struct SigSchedEnumParam
{
    /** Signal number. */
    int             iSignalNo;
    /** Current thread. */
    __LIBC_PTHREAD  pThrd;
} SIGSCHEDENUMPARAM, *PSIGSCHEDENUMPARAM;


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/

/** Signal Mutex.
 * This mutex basically protects all the signal constructs in the process. A
 * complete list of what this is should be provided later.
 *
 * An OS/2 semaphore is used here because speed is less important than
 * reliability. Don't access the mutex directly but use the methods
 * defined for such purposes.
 */
static HMTX             ghmtxSignals;

/** Signal Wait Event Semaphore.
 * This is an event semaphore which will never be posted (unless we wanna resolve
 * some nasty deadlock in the future) but is used to wait for a signal to arrive.
 * When a signal arrives the wait will be interrupted to allow for execution of
 * the exception and signal processing. DosWaitEventSem will return ERROR_INTERRUPT.
 */
static HEV              ghevWait;

/** Signal Properties.
 * Describes the default actions of a signal.
 */
static const unsigned short    gafSignalProperties[__SIGSET_MAXSIGNALS] =
{
    /* return action | default action | property [|anotherprop] */
    SPR_CONTINUE | SPA_IGNORE,                                          /* SIG0 */
    SPR_CONTINUE | SPA_KILL    | SPP_ANYTHRD,                           /* SIGHUP    1     /-- POSIX: Hangup */
    SPR_CONTINUE | SPA_KILL    | SPP_ANYTHRD,                           /* SIGINT    2     /-- ANSI: Interrupt (Ctrl-C) */
    SPR_CONTINUE | SPA_CORE    | SPP_ANYTHRD,                           /* SIGQUIT   3     /-- POSIX: Quit */
    SPR_CONTINUE | SPA_NEXT_CORE | SPP_NORESET,                         /* SIGILL    4     /-- ANSI: Illegal instruction */
    SPR_CONTINUE | SPA_NEXT_KILL | SPP_NORESET,                         /* SIGTRAP   5     /-- POSIX: Single step (debugging) */
    SPR_KILL     | SPA_CORE,                                            /* SIGABRT   6     /-- ANSI: abort () */
    SPR_CONTINUE | SPA_CORE    | SPP_ANYTHRD,                           /* SIGEMT    7     /-- BSD: EMT instruction */
    SPR_CONTINUE | SPA_NEXT_CORE,                                       /* SIGFPE    8     /-- ANSI: Floating point */
    SPR_KILL     | SPA_KILL | SPP_ANYTHRD | SPP_NOBLOCK | SPP_NOCATCH,  /* SIGKILL   9     /-- POSIX: Kill process */
    SPR_CONTINUE | SPA_CORE,                                            /* SIGBUS   10     /-- BSD 4.2: Bus error */
    SPR_CONTINUE | SPA_NEXT_CORE,                                       /* SIGSEGV  11     /-- ANSI: Segmentation fault */
    SPR_CONTINUE | SPA_CORE,                                            /* SIGSYS   12     /-- Invalid argument to system call */
    SPR_CONTINUE | SPA_KILL    | SPP_ANYTHRD,                           /* SIGPIPE  13     /-- POSIX: Broken pipe. */
    SPR_CONTINUE | SPA_KILL    | SPP_ANYTHRD,                           /* SIGALRM  14     /-- POSIX: Alarm. */
    SPR_CONTINUE | SPA_NEXT_KILL | SPP_ANYTHRD,                         /* SIGTERM  15     /-- ANSI: Termination, process killed */
    SPR_CONTINUE | SPA_IGNORE,                                          /* SIGURG   16     /-- POSIX/BSD: urgent condition on IO channel */
    SPR_CONTINUE | SPA_STOP | SPP_ANYTHRD | SPP_NOBLOCK | SPP_NOCATCH,  /* SIGSTOP  17     /-- POSIX: Sendable stop signal not from tty. unblockable. */
    SPR_CONTINUE | SPA_STOPTTY | SPP_ANYTHRD,                           /* SIGTSTP  18     /-- POSIX: Stop signal from tty. */
    SPR_CONTINUE | SPA_RESUME  | SPP_ANYTHRD,                           /* SIGCONT  19     /-- POSIX: Continue a stopped process. */
    SPR_CONTINUE | SPA_IGNORE  | SPP_ANYTHRD | SPP_QUEUED,              /* SIGCHLD  20     /-- POSIX: Death or stop of a child process. (EMX: 18) */
    SPR_CONTINUE | SPA_STOPTTY | SPP_ANYTHRD,                           /* SIGTTIN  21     /-- POSIX: To readers pgrp upon background tty read. */
    SPR_CONTINUE | SPA_STOPTTY | SPP_ANYTHRD,                           /* SIGTTOU  22     /-- POSIX: To readers pgrp upon background tty write. */
    SPR_CONTINUE | SPA_IGNORE  | SPP_ANYTHRD,                           /* SIGIO    23     /-- BSD: Input/output possible signal. */
    SPR_CONTINUE | SPA_KILL,                                            /* SIGXCPU  24     /-- BSD 4.2: Exceeded CPU time limit. */
    SPR_CONTINUE | SPA_KILL,                                            /* SIGXFSZ  25     /-- BSD 4.2: Exceeded file size limit. */
    SPR_CONTINUE | SPA_KILL    | SPP_ANYTHRD,                           /* SIGVTALRM 26    /-- BSD 4.2: Virtual time alarm. */
    SPR_CONTINUE | SPA_KILL    | SPP_ANYTHRD,                           /* SIGPROF  27     /-- BSD 4.2: Profiling time alarm. */
    SPR_CONTINUE | SPA_IGNORE  | SPP_ANYTHRD,                           /* SIGWINCH 28     /-- BSD 4.3: Window size change (not implemented). */
    SPR_CONTINUE | SPA_NEXT_KILL | SPP_ANYTHRD,                         /* SIGBREAK 29     /-- OS/2: Break (Ctrl-Break). (EMX: 21) */
    SPR_CONTINUE | SPA_KILL    | SPP_ANYTHRD,                           /* SIGUSR1  30     /-- POSIX: User-defined signal #1 */
    SPR_CONTINUE | SPA_KILL    | SPP_ANYTHRD,                           /* SIGUSR2  31     /-- POSIX: User-defined signal #2 */
    SPR_CONTINUE | SPA_NEXT_KILL | SPP_ANYTHRD,                         /* SIGBREAK 32     /-- OS/2: Break (Ctrl-Break). (EMX: 21) */
    /* real time signals */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  0 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  1 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  2 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  3 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  4 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  5 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  6 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  7 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  8 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN +  9 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 10 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 11 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 12 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 13 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 14 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 15 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 16 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 17 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 18 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 19 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 20 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 21 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 22 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 23 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 24 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 25 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 26 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 27 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 28 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 29 */
    SPR_CONTINUE | SPA_CORE | SPP_ANYTHRD | SPP_QUEUED,                 /* SIGRTMIN + 30 == SIGRTMAX */
};

/** The signal actions for all signals in the system.
 * All access to this array is protected by the signal semaphore.
 */
static struct sigaction    gaSignalActions[__SIGSET_MAXSIGNALS] =
{
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIG0 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGHUP    1     /-- POSIX: Hangup */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGINT    2     /-- ANSI: Interrupt (Ctrl-C) */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGQUIT   3     /-- POSIX: Quit */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGILL    4     /-- ANSI: Illegal instruction */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGTRAP   5     /-- POSIX: Single step (debugging) */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGABRT   6     /-- ANSI: abort () */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGEMT    7     /-- BSD: EMT instruction */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGFPE    8     /-- ANSI: Floating point */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGKILL   9     /-- POSIX: Kill process */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGBUS   10     /-- BSD 4.2: Bus error */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGSEGV  11     /-- ANSI: Segmentation fault */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGSYS   12     /-- Invalid argument to system call */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGPIPE  13     /-- POSIX: Broken pipe. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGALRM  14     /-- POSIX: Alarm. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGTERM  15     /-- ANSI: Termination, process killed */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGURG   16     /-- POSIX/BSD: urgent condition on IO channel */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGSTOP  17     /-- POSIX: Sendable stop signal not from tty. unblockable. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGTSTP  18     /-- POSIX: Stop signal from tty. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGCONT  19     /-- POSIX: Continue a stopped process. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGCHLD  20     /-- POSIX: Death or stop of a child process. (EMX: 18) */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGTTIN  21     /-- POSIX: To readers pgrp upon background tty read. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGTTOU  22     /-- POSIX: To readers pgrp upon background tty write. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGIO    23     /-- BSD: Input/output possible signal. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGXCPU  24     /-- BSD 4.2: Exceeded CPU time limit. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGXFSZ  25     /-- BSD 4.2: Exceeded file size limit. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGVTALRM 26    /-- BSD 4.2: Virtual time alarm. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGPROF  27     /-- BSD 4.2: Profiling time alarm. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGWINCH 28     /-- BSD 4.3: Window size change (not implemented). */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGINFO  29     /-- BSD 4.3: Information request. */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGUSR1  30     /-- POSIX: User-defined signal #1 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGUSR2  31     /-- POSIX: User-defined signal #2 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGBREAK 32     /-- OS/2: Break (Ctrl-Break). (EMX: 21) */
    /* realtime signals */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  0 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  1 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  2 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  3 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  4 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  5 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  6 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  7 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  8 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN +  9 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 10 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 11 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 12 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 13 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 14 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 15 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 16 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 17 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 18 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 19 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 20 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 21 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 22 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 23 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 24 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 25 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 26 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 27 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 28 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 29 */
    { { .__sa_handler = SIG_DFL }, {{0, 0}}, 0 },   /* SIGRTMIN + 30 == SIGRTMAX */
};


/** Set of signals pending on this process.
 * All access to this set is protected by the signal semaphore.
 */
sigset_t            __libc_gSignalPending = {{0,0}};

/** Head of the queue of pending signals.
 * All access to this set is protected by the signal semaphore.
 */
PSIGQUEUED          gpSigQueueHead = NULL;
/** Tail of the queue of pending signals.
 * All access to this set is protected by the signal semaphore.
 */
PSIGQUEUED          gpSigQueueTail = NULL;

/** Array of statically allocated queued signals. */
static SIGQUEUED    gpaSigQueuedPreAlloced[64] =
{
    { .pNext = &gpaSigQueuedPreAlloced[ 1], .enmHowFree = enmHowFree_PreAllocFree },    /* 0 */
    { .pNext = &gpaSigQueuedPreAlloced[ 2], .enmHowFree = enmHowFree_PreAllocFree },    /* 1 */
    { .pNext = &gpaSigQueuedPreAlloced[ 3], .enmHowFree = enmHowFree_PreAllocFree },    /* 2 */
    { .pNext = &gpaSigQueuedPreAlloced[ 4], .enmHowFree = enmHowFree_PreAllocFree },    /* 3 */
    { .pNext = &gpaSigQueuedPreAlloced[ 5], .enmHowFree = enmHowFree_PreAllocFree },    /* 4 */
    { .pNext = &gpaSigQueuedPreAlloced[ 6], .enmHowFree = enmHowFree_PreAllocFree },    /* 5 */
    { .pNext = &gpaSigQueuedPreAlloced[ 7], .enmHowFree = enmHowFree_PreAllocFree },    /* 6 */
    { .pNext = &gpaSigQueuedPreAlloced[ 8], .enmHowFree = enmHowFree_PreAllocFree },    /* 7 */
    { .pNext = &gpaSigQueuedPreAlloced[ 9], .enmHowFree = enmHowFree_PreAllocFree },    /* 8 */
    { .pNext = &gpaSigQueuedPreAlloced[10], .enmHowFree = enmHowFree_PreAllocFree },    /* 9 */
    { .pNext = &gpaSigQueuedPreAlloced[11], .enmHowFree = enmHowFree_PreAllocFree },    /* 10 */
    { .pNext = &gpaSigQueuedPreAlloced[12], .enmHowFree = enmHowFree_PreAllocFree },    /* 11 */
    { .pNext = &gpaSigQueuedPreAlloced[13], .enmHowFree = enmHowFree_PreAllocFree },    /* 12 */
    { .pNext = &gpaSigQueuedPreAlloced[14], .enmHowFree = enmHowFree_PreAllocFree },    /* 13 */
    { .pNext = &gpaSigQueuedPreAlloced[15], .enmHowFree = enmHowFree_PreAllocFree },    /* 14 */
    { .pNext = &gpaSigQueuedPreAlloced[16], .enmHowFree = enmHowFree_PreAllocFree },    /* 15 */
    { .pNext = &gpaSigQueuedPreAlloced[17], .enmHowFree = enmHowFree_PreAllocFree },    /* 16 */
    { .pNext = &gpaSigQueuedPreAlloced[18], .enmHowFree = enmHowFree_PreAllocFree },    /* 17 */
    { .pNext = &gpaSigQueuedPreAlloced[19], .enmHowFree = enmHowFree_PreAllocFree },    /* 18 */
    { .pNext = &gpaSigQueuedPreAlloced[20], .enmHowFree = enmHowFree_PreAllocFree },    /* 19 */
    { .pNext = &gpaSigQueuedPreAlloced[21], .enmHowFree = enmHowFree_PreAllocFree },    /* 20 */
    { .pNext = &gpaSigQueuedPreAlloced[22], .enmHowFree = enmHowFree_PreAllocFree },    /* 21 */
    { .pNext = &gpaSigQueuedPreAlloced[23], .enmHowFree = enmHowFree_PreAllocFree },    /* 22 */
    { .pNext = &gpaSigQueuedPreAlloced[24], .enmHowFree = enmHowFree_PreAllocFree },    /* 23 */
    { .pNext = &gpaSigQueuedPreAlloced[25], .enmHowFree = enmHowFree_PreAllocFree },    /* 24 */
    { .pNext = &gpaSigQueuedPreAlloced[26], .enmHowFree = enmHowFree_PreAllocFree },    /* 25 */
    { .pNext = &gpaSigQueuedPreAlloced[27], .enmHowFree = enmHowFree_PreAllocFree },    /* 26 */
    { .pNext = &gpaSigQueuedPreAlloced[28], .enmHowFree = enmHowFree_PreAllocFree },    /* 27 */
    { .pNext = &gpaSigQueuedPreAlloced[29], .enmHowFree = enmHowFree_PreAllocFree },    /* 28 */
    { .pNext = &gpaSigQueuedPreAlloced[30], .enmHowFree = enmHowFree_PreAllocFree },    /* 29 */
    { .pNext = &gpaSigQueuedPreAlloced[31], .enmHowFree = enmHowFree_PreAllocFree },    /* 30 */
    { .pNext = &gpaSigQueuedPreAlloced[32], .enmHowFree = enmHowFree_PreAllocFree },    /* 31 */
    { .pNext = &gpaSigQueuedPreAlloced[33], .enmHowFree = enmHowFree_PreAllocFree },    /* 32 */
    { .pNext = &gpaSigQueuedPreAlloced[34], .enmHowFree = enmHowFree_PreAllocFree },    /* 33 */
    { .pNext = &gpaSigQueuedPreAlloced[35], .enmHowFree = enmHowFree_PreAllocFree },    /* 34 */
    { .pNext = &gpaSigQueuedPreAlloced[36], .enmHowFree = enmHowFree_PreAllocFree },    /* 35 */
    { .pNext = &gpaSigQueuedPreAlloced[37], .enmHowFree = enmHowFree_PreAllocFree },    /* 36 */
    { .pNext = &gpaSigQueuedPreAlloced[38], .enmHowFree = enmHowFree_PreAllocFree },    /* 37 */
    { .pNext = &gpaSigQueuedPreAlloced[39], .enmHowFree = enmHowFree_PreAllocFree },    /* 38 */
    { .pNext = &gpaSigQueuedPreAlloced[40], .enmHowFree = enmHowFree_PreAllocFree },    /* 39 */
    { .pNext = &gpaSigQueuedPreAlloced[41], .enmHowFree = enmHowFree_PreAllocFree },    /* 40 */
    { .pNext = &gpaSigQueuedPreAlloced[42], .enmHowFree = enmHowFree_PreAllocFree },    /* 41 */
    { .pNext = &gpaSigQueuedPreAlloced[43], .enmHowFree = enmHowFree_PreAllocFree },    /* 42 */
    { .pNext = &gpaSigQueuedPreAlloced[44], .enmHowFree = enmHowFree_PreAllocFree },    /* 43 */
    { .pNext = &gpaSigQueuedPreAlloced[45], .enmHowFree = enmHowFree_PreAllocFree },    /* 44 */
    { .pNext = &gpaSigQueuedPreAlloced[46], .enmHowFree = enmHowFree_PreAllocFree },    /* 45 */
    { .pNext = &gpaSigQueuedPreAlloced[47], .enmHowFree = enmHowFree_PreAllocFree },    /* 46 */
    { .pNext = &gpaSigQueuedPreAlloced[48], .enmHowFree = enmHowFree_PreAllocFree },    /* 47 */
    { .pNext = &gpaSigQueuedPreAlloced[49], .enmHowFree = enmHowFree_PreAllocFree },    /* 48 */
    { .pNext = &gpaSigQueuedPreAlloced[50], .enmHowFree = enmHowFree_PreAllocFree },    /* 49 */
    { .pNext = &gpaSigQueuedPreAlloced[51], .enmHowFree = enmHowFree_PreAllocFree },    /* 50 */
    { .pNext = &gpaSigQueuedPreAlloced[52], .enmHowFree = enmHowFree_PreAllocFree },    /* 51 */
    { .pNext = &gpaSigQueuedPreAlloced[53], .enmHowFree = enmHowFree_PreAllocFree },    /* 52 */
    { .pNext = &gpaSigQueuedPreAlloced[54], .enmHowFree = enmHowFree_PreAllocFree },    /* 53 */
    { .pNext = &gpaSigQueuedPreAlloced[55], .enmHowFree = enmHowFree_PreAllocFree },    /* 54 */
    { .pNext = &gpaSigQueuedPreAlloced[56], .enmHowFree = enmHowFree_PreAllocFree },    /* 55 */
    { .pNext = &gpaSigQueuedPreAlloced[57], .enmHowFree = enmHowFree_PreAllocFree },    /* 56 */
    { .pNext = &gpaSigQueuedPreAlloced[58], .enmHowFree = enmHowFree_PreAllocFree },    /* 57 */
    { .pNext = &gpaSigQueuedPreAlloced[59], .enmHowFree = enmHowFree_PreAllocFree },    /* 58 */
    { .pNext = &gpaSigQueuedPreAlloced[60], .enmHowFree = enmHowFree_PreAllocFree },    /* 59 */
    { .pNext = &gpaSigQueuedPreAlloced[61], .enmHowFree = enmHowFree_PreAllocFree },    /* 60 */
    { .pNext = &gpaSigQueuedPreAlloced[62], .enmHowFree = enmHowFree_PreAllocFree },    /* 61 */
    { .pNext = &gpaSigQueuedPreAlloced[63], .enmHowFree = enmHowFree_PreAllocFree },    /* 62 */
    { .pNext = NULL,                        .enmHowFree = enmHowFree_PreAllocFree },    /* 63 */
};

/** LIFO of free queued signal structures.
 * Queued by the pNext pointer. (These of course belongs to this process.)
 * All access to this array is protected by the signal semaphore.
 */
static PSIGQUEUED   gpSigQueuedFree = &gpaSigQueuedPreAlloced[0];
/** Number of structures in the free queue. */
static unsigned     gcSigQueuedFree = sizeof(gpaSigQueuedPreAlloced) / sizeof(gpaSigQueuedPreAlloced[0]);


/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
static int              signalSchedule(__LIBC_PTHREAD pThrd, int iSignalNo, siginfo_t *pSigInfo, unsigned fFlags, PSIGQUEUED pSigQueued);
static void             signalScheduleSPM(__LIBC_PTHREAD pThrd);
static void             signalScheduleProcess(__LIBC_PTHREAD pThrd);
static void             signalScheduleToThread(__LIBC_PTHREAD pThrd, __LIBC_PTHREAD pThrdSig, int iSignalNo, PSIGQUEUED pSig);
static __LIBC_PTHREAD   signalScheduleThread(int iSignalNo, __LIBC_PTHREAD pThrdCur);
static int              signalScheduleThreadWorker(__LIBC_PTHREAD pCur, __LIBC_PTHREAD pBest, void *pvParam);
static void             signalScheduleKickThreads(__LIBC_PTHREAD pThrd);
static int              signalScheduleKickThreadsWorker(__LIBC_PTHREAD pCur, void *pvParam);
static int              signalDeliver(__LIBC_PTHREAD pThrd, int iSignalNo, void *pvXcptParams);
static void             signalTerminate(int iSignalNo) __dead2;
static int              signalJobStop(int iSignalNo);
static int              signalJobResume(void);
static void             signalPokeThread(__LIBC_PTHREAD pThrdPoke);
static int              signalActionWorker(__LIBC_PTHREAD pCur, void *pvParam);
static unsigned         signalTimestamp(void);


/**
 * Initializes the signal handling for this process.
 *
 * This is called at DLL/LIBC init.
 *
 * @returns 0 on success.
 * @returns -1 on failure.
 */
int __libc_back_signalInit(void)
{
    LIBCLOG_ENTER("\n");
    int rc;
    FS_VAR();

    /*
     * Create the semaphore.
     */
    FS_SAVE_LOAD();
    rc = DosCreateMutexSemEx(NULL, &ghmtxSignals, 0, FALSE);
    if (!rc)
    {
        /*
         * Create the wait semaphore.
         */
        rc = DosCreateEventSemEx(NULL, &ghevWait, 0, 0);
        if (!rc)
        {
            FS_RESTORE();
            LIBCLOG_RETURN_INT(0);
        }
        else
            LIBC_ASSERTM_FAILED("DosCreateEventSemEx failed with rc=%d\n", rc);

        DosCloseMutexSemEx(ghmtxSignals);
        ghmtxSignals = NULLHANDLE;
    }
    else
        LIBC_ASSERTM_FAILED("DosCreateMutexSemEx failed with rc=%d\n", rc);

    FS_RESTORE();
    LIBCLOG_RETURN_INT(-1);
}


/**
 * Checks if the caller is the current semaphore owner.
 *
 * @returns Nesting level if owner.
 * @returns 0 if not owner.
 */
unsigned    __libc_back_signalSemIsOwner(void)
{
    FS_VAR()
    PTIB    pTib;
    PPIB    pPib;
    int     rc;
    PID     pid = 0;
    TID     tid = 0;
    ULONG   cNesting = 1;

    FS_SAVE_LOAD();
    DosGetInfoBlocks(&pTib, &pPib);

    rc = DosQueryMutexSem(ghmtxSignals, &pid, &tid, &cNesting);
    FS_RESTORE();
    LIBC_ASSERTM(!rc, "DosQueryMutexSem(%#lx,,,,) -> rc=%d\n", ghmtxSignals, rc);
    if (!rc && pTib->tib_ptib2->tib2_ultid == tid && pPib->pib_ulpid == pid)
        return cNesting ? cNesting : 1; /* paranoia */
    return 0;
}


/**
 * Requests the signal semaphore and enters a must complete section.
 *
 * @returns 0 if we got the semaphore.
 * @returns -1 if we didn't get it (pretty fatal).
 */
int __libc_back_signalSemRequest(void)
{
    LIBCLOG_ENTER("\n");
    int     rc;
    PPIB    pPib;
    PTIB    pTib;
    ULONG   cNesting;
    FS_VAR();

    FS_SAVE_LOAD();
    do
    {
        HMTX    hmtx = ghmtxSignals;
        rc = DosRequestMutexSem(hmtx, SEM_INDEFINITE_WAIT);
        if (!rc)
            break;
        if (rc == ERROR_INTERRUPT)
        {
            /*
             * The waiting was interrupted, probably because someone
             * signalling us.
             *
             * Check if the process is dying before retrying to get the
             * semaphore. Being stubborn during process termination usually
             * lead to bad deadlocks.
             */
            DosGetInfoBlocks(&pTib, &pPib);
            if (pPib->pib_flstatus & (0x40/*dying*/ | 0x04/*exiting all*/ | 0x02/*Exiting Thread 1*/ | 0x01/* ExitList */))
            {
                /* we're terminating, don't give a shit about signals now, just hurry up and die! */
                FS_RESTORE();
                LIBCLOG_RETURN_MSG(-1, "ret -1 (we're dying)\n");
            }
            /* Not dying, retry. */
        }
        else if (rc == ERROR_SEM_OWNER_DIED)
        {
            HMTX hmtxNew;
            LIBCLOG_MSG("semaphore owner died. !SHIT!\n");
            /* we'll try fix this situation */
            rc = DosCreateMutexSem(NULL, &hmtxNew, 0, TRUE);
            if (rc)
            {
                FS_RESTORE();
                LIBCLOG_RETURN_MSG(-1, "ret -1 (create mutex failed, rc=%d)\n", rc);
            }
            /* we race here. */
            if (__atomic_cmpxchg32((uint32_t *)(void *)&ghmtxSignals, (uint32_t)hmtx, (uint32_t)hmtxNew))
                break;

            /* beaten. */
            DosCloseMutexSem(hmtxNew);
        }
        else
        {
            FS_RESTORE();
            LIBC_ASSERTM_FAILED("DosRequestMutexSem retured odd rc=%d\n", rc);
            LIBCLOG_RETURN_MSG(-1, "ret -1 (unknown reason, rc=%d)\n", rc);
        }

        LIBCLOG_MSG("interrupted, retry\n");
    } while (1);

    /*
     * Got the semaphore, now enter must complete.
     */
    rc = DosEnterMustComplete(&cNesting);
    LIBC_ASSERTM(!rc, "DosEnterMustComplete failed with rc=%d\n", rc);

    LIBCLOG_RETURN_MSG(0, "ret 0 (must complete nesting %lu)\n", cNesting);
}


/**
 * Releases the signal semaphore and the must complete section.
 */
void __libc_back_signalSemRelease(void)
{
    LIBCLOG_ENTER("\n");
    ULONG   cNesting;
    int     rc;
    FS_VAR();

    /*
     * First the semphore.
     */
    FS_SAVE_LOAD();
    rc = DosReleaseMutexSem(ghmtxSignals);
    if (rc)
    {
        LIBC_ASSERTM_FAILED("DosReleaseMutexSem failed with rc=%d\n", rc);
        FS_RESTORE();
        LIBCLOG_RETURN_VOID();
    }

    /*
     * Now we can leave the must complete stuff.
     */
    rc = DosExitMustComplete(&cNesting);
    LIBC_ASSERTM(!rc, "DosExitMustComplete failed with rc=%d\n", rc);

    FS_RESTORE();
    LIBCLOG_RETURN_MSG_VOID("ret 0 (must complete nesting %lu)\n", cNesting);
}


/**
 * Raises a signal in the current process.
 *
 * @returns On success a flag mask out of the __LIBC_BSRR_* #defines is returned.
 * @returns On failure a negative error code (errno.h) is returned.
 * @param   iSignalNo           Signal to raise.
 * @param   pSigInfo            Pointer to signal info for this signal.
 *                              NULL is allowed.
 * @param   pvXcptOrQueued      Exception handler parameter list.
 *                              Or if __LIBC_BSRF_QUEUED is set, a pointer to locally malloced
 *                              SIGQUEUED node.
 * @param   fFlags              Flags of the #defines __LIBC_BSRF_* describing how to
 *                              deliver the signal.
 *
 * @remark  This Backend Signal API does NOT require the caller to own the signal semaphore.
 */
int __libc_Back_signalRaise(int iSignalNo, siginfo_t *pSigInfo, void *pvXcptOrQueued, unsigned fFlags)
{
    LIBCLOG_ENTER("iSignalNo=%d pSigInfo=%p pvXcptOrQueued=%p fFlags=%#x\n",
                  iSignalNo, (void *)pSigInfo, (void *)pvXcptOrQueued, fFlags);

    /*
     * Can't be to careful!
     */
    LIBC_ASSERTM(!__libc_back_signalSemIsOwner(), "Thread owns the signal semaphore!!! Bad boy!!\n");
    if (__SIGSET_SIG_VALID(iSignalNo) || iSignalNo == 0)
    {
        LIBC_ASSERTM_FAILED("Invalid signal %d\n", iSignalNo);
        return -EINVAL;
    }

    /*
     * Get the current thread.
     * There *MUST* be a current thread, if not, we're toast!
     */
    __LIBC_PTHREAD  pThrd = __libc_threadCurrentNoAuto();
    if (!pThrd)
    {
        if (fFlags & (__LIBC_BSRF_EXTERNAL | __LIBC_BSRF_HARDWARE))
        {
            LIBC_ASSERTM_FAILED("No thread structure and we cannot safely create one!\n");
            return __LIBC_BSRR_PASSITON | (__SIGSET_ISSET(&__libc_gSignalRestartMask, iSignalNo) ? __LIBC_BSRR_RESTART : __LIBC_BSRR_INTERRUPT);
        }
        pThrd = __libc_threadCurrent();
        if (!pThrd)
        {
            LIBC_ASSERTM_FAILED("Failed to get thread structure!\n");
            return -ENOMEM;
        }
    }

    /*
     * Fill in the rest of the SigInfo packet if we've got one.
     */
    if (pSigInfo)
    {
        if (!pSigInfo->si_timestamp)
            pSigInfo->si_timestamp = signalTimestamp();
        if (fFlags & __LIBC_BSRF_QUEUED)
            pSigInfo->si_flags |= __LIBC_SI_QUEUED;
        if (!pSigInfo->si_pid)
            pSigInfo->si_pid = _sys_pid;
        if (!pSigInfo->si_tid && pThrd)
            pSigInfo->si_tid = pThrd->tid;
    }

    /*
     * Take the semaphore.
     */
    if (__libc_back_signalSemRequest())
    {
        /* we're toast. */
        static const char szMsg[] = "\r\nLIBC: internal error!! signal sem is dead/deadlocked!!!\n\r";
        ULONG   cb;
        LIBCLOG_MSG("we're toast!\n");
        DosWrite(HFILE_STDERR, szMsg, sizeof(szMsg) - 1, &cb);
        signalTerminate(iSignalNo);
    }

    /*
     * Schedule the signal.
     */
    int rc = signalSchedule(pThrd, iSignalNo, pSigInfo, fFlags, fFlags & __LIBC_BSRF_QUEUED ? pvXcptOrQueued : NULL);
    if (rc >= 0)
    {
        /*
         * Paranoia scheduling of the process.
         */
        signalScheduleSPM(pThrd);
        signalScheduleProcess(pThrd);

        /*
         * Deliver signals pending on this thread.
         */
        int rc = signalDeliver(pThrd, fFlags & __LIBC_BSRF_HARDWARE ? iSignalNo : 0,  fFlags & __LIBC_BSRF_QUEUED ? NULL : pvXcptOrQueued);
        if (rc >= 0 && __SIGSET_ISSET(&__libc_gSignalRestartMask, iSignalNo))
            rc = (rc & ~(__LIBC_BSRR_INTERRUPT)) | __LIBC_BSRR_RESTART;
    }

    return rc;
}


/**
 * Schedules a signal to the appropriate thread or leave it pending on the process.
 *
 * @returns 0 on success
 * @returns Negative error code (errno.h) on failure.
 * @param   pThrd               Current thread.
 * @param   iSignalNo           Signal number of the signal to be scheduled.
 * @param   pSigInfo            Pointer to signal info for this signal.
 *                              NULL is allowed.
 * @param   fFlags              Flags of the #defines __LIBC_BSRF_* describing how to
 *                              deliver the signal.
 * @param   pSigQueued          A pointer to locally malloced SIGQUEUED node. NULL is allowed and common.
 */
static int signalSchedule(__LIBC_PTHREAD pThrd, int iSignalNo, siginfo_t *pSigInfo, unsigned fFlags, PSIGQUEUED pSigQueued)
{
    LIBCLOG_ENTER("pThrd=%p iSignalNo=%d pSigInfo=%p fFlags=%#x pSigQueued=%p\n",
                  (void *)pThrd, iSignalNo, (void *)pSigInfo, fFlags, (void *)pSigQueued);

    LIBC_ASSERTM(__libc_back_signalSemIsOwner(), "Thread doesn't own the signal semaphore!!! Bad boy!!\n");

    /*
     * If we're ignoring the signal we just eat it up and return immediately.
     */
    if (    gaSignalActions[iSignalNo].__sigaction_u.__sa_handler == SIG_IGN
        ||  (   gaSignalActions[iSignalNo].__sigaction_u.__sa_handler == SIG_DFL
             && (gafSignalProperties[iSignalNo] & SPA_MASK) == SPA_IGNORE)
        )
    {
        /* free any preallocated queued node. */
        if (pSigQueued)
        {
            pSigQueued->pNext = gpSigQueuedFree;
            gpSigQueuedFree = pSigQueued->pNext;
            gcSigQueuedFree++;
        }

        LIBCLOG_MSG("Ignoring signal %d\n", iSignalNo);
        LIBCLOG_RETURN_INT(0);
    }


    /*
     * Try schedule the signal for a thread.
     */
    __LIBC_PTHREAD  pThrdSig = NULL;
    if (    (fFlags & (__LIBC_BSRF_THREAD | __LIBC_BSRF_HARDWARE))
        || !(gafSignalProperties[iSignalNo] & SPP_ANYTHRD))
    {
        /*
         * Must be scheduled for the current thread.
         */
        pThrdSig = pThrd;
    }
    else
    {
        /*
         * Schedule for any thread in the process (but obviously
         * prefering the current thread when ever possible).
         */
         pThrdSig = signalScheduleThread(iSignalNo, pThrd);
    }

    int rc = 0;
    if (pThrdSig)
    {
        /*
         * Was it waited for?
         */
        if (   pThrdSig->enmStatus == enmLIBCThreadStatus_sigwait
            && __SIGSET_ISSET(&pThrdSig->u.SigWait.SigSetWait, iSignalNo))
        {
            /*
             * Wake up a sigwait call.
             */
            /** @todo install extra exception handler while accessing this. */
            if (pSigInfo)
                *pThrdSig->u.SigWait.pSigInfo = *pSigInfo;
            else
            {
                bzero(pThrdSig->u.SigWait.pSigInfo, sizeof(*pThrdSig->u.SigWait.pSigInfo));
                pThrdSig->u.SigWait.pSigInfo->si_signo = iSignalNo;
            }
            pThrdSig->enmStatus = enmLIBCThreadStatus_unknown;
            if (pThrdSig != pThrd)
                signalPokeThread(pThrdSig);
            LIBCLOG_MSG("wokeup sigwait in thread %d on signal %d.\n", pThrdSig->tid, iSignalNo);
        }
        else
        {
            /*
             * Not waited for, so just queue it if possible and mark it pending.
             */
            if (    (fFlags & __LIBC_BSRF_QUEUED)
                ||  (gafSignalProperties[iSignalNo] & SPP_QUEUED))
            {
                /*
                 * Queue the signal.
                 */
                if (!pSigQueued)
                {   /* 'allocate' it. */
                    pSigQueued = gpSigQueuedFree;
                    if (pSigQueued)
                        gpSigQueuedFree = pSigQueued->pNext;
                }

                /* got a sigqueued node? */
                if (pSigQueued)
                {
                    if (pSigInfo)
                        pSigQueued->SigInfo = *pSigInfo;
                    else
                        bzero(pSigQueued, sizeof(*pSigQueued));
                    pSigQueued->SigInfo.si_signo = iSignalNo;

                    /* insert into the queue, give hardware and kill priority (see signalDeliver()). */
                    if (    (fFlags & __LIBC_BSRF_HARDWARE)
                        ||  iSignalNo == SIGKILL)
                    {
                        pSigQueued->pPrev = NULL;
                        pSigQueued->pNext = pThrdSig->SigQueue.pHead;
                        if (pThrdSig->SigQueue.pHead)
                            pThrdSig->SigQueue.pHead = pThrdSig->SigQueue.pHead->pPrev = pSigQueued;
                        else
                            pThrdSig->SigQueue.pHead = pThrdSig->SigQueue.pTail = pSigQueued;
                    }
                    else
                    {
                        pSigQueued->pNext = NULL;
                        pSigQueued->pPrev = pThrdSig->SigQueue.pTail;
                        if (pThrdSig->SigQueue.pTail)
                            pThrdSig->SigQueue.pTail = pThrdSig->SigQueue.pTail->pNext = pSigQueued;
                        else
                            pThrdSig->SigQueue.pTail = pThrdSig->SigQueue.pHead = pSigQueued;
                    }
                    LIBCLOG_MSG("enqueued signal %d.\n", iSignalNo);
                }
                else
                {   /* arg! out of nodes :/ */
                    LIBC_ASSERTM_FAILED("Out of queue nodes! iSignalNo=%d\n", iSignalNo);
                    rc = -EAGAIN;
                }
            }

            /*
             * Set pending and poke the thread (if it's not ourself).
             */
            if (!rc)
            {
                __SIGSET_SET(&pThrd->SigSetPending, iSignalNo);
                if (pThrdSig != pThrd)
                    signalPokeThread(pThrdSig);
                LIBCLOG_MSG("setting %d pending on thread %d's and poking it.\n", pThrdSig->tid, iSignalNo);
            }
        }
    }
    else
    {
        /*
         * No thread ready for it, leave the signal on process level.
         * Queue it if required and then mark it pending.
         */
        if (    (fFlags & __LIBC_BSRF_QUEUED)
            ||  (gafSignalProperties[iSignalNo] & SPP_QUEUED))
        {
            /*
             * Queue the signal.
             */
            if (!pSigQueued)
            {   /* 'allocate' it. */
                pSigQueued = gpSigQueuedFree;
                if (pSigQueued)
                    gpSigQueuedFree = pSigQueued->pNext;
            }

            /* got a sigqueued node? */
            if (pSigQueued)
            {
                if (pSigInfo)
                    pSigQueued->SigInfo = *pSigInfo;
                else
                    bzero(pSigQueued, sizeof(*pSigQueued));
                pSigQueued->SigInfo.si_signo = iSignalNo;

                /* insert into the queue */
                pSigQueued->pNext = NULL;
                pSigQueued->pPrev = gpSigQueueTail;
                if (gpSigQueueTail)
                    gpSigQueueTail = gpSigQueueTail->pNext = pSigQueued;
                else
                    gpSigQueueTail = gpSigQueueHead = pSigQueued;
                LIBCLOG_MSG("enqueued signal %d.\n", iSignalNo);
            }
            else
            {   /* arg! out of nodes :/ */
                rc = -EAGAIN;
                LIBC_ASSERTM_FAILED("Out of queue nodes! iSignalNo=%d\n", iSignalNo);
            }
        }

        /*
         * Add to the pending signals of this process.
         */
        if (rc >= 0)
        {
            __SIGSET_SET(&__libc_gSignalPending, iSignalNo);
            LIBCLOG_MSG("adding %d to pending signals of the current process.\n", iSignalNo);
        }
    }

    return rc;
}


/**
 * Checks if we can schedule any of the signals queued on the process in SPM.
 *
 * This means discarding (when ignored) or moving signals from pending at 1st
 * level to 3rd level (thread). signalSchedule() is used to do the actual
 * thead scheduling of the signals, which means that the caller must check
 * the current thread for any deliverable signals upon return.
 *
 * @param   pThrd   Current thread.
 */
static void signalScheduleSPM(__LIBC_PTHREAD pThrd)
{
    LIBCLOG_ENTER("pThrd=%p\n", (void *)pThrd);
    LIBC_ASSERTM(__libc_back_signalSemIsOwner(), "Thread doesn't own the signal semaphore!!! Bad boy!!\n");

    /*
     * Get the mask of pending signals.
     */
    sigset_t    SigSetPending;
    int         cSignals = __libc_spmSigPending(&SigSetPending);

    /*
     * Iterate thru the pending signals and see if any of
     * them can be delivered.
     */
    int         iSignalNo = 1;
    while (cSignals > 0 && iSignalNo < __SIGSET_MAXSIGNALS)
    {
        if (__SIGSET_ISSET(&SigSetPending, iSignalNo))
        {
            /*
             * Let's see if we can ignore it or successfully schedule it to a thread.
             */
            __LIBC_PTHREAD pThrdSig = NULL;
            if (    gaSignalActions[iSignalNo].__sigaction_u.__sa_handler == SIG_IGN
                ||  (   gaSignalActions[iSignalNo].__sigaction_u.__sa_handler == SIG_DFL
                     && (gafSignalProperties[iSignalNo] & SPA_MASK) == SPA_IGNORE)
                || (pThrdSig = signalScheduleThread(iSignalNo, pThrd)))
            {
                /*
                 * Fetch signals of this type.
                 */
                siginfo_t SigInfo;
                while (__libc_spmSigDequeue(iSignalNo, &SigInfo, 1, sizeof(SigInfo)) > 0)
                {
                    int rc = signalSchedule(pThrd, iSignalNo, &SigInfo,
                                            __LIBC_BSRF_EXTERNAL | ((SigInfo.si_flags & __LIBC_SI_QUEUED) ? __LIBC_BSRF_QUEUED : 0),
                                            NULL);
                    if (rc >= 0)
                        cSignals--;
                    else
                    {
                        /* Shit! We failed to queue it on a thread. Try put it back in the 1st level queue. */
                        LIBCLOG_MSG("failed to schedule signal %d for any thread!\n", iSignalNo);
                        __libc_spmSigQueue(&SigInfo, _sys_pid, !!(SigInfo.si_flags & __LIBC_SI_QUEUED));
                        break;
                    }
                }
            }
        }

        /* next */
        iSignalNo++;
    }
    LIBCLOG_RETURN_VOID();
}


/**
 * Schedules all signals currently pending on the process (2nd level).
 *
 * @param   pThrd   Current Thread.
 */
static void signalScheduleProcess(__LIBC_PTHREAD pThrd)
{
    LIBCLOG_ENTER("pThrd=%p\n", (void *)pThrd);
    LIBC_ASSERTM(__libc_back_signalSemIsOwner(), "Thread doesn't own the signal semaphore!!! Bad boy!!\n");

    /*
     * Process queued signals first (to get a more correct order).
     */
    sigset_t    SigDone;
    __SIGSET_EMPTY(&SigDone);
    PSIGQUEUED pSig = gpSigQueueHead;
    while (pSig)
    {
        int iSignalNo = pSig->SigInfo.si_signo;
        if (!__SIGSET_ISSET(&SigDone, iSignalNo))
        {
            __LIBC_PTHREAD pThrdSig = signalScheduleThread(iSignalNo, pThrd);
            if (!pThrdSig)
                __SIGSET_SET(&SigDone, iSignalNo);
            else
            {
                /* unlink */
                PSIGQUEUED pSigNext = pSig->pNext;
                if (pSig->pPrev)
                    pSig->pPrev->pNext = pSig->pNext;
                else
                    gpSigQueueHead     = pSig->pNext;
                if (pSig->pNext)
                    pSig->pNext->pPrev = pSig->pPrev;
                else
                    gpSigQueueTail     = pSig->pPrev;

                /* schedule */
                signalScheduleToThread(pThrd, pThrdSig, iSignalNo, pSig);
                __SIGSET_CLEAR(&__libc_gSignalPending, iSignalNo);
                __libc_threadDereference(pThrdSig);

                /* next */
                pSig = pSigNext;
                continue;
            }
        }

        /* next */
        pSig = pSig->pNext;
    }

    /*
     * Process pending signals (which was not queued).
     */
    int iSignalNo = 0;
    while (iSignalNo++ < __SIGSET_MAXSIGNALS)
    {
        if (    !__SIGSET_ISSET(&SigDone, iSignalNo)
            &&  __SIGSET_ISSET(&__libc_gSignalPending, iSignalNo))
        {
            __LIBC_PTHREAD pThrdSig = signalScheduleThread(iSignalNo, pThrd);
            if (pThrdSig)
            {
                signalScheduleToThread(pThrd, pThrdSig, iSignalNo, NULL);
                __SIGSET_CLEAR(&__libc_gSignalPending, iSignalNo);
                __libc_threadDereference(pThrdSig);
            }
        }
    }

    /*
     * Now kick all thread with pending signals which can be delivered.
     */
    signalScheduleKickThreads(pThrd);
    LIBCLOG_RETURN_VOID();
}


/**
 * Schedules a signal to a thread and notifies the thread.
 *
 * @param   pThrd       Current thread.
 * @param   pThrdSig    Thread to schedule the signal to.
 * @param   iSignalNo   Signal to schedule.
 * @param   pSig        Signal queue node containing more info about the signal. (Optional)
 */
static void signalScheduleToThread(__LIBC_PTHREAD pThrd, __LIBC_PTHREAD pThrdSig, int iSignalNo, PSIGQUEUED pSig)
{
    LIBCLOG_ENTER("pThrd=%p pThrdSig=%p {.tid=%#x} iSignalNo=%d pSig=%p\n", (void *)pThrd, (void *)pThrdSig, pThrdSig->tid, iSignalNo, (void *)pSig);
    /*
     * If we've got a queue node, queue it.
     */
    if (pSig)
    {
        pSig->pNext = NULL;
        pSig->pPrev = pThrdSig->SigQueue.pTail;
        if (pThrdSig->SigQueue.pTail)
            pThrdSig->SigQueue.pTail->pNext = pSig;
        else
            pThrdSig->SigQueue.pTail = pThrdSig->SigQueue.pHead = pSig;
    }

    /*
     * In any case, we set the signal pending and poke the thread.
     */
    __SIGSET_SET(&pThrd->SigSetPending, iSignalNo);
    if (pThrdSig != pThrd)
        signalPokeThread(pThrdSig);
    LIBCLOG_MSG("setting %d pending on thread %d's and poking it.\n", pThrdSig->tid, iSignalNo);
    LIBCLOG_RETURN_VOID();
}


/**
 * Schedules the signal on a suitable thread.
 *
 * @returns Pointer to a thread structure (referenced).
 * @returns NULL if no thread can handle the signal.
 * @param   iSignalNo   Signal number in question.
 * @param   pThrdCur    Pointer to the current thread.
 */
__LIBC_PTHREAD  signalScheduleThread(int iSignalNo, __LIBC_PTHREAD pThrdCur)
{
    LIBC_ASSERTM(__libc_back_signalSemIsOwner(), "Thread does not own the signal semaphore!!!\n");

    /*
     * Enumerate all thread in the process and figure out which on
     * is the best suited to receive the signal.
     */
    SIGSCHEDENUMPARAM   EnumParam;
    EnumParam.iSignalNo = iSignalNo;
    EnumParam.pThrd     = pThrdCur;
    return __libc_threadLookup2(signalScheduleThreadWorker, &EnumParam);
}


/**
 * Thread enumerator which selects the best thread for a given signal.
 *
 * @returns 1 if the current thread should be returned.
 * @returns 2 if the current thread should be returned immediately.
 * @returns 0 if the current best thread should remain unchanged.
 * @returns -1 if the enumeration should fail.
 * @param   pCur        The current thread.
 * @param   pBest       The current best thread.
 * @param   pvParam     User parameters (PSIGSCHEDENUMPARAM in our case).
 */
static int signalScheduleThreadWorker(__LIBC_PTHREAD pCur, __LIBC_PTHREAD pBest, void *pvParam)
{
    PSIGSCHEDENUMPARAM pParam = (PSIGSCHEDENUMPARAM)pvParam;

    /*
     * Skip internal threads.
     */
    if (pCur->fInternalThread)
        return 0;

    /*
     * Look for sig*wait*().
     */
    if (   pCur->enmStatus == enmLIBCThreadStatus_sigwait
        && __SIGSET_ISSET(&pCur->u.SigWait.SigSetWait, pParam->iSignalNo))
    {
        if (pCur == pParam->pThrd)
            return 2;                   /* we're done. */
        return 1;                       /* the last wait thread gets it... */
    }

    /*
     * Skip anyone who blocks the signal.
     */
    if (__SIGSET_ISSET(&pCur->SigSetBlocked, pParam->iSignalNo))
        return 0;

    /*
     * Not blocked, check if we've got a better choice.
     * (Better means cur thread, or a sig*wait*() thread.)
     */
    if (    pParam->pThrd == pBest
        ||  (   pBest
             && pBest->enmStatus == enmLIBCThreadStatus_sigwait
             && __SIGSET_ISSET(&pCur->u.SigWait.SigSetWait, pParam->iSignalNo)))
        return 0;
    /* ok, it's not blocking it so it's ok to use it. */
    return 1;
}


/**
 * Enumerates all the threads in the process and poke the ones
 * which have signals that can be process.
 *
 * @param   pThrd   Current thread.
 */
static void signalScheduleKickThreads(__LIBC_PTHREAD pThrd)
{
    __libc_threadEnum(signalScheduleKickThreadsWorker, (void *)pThrd);
}


/**
 * Thread enumeration worker for signalScheduleKickThreads.
 *
 * It poke threads which have signals that can be delivered.
 *
 * @returns 0.
 * @param   pCur        The current thread in the enumration.
 * @param   pvParam     This thread.
 */
static int signalScheduleKickThreadsWorker(__LIBC_PTHREAD pCur, void *pvParam)
{
    /*
     * Skip internal threads.
     */
    if (pCur->fInternalThread)
        return 0;

    /*
     * Walk the queued signals and make sure they're pending.
     * (Self check.)
     */
    PSIGQUEUED pSig = pCur->SigQueue.pHead;
    while (pSig)
    {
        if (!__SIGSET_ISSET(&pCur->SigSetPending, pSig->SigInfo.si_signo))
        {
            LIBC_ASSERTM_FAILED("Signal %d is queued but not marked pending on thread %#x!\n", pSig->SigInfo.si_signo, pCur->tid);
            __SIGSET_SET(&pCur->SigSetPending, pSig->SigInfo.si_signo);
        }

        /* next */
        pSig = pSig->pNext;
    }

    /*
     * Skip further processing of threads which have been poked
     * or if if it's the current thread
     */
    if (    pCur->fSigBeingPoked
        ||  pCur != (__LIBC_PTHREAD)pvParam)
        return 0;

    /*
     * See if there are pending signals that can be delivered.
     */
    sigset_t    SigSet = pCur->SigSetPending;
    __SIGSET_AND(&SigSet, &SigSet, &pCur->SigSetBlocked);
    if (!__SIGSET_EMPTY(&SigSet))
        __libc_back_signalPokeThread(pCur->tid);

    return 0;
}


/**
 * Deliver signals pending on the current thread.
 *
 * Caller must own the semaphore before calling, this function
 * will release the semaphore no matter what happens!
 *
 * @returns On success a flag mask out of the __LIBC_BSRR_* #defines is returned.
 * @returns On failure a negative error code (errno.h) is returned.
 * @param   pThrd               Current thread.
 * @param   iSignalNo           Deliver the first signal of this type no matter if it's
 *                              blocked or not. This is for use with hardware signals only!
 * @param   pvXcptParams        sPointer to exception parameter list is any.
 */
static int signalDeliver(__LIBC_PTHREAD pThrd, int iSignalNo, void *pvXcptParams)
{
    LIBCLOG_ENTER("pThrd=%p pvXcptParams=%p\n", (void *)pThrd, pvXcptParams);
    int rcRet = __LIBC_BSRR_CONTINUE | __LIBC_BSRR_INTERRUPT;

    /*
     * Assert that we do own the semaphore upon entry.
     */
    LIBC_ASSERTM(__libc_back_signalSemIsOwner(), "Thread does not own the signal semaphore!!!\n");

    for (;;)
    {
        /*
         * Anything pending?
         */
        __sigset_t  SigDeliver = pThrd->SigSetPending;
        __SIGSET_AND(&SigDeliver, &SigDeliver, &pThrd->SigSetBlocked);
        if (__SIGSET_ISEMPTY(&SigDeliver))
            break;

        /*
         * Deliver signals /execute signal default actions in this order:
         *   0. Passed in signal (ignores blocking).
         *   1. SIGKILL
         *   2. Realtime signals. (queued)
         *   3. SIGCHLD
         *   4. SIGCONT
         *   5. Ascending from pending mask.
         *
         * Note that 0 and 1 may be violating the POSIX specs.
         */
        if (!iSignalNo || !__SIGSET_ISSET(&pThrd->SigSetPending, iSignalNo))
        {
            iSignalNo = SIGKILL;
            if (!__SIGSET_ISSET(&SigDeliver, SIGKILL))
            {
                iSignalNo = SIGRTMIN;
                while (iSignalNo <= SIGRTMAX && !__SIGSET_ISSET(&SigDeliver, iSignalNo))
                    iSignalNo++;
                if (iSignalNo > SIGRTMAX)
                {
                    iSignalNo = SIGCHLD;
                    if (!__SIGSET_ISSET(&SigDeliver, SIGCHLD))
                    {
                        iSignalNo = SIGCONT;
                        if (!__SIGSET_ISSET(&SigDeliver, SIGCONT))
                        {
                            iSignalNo = 1;
                            while (iSignalNo < SIGRTMIN && !__SIGSET_ISSET(&SigDeliver, iSignalNo))
                                iSignalNo++;
                            if (iSignalNo >= SIGRTMIN)
                            {
                                LIBC_ASSERTM_FAILED("Internal error!!!\n");
                                LIBCLOG_RETURN_INT(__LIBC_BSRR_CONTINUE);
                            }
                        }
                    }
                }
            }
        }

        /*
         * Check if there is a queue element for the signal.
         */
        siginfo_t   SigInfo = {0};
        PSIGQUEUED pSig = pThrd->SigQueue.pHead;
        while (pSig)
        {
            if (pSig->SigInfo.si_signo == iSignalNo)
            {
                /* unlink */
                if (pSig->pPrev)
                    pSig->pPrev->pNext    = pSig->pNext;
                else
                    pThrd->SigQueue.pHead = pSig->pNext;
                if (pSig->pNext)
                    pSig->pNext->pPrev    = pSig->pPrev;
                else
                    pThrd->SigQueue.pTail = pSig->pPrev;

                /*
                 * Check if more signals of this type so we can update the
                 * pending mask correctly.
                 */
                PSIGQUEUED pSigMore = pSig->pNext;
                while (pSigMore && pSigMore->SigInfo.si_signo != iSignalNo)
                    pSigMore = pSigMore->pNext;
                if (!pSigMore)
                    __SIGSET_CLEAR(&pThrd->SigSetPending, iSignalNo);

                /*
                 * Copy the signal to the SigInfo structure on the stack
                 * and then release the signal queue structure.
                 */
                SigInfo = pSig->SigInfo;
                break;
            }
        }

        /*
         * If no queued signal data, then we can simply clear it in the pending mask.
         */
        if (!pSig)
        {
            SigInfo.si_signo = iSignalNo;
            __SIGSET_CLEAR(&pThrd->SigSetPending, iSignalNo);
        }


        /*
         * What to do for this signal?
         */
        struct sigaction SigAction = gaSignalActions[iSignalNo];
        if (    SigAction.__sigaction_u.__sa_handler == SIG_IGN
            ||  (   SigAction.__sigaction_u.__sa_handler == SIG_DFL
                 && (gafSignalProperties[iSignalNo] & SPA_MASK) == SPA_IGNORE)
            )
        {
            /* Ignore the signal. */
            iSignalNo = 0;
            continue;
        }

        if (SigAction.__sigaction_u.__sa_handler == SIG_DFL)
        {
            /*
             * Perform default action.
             */
            switch (gafSignalProperties[iSignalNo] & SPA_MASK)
            {
                /*
                 * Ignore the signal in one of another way.
                 */
                case SPA_NEXT:
                case SPA_IGNORE:
                    break;

                /*
                 * Kill process / dump core (no cores for LIBC!).
                 */
                case SPA_NEXT_KILL:
                case SPA_NEXT_CORE:
                case SPA_KILL:
                case SPA_CORE:
                    signalTerminate(iSignalNo);
                    break;

                /*
                 * Stop the current process.
                 */
                case SPA_STOP:
                case SPA_STOPTTY:
                    signalJobStop(iSignalNo);
                    break;

                /*
                 * Resume the current process.
                 */
                case SPA_RESUME:
                    signalJobResume();
                    break;

                default:
                    LIBC_ASSERTM_FAILED("Invalid signal action property! iSignalNo=%d\n", iSignalNo);
                    break;
            }
        }
        else
        {
            /*
             * Prepare call.
             */
            /* reset action in SysV fashion? */
            if (    (SigAction.sa_flags & SA_RESETHAND)
                && !(gafSignalProperties[iSignalNo] & SPP_NORESET))
            {
                gaSignalActions[iSignalNo].__sigaction_u.__sa_handler = SIG_DFL;
                gaSignalActions[iSignalNo].sa_flags &= ~SA_SIGINFO;
                SigAction.sa_flags |= SA_NODEFER;
            }

            /* adjust and apply the signal mask. */
            if ((SigAction.sa_flags & (SA_ACK | SA_NODEFER)) != SA_NODEFER)
                __SIGSET_SET(&SigAction.sa_mask, iSignalNo);
            if (gafSignalProperties[iSignalNo] & SPP_NOBLOCK)
                __SIGSET_CLEAR(&SigAction.sa_mask, iSignalNo);
            sigset_t SigSetOld = pThrd->SigSetBlocked;
            __SIGSET_OR(&pThrd->SigSetBlocked, &pThrd->SigSetBlocked, &SigAction.sa_mask);

            void *pvCtx = NULL;
            if (SigAction.sa_flags & SA_SIGINFO)
            {
                /** @todo Implement x86 signal context. */
            }

            /*
             * Call the signal handler.
             */
            if (    !(SigAction.sa_flags & SA_ONSTACK)
                ||  pThrd->fSigStackActive
                ||  !pThrd->pvSigStack
                ||  !pThrd->cbSigStack)
            {
                __libc_back_signalSemRelease();
                SigAction.__sigaction_u.__sa_sigaction(iSignalNo, &SigInfo, pvCtx);
            }
            else
            {
                /*
                 * Deliver on an alternative stack.
                 */

                /* get and save the tib stack pointer entries. */
                PTIB pTib;
                PPIB pPib;
                FS_VAR();
                FS_SAVE_LOAD();
                DosGetInfoBlocks(&pTib, &pPib);
                FS_RESTORE();
                PVOID pvOldStack      = pTib->tib_pstack;
                PVOID pvOldStackLimit = pTib->tib_pstacklimit;

                /* Mark active before we touch the stack so we don't make the same mistake twice... */
                pThrd->fSigStackActive = 1;
                /* Setup stack frame for the call. */
                uintptr_t *pStack = (uintptr_t *)((char *)pThrd->pvSigStack + pThrd->cbSigStack - sizeof(uintptr_t));
                *pStack-- = 0; /* where we save esp */
                *pStack-- = (uintptr_t)pvCtx;
                *pStack-- = (uintptr_t)&SigInfo;
                *pStack-- = iSignalNo;
                pTib->tib_pstack      = pThrd->pvSigStack;
                pTib->tib_pstacklimit = (char *)pThrd->pvSigStack + pThrd->cbSigStack;

                /* release, switch and call. */
                __libc_back_signalSemRelease();
                __asm__ __volatile__ (
                    "movl %%esp, %0\n\t"
                    "movl %1, %%esp\n\t"
                    "call *%2\n\t"
                    "movl 12(%%esp), %%esp\n\t"
                    : "=m" (*(uintptr_t *)((char *)pThrd->pvSigStack + pThrd->cbSigStack - sizeof(uintptr_t)))
                    : "a" (pStack),
                      "d" (SigAction.__sigaction_u.__sa_sigaction)
                    : "ecx" );

                /* Restore tib and release the stack. */
                pTib->tib_pstack      = pvOldStack;
                pTib->tib_pstacklimit = pvOldStackLimit;
                pThrd->fSigStackActive = 0;
            }

            /*
             * Handler returned, what to do now?
             */
            switch (gafSignalProperties[iSignalNo] & SPR_MASK)
            {
                /*
                 * Kill the process.
                 */
                case SPR_KILL:
                    signalTerminate(iSignalNo);
                    break;

                /*
                 * Execution should continue on return, which is our default return.
                 */
                case SPR_CONTINUE:
                    break;

                default:
                    LIBC_ASSERTM_FAILED("Invalid signal return action property! iSignalNo=%d\n", iSignalNo);
                    break;
            }

            /*
             * Re-take the signal semaphore and restore the signal mask.
             */
            if (__libc_back_signalSemRequest())
                break;
            pThrd->SigSetBlocked = SigSetOld;
        }

        iSignalNo = 0;
    } /* forever */

    LIBCLOG_RETURN_INT(rcRet);
}


/**
 * Cause immediate process termination because of the given signal.
 *
 * @param   iSignalNo   Signal causing the termination.
 */
static void signalTerminate(int iSignalNo)
{
    /** @todo implement process termination by signal properly! */
    /*
     * Before we go crazy here, let's release the semaphore.
     *
     * If possible we'll stay in the must complete section to
     * be sure we get to the exit. If not we'll have to mess with
     * exception handlers and such now.
     */
    //if (__libc_back_signalSemIsOwner())
    //    DosReleaseMutexSem(ghmtxSignals);

    for (;;)
        DosExit(EXIT_THREAD, 127);
}

/**
 * Stop the process (job control emulation).
 *
 * @param   iSignalNo   Signal causing the stop.
 */
static int signalJobStop(int iSignalNo)
{
    /** @todo Job control: tell parent and ensure that thread 1 is frozen and unfrozen. */
    DosWaitEventSem(ghevWait, SEM_INDEFINITE_WAIT);
    return 0;
}

/**
 * Resume the process (job control emulation).
 *
 * @param   iSignalNo   Signal causing the termination.
 */
static int signalJobResume(void)
{
    /** @todo Job control: tell parent and ensure that thread 1 is unfrozen. */
    return 0;
}


/**
 * Send signal to other process.
 *
 * Practically speaking a kill() implementation, but only for a single process.
 *
 * @return 0 on success.
 * @return -errno on failure.
 * @param   iSignalNo   Signal number to send.
 */
int __libc_back_signalSendPidOther(pid_t pid, int iSignalNo)
{
    LIBCLOG_ENTER("pid=%d iSignalNo=%d\n", pid, iSignalNo);
    int     rc;
    int     rcOS2;
    FS_VAR();

    /*
     * Send the signal.
     */
    switch (iSignalNo)
    {
        /*
         * SIGKILL means kill process.
         */
        case SIGKILL:
            FS_SAVE_LOAD();
            rcOS2 = DosKillProcess(DKP_PROCESS, pid);
            FS_RESTORE();
            switch (rcOS2)
            {
                case NO_ERROR:
                    LIBCLOG_RETURN_INT(0);

                case ERROR_INVALID_PROCID:
                    rc = -ESRCH;
                    break;
                default:
                    rc = -EPERM;
                    break;
            }
            LIBCLOG_RETURN_MSG(rc, "ret %d (DosKillProcess rcOS2=%d)\n", rc, rcOS2);

        /*
         * We can send these the normal way to decentants.
         */
        case SIGINT:
        case SIGBREAK:
            FS_SAVE_LOAD();
            rcOS2 = DosSendSignalException(pid, iSignalNo == SIGINT ? XCPT_SIGNAL_INTR : XCPT_SIGNAL_BREAK);
            FS_RESTORE();
            switch (rcOS2)
            {
                case NO_ERROR:
                    LIBCLOG_RETURN_INT(0);

                case ERROR_INVALID_PROCID:
                    LIBCLOG_RETURN_INT(-ESRCH);
            }
            /* try flag it */
            break;

        default:
            break;
    }

    /*
     * Try flag the process.
     *
     * If the target is a LIBC process we'll queue a signal on it and poke it.
     * If not we'll convert the signal to EMX and flag it in the EMX compatible way.
     */
    siginfo_t SigInfo = {0};
    SigInfo.si_signo    = iSignalNo;
    SigInfo.si_pid      = _sys_pid;
    __LIBC_PTHREAD pThrd = __libc_threadCurrentNoAuto();
    if (pThrd)
        SigInfo.si_tid  = pThrd->tid;
    SigInfo.si_timestamp= signalTimestamp();
    if (gafSignalProperties[iSignalNo] & SPP_QUEUED)
        SigInfo.si_flags |= __LIBC_SI_QUEUED;

    rc = __libc_spmSigQueue(&SigInfo, pid, SigInfo.si_flags & __LIBC_SI_QUEUED);
    if (!rc)
    {
        /* poke it... */
        FS_SAVE_LOAD();
        rc = DosFlagProcess(pid, FLGP_PID, PFLG_A, 0);
        if (rc == ERROR_SIGNAL_REFUSED)
        {
            /* Try again... no sure what would cause this condition... */
            DosSleep(0);
            DosSleep(1);
            rc = DosFlagProcess(pid, FLGP_PID, PFLG_A, 0);
        }
        FS_RESTORE();
    }
    else if (rc == -ESRCH)
    {
        /*
         * EMX style signal.
         * Doesn't work for all types of signals.
         */
        unsigned uSignalEMX = 0;
        switch (iSignalNo)
        {
            case SIGHUP  :  uSignalEMX = EMX_SIGHUP;   break;
            case SIGINT  :  uSignalEMX = EMX_SIGINT;   break;
            case SIGQUIT :  uSignalEMX = EMX_SIGQUIT;  break;
            case SIGILL  :  uSignalEMX = EMX_SIGILL;   break;
            case SIGTRAP :  uSignalEMX = EMX_SIGTRAP;  break;
            case SIGABRT :  uSignalEMX = EMX_SIGABRT;  break;
            case SIGEMT  :  uSignalEMX = EMX_SIGEMT;   break;
            case SIGFPE  :  uSignalEMX = EMX_SIGFPE;   break;
            case SIGKILL :  uSignalEMX = EMX_SIGKILL;  break;
            case SIGBUS  :  uSignalEMX = EMX_SIGBUS;   break;
            case SIGSEGV :  uSignalEMX = EMX_SIGSEGV;  break;
            case SIGSYS  :  uSignalEMX = EMX_SIGSYS;   break;
            case SIGPIPE :  uSignalEMX = EMX_SIGPIPE;  break;
            case SIGALRM :  uSignalEMX = EMX_SIGALRM;  break;
            case SIGTERM :  uSignalEMX = EMX_SIGTERM;  break;
            case SIGUSR1 :  uSignalEMX = EMX_SIGUSR1;  break;
            case SIGUSR2 :  uSignalEMX = EMX_SIGUSR2;  break;
            case SIGCHLD :  uSignalEMX = EMX_SIGCHLD;  break;
            case SIGBREAK:  uSignalEMX = EMX_SIGBREAK; break;
            case SIGWINCH:  uSignalEMX = EMX_SIGWINCH; break;

        }
        if (uSignalEMX)
        {
            FS_SAVE_LOAD();
            rc = DosFlagProcess(pid, FLGP_PID, PFLG_A, uSignalEMX);
            if (rc == ERROR_SIGNAL_PENDING || rc == ERROR_SIGNAL_REFUSED)
            {
                DosSleep(0);
                DosSleep(1);
                rc = DosFlagProcess(pid, FLGP_PID, PFLG_A, uSignalEMX);
            }
            FS_RESTORE();
        }
        else
            LIBCLOG_RETURN_INT(-EPERM);
    }
    else
        LIBCLOG_RETURN_INT(rc);

    /*
     * Check for DosFlagProcess errors.
     */
    if (!rc)
        LIBCLOG_RETURN_INT(0);
    rc = -__libc_native2errno(rc);
    LIBCLOG_RETURN_INT(rc);
}


/**
 * OS/2 V1 signal handler used for interprocess signaling.
 * This is called by the 16-bit thunker.
 *
 * @param   uOS2Signal  Signal number - SIG_PFLG_A.
 * @param   uArg        LIBC sets this to 0 when the target is a LIBC process.
 *                      If the target is a non-LIBC process it's an EMX signal
 *                      number.
 */
void __libc_back_signalOS2V1Handler32bit(unsigned uOS2Signal, unsigned uArg)
{
    LIBCLOG_ENTER("uOS2Signal=%u uArg=%u\n", uOS2Signal, uArg);

    /*
     * Can't be to careful!
     */
    LIBC_ASSERTM(!__libc_back_signalSemIsOwner(), "Thread owns the signal semaphore!!! That should be impossible...\n");

    /*
     * Get the current thread (1st thread, so that should be safe).
     */
    __LIBC_PTHREAD  pThrd = __libc_threadCurrentNoAuto();
    if (!pThrd)
    {
        LIBC_ASSERTM_FAILED("Failed to get thread structure!\n");
        LIBCLOG_RETURN_VOID();
    }

    /*
     * Try schedule signals pending on 1st (SPM) and 2nd level (private).
     */
    signalScheduleSPM(pThrd);
    signalScheduleProcess(pThrd);

    /*
     * This could be an EMX signal.
     */
    if (uArg)
    {
        unsigned iSignalNo = 0;
        switch (uArg)
        {
            case EMX_SIGHUP  :  iSignalNo = SIGHUP;   break;
            case EMX_SIGINT  :  iSignalNo = SIGINT;   break;
            case EMX_SIGQUIT :  iSignalNo = SIGQUIT;  break;
            case EMX_SIGILL  :  iSignalNo = SIGILL;   break;
            case EMX_SIGTRAP :  iSignalNo = SIGTRAP;  break;
            case EMX_SIGABRT :  iSignalNo = SIGABRT;  break;
            case EMX_SIGEMT  :  iSignalNo = SIGEMT;   break;
            case EMX_SIGFPE  :  iSignalNo = SIGFPE;   break;
            case EMX_SIGKILL :  iSignalNo = SIGKILL;  break;
            case EMX_SIGBUS  :  iSignalNo = SIGBUS;   break;
            case EMX_SIGSEGV :  iSignalNo = SIGSEGV;  break;
            case EMX_SIGSYS  :  iSignalNo = SIGSYS;   break;
            case EMX_SIGPIPE :  iSignalNo = SIGPIPE;  break;
            case EMX_SIGALRM :  iSignalNo = SIGALRM;  break;
            case EMX_SIGTERM :  iSignalNo = SIGTERM;  break;
            case EMX_SIGUSR1 :  iSignalNo = SIGUSR1;  break;
            case EMX_SIGUSR2 :  iSignalNo = SIGUSR2;  break;
            case EMX_SIGCHLD :  iSignalNo = SIGCHLD;  break;
            case EMX_SIGBREAK:  iSignalNo = SIGBREAK; break;
            case EMX_SIGWINCH:  iSignalNo = SIGWINCH; break;
        }
        if (iSignalNo)
        {
            /*
             * Shcedule the signal, ignore return code.
             */
            signalSchedule(pThrd, iSignalNo, NULL, __LIBC_BSRF_EXTERNAL, NULL);
        }
    }

    /*
     * Acknowledge the signal.
     */
    DosSetSigHandler(NULL, NULL, NULL, SIGA_ACKNOWLEDGE, uOS2Signal);

    /*
     * Deliver signals.
     */
    signalDeliver(pThrd, 0, NULL);
    LIBCLOG_RETURN_VOID();
}


/**
 * Pokes a thread in the current process, forcing it to
 * evaluate pending signals.
 *
 * @param   tid     Thread to poke. 0 means current thread.
 */
void        __libc_back_signalPokeThread(int tid)
{
    LIBC_ASSERTM(__libc_back_signalSemIsOwner(), "Thread must own signal sem!\n");

    /*
     * Get thread structure of target thread.
     */
    __LIBC_PTHREAD pThrdPoke = __libc_threadLookup(tid);
    if (pThrdPoke)
    {
        signalPokeThread(pThrdPoke);
        __libc_threadDereference(pThrdPoke);
    }
}


/**
 * Pokes a thread in the current process, forcing it to
 * evaluate pending signals.
 *
 * @param   pThrdPoke       Thread to poke.
 */
static void     signalPokeThread(__LIBC_PTHREAD pThrdPoke)
{
    LIBC_ASSERTM(__libc_back_signalSemIsOwner(), "Thread must own signal sem!\n");

    /*
     * Kick it if it's not already poked.
     */
    if (!pThrdPoke->fSigBeingPoked)
    {
        __atomic_xchg(&pThrdPoke->fSigBeingPoked, 1);
        int rc = DosKillThread(pThrdPoke->tid);
        if (rc)
            LIBC_ASSERTM_FAILED("DosKillThread(%d) -> rc=%d\n", pThrdPoke->tid, rc);
    }
}



/**
 * Thread was poked, deliver signals for this thread.
 */
int         __libc_back_signalRaisePoked(void *pvXcptParams, int tidPoker)
{
    LIBC_ASSERTM(!__libc_back_signalSemIsOwner(), "Thread owns the signal semaphore!!! Bad boy!!\n");

    /*
     * Get thread structure.
     */
    __LIBC_PTHREAD pThrd = __libc_threadCurrentNoAuto();
    if (pThrd)
    {
        /*
         * Take signal semaphore.
         */
        int rc = __libc_back_signalSemRequest();
        if (!rc)
        {
            if (pThrd->fSigBeingPoked)
            {
                /*
                 * Clear the indicator and deliver signals.
                 */
                __atomic_xchg(&pThrd->fSigBeingPoked, 0);
                rc = signalDeliver(pThrd, 0, pvXcptParams);
                if (rc < 0)
                    rc = __LIBC_BSRR_CONTINUE | __LIBC_BSRR_INTERRUPT;
                return rc;
            }

            /*
             * We weren't being poked.
             */
            __libc_back_signalSemRelease();
        }
    }
    else
        LIBC_ASSERTM_FAILED("No thread structure! This is really impossible!\n");

    return -EINVAL;
}


/**
 * sigaction worker; queries and/or sets the action for a signal.
 *
 * The caller MUST own the signal semaphore when calling this function.
 *
 * @returns 0 on success.
 * @returns -1 on failure, errno set.
 * @param   iSignalNo   Signal number.
 * @param   pSigAct     Pointer to new signal action.
 *                      If NULL no update is done.
 * @param   pSigActOld  Where to store the old signal action.
 *                      If NULL nothing is attempted stored.
 */
int         __libc_back_signalAction(int iSignalNo, const struct sigaction *pSigAct, struct sigaction *pSigActOld)
{
    LIBCLOG_ENTER("iSignalNo=%d pSigAct=%p {sa_handler=%p, sa_mask={%#08lx%#08lx}, sa_flags=%#x} pSigActOld=%p\n",
                  iSignalNo, (void *)pSigAct,
                  pSigAct ? (void*)pSigAct->__sigaction_u.__sa_sigaction : NULL,
                  pSigAct ? pSigAct->sa_mask.__bitmap[0] : 0,
                  pSigAct ? pSigAct->sa_mask.__bitmap[1] : 0,
                  pSigAct ? pSigAct->sa_flags : 0,
                  (void *)pSigActOld);
    int rc = 0;

    /*
     * Assert state.
     */
    LIBC_ASSERTM(__libc_back_signalSemIsOwner(), "Thread does not own the signal semaphore!!!\n");

    /*
     * Check input.
     */
    if (!__SIGSET_SIG_VALID(iSignalNo))
    {
        errno = EINVAL;
        LIBCLOG_RETURN_INT(-1);
    }

    /*
     * Query.
     */
    if (pSigActOld)
    {
        *pSigActOld = gaSignalActions[iSignalNo];
    }

    /*
     * Set.
     */
    if (pSigAct)
    {
        /*
         * Validate the new action.
         */
        if (!(pSigAct->sa_flags & ~(  SA_ONSTACK | SA_RESTART | SA_RESETHAND | SA_NODEFER | SA_NOCLDWAIT
                                    | SA_SIGINFO | SA_NOCLDSTOP | SA_ACK | SA_SYSV)))
        {
            if (    pSigAct->__sigaction_u.__sa_handler != SIG_ERR
                &&  pSigAct->__sigaction_u.__sa_handler != SIG_ACK)
            {
                /*
                 * Update the handler.
                 */
                gaSignalActions[iSignalNo] = *pSigAct;

                /*
                 * If set to ignore, we'll remove any pending signals of this type.
                 */
                if (    gaSignalActions[iSignalNo].__sigaction_u.__sa_handler == SIG_IGN
                    ||  (   gaSignalActions[iSignalNo].__sigaction_u.__sa_handler == SIG_DFL
                         && (gafSignalProperties[iSignalNo] & SPA_MASK) == SPA_IGNORE))
                {
                    __libc_threadEnum(signalActionWorker, (void *)iSignalNo);
                }
            }
            else
            {
                LIBC_ASSERTM_FAILED("Invalid sa_handler=%p\n", (void *)pSigAct->__sigaction_u.__sa_handler);
                errno = EINVAL;
            }
        }
        else
        {
            LIBC_ASSERTM_FAILED("Invalid sa_flags=%#x\n", pSigAct->sa_flags);
            errno = EINVAL;
        }

    }

    LIBCLOG_RETURN_MSG(rc, "ret %d (pSigActOld=%p {sa_handler=%p, sa_mask={%#08lx%#08lx}, sa_flags=%#x})\n",
                       rc, (void *)pSigActOld,
                       pSigActOld ? (void*)pSigActOld->__sigaction_u.__sa_sigaction : NULL,
                       pSigActOld ? pSigActOld->sa_mask.__bitmap[0] : 0,
                       pSigActOld ? pSigActOld->sa_mask.__bitmap[1] : 0,
                       pSigActOld ? pSigActOld->sa_flags : 0);
}



/**
 * Thread enumeration worker for __libc_back_signalAction.
 *
 * It discards pending and queued signals which now are ignored.
 * @returns 0.
 * @param   pCur        The current thread.
 * @param   pvParam     The signal number.
 */
static int signalActionWorker(__LIBC_PTHREAD pCur, void *pvParam)
{
    /*
     * Skip internal threads.
     */
    if (pCur->fInternalThread)
        return 0;

    /*
     * Check if the thread have iSignalNo pending.
     */
    int iSignalNo = (int)pvParam;
    if (__SIGSET_ISSET(&pCur->SigSetPending, iSignalNo))
    {
        __SIGSET_CLEAR(&pCur->SigSetPending, iSignalNo);

        /*
         * Check if there are any queued signals of the sort,
         * remove them from the queue.
         */
        PSIGQUEUED   pSig = pCur->SigQueue.pHead;
        while (pSig)
        {
            PSIGQUEUED   pSigNext = pSig->pNext;
            if (pSig->SigInfo.si_signo == iSignalNo)
            {
                /* unlink */
                if (pSig->pPrev)
                    pSig->pPrev->pNext = pSig->pNext;
                else
                    pCur->SigQueue.pHead = pSig->pNext;
                if (pSig->pNext)
                    pSig->pNext->pPrev = pSig->pPrev;
                else
                    pCur->SigQueue.pTail = pSig->pPrev;

                /* free it */
                switch (pSig->enmHowFree)
                {
                    case enmHowFree_PreAllocFree:
                        pSig->pNext = gpSigQueuedFree;
                        gpSigQueuedFree = pSig;
                        break;
                    case enmHowFree_HeapFree:
                        free(pSig);
                        break;
                    case enmHowFree_Temporary:
                        /* do nothing */
                        break;
                }
            }

            /* next */
            pSig = pSigNext;
        }
    }

    return 0;
}


/**
 * Gets the current timestamp.
 *
 * @returns The current timestamp.
 */
static unsigned signalTimestamp(void)
{
    ULONG   ul;
    int     rc;
    FS_VAR();
    FS_SAVE_LOAD();
    ul = 0;
    rc = DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT, &ul, sizeof(ul));
    LIBC_ASSERTM(!rc, "DosQuerySysInfo failed rc=%d\n", rc);
    FS_RESTORE();
    rc = rc;
    return (unsigned)ul;
}

