/* $Id: signals.h 1615 2004-11-06 08:35:32Z bird $ */
/** @file
 *
 * InnoTek LIBC - Signal Internals.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __InnoTekLIBC_signals_h__
#define __InnoTekLIBC_signals_h__

/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
/** @defgroup __libc_back_signalRaise_return    __libc_back_signalRaise() returns.
 * These are only valid for positive return values.
 * @{ */
/** Try restart any interrupted system call. */
#define __LIBC_BSRR_RESTART     0x01
/** Go ahead interrupt system call in progress. */
#define __LIBC_BSRR_INTERRUPT   0x02
/** If set execution should be resumed. */
#define __LIBC_BSRR_CONTINUE    0x10
/** If set execution should not be resumed but the signal should be passed
 * on to the system. */
#define __LIBC_BSRR_PASSITON    0x20
/** If set the passed in SIGQUEUED structure was used. */
#define __LIBC_BSRR_USED_QUEUED 0x40
/** @} */

/** @defgroup __libc_back_signalRaise_flags     __libc_back_signalRaise() flags.
 * @{ */
/** The signal is thread specific and must be delivered to the current thread. */
#define __LIBC_BSRF_THREAD      0x01
/** The signal was send from an unknown process. */
#define __LIBC_BSRF_EXTERNAL    0x02
/** The signal was generated by the hardware (i.e. CPUs and such). */
#define __LIBC_BSRF_HARDWARE    0x04
/** The signal should be queued. */
#define __LIBC_BSRF_QUEUED      0x08
/** @} */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include <sys/cdefs.h>
#include <signal.h>


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/** Signal set of the signals which will interrupt system call execution.
 * By default all signals will interrupt syscall execution, since OS/2 can't really
 * restart system calls easily.
 * Update is protected by the signal semaphore, however read access isn't.
 */
extern sigset_t     __libc_gSignalRestartMask;

/** Signal set for the signals which are pending on a process scope.
 * These signals are blocked by all threads and waiting to be delivered
 * to a thread. */
extern sigset_t     __libc_gSignalPending;


__BEGIN_DECLS

int         __libc_back_signalInit(void);
int         __libc_back_signalSemRequest(void);
void        __libc_back_signalSemRelease(void);
unsigned    __libc_back_signalSemIsOwner(void);
void        __libc_back_signalPokeThread(int tid);
void        __libc_back_signalPokeProcess(void);
int         __libc_back_signalSuspend(void);
int         __libc_back_signalWait(const struct timespec *pTimeout);
int         __libc_back_signalSendPidOther(pid_t pid, int iSignalNo);
int         __libc_back_signalAction(int iSignalNo, const struct sigaction *pSigAct, struct sigaction *pSigActOld);
int         __libc_back_signalRaisePoked(void *pvXcptParams, int tidPoker);
void        __libc_back_signalOS2V1Handler16bit(unsigned short uSignal, unsigned short uArg);
void        __libc_back_signalOS2V1Handler32bit(unsigned uSignal, unsigned uArg);


/**
 * Raises a signal in the current process.
 *
 * @returns On success a flag mask out of the __LIBC_BSRR_* #defines is returned.
 * @returns On failure a negative error code (errno.h) is returned.
 * @param   iSignalNo           Signal to raise.
 * @param   pSigInfo            Pointer to signal info for this signal.
 *                              NULL is allowed.
 * @param   pvXcptOrQueued      Exception handler parameter list.
 *                              Or if __LIBC_BSRF_QUEUED is set, a pointer to locally malloced
 *                              SIGQUEUED node.
 * @param   fFlags              Flags of the #defines __LIBC_BSRF_* describing how to
 *                              deliver the signal.
 *
 * @remark  This Backend Signal API does NOT require the caller to own the signal semaphore.
 */
int __libc_Back_signalRaise(int iSignalNo, siginfo_t *pSigInfo, void *pvXcptOrQueued, unsigned fFlags);

/**
 * Send a signal to a process.
 *
 * @returns 0 on if signal sent.
 * @returns -errno on failure.
 *
 * @param   pid         Process Id of the process which the signal is to be sent to.
 * @param   iSignalNo   The signal to send.
 * @remark  This Backend Signal API does NOT require the caller to own the signal semaphore.
 */
int     __libc_Back_signalSendPid(pid_t pid, int iSignalNo);

/**
 * Verify the existance of another process and that the current process
 * is allowed to signal it.
 *
 * @return 0 on success.
 * @return -ESRCH if pid doesn't exist.
 * @return -EPERM if we aren't allowed to signal the pid.
 * @param   pid     Process Id for the process which we wanna signal.
 * @todo    Do EPERM check, no ideas here yet.
 * @remark  This Backend Signal API does NOT require the caller to own the signal semaphore.
 */
int     __libc_Back_signalVerifyPid(pid_t pid);

/**
 * Not implemented.
 *
 * @returns -ENOSYS.
 * @param   pgrp        Process group (positive).
 *                      0 means the process group of this process.
 *                      1 means all process in the system.
 * @param   iSignalNo   Signal to send to all the processes in the group.
 */
int     __libc_Back_signalSendPGrp(pid_t pgrp, int iSignalNo);

/**
 * Verify the existance of a process group and that the current process
 * is allowed to signal it.
 *
 * @return 0 on success.
 * @return -ESRCH if pgid doesn't exist.
 * @return -EPERM if we aren't allowed to signal the pgid.
 * @param   pgid    Process group id which the current process intend to signal.
 * @todo    Do EPERM check, no ideas here yet.
 * @remark  This Backend Signal API does NOT require the caller to own the signal semaphore.
 */
int     __libc_Back_signalVerifyPGrp(pid_t pgid);


__END_DECLS

#endif
