/* $Id: sigprocmask.c 1614 2004-11-05 02:16:17Z bird $ */
/** @file
 *
 * LIBC - sigprocmask().
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#include <signal.h>
#include <sys/_sigset.h>
#include <errno.h>
#include <InnoTekLIBC/signals.h>
#include <InnoTekLIBC/thread.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_SIGNAL
#include <InnoTekLIBC/logstrict.h>


/**
 * Block or unblock signal deliveries.
 *
 * This API was by the POSIX specs only intended for use in a single threaded process,
 * however we define it as equivalent to pthread_sigmask().
 *
 * @returns 0 on success.
 * @returns -1 and errno set to EINVAL on failure.
 * @param   iHow        Describes the action taken if pSigSetNew not NULL. Recognized
 *                      values are SIG_BLOCK, SIG_UNBLOCK or SIG_SETMASK.
 *
 *                      SIG_BLOCK means to or the sigset pointed to by pSigSetNew with
 *                          the signal mask for the current thread.
 *                      SIG_UNBLOCK means to and the 0 complement of the sigset pointed
 *                          to by pSigSetNew with the signal mask of the current thread.
 *                      SIG_SETMASK means to set the signal mask of the current thread
 *                          to the sigset pointed to by pSigSetNew.
 *
 * @param   pSigSetNew  Pointer to signal set which will be applied to the current
 *                      threads signal mask according to iHow. If NULL no change
 *                      will be made the the current threads signal mask.
 * @param   pSigSetOld  Where to store the current threads signal mask prior to applying
 *                      pSigSetNew to it. This parameter can be NULL.
 */
int _STD(sigprocmask)(int iHow, const sigset_t * __restrict pSigSetNew, sigset_t * __restrict pSigSetOld)
{
    LIBCLOG_ENTER("iHow=%d pSigSetNew=%p {%08lx%08lx} pSigSetOld=%p\n",
                  iHow,
                  (void *)pSigSetNew,
                  pSigSetNew ? pSigSetNew->__bitmap[1] : 0, pSigSetNew ? pSigSetNew->__bitmap[0] : 0,
                  (void *)pSigSetOld);
    __LIBC_THREAD  *pThrd = __libc_threadCurrent();
    sigset_t        SigSetNew;
    sigset_t        SigSetOld;
    sigset_t        SigSet;

    /*
     * Validate input.
     * ASSUMES SIG_BLOCK 1 and SIG_SETMASK 3.
     */
    if (iHow < SIG_BLOCK || iHow > SIG_SETMASK)
    {
        LIBC_ASSERTM_FAILED("iHow=%d is wrong\n", iHow);
        errno = EINVAL;
        LIBCLOG_RETURN_INT(-1);
    }


    /*
     * Prepare action.
     * We make copies on the stack so we will not crash accessing
     * bogus stuff inside the signal mutex.
     */
    __SIGSET_EMPTY(&SigSetOld);         /* touch it. (paranoia) */
    if (pSigSetNew)
        SigSetNew = *pSigSetNew;
    else
        __SIGSET_EMPTY(&SigSetNew);

    /*
     * Gain exclusive access to the signal stuff.
     */
    if (__libc_back_signalSemRequest())
        LIBCLOG_RETURN_INT(-1);

    /*
     * Save old set of blocked signals and apply any changes.
     */
    SigSetOld = pThrd->SigSetBlocked;
    if (pSigSetNew)
    {
        switch (iHow)
        {
            case SIG_BLOCK:
                __SIGSET_OR(&pThrd->SigSetBlocked, &pThrd->SigSetBlocked, &SigSetNew);
                break;

            case SIG_UNBLOCK:
                __SIGSET_NOT(&SigSetNew);
                __SIGSET_AND(&pThrd->SigSetBlocked, &pThrd->SigSetBlocked, &SigSetNew);
                break;

            case SIG_SETMASK:
                pThrd->SigSetBlocked = SigSetNew;
                break;
        }

        /*
         * Check if we've unblocked anything and thus made it possible for
         * pending signals to be delivered.
         */
        SigSetNew = pThrd->SigSetBlocked;
        __SIGSET_AND(&SigSet, &SigSetOld, &SigSetNew);
        if (!__SIGSET_ISEMPTY(&SigSetNew))
        {
            sigset_t    SigSetPending;
            __SIGSET_OR(&SigSetPending, &pThrd->SigSetPending, &__libc_gSignalPending);
            __SIGSET_AND(&SigSetPending, &SigSetPending, &SigSetNew);
            if (!__SIGSET_ISEMPTY(&SigSetPending))
                __libc_back_signalPokeThread(0);
        }
    }

    /*
     * Release semaphore.
     */
    __libc_back_signalSemRelease();

    /*
     * Copy old signal set if requested.
     */
    if (pSigSetOld)
        *pSigSetOld = SigSetOld;

    LIBCLOG_MSG("old={%08lx%08lx} new={%08lx%08lx}\n",
                SigSetOld.__bitmap[1], SigSetOld.__bitmap[0],
                SigSetNew.__bitmap[1], SigSetNew.__bitmap[0]);
    LIBCLOG_RETURN_INT(0);
}
