/* $Id: sigaltstack.c 1614 2004-11-05 02:16:17Z bird $ */
/** @file
 *
 * LIBC - sigaltstack().
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#include <signal.h>
#include <errno.h>
#include <InnoTekLIBC/signals.h>
#include <InnoTekLIBC/thread.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_SIGNAL
#include <InnoTekLIBC/logstrict.h>



int _STD(sigaltstack)(const stack_t *pStack, stack_t *pOldStack)
{
    LIBCLOG_ENTER("pStack=%p {ss_flags=%#x, ss_sp=%p, ss_size=%d} pOldStack=%p\n",
                  (void *)pStack,
                  pStack ? pStack->ss_flags : 0,
                  pStack ? pStack->ss_sp    : NULL,
                  pStack ? pStack->ss_size  : 0,
                  (void *)pOldStack);
    __LIBC_THREAD  *pThrd = __libc_threadCurrent();
    stack_t         OldStack = {0};
    stack_t         Stack;
    int             rc = 0;

    /*
     * Validate input.
     */
    if (pStack)
    {
        if (pStack->ss_flags & ~SS_DISABLE)
        {
            LIBC_ASSERTM_FAILED("ss_flags=%#x is invalid\n", pStack->ss_flags);
            errno = EINVAL;
            LIBCLOG_RETURN_INT(-1);
        }
        if (pStack->ss_flags != SS_DISABLE)
        {
            if (!pStack->ss_sp)
            {
                LIBC_ASSERTM_FAILED("ss_sp=%p is NULL when SS_DISABLE is not set\n", pStack->ss_sp);
                errno = EINVAL;
                LIBCLOG_RETURN_INT(-1);
            }
            if (!pStack->ss_size < MINSIGSTKSZ)
            {
                LIBC_ASSERTM_FAILED("ss_size=%d is too small. minimum size is %d\n", pStack->ss_size, MINSIGSTKSZ);
                errno = ENOMEM;
                LIBCLOG_RETURN_INT(-1);
            }
        }
    }

    /*
     * Copy input.
     */
    if (pStack)
        Stack = *pStack;

    /*
     * Gain exclusive access to the signal stuff.
     */
    if (__libc_back_signalSemRequest())
        LIBCLOG_RETURN_INT(-1);

    /*
     * Get the old stack.
     */
    OldStack.ss_flags = pThrd->pvSigStack ? 0 : SS_DISABLE;
    if (pThrd->fSigStackActive)
        OldStack.ss_flags |= SS_ONSTACK;
    OldStack.ss_sp    = pThrd->pvSigStack;
    OldStack.ss_size  = pThrd->cbSigStack;

    /*
     * Change the stack.
     */
    if (pStack)
    {
        if (!pThrd->fSigStackActive)
        {
            if (Stack.ss_flags == SS_ONSTACK)
            {
                pThrd->pvSigStack = Stack.ss_sp;
                pThrd->cbSigStack = Stack.ss_size;
            }
            else
            {
                pThrd->pvSigStack = NULL;
                pThrd->cbSigStack = 0;
            }
        }
        else
            rc = EPERM;
    }

    /*
     * Release semaphore.
     */
    __libc_back_signalSemRelease();


    /*
     * Check for failure and return.
     */
    if (rc)
    {
        LIBC_ASSERTM_FAILED("Stack is in use\n");
        errno = rc;
        LIBCLOG_RETURN_INT(-1);
    }

    if (pOldStack)
        *pOldStack = OldStack;

    LIBCLOG_RETURN_MSG(0, "ret 0 (*pOldStack {ss_flags=%#x, ss_sp=%p, ss_size=%d}\n",
                       OldStack.ss_flags, OldStack.ss_sp, OldStack.ss_size);
}
