/* $Id: sigtimedwait.c 1614 2004-11-05 02:16:17Z bird $ */
/** @file
 *
 * LIBC - sigtimedwait().
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#include <signal.h>
#include <assert.h>
#include <InnoTekLIBC/signals.h>
#include <InnoTekLIBC/thread.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_SIGNAL
#include <InnoTekLIBC/logstrict.h>



/**
 * Wait for a signal to become pending.
 *
 * @returns 0 on success.
 * @returns -1 on failure, errno set.
 * @param   pSigSet     Signals to wait for.
 * @param   pSigInfo    Where to store the signal info for the signal
 *                      that we accepted.
 */
int _STD(sigtimedwait)(const sigset_t *pSigSet, siginfo_t *pSigInfo, const struct timespec *pTimeout)
{
    LIBCLOG_ENTER("pSigSet=%p {%#08lx%#08lx} pSigInfo=%p pTimeout=%p {%d, %ld}\n",
                  (void *)pSigSet, pSigSet ? pSigSet->__bitmap[1] : 0, pSigSet ? pSigSet->__bitmap[0] : 0,
                  (void *)pSigInfo,
                  (void *)pTimeout, pTimeout ? pTimeout->tv_sec : ~0, pTimeout ? pTimeout->tv_nsec : ~0);
    __LIBC_PTHREAD  pThrd = __libc_threadCurrent();
    sigset_t        SigSet;
    siginfo_t       SigInfo = {0};
    struct timespec Timeout;
    int             rc;

    /*
     * Make copy of the input (we can safely crash here).
     */
    SigSet = *pSigSet;
    if (pTimeout)
        Timeout = *pTimeout;

    /*
     * Gain exclusive access to the signal stuff.
     */
    if (__libc_back_signalSemRequest())
        LIBCLOG_RETURN_INT(-1);

    /*
     * Mark thread as blocked in begin in sigtimedwait().
     */
    if (pThrd->enmStatus != enmLIBCThreadStatus_unknown)
    {
        LIBC_ASSERTM_FAILED("Must push thread state!\n");
        __libc_back_signalSemRelease();
        assert(0);
        LIBCLOG_RETURN_INT(-1);
    }
    pThrd->enmStatus            = enmLIBCThreadStatus_sigwait;
    pThrd->u.SigWait.SigSetWait = SigSet;
    pThrd->u.SigWait.pSigInfo   = &SigInfo;

    /*
     * Go to sleep.
     *
     * We have to do this call from within the sem+mustcomplete section.
     * The backend function will leave the section.
     */
    rc = __libc_back_signalWait(pTimeout ? &Timeout : NULL);

    /*
     * Store result.
     */
    if (pSigInfo)
        *pSigInfo = SigInfo;

    LIBCLOG_RETURN_MSG(rc, "ret %d (*pSigInfo={si_signo=%d, si_errno=%d, si_code=%#x, si_pid=%d, si_uid=%d, si_status=%d, si_addr=%p, si_value=%#08x, si_band=%ld})\n",
                       rc, SigInfo.si_signo, SigInfo.si_errno, SigInfo.si_code, SigInfo.si_pid, SigInfo.si_uid,
                       SigInfo.si_status, SigInfo.si_addr, SigInfo.si_value.sigval_int, SigInfo.si_band);
}

